/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.schema;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import net.morilib.db.fichier.FabriqueDeFichier;
import net.morilib.db.fichier.Fichier;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.HTMLTableHandler;
import net.morilib.db.misc.HTMLTermException;
import net.morilib.db.misc.NullBoolean;
import net.morilib.db.relations.DefaultRelationTuple;
import net.morilib.db.relations.NamedRelation;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.relations.SingleTableRelation;
import net.morilib.db.relations.TableRenameRelation;
import net.morilib.db.sqlcs.ddl.SqlColumnDefinition;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.parser.html.HTMLHandler;
import net.morilib.parser.html.HTMLParser;

public class HTMLSqlSchema implements SqlSchema {

	private String html;
	private Map<String, Relation> rels;

	public HTMLSqlSchema(String htmlfile) {
		html = htmlfile;
	}

	SqlCreateTable _getCreateTable(
			String name) throws IOException, SQLException {
		HTMLTableHandler h = new HTMLTableHandler(name);
		BufferedReader b = null;

		try {
			b = new BufferedReader(
					fabrique().newInstance(html).openReader());
			try {
				HTMLParser.parse(h, b);
			} catch(HTMLTermException e) {
				// do nothing
			}
			if(h.getValues().size() == 0)  return null;
			return SqlSchemata.guessTable(name, h.getValues());
		} finally {
			if(b != null)  b.close();
		}
	}

	@Override
	public NamedRelation readRelation(String name,
			String as) throws IOException, SQLException {
		HTMLTableHandler h = new HTMLTableHandler(name);
		List<SqlColumnDefinition> d;
		BufferedReader b = null;
		List<RelationTuple> r;
		Map<String, Object> m;
		List<List<String>> l;
		SqlCreateTable t;

		// with clause
		if(rels != null && rels.containsKey(name.toUpperCase())) {
			return new TableRenameRelation(
					rels.get(name.toUpperCase()),
					(as != null ? as : name).toUpperCase());
		}

		// from HTML
		try {
			b = new BufferedReader(
					fabrique().newInstance(html).openReader());
			try {
				HTMLParser.parse(h, b);
			} catch(HTMLTermException e) {
				// do nothing
			}

			if(h.getValues().size() == 0) {
				throw ErrorBundle.getDefault(10015, name);
			}
			l = h.getValues();
			t = SqlSchemata.guessTable(name, l);

			r = new ArrayList<RelationTuple>();
			m = new LinkedHashMap<String, Object>();
			d = t.getColumnDefinitions();
			for(int i = 1; i < l.size(); i++) {
				for(int j = 0; j < d.size(); j++) {
					m.put(d.get(j).getName(),
							d.get(j).getType().cast(l.get(i).get(j)));
				}
				r.add(new DefaultRelationTuple(m));
			}
			return new SingleTableRelation(t,
					as != null ? as : name,
					r);
		} finally {
			if(b != null)  b.close();
		}
	}

	@Override
	public void writeRelation(String name,
			Collection<RelationTuple> z
			) throws IOException, SQLException {
		char[] a = new char[1024];
		BufferedReader b = null;
		HTMLWriteTbls t;
		Writer w = null;
		Fichier f;
		int j;

		// write a temporary file
		try {
			b = new BufferedReader(
					fabrique().newInstance(html).openReader());
			f = fabrique().createTempFile("html", ".html.tmp");
			w = f.openWriter();
			t = new HTMLWriteTbls(w, getCreateTable(name), z);
			HTMLParser.parse(t, b);
		} finally {
			if(b != null)  b.close();
			if(w != null)  w.close();
		}

		// write a HTML file from the temporary file
		try {
			b = new BufferedReader(f.openReader());
			w = new BufferedWriter(
					fabrique().newInstance(html).openWriter());
			while((j = b.read(a)) >= 0)  w.write(a, 0, j);
		} finally {
			if(b != null)  b.close();
			if(w != null)  w.close();
		}
	}

	@Override
	public SqlCreateTable getCreateTable(
			String name) throws IOException, SQLException {
		SqlCreateTable t;

		if((t = _getCreateTable(name)) == null) {
			throw ErrorBundle.getDefault(10015, name);
		}
		return t;
	}

	@Override
	public Collection<String> getTableNames(
			) throws IOException, SQLException {
		HTMLTbls t = new HTMLTbls();
		BufferedReader b = null;

		try {
			b = new BufferedReader(
					fabrique().newInstance(html).openReader());
			HTMLParser.parse(t, b);
			return Collections.unmodifiableList(t.names);
		} finally {
			if(b != null)  b.close();
		}
	}

	@Override
	public boolean isTable(
			String name) throws IOException, SQLException {
		return _getCreateTable(name) != null;
	}

	@Override
	public void putCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public void alterCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public void truncateTable(
			String name) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public void removeCreateTable(
			String name) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#fork()
	 */
	@Override
	public SqlSchema fork() {
		return this;
	}

	@Override
	public NullBoolean isReadonly() {
		return NullBoolean.TRUE;
	}

	@Override
	public NullBoolean usesLocalFiles() {
		return NullBoolean.FALSE;
	}

	@Override
	public NullBoolean usesLocalFilePerTable() {
		return NullBoolean.FALSE;
	}

	@Override
	public FabriqueDeFichier fabrique() {
		return FabriqueDeFichier.getDefault();
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#bindSchema(java.lang.String, net.morilib.db.relations.Relation)
	 */
	@Override
	public void bindSchema(String name, Relation r) {
		rels.put(name, r);
	}

}

class HTMLTbls implements HTMLHandler {

	List<String> names = new ArrayList<String>();
	private int tblno = 0, nest = 0;
	private String tn;

	@Override
	public void string(String s) throws SQLException {
		// do nothing
	}

	@Override
	public void startTag(String s) throws SQLException {
		if(s.equalsIgnoreCase("TABLE")) {
			if(nest++ == 0) {
				tblno++;
				tn = null;
			}
		}
	}

	@Override
	public void endTag(String s) throws SQLException {
		if(s.equalsIgnoreCase("TABLE")) {
			if(--nest == 0 && tn == null) {
				names.add("TABLE" + tblno);
			}
		}
	}

	@Override
	public void tagAttribute(String k, String v) throws SQLException {
		if(nest != 1) {
			// do nothing
		} else if(k.equalsIgnoreCase("ID")) {
			names.add(tn = v.toUpperCase());
		} else if(k.equalsIgnoreCase("TABLE" + tblno)) {
			names.add(tn = v.toUpperCase());
		}
	}

}

class HTMLWriteTbls implements HTMLHandler {

	private static enum S {
		INIT, TABLE_P, TABLE, TR, TD, EXTRA
	}

	private PrintWriter wr;
	private Iterator<RelationTuple> itr;
	private SqlCreateTable table;
	private boolean tagw = false;
	private RelationTuple tuple;
	private List<String> extra;
	private int tblno, colno;
	private String name;
	private S stat;

	HTMLWriteTbls(Writer w, SqlCreateTable t,
			Iterable<RelationTuple> i) {
		wr = new PrintWriter(w);
		table = t;
		itr = i.iterator();
	}

	private void puttagw() {
		if(tagw) {
			wr.print('>');
			tagw = false;
		}
	}

	private void putdata() throws SQLException {
		SqlColumnDefinition d;

		d = table.getColumnDefinitions().get(colno);
		wr.print(d.getType().string(tuple.get(d.getName())));
	}

	@Override
	public void string(String s) throws SQLException {
		puttagw();
		if(stat != S.TD) {
			wr.print(s);
		}
	}

	@Override
	public void startTag(String s) throws SQLException {
		puttagw();
		wr.format("<%s", s);  tagw = true;
		switch(stat) {
		case INIT:
			if(s.equalsIgnoreCase("TABLE")) {
				stat = S.TABLE_P;
				tblno++;
			}
			break;
		case TABLE_P:
			if(name.equalsIgnoreCase("TABLE" + tblno)) {
				stat = S.TABLE;
				// go next
			} else {
				break;
			}
		case TABLE:
			if(s.equalsIgnoreCase("TR")) {
				colno = 0;
				tuple = itr.next();
				stat = S.TR;
			}
			break;
		case TR:
			if(s.equalsIgnoreCase("TD") ||
					s.equalsIgnoreCase("TH")) {
				// start TD
				extra = new ArrayList<String>();
				extra.add(s);
				colno++;
				putdata();
				stat = S.TD;
			}
			break;
		case TD:
			if(s.equalsIgnoreCase(extra.get(0))) {
				// end TD and start TD
				colno++;
				putdata();
			} else if(s.equalsIgnoreCase("TR")) {
				colno = 0;
				tuple = itr.next();
				stat = S.TR;
			} else {
				extra.add(s);
				stat = S.EXTRA;
			}
			break;
		case EXTRA:
			extra.add(s);
			break;
		}
	}

	@Override
	public void endTag(String s) {
		puttagw();
		wr.format("</%s>", s);
		switch(stat) {
		case INIT:
			break;
		case TABLE_P:
			stat = S.INIT;
			break;
		case TABLE:
			stat = S.INIT;
			break;
		case TR:
			if(s.equalsIgnoreCase("TR")) {
				stat = S.TABLE;
			}
			break;
		case TD:
			if(s.equalsIgnoreCase(extra.get(0))) {
				stat = S.TR;
			} else if(s.equalsIgnoreCase("TR")) {
				stat = S.TABLE;
			}
			break;
		case EXTRA:
			while(extra.size() > 0 &&
					!extra.get(extra.size() - 1).equalsIgnoreCase(s)) {
				extra.remove(extra.size() - 1);
			}

			if(extra.size() > 1) {
				// do nothing
			} else if(extra.size() == 1) {
				stat = S.TR;
			} else if(s.equalsIgnoreCase("TR")) {
				stat = S.TABLE;
			} else if(s.equalsIgnoreCase("TABLE")) {
				throw new HTMLTermException();
			}
			break;
		}
	}

	@Override
	public void tagAttribute(String k, String v) {
		wr.format(" %s='%s'", k, v);
		switch(stat) {
		case INIT:  break;
		case TABLE_P:
			if(!k.equalsIgnoreCase("ID")) {
				// do nothing
			} else if(v.equalsIgnoreCase(name) ||
					name.equalsIgnoreCase("TABLE" + tblno)) {
				stat = S.TABLE;
			}
			break;
		case TABLE:  break;
		case TR:  break;
		case TD:  break;
		case EXTRA:  break;
		}
	}

}
