/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.sql;

import java.io.IOException;
import java.io.StringReader;
import java.sql.SQLException;

import junit.framework.TestCase;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/07/28
 */
public class DbSqlExprParserTest extends TestCase {

	void okp(String s) {
		DbSqlLexer l;

		try {
			l = new DbSqlLexer(new StringReader(s));
			new DbSqlExprParser().parse(l, null);
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	void eqp(String s, String t) {
		DbSqlLexer l;

		try {
			l = new DbSqlLexer(new StringReader(s));
			assertEquals(t, new DbSqlExprParser().parse(l, null).toString());
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	public void testExprParser001() {
		okp("1");
		okp("'1'");
	}

	public void testExprParser002() {
		okp("1 + 2");
		okp("1 - 2");
		okp("1 * 2");
		okp("1 / 2");
		okp("'1' || '2'");
		okp("1 and 2");
		okp("1 or 2");
		okp("1 = 2");  okp("1 != 2");  okp("1 <> 2");
		okp("1 < 2");  okp("1 <= 2");
		okp("1 > 2");  okp("1 >= 2");
		okp("'aaa' like 'a%'");
		okp("1 + 2 * 3");
	}

	public void testExprParser003() {
		okp("-1");
		okp("+1");
		okp("not 1");
	}

	public void testExprParser004() {
		okp("(1)");
		okp("(1 + 2) * 3");
	}

	public void testExprParser005() {
		okp("'aa' in ( 'aa' )");
		okp("'aa' in ( 'aa', 'bb' )");
		okp("'aa' in ( 'aa' || 'bb' )");
		okp("'aa' in ( 'aa' || 'bb', 'bb' || 'bb' )");
	}

	public void testExprParser006() {
		okp("count(*)");
		okp("rand()");
		okp("aaaa(*, 1, 2 + 3)");
	}

	public void testExprParser007() {
		okp("A.AAA between 1 and 3");
		okp("A.AAA between 1 + 2 and 3 + 4");
	}

	public void testExprParser008() {
		okp("A.AA IS NULL");
		okp("A.AA IS NOT NULL");
	}

	public void testExprParser009() {
		okp("'aa' in ( 'aa' ) AND A.AA IS NULL");
	}

	public void testExprParser101() {
		eqp("1", "1");
		eqp("'aaa'", "\"aaa\"");
		eqp("aaa", "AAA");
	}

	public void testExprParser102() {
		eqp("1 + 2", "(+ 1 2)");
		eqp("1 - 2", "(- 1 2)");
		eqp("1 * 2", "(* 1 2)");
		eqp("1 / 2", "(/ 1 2)");
		eqp("'1' || '2'", "(|| \"1\" \"2\")");
		eqp("1 and 2", "(and 1 2)");
		eqp("1 or 2", "(or 1 2)");
		eqp("1 = 2", "(= 1 2)");
		eqp("1 != 2", "(<> 1 2)");  eqp("1 <> 2", "(<> 1 2)");
		eqp("1 < 2", "(< 1 2)");  eqp("1 <= 2", "(<= 1 2)");
		eqp("1 > 2", "(> 1 2)");  eqp("1 >= 2", "(>= 1 2)");
		eqp("'aaa' like 'a%'", "(like \"aaa\" \"a%\")");
		eqp("1 + 2 * 3", "(+ 1 (* 2 3))");
	}

	public void testExprParser103() {
		eqp("-1", "(- 1)");
		eqp("+1", "(+ 1)");
		eqp("not 1", "(not 1)");
	}

	public void testExprParser104() {
		eqp("(1)", "1");
		eqp("(1 + 2) * 3", "(* (+ 1 2) 3)");
	}

	public void testExprParser105() {
		eqp("'aa' in ( 'aa' )", "(in \"aa\" \"aa\")");
		eqp("'aa' in ( 'aa', 'bb' )", "(in \"aa\" \"aa\" \"bb\")");
		eqp("'aa' in ( 'aa' || 'bb' )", "(in \"aa\" (|| \"aa\" \"bb\"))");
		eqp("'aa' in ( 'aa' || 'bb', 'bb' || 'bb' )",
				"(in \"aa\" (|| \"aa\" \"bb\") (|| \"bb\" \"bb\"))");
	}

	public void testExprParser106() {
		eqp("count(*)", "(COUNT *)");
		eqp("rand()", "(RAND)");
		eqp("aaaa(*, 1, 2 + 3)", "(AAAA * 1 (+ 2 3))");
	}

	public void testExprParser107() {
		eqp("A.AAA between 1 and 3", "(between A.AAA 1 3)");
		eqp("A.AAA between 1 + 2 and 3 + 4", "(between A.AAA (+ 1 2) (+ 3 4))");
	}

	public void testExprParser108() {
		eqp("A.AA IS NULL", "(is-null A.AA)");
		eqp("A.AA IS NOT NULL", "(is-not-null A.AA)");
	}

	public void testExprParser109() {
		eqp("'aa' in ( 'aa' ) AND A.AA IS NULL", "(and (in \"aa\" \"aa\") (is-null A.AA))");
	}

	public void testExprParser111() {
		eqp("+1 * 2",                "(* (+ 1) 2)");
		eqp("-1 * 2",                "(* (- 1) 2)");
		eqp("1 * 2 * 3",             "(* (* 1 2) 3)");
		eqp("1 * 2 / 3",             "(/ (* 1 2) 3)");
		eqp("1 * 2 + 3",             "(+ (* 1 2) 3)");
		eqp("1 * 2 - 3",             "(- (* 1 2) 3)");
		eqp("1 * 2 || 3",            "(|| (* 1 2) 3)");
		eqp("1 * 2 = 3",             "(= (* 1 2) 3)");
		eqp("1 * 2 <> 3",            "(<> (* 1 2) 3)");
		eqp("1 * 2 < 3",             "(< (* 1 2) 3)");
		eqp("1 * 2 <= 3",            "(<= (* 1 2) 3)");
		eqp("1 * 2 > 3",             "(> (* 1 2) 3)");
		eqp("1 * 2 >= 3",            "(>= (* 1 2) 3)");
		eqp("1 * 2 LIKE 3",          "(like (* 1 2) 3)");
		eqp("1 * 2 IS NULL",         "(is-null (* 1 2))");
		eqp("1 * 2 IS NOT NULL",     "(is-not-null (* 1 2))");
		eqp("1 * 2 BETWEEN 3 AND 4", "(between (* 1 2) 3 4)");
		eqp("not 1 * 2",             "(not (* 1 2))");
		eqp("1 * 2 AND 3",           "(and (* 1 2) 3)");
		eqp("1 * 2 OR 3",            "(or (* 1 2) 3)");

		eqp("1 * +2",                "(* 1 (+ 2))");
		eqp("1 * -2",                "(* 1 (- 2))");
		eqp("1 *    2 * 3",          "(* (* 1 2) 3)");
		eqp("1 /    2 * 3",          "(* (/ 1 2) 3)");
		eqp("1 +    2 * 3",          "(+ 1 (* 2 3))");
		eqp("1 -    2 * 3",          "(- 1 (* 2 3))");
		eqp("1 ||   2 * 3",          "(|| 1 (* 2 3))");
		eqp("1 =    2 * 3",          "(= 1 (* 2 3))");
		eqp("1 <>   2 * 3",          "(<> 1 (* 2 3))");
		eqp("1 <    2 * 3",          "(< 1 (* 2 3))");
		eqp("1 <=   2 * 3",          "(<= 1 (* 2 3))");
		eqp("1 >    2 * 3",          "(> 1 (* 2 3))");
		eqp("1 >=   2 * 3",          "(>= 1 (* 2 3))");
		eqp("1 LIKE 2 * 3",          "(like 1 (* 2 3))");
		eqp("1 IS NULL * 2",         "(* (is-null 1) 2)");
		eqp("1 IS NOT NULL * 2",     "(* (is-not-null 1) 2)");
		eqp("1 BETWEEN 3 AND 4 * 2", "(between 1 3 (* 4 2))");
		eqp("1 * not 2",             "(* 1 (not 2))");
		eqp("1 AND  2 * 3",          "(and 1 (* 2 3))");
		eqp("1 OR   2 * 3",          "(or 1 (* 2 3))");
	}

	public void testExprParser112() {
		eqp("+1 / 2",                "(/ (+ 1) 2)");
		eqp("-1 / 2",                "(/ (- 1) 2)");
		eqp("1 / 2 * 3",             "(* (/ 1 2) 3)");
		eqp("1 / 2 / 3",             "(/ (/ 1 2) 3)");
		eqp("1 / 2 + 3",             "(+ (/ 1 2) 3)");
		eqp("1 / 2 - 3",             "(- (/ 1 2) 3)");
		eqp("1 / 2 || 3",            "(|| (/ 1 2) 3)");
		eqp("1 / 2 = 3",             "(= (/ 1 2) 3)");
		eqp("1 / 2 <> 3",            "(<> (/ 1 2) 3)");
		eqp("1 / 2 < 3",             "(< (/ 1 2) 3)");
		eqp("1 / 2 <= 3",            "(<= (/ 1 2) 3)");
		eqp("1 / 2 > 3",             "(> (/ 1 2) 3)");
		eqp("1 / 2 >= 3",            "(>= (/ 1 2) 3)");
		eqp("1 / 2 LIKE 3",          "(like (/ 1 2) 3)");
		eqp("1 / 2 IS NULL",         "(is-null (/ 1 2))");
		eqp("1 / 2 IS NOT NULL",     "(is-not-null (/ 1 2))");
		eqp("1 / 2 BETWEEN 3 AND 4", "(between (/ 1 2) 3 4)");
		eqp("not 1 / 2",             "(not (/ 1 2))");
		eqp("1 / 2 AND 3",           "(and (/ 1 2) 3)");
		eqp("1 / 2 OR 3",            "(or (/ 1 2) 3)");

		eqp("1 / +2",                "(/ 1 (+ 2))");
		eqp("1 / -2",                "(/ 1 (- 2))");
		eqp("1 *    2 / 3",          "(/ (* 1 2) 3)");
		eqp("1 /    2 / 3",          "(/ (/ 1 2) 3)");
		eqp("1 +    2 / 3",          "(+ 1 (/ 2 3))");
		eqp("1 -    2 / 3",          "(- 1 (/ 2 3))");
		eqp("1 ||   2 / 3",          "(|| 1 (/ 2 3))");
		eqp("1 =    2 / 3",          "(= 1 (/ 2 3))");
		eqp("1 <>   2 / 3",          "(<> 1 (/ 2 3))");
		eqp("1 <    2 / 3",          "(< 1 (/ 2 3))");
		eqp("1 <=   2 / 3",          "(<= 1 (/ 2 3))");
		eqp("1 >    2 / 3",          "(> 1 (/ 2 3))");
		eqp("1 >=   2 / 3",          "(>= 1 (/ 2 3))");
		eqp("1 IS NULL / 2",         "(/ (is-null 1) 2)");
		eqp("1 IS NOT NULL / 2",     "(/ (is-not-null 1) 2)");
		eqp("1 LIKE 2 / 3",          "(like 1 (/ 2 3))");
		eqp("1 BETWEEN 3 AND 4 / 2", "(between 1 3 (/ 4 2))");
		eqp("1 / not 2",             "(/ 1 (not 2))");
		eqp("1 AND  2 / 3",          "(and 1 (/ 2 3))");
		eqp("1 OR   2 / 3",          "(or 1 (/ 2 3))");
	}

	public void testExprParser113() {
		eqp("+1 + 2",                "(+ (+ 1) 2)");
		eqp("-1 + 2",                "(+ (- 1) 2)");
		eqp("1 + 2 * 3",             "(+ 1 (* 2 3))");
		eqp("1 + 2 / 3",             "(+ 1 (/ 2 3))");
		eqp("1 + 2 + 3",             "(+ (+ 1 2) 3)");
		eqp("1 + 2 - 3",             "(- (+ 1 2) 3)");
		eqp("1 + 2 || 3",            "(|| (+ 1 2) 3)");
		eqp("1 + 2 = 3",             "(= (+ 1 2) 3)");
		eqp("1 + 2 <> 3",            "(<> (+ 1 2) 3)");
		eqp("1 + 2 < 3",             "(< (+ 1 2) 3)");
		eqp("1 + 2 <= 3",            "(<= (+ 1 2) 3)");
		eqp("1 + 2 > 3",             "(> (+ 1 2) 3)");
		eqp("1 + 2 >= 3",            "(>= (+ 1 2) 3)");
		eqp("1 + 2 LIKE 3",          "(like (+ 1 2) 3)");
		eqp("1 + 2 IS NULL",         "(is-null (+ 1 2))");
		eqp("1 + 2 IS NOT NULL",     "(is-not-null (+ 1 2))");
		eqp("1 + 2 BETWEEN 3 AND 4", "(between (+ 1 2) 3 4)");
		eqp("not 1 + 2",             "(not (+ 1 2))");
		eqp("1 + 2 AND 3",           "(and (+ 1 2) 3)");
		eqp("1 + 2 OR 3",            "(or (+ 1 2) 3)");

		eqp("1 + +2",                "(+ 1 (+ 2))");
		eqp("1 + -2",                "(+ 1 (- 2))");
		eqp("1 *    2 + 3",          "(+ (* 1 2) 3)");
		eqp("1 /    2 + 3",          "(+ (/ 1 2) 3)");
		eqp("1 +    2 + 3",          "(+ (+ 1 2) 3)");
		eqp("1 -    2 + 3",          "(+ (- 1 2) 3)");
		eqp("1 ||   2 + 3",          "(|| 1 (+ 2 3))");
		eqp("1 =    2 + 3",          "(= 1 (+ 2 3))");
		eqp("1 <>   2 + 3",          "(<> 1 (+ 2 3))");
		eqp("1 <    2 + 3",          "(< 1 (+ 2 3))");
		eqp("1 <=   2 + 3",          "(<= 1 (+ 2 3))");
		eqp("1 >    2 + 3",          "(> 1 (+ 2 3))");
		eqp("1 >=   2 + 3",          "(>= 1 (+ 2 3))");
		eqp("1 LIKE 2 + 3",          "(like 1 (+ 2 3))");
		eqp("1 IS NULL + 2",         "(+ (is-null 1) 2)");
		eqp("1 IS NOT NULL + 2",     "(+ (is-not-null 1) 2)");
		eqp("1 BETWEEN 3 AND 4 + 2", "(between 1 3 (+ 4 2))");
		eqp("1 + not 2",             "(+ 1 (not 2))");
		eqp("1 AND  2 + 3",          "(and 1 (+ 2 3))");
		eqp("1 OR   2 + 3",          "(or 1 (+ 2 3))");
	}

	public void testExprParser114() {
		eqp("+1 - 2",                "(- (+ 1) 2)");
		eqp("-1 - 2",                "(- (- 1) 2)");
		eqp("1 - 2 * 3",             "(- 1 (* 2 3))");
		eqp("1 - 2 / 3",             "(- 1 (/ 2 3))");
		eqp("1 - 2 + 3",             "(+ (- 1 2) 3)");
		eqp("1 - 2 - 3",             "(- (- 1 2) 3)");
		eqp("1 - 2 || 3",            "(|| (- 1 2) 3)");
		eqp("1 - 2 = 3",             "(= (- 1 2) 3)");
		eqp("1 - 2 <> 3",            "(<> (- 1 2) 3)");
		eqp("1 - 2 < 3",             "(< (- 1 2) 3)");
		eqp("1 - 2 <= 3",            "(<= (- 1 2) 3)");
		eqp("1 - 2 > 3",             "(> (- 1 2) 3)");
		eqp("1 - 2 >= 3",            "(>= (- 1 2) 3)");
		eqp("1 - 2 LIKE 3",          "(like (- 1 2) 3)");
		eqp("1 - 2 IS NULL",         "(is-null (- 1 2))");
		eqp("1 - 2 IS NOT NULL",     "(is-not-null (- 1 2))");
		eqp("1 - 2 BETWEEN 3 AND 4", "(between (- 1 2) 3 4)");
		eqp("not 1 - 2",             "(not (- 1 2))");
		eqp("1 - 2 AND 3",           "(and (- 1 2) 3)");
		eqp("1 - 2 OR 3",            "(or (- 1 2) 3)");

		eqp("1 - +2",                "(- 1 (+ 2))");
		eqp("1 - -2",                "(- 1 (- 2))");
		eqp("1 *    2 - 3",          "(- (* 1 2) 3)");
		eqp("1 /    2 - 3",          "(- (/ 1 2) 3)");
		eqp("1 +    2 - 3",          "(- (+ 1 2) 3)");
		eqp("1 -    2 - 3",          "(- (- 1 2) 3)");
		eqp("1 ||   2 - 3",          "(|| 1 (- 2 3))");
		eqp("1 =    2 - 3",          "(= 1 (- 2 3))");
		eqp("1 <>   2 - 3",          "(<> 1 (- 2 3))");
		eqp("1 <    2 - 3",          "(< 1 (- 2 3))");
		eqp("1 <=   2 - 3",          "(<= 1 (- 2 3))");
		eqp("1 >    2 - 3",          "(> 1 (- 2 3))");
		eqp("1 >=   2 - 3",          "(>= 1 (- 2 3))");
		eqp("1 LIKE 2 - 3",          "(like 1 (- 2 3))");
		eqp("1 IS NULL - 2",         "(- (is-null 1) 2)");
		eqp("1 IS NOT NULL - 2",     "(- (is-not-null 1) 2)");
		eqp("1 BETWEEN 3 AND 4 - 2", "(between 1 3 (- 4 2))");
		eqp("1 - not 2",             "(- 1 (not 2))");
		eqp("1 AND  2 - 3",          "(and 1 (- 2 3))");
		eqp("1 OR   2 - 3",          "(or 1 (- 2 3))");
	}

	public void testExprParser115() {
		eqp("+1 || 2",                "(|| (+ 1) 2)");
		eqp("-1 || 2",                "(|| (- 1) 2)");
		eqp("1 || 2 * 3",             "(|| 1 (* 2 3))");
		eqp("1 || 2 / 3",             "(|| 1 (/ 2 3))");
		eqp("1 || 2 + 3",             "(|| 1 (+ 2 3))");
		eqp("1 || 2 - 3",             "(|| 1 (- 2 3))");
		eqp("1 || 2 || 3",            "(|| (|| 1 2) 3)");
		eqp("1 || 2 = 3",             "(= (|| 1 2) 3)");
		eqp("1 || 2 <> 3",            "(<> (|| 1 2) 3)");
		eqp("1 || 2 < 3",             "(< (|| 1 2) 3)");
		eqp("1 || 2 <= 3",            "(<= (|| 1 2) 3)");
		eqp("1 || 2 > 3",             "(> (|| 1 2) 3)");
		eqp("1 || 2 >= 3",            "(>= (|| 1 2) 3)");
		eqp("1 || 2 LIKE 3",          "(like (|| 1 2) 3)");
		eqp("1 || 2 IS NULL",         "(is-null (|| 1 2))");
		eqp("1 || 2 IS NOT NULL",     "(is-not-null (|| 1 2))");
		eqp("1 || 2 BETWEEN 3 AND 4", "(between (|| 1 2) 3 4)");
		eqp("not 1 || 2",             "(not (|| 1 2))");
		eqp("1 || 2 AND 3",           "(and (|| 1 2) 3)");
		eqp("1 || 2 OR 3",            "(or (|| 1 2) 3)");

		eqp("1 || +2",                "(|| 1 (+ 2))");
		eqp("1 || -2",                "(|| 1 (- 2))");
		eqp("1 *    2 || 3",          "(|| (* 1 2) 3)");
		eqp("1 /    2 || 3",          "(|| (/ 1 2) 3)");
		eqp("1 +    2 || 3",          "(|| (+ 1 2) 3)");
		eqp("1 -    2 || 3",          "(|| (- 1 2) 3)");
		eqp("1 ||   2 || 3",          "(|| (|| 1 2) 3)");
		eqp("1 =    2 || 3",          "(= 1 (|| 2 3))");
		eqp("1 <>   2 || 3",          "(<> 1 (|| 2 3))");
		eqp("1 <    2 || 3",          "(< 1 (|| 2 3))");
		eqp("1 <=   2 || 3",          "(<= 1 (|| 2 3))");
		eqp("1 >    2 || 3",          "(> 1 (|| 2 3))");
		eqp("1 >=   2 || 3",          "(>= 1 (|| 2 3))");
		eqp("1 LIKE 2 || 3",          "(like 1 (|| 2 3))");
		eqp("1 IS NULL || 2",         "(|| (is-null 1) 2)");
		eqp("1 IS NOT NULL || 2",     "(|| (is-not-null 1) 2)");
		eqp("1 BETWEEN 3 AND 4 || 2", "(between 1 3 (|| 4 2))");
		eqp("1 || not 2",             "(|| 1 (not 2))");
		eqp("1 AND  2 || 3",          "(and 1 (|| 2 3))");
		eqp("1 OR   2 || 3",          "(or 1 (|| 2 3))");
	}

	public void testExprParser116() {
		eqp("+1 <> 2",                "(<> (+ 1) 2)");
		eqp("-1 <> 2",                "(<> (- 1) 2)");
		eqp("1 <> 2 * 3",             "(<> 1 (* 2 3))");
		eqp("1 <> 2 / 3",             "(<> 1 (/ 2 3))");
		eqp("1 <> 2 + 3",             "(<> 1 (+ 2 3))");
		eqp("1 <> 2 - 3",             "(<> 1 (- 2 3))");
		eqp("1 <> 2 || 3",            "(<> 1 (|| 2 3))");
		eqp("1 <> 2 = 3",             "(= (<> 1 2) 3)");
		eqp("1 <> 2 <> 3",            "(<> (<> 1 2) 3)");
		eqp("1 <> 2 < 3",             "(< (<> 1 2) 3)");
		eqp("1 <> 2 <= 3",            "(<= (<> 1 2) 3)");
		eqp("1 <> 2 > 3",             "(> (<> 1 2) 3)");
		eqp("1 <> 2 >= 3",            "(>= (<> 1 2) 3)");
		eqp("1 <> 2 LIKE 3",          "(like (<> 1 2) 3)");
		eqp("1 <> 2 IS NULL",         "(is-null (<> 1 2))");
		eqp("1 <> 2 IS NOT NULL",     "(is-not-null (<> 1 2))");
		eqp("1 <> 2 BETWEEN 3 AND 4", "(between (<> 1 2) 3 4)");
		eqp("not 1 <> 2",             "(not (<> 1 2))");
		eqp("1 <> 2 AND 3",           "(and (<> 1 2) 3)");
		eqp("1 <> 2 OR 3",            "(or (<> 1 2) 3)");

		eqp("1 <> +2",                "(<> 1 (+ 2))");
		eqp("1 <> -2",                "(<> 1 (- 2))");
		eqp("1 *    2 <> 3",          "(<> (* 1 2) 3)");
		eqp("1 /    2 <> 3",          "(<> (/ 1 2) 3)");
		eqp("1 +    2 <> 3",          "(<> (+ 1 2) 3)");
		eqp("1 -    2 <> 3",          "(<> (- 1 2) 3)");
		eqp("1 ||   2 <> 3",          "(<> (|| 1 2) 3)");
		eqp("1 =    2 <> 3",          "(<> (= 1 2) 3)");
		eqp("1 <>   2 <> 3",          "(<> (<> 1 2) 3)");
		eqp("1 <    2 <> 3",          "(<> (< 1 2) 3)");
		eqp("1 <=   2 <> 3",          "(<> (<= 1 2) 3)");
		eqp("1 >    2 <> 3",          "(<> (> 1 2) 3)");
		eqp("1 >=   2 <> 3",          "(<> (>= 1 2) 3)");
		eqp("1 LIKE 2 <> 3",          "(<> (like 1 2) 3)");
		eqp("1 IS NULL <> 2",         "(<> (is-null 1) 2)");
		eqp("1 IS NOT NULL <> 2",     "(<> (is-not-null 1) 2)");
		eqp("1 BETWEEN 3 AND 4 <> 2", "(between 1 3 (<> 4 2))");
		eqp("1 <> not 2",             "(<> 1 (not 2))");
		eqp("1 AND  2 <> 3",          "(and 1 (<> 2 3))");
		eqp("1 OR   2 <> 3",          "(or 1 (<> 2 3))");
	}

	public void testExprParser117() {
		eqp("+1 BETWEEN 2 and 3",     "(between (+ 1) 2 3)");
		eqp("-1 BETWEEN 2 and 3",     "(between (- 1) 2 3)");
		eqp("1 BETWEEN 4 and 2 * 3",  "(between 1 4 (* 2 3))");
		eqp("1 BETWEEN 4 and 2 / 3",             "(between 1 4 (/ 2 3))");
		eqp("1 BETWEEN 4 and 2 + 3",             "(between 1 4 (+ 2 3))");
		eqp("1 BETWEEN 4 and 2 - 3",             "(between 1 4 (- 2 3))");
		eqp("1 BETWEEN 4 and 2 || 3",            "(between 1 4 (|| 2 3))");
		eqp("1 BETWEEN 4 and 2 = 3",             "(between 1 4 (= 2 3))");
		eqp("1 BETWEEN 4 and 2 <> 3",            "(between 1 4 (<> 2 3))");
		eqp("1 BETWEEN 4 and 2 < 3",             "(between 1 4 (< 2 3))");
		eqp("1 BETWEEN 4 and 2 <= 3",            "(between 1 4 (<= 2 3))");
		eqp("1 BETWEEN 4 and 2 > 3",             "(between 1 4 (> 2 3))");
		eqp("1 BETWEEN 4 and 2 >= 3",            "(between 1 4 (>= 2 3))");
		eqp("1 BETWEEN 4 and 2 LIKE 3",          "(between 1 4 (like 2 3))");
		eqp("1 BETWEEN 4 and 2 IS NULL",         "(between 1 4 (is-null 2))");
		eqp("1 BETWEEN 4 and 2 IS NOT NULL",     "(between 1 4 (is-not-null 2))");
		eqp("1 BETWEEN 4 and 2 BETWEEN 3 AND 4", "(between 1 4 (between 2 3 4))");
		eqp("not 1 BETWEEN 4 and 2",             "(not (between 1 4 2))");
		eqp("1 BETWEEN 4 and 2 AND 3",           "(and (between 1 4 2) 3)");
		eqp("1 BETWEEN 4 and 2 OR 3",            "(or (between 1 4 2) 3)");

		eqp("1 BETWEEN 4 and +2",                "(between 1 4 (+ 2))");
		eqp("1 BETWEEN 4 and -2",                "(between 1 4 (- 2))");
		eqp("1 *    2 BETWEEN 4 and 3",          "(between (* 1 2) 4 3)");
		eqp("1 /    2 BETWEEN 4 and 3",          "(between (/ 1 2) 4 3)");
		eqp("1 +    2 BETWEEN 4 and 3",          "(between (+ 1 2) 4 3)");
		eqp("1 -    2 BETWEEN 4 and 3",          "(between (- 1 2) 4 3)");
		eqp("1 ||   2 BETWEEN 4 and 3",          "(between (|| 1 2) 4 3)");
		eqp("1 =    2 BETWEEN 4 and 3",          "(between (= 1 2) 4 3)");
		eqp("1 <>   2 BETWEEN 4 and 3",          "(between (<> 1 2) 4 3)");
		eqp("1 <    2 BETWEEN 4 and 3",          "(between (< 1 2) 4 3)");
		eqp("1 <=   2 BETWEEN 4 and 3",          "(between (<= 1 2) 4 3)");
		eqp("1 >    2 BETWEEN 4 and 3",          "(between (> 1 2) 4 3)");
		eqp("1 >=   2 BETWEEN 4 and 3",          "(between (>= 1 2) 4 3)");
		eqp("1 LIKE 2 BETWEEN 4 and 3",          "(between (like 1 2) 4 3)");
		eqp("1 IS NULL BETWEEN 4 and 2",         "(between (is-null 1) 4 2)");
		eqp("1 IS NOT NULL BETWEEN 4 and 2",     "(between (is-not-null 1) 4 2)");
		eqp("1 BETWEEN 3 AND 4 BETWEEN 4 and 2", "(between 1 3 (between 4 4 2))");
		eqp("1 BETWEEN 4 and not 2",             "(between 1 4 (not 2))");
		eqp("1 AND  2 BETWEEN 4 and 3",          "(and 1 (between 2 4 3))");
		eqp("1 OR   2 BETWEEN 4 and 3",          "(or 1 (between 2 4 3))");
	}

	public void testExprParser118() {
		eqp("+1 AND 2",                "(and (+ 1) 2)");
		eqp("-1 AND 2",                "(and (- 1) 2)");
		eqp("1 AND 2 * 3",             "(and 1 (* 2 3))");
		eqp("1 AND 2 / 3",             "(and 1 (/ 2 3))");
		eqp("1 AND 2 + 3",             "(and 1 (+ 2 3))");
		eqp("1 AND 2 - 3",             "(and 1 (- 2 3))");
		eqp("1 AND 2 || 3",            "(and 1 (|| 2 3))");
		eqp("1 AND 2 = 3",             "(and 1 (= 2 3))");
		eqp("1 AND 2 <> 3",            "(and 1 (<> 2 3))");
		eqp("1 AND 2 < 3",             "(and 1 (< 2 3))");
		eqp("1 AND 2 <= 3",            "(and 1 (<= 2 3))");
		eqp("1 AND 2 > 3",             "(and 1 (> 2 3))");
		eqp("1 AND 2 >= 3",            "(and 1 (>= 2 3))");
		eqp("1 AND 2 LIKE 3",          "(and 1 (like 2 3))");
		eqp("1 AND 2 IS NULL",         "(and 1 (is-null 2))");
		eqp("1 AND 2 IS NOT NULL",     "(and 1 (is-not-null 2))");
		eqp("1 AND 2 BETWEEN 3 AND 4", "(and 1 (between 2 3 4))");
		eqp("not 1 AND 2",             "(and (not 1) 2)");
		eqp("1 AND 2 AND 3",           "(and (and 1 2) 3)");
		eqp("1 AND 2 OR 3",            "(or (and 1 2) 3)");

		eqp("1 AND +2",                "(and 1 (+ 2))");
		eqp("1 AND -2",                "(and 1 (- 2))");
		eqp("1 *    2 AND 3",          "(and (* 1 2) 3)");
		eqp("1 /    2 AND 3",          "(and (/ 1 2) 3)");
		eqp("1 +    2 AND 3",          "(and (+ 1 2) 3)");
		eqp("1 -    2 AND 3",          "(and (- 1 2) 3)");
		eqp("1 ||   2 AND 3",          "(and (|| 1 2) 3)");
		eqp("1 =    2 AND 3",          "(and (= 1 2) 3)");
		eqp("1 <>   2 AND 3",          "(and (<> 1 2) 3)");
		eqp("1 <    2 AND 3",          "(and (< 1 2) 3)");
		eqp("1 <=   2 AND 3",          "(and (<= 1 2) 3)");
		eqp("1 >    2 AND 3",          "(and (> 1 2) 3)");
		eqp("1 >=   2 AND 3",          "(and (>= 1 2) 3)");
		eqp("1 LIKE 2 AND 3",          "(and (like 1 2) 3)");
		eqp("1 IS NULL AND 2",         "(and (is-null 1) 2)");
		eqp("1 IS NOT NULL AND 2",     "(and (is-not-null 1) 2)");
		eqp("1 BETWEEN 3 AND 4 AND 2", "(and (between 1 3 4) 2)");
		eqp("1 AND not 2",             "(and 1 (not 2))");
		eqp("1 AND  2 AND 3",          "(and (and 1 2) 3)");
		eqp("1 OR   2 AND 3",          "(or 1 (and 2 3))");
	}

	public void testExprParser119() {
		eqp("+1 OR 2",                "(or (+ 1) 2)");
		eqp("-1 OR 2",                "(or (- 1) 2)");
		eqp("1 OR 2 * 3",             "(or 1 (* 2 3))");
		eqp("1 OR 2 / 3",             "(or 1 (/ 2 3))");
		eqp("1 OR 2 + 3",             "(or 1 (+ 2 3))");
		eqp("1 OR 2 - 3",             "(or 1 (- 2 3))");
		eqp("1 OR 2 || 3",            "(or 1 (|| 2 3))");
		eqp("1 OR 2 = 3",             "(or 1 (= 2 3))");
		eqp("1 OR 2 <> 3",            "(or 1 (<> 2 3))");
		eqp("1 OR 2 < 3",             "(or 1 (< 2 3))");
		eqp("1 OR 2 <= 3",            "(or 1 (<= 2 3))");
		eqp("1 OR 2 > 3",             "(or 1 (> 2 3))");
		eqp("1 OR 2 >= 3",            "(or 1 (>= 2 3))");
		eqp("1 OR 2 LIKE 3",          "(or 1 (like 2 3))");
		eqp("1 OR 2 IS NULL",         "(or 1 (is-null 2))");
		eqp("1 OR 2 IS NOT NULL",     "(or 1 (is-not-null 2))");
		eqp("1 OR 2 BETWEEN 3 AND 4", "(or 1 (between 2 3 4))");
		eqp("not 1 OR 2",             "(or (not 1) 2)");
		eqp("1 OR 2 AND 3",           "(or 1 (and 2 3))");
		eqp("1 OR 2 OR 3",            "(or (or 1 2) 3)");

		eqp("1 OR +2",                "(or 1 (+ 2))");
		eqp("1 OR -2",                "(or 1 (- 2))");
		eqp("1 *    2 OR 3",          "(or (* 1 2) 3)");
		eqp("1 /    2 OR 3",          "(or (/ 1 2) 3)");
		eqp("1 +    2 OR 3",          "(or (+ 1 2) 3)");
		eqp("1 -    2 OR 3",          "(or (- 1 2) 3)");
		eqp("1 ||   2 OR 3",          "(or (|| 1 2) 3)");
		eqp("1 =    2 OR 3",          "(or (= 1 2) 3)");
		eqp("1 <>   2 OR 3",          "(or (<> 1 2) 3)");
		eqp("1 <    2 OR 3",          "(or (< 1 2) 3)");
		eqp("1 <=   2 OR 3",          "(or (<= 1 2) 3)");
		eqp("1 >    2 OR 3",          "(or (> 1 2) 3)");
		eqp("1 >=   2 OR 3",          "(or (>= 1 2) 3)");
		eqp("1 LIKE 2 OR 3",          "(or (like 1 2) 3)");
		eqp("1 IS NULL OR 2",         "(or (is-null 1) 2)");
		eqp("1 IS NOT NULL OR 2",     "(or (is-not-null 1) 2)");
		eqp("1 BETWEEN 3 AND 4 OR 2", "(or (between 1 3 4) 2)");
		eqp("1 OR not 2",             "(or 1 (not 2))");
		eqp("1 AND  2 OR 3",          "(or (and 1 2) 3)");
		eqp("1 OR   2 OR 3",          "(or (or 1 2) 3)");
	}

}
