/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.schema;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import net.morilib.db.fichier.FabriqueDeFichier;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.NullBoolean;
import net.morilib.db.misc.XlsxSharedString;
import net.morilib.db.relations.NamedRelation;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.relations.TableRenameRelation;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.parser.html.HTMLHandler;
import net.morilib.parser.html.HTMLParser;

public class XlsxSqlSchema implements SqlSchema {

	private String xlsx;
	private Map<String, Relation> rels;

	/**
	 * 
	 * @param xlsxfile
	 */
	public XlsxSqlSchema(String xlsxfile) {
		xlsx = xlsxfile;
	}

	private List<List<String>> gettbl(
			String name) throws SQLException, IOException {
		BufferedReader b = null;
		ZipInputStream z = null;
		InputStream m = null;
		XlsxSharedString s;
		XlsxReadTable h;
		XlsxTables y;
		ZipEntry t;
		String n;

		y = _getTableNames();
		if((n = y.getSheet(name)) == null) {
			throw ErrorBundle.getDefault(10015, name);
		}

		try {
			m = fabrique().newInstance(xlsx).openInputStream();
			s = XlsxSharedString.fromXlsx(m);
		} finally {
			if(m != null)  m.close();
		}

		try {
			z = new ZipInputStream(
					fabrique().newInstance(xlsx).openInputStream());
			while((t = z.getNextEntry()) != null) {
				if(t.getName().equalsIgnoreCase("xl/worksheets/" + n)) {
					b = new BufferedReader(new InputStreamReader(
							z, "UTF-8"));
					h = new XlsxReadTable(s);
					HTMLParser.parse(h, b);
					return h.values.size() > 0 ? h.values : null;
				}
			}
			return null;
		} finally {
			if(z != null)  z.close();
		}
	}

	@Override
	public NamedRelation readRelation(String name,
			String as) throws IOException, SQLException {
		List<List<String>> l;

		if(rels != null && rels.containsKey(name.toUpperCase())) {
			// with clause
			return new TableRenameRelation(
					rels.get(name.toUpperCase()),
					(as != null ? as : name).toUpperCase());
		} if((l = gettbl(name)) == null) {
			throw ErrorBundle.getDefault(10015, name);
		} else {
			return SqlSchemata.readRelation(name, as, l);
		}
	}

	@Override
	public void writeRelation(String name,
			Collection<RelationTuple> z
			) throws IOException, SQLException {
		// TODO
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public SqlCreateTable getCreateTable(
			String name) throws IOException, SQLException {
		List<List<String>> l;

		if((l = gettbl(name)) == null) {
			throw ErrorBundle.getDefault(10015, name);
		}
		return SqlSchemata.guessTable(name, l);
	}

	@Override
	public boolean isTable(
			String name) throws IOException, SQLException {
		return getTableNames().contains(name);
	}

	@Override
	public void putCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		// TODO
		throw ErrorBundle.getDefault(10017);
	}

	XlsxTables _getTableNames() throws IOException, SQLException {
		BufferedReader b = null;
		ZipInputStream z = null;
		XlsxTables h;
		ZipEntry t;

		try {
			z = new ZipInputStream(
					fabrique().newInstance(xlsx).openInputStream());
			while((t = z.getNextEntry()) != null) {
				if(t.getName().equalsIgnoreCase("xl/workbook.xml")) {
					b = new BufferedReader(new InputStreamReader(
							z, "UTF-8"));
					h = new XlsxTables();
					HTMLParser.parse(h, b);
					return h;
				}
			}
			return null;
		} finally {
			if(z != null)  z.close();
		}
	}

	@Override
	public Collection<String> getTableNames(
			) throws IOException, SQLException {
		return Collections.unmodifiableSet(
				_getTableNames().map.keySet());
	}

	@Override
	public void truncateTable(
			String name) throws IOException, SQLException {
		// TODO
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public void removeCreateTable(
			String name) throws IOException, SQLException {
		// TODO
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public void alterCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		// TODO
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public SqlSchema fork() {
		return this;
	}

	@Override
	public NullBoolean isReadonly() {
		return NullBoolean.TRUE;
	}

	@Override
	public NullBoolean usesLocalFiles() {
		return NullBoolean.TRUE;
	}

	@Override
	public NullBoolean usesLocalFilePerTable() {
		return NullBoolean.FALSE;
	}

	@Override
	public FabriqueDeFichier fabrique() {
		return FabriqueDeFichier.getDefault();
	}

	@Override
	public void bindSchema(String name, Relation r) {
		rels.put(name, r);
	}

}

class XlsxReadTable implements HTMLHandler {

	private static enum S {
		INIT, WORKSHEET, SHEET_DATA, ROW, C, V
	}

	private static final Pattern PT1 = Pattern.compile("[A-Z]+");

	List<List<String>> values;
	private XlsxSharedString share;
	private List<String> value;
	private boolean sflg;
	private int colno;
	private S stat;

	XlsxReadTable(XlsxSharedString s) {
		values = new ArrayList<List<String>>();
		stat = S.INIT;
		share = s;
	}

	private static int colno(String s) throws SQLException {
		Matcher m;
		int x = 0;
		String v;

		if((m = PT1.matcher(s.toUpperCase())).lookingAt()) {
			v = m.group();
			for(int i = v.length() - 1; i >= 0; i--) {
				x = x * 26 + (v.charAt(i) - 'A');
			}
			return x;
		} else {
			throw ErrorBundle.getDefault(10046);
		}
	}

	private void _put(int x, String s) {
		if(x < value.size()) {
			value.set(x, s);
		} else {
			for(int i = value.size(); i < x; i++)  value.set(i, "");
			value.add(s);
		}
	}

	@Override
	public void string(String s) throws SQLException {
		switch(stat) {
		case INIT:
		case WORKSHEET:
		case SHEET_DATA:
		case ROW:
		case C:
			break;
		case V:
			if(colno < 0) {
				throw ErrorBundle.getDefault(10046);
			} else if(sflg) {
				try {
					_put(colno, share.getString(Integer.parseInt(s)));
				} catch(NumberFormatException e) {
					throw ErrorBundle.getDefault(10046);
				}
			} else {
				_put(colno, s);
			}
			break;
		}
	}

	@Override
	public void startTag(String s) throws SQLException {
		switch(stat) {
		case INIT:
			if(s.equals("worksheet"))  stat = S.WORKSHEET;
			break;
		case WORKSHEET:
			if(s.equals("sheetData"))  stat = S.SHEET_DATA;
			break;
		case SHEET_DATA:
			if(s.equals("row")) {
				stat = S.ROW;
				value = new ArrayList<String>();
			}
			break;
		case ROW:
			if(s.equals("c")) {
				stat = S.C;
				sflg = false;
				colno = -1;
			}
			break;
		case C:
			if(s.equals("v"))  stat = S.V;
			break;
		case V:
			break;
		}
	}

	@Override
	public void endTag(String s) throws SQLException {
		switch(stat) {
		case INIT:
			break;
		case WORKSHEET:
			if(s.equals("worksheet"))  stat = S.INIT;
			break;
		case SHEET_DATA:
			if(s.equals("sheetData"))  stat = S.WORKSHEET;
			break;
		case ROW:
			if(s.equals("row")) {
				values.add(value);
				stat = S.SHEET_DATA;
			}
			break;
		case C:
			if(s.equals("c"))  stat = S.ROW;
			break;
		case V:
			if(s.equals("v"))  stat = S.C;
			break;
		}
	}

	@Override
	public void tagAttribute(String k, String v) throws SQLException {
		switch(stat) {
		case INIT:
		case WORKSHEET:
		case SHEET_DATA:
		case ROW:
		case V:
			break;
		case C:
			if(k.equals("r")) {
				colno = colno(v);
			} else if(k.equals("t")) {
				if(v.equals("s")) {
					sflg = true;
				}
			}
			break;
		}
	}

	@Override
	public void meta(String s) throws SQLException {
		// do nothing
	}

}

class XlsxTables implements HTMLHandler {

	private static enum S {
		INIT, WORKBOOK, SHEETS, SHEET
	}

	Map<String, Integer> map = new HashMap<String, Integer>();
	Map<Integer, String> inv = new HashMap<Integer, String>();
	private S stat = S.INIT;
	private String name;
	private int id;

	String getSheet(String name) {
		return "sheet" + map.get(name) + ".xml";
	}

	@Override
	public void string(String s) throws SQLException {
		// do nothing
	}

	@Override
	public void startTag(String s) throws SQLException {
		switch(stat) {
		case INIT:
			if(s.equals("workbook"))  stat = S.WORKBOOK;
			break;
		case WORKBOOK:
			if(s.equals("sheets"))  stat = S.SHEETS;
			break;
		case SHEETS:
			if(s.equals("sheet")) {
				id = -1;
				stat = S.SHEET;
			}
			break;
		case SHEET:
			break;
		}
	}

	@Override
	public void endTag(String s) throws SQLException {
		switch(stat) {
		case INIT:
			break;
		case WORKBOOK:
			if(s.equals("workbook"))  stat = S.INIT;
			break;
		case SHEETS:
			if(s.equals("sheets"))  stat = S.WORKBOOK;
			break;
		case SHEET:
			if(s.equals("sheet")) {
				if(id < 0) {
					throw ErrorBundle.getDefault(10046);
				}
				map.put(name, id);
				inv.put(id, name);
				stat = S.SHEETS;
			}
			break;
		}
	}

	@Override
	public void tagAttribute(String k, String v) throws SQLException {
		if(stat != S.SHEET) {
			// do nothing
		} else if(k.equals("sheetId")) {
			try {
				id = Integer.parseInt(v);
			} catch(NumberFormatException e) {
				throw ErrorBundle.getDefault(10046);
			}
		} else if(k.equals("name")) {
			name = v;
		}
	}

	@Override
	public void meta(String s) throws SQLException {
		// do nothing
	}

}
