﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Text;
using System.Security.Cryptography;
using System.Xml;
using DPLib;

namespace Quicker
{
    public class Settings
    {
        #region XML Tag
        const string ElemConfigRoot = "Configs";
        const string ElemKey = "Key";
        const string ElemShift = "Shift";
        const string ElemCtrl = "Ctrl";
        const string ElemAlt = "Alt";
        const string ElemGrayback = "Grayback";

        const string ElemAcountRoot = "Acounts";
        const string ElemAcount = "Acount";
        const string ElemPassword = "Password";
        #endregion

        #region 設定値格納領域

        /// <summary>
        /// ショートカットキー
        /// </summary>
        string Key_;
        public string Key
        {
            set { Key_ = value; }
            get { return Key_; }
        }

        /// <summary>
        /// ショートカットにShiftを含む
        /// </summary>
        bool Shift_;
        public bool Shift
        {
            set { Shift_ = value; }
            get { return Shift_; }
        }

        /// <summary>
        /// ショートカットにCtrlを含む
        /// </summary>
        bool Ctrl_;
        public bool Ctrl
        {
            set { Ctrl_ = value; }
            get { return Ctrl_; }
        }

        /// <summary>
        /// 入力時に背景を暗くする
        /// </summary>
        bool Grayback_;
        public bool Grayback
        {
            set { Grayback_ = value; }
            get { return Grayback_; }
        }

        /// <summary>
        /// ショートカットにAltを含む
        /// </summary>
        bool Alt_;
        public bool Alt
        {
            set { Alt_ = value; }
            get { return Alt_; }
        }

        /// <summary>
        /// アカウント
        /// </summary>
        string Acount_;
        public string Acount
        {
            set { Acount_ = value; }
            get { return Acount_; }
        }

        /// <summary>
        /// パスワード
        /// </summary>
        string Password_;
        public string Password
        {
            set { Password_ = value; }
            get { return Password_; }
        }
        #endregion

        /// <summary>
        /// ハッシュキー
        /// </summary>
        string HashKey_ = "";
        private string HashKey
        {
            set
            {
                byte[] tmpSource;
                byte[] tmpHash;

                tmpSource = ASCIIEncoding.ASCII.GetBytes(value);
                tmpHash = new MD5CryptoServiceProvider().ComputeHash(tmpSource);

                int i;
                StringBuilder sOutput = new StringBuilder(tmpHash.Length);
                for (i = 0; i < tmpHash.Length; i++)
                {
                    sOutput.Append(tmpHash[i].ToString("X2"));
                }
                HashKey_ = sOutput.ToString();
            }
            get { return HashKey_; }
        }

        /// <summary>
        /// 設定ファイルルートパス
        /// </summary>
        public string RootPath;

        /// <summary>
        /// システム名
        /// </summary>
        public string SystemName;

        /// <summary>
        /// 環境設定データファイル名
        /// </summary>
        public string ConfigFile = "config.xml";

        /// <summary>
        /// アカウントデータファイル名
        /// </summary>
        public string AcountFile = "ac.xml";

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="systemName">システム名称</param>
        public Settings(string systemName)
        {
            // 設定ファイルの格納パスを生成
            this.RootPath = Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData) + "\\" + systemName;

            // パスが存在しない場合は生成
            if (!Directory.Exists(RootPath))
            {
                Directory.CreateDirectory(RootPath);
            }
        }

        /// <summary>
        /// 設定の保存
        /// </summary>
        public void save()
        {
            // 環境設定データの保存
            saveConfig();

            // アカウントデータの保存
            saveAcount();
        }

        /// <summary>
        /// コンフィグデータの保存
        /// </summary>
        public void saveConfig()
        {
            // Xmlドキュメントを開く
            XmlTextWriter writer = new XmlTextWriter(RootPath + "\\" + ConfigFile, System.Text.Encoding.UTF8);
            writer.Formatting = Formatting.Indented;

            // ドキュメントの開始
            writer.WriteStartDocument(true);

            // ルートの生成
            writer.WriteStartElement(ElemConfigRoot);

            // ショートカットキーの格納
            writer.WriteElementString(ElemKey, Key);
            writer.WriteElementString(ElemShift, Shift.ToString());
            writer.WriteElementString(ElemCtrl, Ctrl.ToString());
            writer.WriteElementString(ElemAlt, Alt.ToString());

            // つぶやき時の背景を暗くする設定
            writer.WriteElementString(ElemGrayback, Grayback.ToString());

            // ルートを閉じる
            writer.WriteEndElement();

            // ドキュメントの終了
            writer.WriteEndDocument();

            writer.Close();
        }

        /// <summary>
        /// アカウントデータの保存
        /// </summary>
        public void saveAcount()
        {
            // ハッシュキーの生成
            this.HashKey = this.Acount;

            // Xmlドキュメントを開く
            XmlTextWriter writer = new XmlTextWriter(RootPath + "\\" + AcountFile, System.Text.Encoding.UTF8);
            writer.Formatting = Formatting.Indented;

            // ドキュメントの開始
            writer.WriteStartDocument(true);

            // ルートの生成
            writer.WriteStartElement(ElemAcountRoot);

            // 設定値の格納
            writer.WriteElementString(ElemAcount, Acount.ToString());
            DataProtection dp = new DataProtection();
            writer.WriteElementString(ElemPassword, dp.protect(Password.ToString()));

            // ルートを閉じる
            writer.WriteEndElement();

            // ドキュメントの終了
            writer.WriteEndDocument();


            writer.Close();
        }

        /// <summary>
        /// 設定の読込
        /// </summary>
        public void load()
        {
            // 環境設定データの保存
            loadConfig();

            // アカウントデータの保存
            loadAcount();
        }

        /// <summary>
        /// コンフィグデータの読込
        /// </summary>
        public void loadConfig()
        {
            // ファイルが存在しない場合は、そのままreturn
            if (!File.Exists(RootPath + "\\" + ConfigFile))
            {
                // デフォルト設定を行う(Q + Shift + Ctrl)
                Key = "Q";
                Shift = true;
                Ctrl = true;
                Alt = false;
                Grayback = false;
                return;
            }

            XmlTextReader reader = new XmlTextReader(RootPath + "\\" + ConfigFile);

            string key = "", shift = "", ctrl = "", alt = "", grayback = "";
            bool inConfigs = false;

            while (reader.Read())
            {
                if (reader.NodeType == XmlNodeType.Element)
                {
                    // ルートを探す
                    if (reader.LocalName.Equals(ElemConfigRoot))
                    {
                        inConfigs = true;
                        continue;
                    }

                    if (inConfigs)
                    {
                        if (reader.LocalName.Equals(ElemKey))
                        {
                            key = reader.ReadString();
                        }
                        else if (reader.LocalName.Equals(ElemShift))
                        {
                            shift = reader.ReadString();
                        }
                        else if (reader.LocalName.Equals(ElemCtrl))
                        {
                            ctrl = reader.ReadString();
                        }
                        else if (reader.LocalName.Equals(ElemAlt))
                        {
                            alt = reader.ReadString();
                        }
                        else if (reader.LocalName.Equals(ElemGrayback))
                        {
                            grayback = reader.ReadString();
                        }
                    }
                }
                else if (reader.NodeType == XmlNodeType.EndElement)
                {
                    if (reader.LocalName.Equals(ElemConfigRoot))
                    {
                        bool ret;
                        Key = key != "" ? key : "Q";
                        Shift = bool.TryParse(shift, out ret) ? bool.Parse(shift) : false;
                        Ctrl = bool.TryParse(ctrl, out ret) ? bool.Parse(ctrl) : false;
                        Alt = bool.TryParse(alt, out ret) ? bool.Parse(alt) : false;
                        Grayback = bool.TryParse(grayback, out ret) ? bool.Parse(grayback) : false;

                        inConfigs = false;
                        break;
                    }
                }
            }
            reader.Close();

        }

        /// <summary>
        /// アカウントデータの読込
        /// </summary>
        public void loadAcount()
        {
            // ファイルが存在しない場合は、そのままreturn
            if (!File.Exists(RootPath + "\\" + AcountFile))
            {
                return;
            }

            XmlTextReader reader = new XmlTextReader(RootPath + "\\" + AcountFile);

            string acount = "", password = "";
            bool inAcounts = false;

            while (reader.Read())
            {
                if (reader.NodeType == XmlNodeType.Element)
                {
                    // ルートを探す
                    if (reader.LocalName.Equals(ElemAcountRoot))
                    {
                        inAcounts = true;
                        continue;
                    }

                    if (inAcounts)
                    {
                        if (reader.LocalName.Equals(ElemAcount))
                        {
                            acount = reader.ReadString();
                            HashKey = acount;
                        }
                        else if (reader.LocalName.Equals(ElemPassword))
                        {
                            DataProtection dp = new DataProtection();
                            password = dp.unprotect(reader.ReadString());
                        }
                    }
                }
                else if (reader.NodeType == XmlNodeType.EndElement)
                {
                    if (reader.LocalName.Equals(ElemAcountRoot))
                    {
                        Acount = acount;
                        Password = password;

                        inAcounts = false;
                        break;
                    }
                }
            }
            reader.Close();
        }
    }
}
