//-----------------------------------------------------------------------
//  This file is part of the Microsoft Robotics Studio Code Samples.
// 
//  Copyright (C) Microsoft Corporation.  All rights reserved.
//
//  $File: Packet.cs $ $Revision: 1 $
//-----------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.IO.Ports;
using System.Diagnostics;
using Microsoft.Dss.ServiceModel.Dssp;
using System.Xml;
using Microsoft.Dss.Services.ConsoleOutput;
using Microsoft.Dss.ServiceModel.DsspServiceBase;

namespace Microsoft.Robotics.Services.Sensors.SickLRF
{
    internal class Packet
    {
        private string _data;
        private int[] data;
        private ushort length;
        
        #region Constructors

        public Packet(string cmd)
        {
            if (cmd[cmd.Length-1] != '\n')
                cmd = cmd + "\n";

            _data = cmd;
            //LogInfo("Packet :" + _data); System.Console.WriteLine("Packet :+" + _data);
        }

        /*
        Packet(byte stx, byte address, byte command)
        {
            _data = new byte[7];

            _data[0] = stx;
            _data[1] = address;
            Write(2, 1);
            _data[4] = command;
            Write(5, CalculatedChecksum);
        }

        Packet(byte stx, byte address, byte[] command)
        {
            _data = new byte[6 + command.Length];
            _data[0] = stx;
            _data[1] = address;
            Write(2, (ushort)command.Length);
            command.CopyTo(_data, 4);
            Write(4 + command.Length, CalculatedChecksum);
        }
        */
        #endregion

        #region Component accessors

        public ushort Length
        {

            get { return length; }
            set { length = value; }
        }

        public string Command
        {
            get { return _data.Substring(0, 2); }
        }
        public string Response
        {
            get { return _data.Substring(0, 2); }
        }
        public int[] Data
        {
            get
            {
                if(Length == 0)
                  data = new int[0];
                
                return data;
            }
            set
            {
                data = value;
            }
        }

        public int TIME
        {
            get
            {
                int r = -1;

                if (Response == "MD" || Response == "GD")
                {
                    string[] parts = _data.Split(new char[] { LL.Lf }, StringSplitOptions.RemoveEmptyEntries);
                    if (parts[1] == "99b")
                    {
                        string time = parts[2].Remove(4, 1);

                        string s = String.Empty;
                        for (int i = 0; i < 4; i++)
                        {
                            int x = time[i] - 0x30;
                            string t = Convert.ToString(x, 2);
                            s += t.PadLeft(6, '0');
                        }
                        r = Convert.ToInt32(s, 2);
                    }
                }

                return r;
            }
        }

        public ushort SETDATA
        {
            get
            {
                length = Decoder();
                return length;
            }
        }

        public Packet SETRANGE
        {
            set
            {
                Packet p = value;
                string[] parts = p._data.Split(new char[]{LL.Lf}, StringSplitOptions.RemoveEmptyEntries);

                SensorParameter.AMIN(parts[6].Remove(parts[6].Length - 2, 2).Remove(0,5));
                SensorParameter.AMAX(parts[7].Remove(parts[7].Length - 2, 2).Remove(0,5));
                SensorParameter.ARES(parts[5].Remove(parts[5].Length - 2, 2).Remove(0,5));
                SensorParameter.AFRT(parts[8].Remove(parts[8].Length - 2, 2).Remove(0,5));
                //dmin = parts[3].Remove(parts[3].Length - 2, 2);
                //dmax = parts[4].Remove(parts[4].Length - 2, 2);
            }
        }

        /*
        public ushort Checksum
        {
            get { return ReadUshort(4 + Length); }
        }
        public ushort CalculatedChecksum
        {
            get { return CreateCRC(_data, 0, 4 + Length); }
        }

        public bool GoodChecksum
        {
            get
            {
                return CalculatedChecksum == Checksum;
            }
        }*/

        #endregion

        #region static Constructors

        public static Packet InitializeAndReset
        {
            get
            {
                return new Packet("SCIP2.0");
            }
        }

        public static Packet MonitoringMode(byte mode)
        {
            LogInfo("MonitoringMode:{0}",mode);
            if (mode == 0x25)
                return new Packet("QT");
            else if (mode == 0x41)
                return new Packet("PP");
            else if (mode == 0x24)
            {
                if (SensorParameter.SetParameter(44,725,01,9,0))
                    LogInfo("illegal Sensor Parameter ");

                return new Packet(SensorParameter.SETMD);
            }
            else
                return new Packet("XX");
        }

        public static Packet RequestMeasured(byte mode)
        {
            return new Packet(SensorParameter.SETGD);
        }

        public static Packet Status
        {
            get
            {
                return new Packet("VV");
            }
        }

        #endregion

        #region IO
        public void SendToSensor(SerialPort port)  //ZTiVA|[gjւ̃R}hM
        {
            LogInfo(_data); System.Console.WriteLine(_data);
            port.Write(_data);
        }
/*
        public static Packet ReadFromSensor(SerialPort port) //ZTiVA|[gj̎M(SickLRFł͎gĂȂ)
        {

            LogInfo("ReadFromSensor");
            try
            {
                
                List<byte> data = new List<byte>();

                byte b = 0;

                while (b != 0x02)
                {
                    if (port.BytesToRead == 0)
                    {
                        return null;
                    }
                    b = (byte)port.ReadByte();
                }
                // STX
                data.Add(b);
                // Address
                data.Add((byte)port.ReadByte());
                // Low Length
                data.Add((byte)port.ReadByte());
                // High Length
                data.Add((byte)port.ReadByte());

                ushort length = MakeUshort(data[2], data[3]);

                for (int i = 0; i < length; i++)
                {
                    data.Add((byte)port.ReadByte());
                }

                // Low Checksum
                data.Add((byte)port.ReadByte());
                // High Checksum
                data.Add((byte)port.ReadByte());

                return new Packet(data);
            }
            catch
            {
                return null;
            }
        }
        */

        #endregion

        #region Utility Functions

        private ushort Decoder()
        {
            string[] parts = _data.Split(new char[]{LL.Lf}, StringSplitOptions.RemoveEmptyEntries);

            if (parts.Length < 4)
                return 0;

            String r = String.Empty;
            for (int i = 3; i < parts.Length; i++)
            {
                parts[i] = parts[i].Remove(parts[i].Length - 1, 1);
                r += parts[i];
            }

            data = new int[r.Length / 3];

            for (int i = 0, j=0; i < r.Length; i += 3, j++)
            {
                string s = String.Empty;
                for (int k = 0; k < 3; k++)
                {
                    int    x = r[k+i] - 0x30;
                    string t = Convert.ToString(x, 2);
                    s += t.PadLeft(6, '0');
                }

                int range = Convert.ToInt32(s, 2); //
                if (range < 19)                //f[^19ȉȂG[Ƃ0
                    range = 0;                 //

                data[j] = range;
            }

            return (ushort)data.Length;
        }

        public ushort ReadUshort(int index)
        {
            return (ushort)(_data[index] | (_data[index + 1] << 8));
        }
        /*
        void Write(int index, ushort value)
        {
            _data[index] = (byte)(value & 0xFF);
            _data[index + 1] = (byte)((value >> 8) & 0xFF);
        }
        */
        public static ushort MakeUshort(byte LowByte, byte HighByte)
        {
            return (ushort)(LowByte | (HighByte << 8));
        }
        #endregion

        #region CRC Function
        const ushort CRC16_GEN_POL = 0x8005;
        static ushort CreateCRC(byte[] CommData, int start, int end)
        {
            ushort crc;
            byte low, high;

            crc = 0;
            low = 0;

            for (int index = start; index < end; index++)
            {
                high = low;
                low = CommData[index];

                if ((crc & 0x8000) == 0x8000)
                {
                    crc = (ushort)((crc & 0x7FFF) << 1);
                    crc ^= CRC16_GEN_POL;
                }
                else
                {
                    crc <<= 1;
                }
                crc ^= MakeUshort(low, high);
            }
            return crc;
        }
        #endregion

        #region LogInfo
        static string _parent;
        public string Parent
        {
            get { return _parent; }
            set { _parent = value; }
        }
        static ConsoleOutputPort _console;
        public ConsoleOutputPort Console
        {
            get { return _console; }
            set { _console = value; }
        }
        static void LogInfo(string format, params object[] args)
        {
            string msg = string.Format(format, args);

            DsspServiceBase.Log(
                TraceLevel.Info,
                TraceLevel.Info,
                new XmlQualifiedName("PacketBuilder", Contract.Identifier),
                _parent,
                msg,
                null,
                _console);
        }
        #endregion

    }

    internal class PacketBuilder
    {
        //string _data;
        Queue<Packet> _packets = new Queue<Packet>();

        //int _length;
        ConsoleOutputPort _console;

        public PacketBuilder()
        {
        }

        public bool Add(string buffer)
        {
            string[] parts = buffer.Split(new string[]{LL.Nl2}, StringSplitOptions.RemoveEmptyEntries);
            int lostpacket = 0;

            for (int i = 0; i < parts.Length; i++ )
            {
                string[] parts_ = parts[i].Split(new char[]{LL.Lf}, StringSplitOptions.RemoveEmptyEntries);

                if (parts_.Length > 1)
                {
                    switch (parts_[1])
                    {
                        case "E": return false;
                        case "99b":
                        case "00P":
                        case "0E":
                        case "0": Packet p = new Packet(parts[i]); //packet
                            _packets.Enqueue(p);//packet^̃L[ɂ߂
                            //LogInfo(parts_[0]); System.Console.WriteLine(parts_[0]);
                            break;
                        default: lostpacket++; break;
                    }
                    parts_[1] = String.Empty;
                }
                else
                {
                    lostpacket++;
                }
            }
            if (lostpacket > 0)
            {
                LogInfo("LostPacket : " + lostpacket);
                System.Console.WriteLine("LostPacket : " + lostpacket);
            }

            return true;
        }

        public bool HasPacket
        {
            get
            {
                return _packets.Count > 0;//packcet^L[_packetsł邩ۂԂ
            }
        }

        public Packet RemovePacket()//L[_packets̒gЂƂo
        {
            return _packets.Dequeue();
        }

        string _parent;
        public string Parent
        {
            get { return _parent; }
            set { _parent = value; }
        }
        public ConsoleOutputPort Console
        {
            get { return _console; }
            set { _console = value; }
        }

        void LogInfo(string format, params object[] args)
        {
            string msg = string.Format(format, args);

            DsspServiceBase.Log(
                TraceLevel.Info,
                TraceLevel.Info,
                new XmlQualifiedName("PacketBuilder", Contract.Identifier),
                _parent,
                msg,
                null,
                _console);
        }
    }

    static class LL
    {
        public static char Lf
        {
            get {return Convert.ToChar(0x0a);}
        }



        public static string Nl2
        {
            get
            {
                char[] dammy = { Lf, Lf };
                String str = new string(dammy);
                return str;
            }
        }
    }

    static class SensorParameter
    {
        private static ushort amin_def;
        private  static ushort amax_def;
        private  static ushort ares;
        private  static ushort afrt;

        private static string amin = amin_def.ToString().PadLeft(4,'0');
        private static string amax = amax_def.ToString().PadLeft(4,'0');
        private static string step = "01";
        private static string reso = "0";
        private static string count = "00";

        //private static string dmin = String.Empty;
        //private static string dmax = String.Empty;

        public static string SETMD
        {
            get { return "MD" + amin + amax + step + reso + count + ';' + Path.GetRandomFileName(); }
        }

        public static string SETGD
        {
            get { return "GD" + amin + amax + step + ';' + Path.GetRandomFileName(); }
        }

        public static double STEPUNIT
        {
            get { return 360.0/ares; }
        }

        public static void AMIN(string amin_)
        {
            
               amin_def = UInt16.Parse(amin_);
        }

        public static void AMAX(string amax_)
        {
             amax_def = ushort.Parse(amax_);
        }

        public static void AFRT(string afrt_)
        {
            afrt = ushort.Parse(afrt_);
        }

        public static void ARES(string ares_)
        {
            ares = ushort.Parse(ares_);
        }


        public static bool SetParameter(int angle)
        {
            double step1 = 360.0 / (double)ares;

            if (angle < 1 || angle > (double)(amax_def - amin_def) * step1)
                return false;

            double steps = 0.5 * (double)angle / step1;

            System.Console.WriteLine("step1:{0}, steps:{1}, (int)stes:{2}, ares{3}",step1,steps,(int)steps, ares);


            int min = afrt - (int)steps;
            int max = afrt + (int)steps;

            amin = min.ToString().PadLeft(4, '0');
            amax = max.ToString().PadLeft(4, '0');

            return true;
        }


        public static bool SetParameter(int _amin, int _amax)
        {
            if (_amin < amin_def || _amin > amax_def-1 ||
                _amax > amax_def || _amax < amin_def-1)
                return false;

            amin = _amin.ToString().PadLeft(4, '0');
            amax = _amax.ToString().PadLeft(4, '0');

            return true;
        }

        public static bool SetParameter(int _amin, int _amax, int _count)
        {
            if (_amin < amin_def || _amin > amax_def - 1 ||
                _amax > amax_def || _amax < amin_def - 1 ||
                _count < 0 || _count > 99)
                return false;

            amin = _amin.ToString().PadLeft(4, '0');
            amax = _amax.ToString().PadLeft(4, '0');
            count = _count.ToString().PadLeft(2, '0');

            return true;
        }

        public static bool SetParameter(int _amin, int _amax, int _step, int _reso, int _count)
        {
            if (_amin  < amin_def || _amin  > amax_def - 1 ||
                _amax  > amax_def || _amax  < amin_def - 1 ||
                _step  < 0        || _step  > 99           ||
                _reso  < 0        || _reso  > 9            ||
                _count < 0        || _count > 99)
                return false;

            amin  = _amin.ToString().PadLeft(4, '0');
            amax  = _amax.ToString().PadLeft(4, '0');
            step  = _step.ToString().PadLeft(2, '0');
            reso  = _reso.ToString();
            count = _count.ToString().PadLeft(2, '0');

            return true;
        }

    }

}
