/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.LinkedList;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import org.apache.commons.logging.Log;
import org.apache.hadoop.fs.FSInputChecker;
import org.apache.hadoop.fs.FSOutputSummer;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.FSConstants;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.server.datanode.BlockAlreadyExistsException;
import org.apache.hadoop.hdfs.server.datanode.BlockMetadataHeader;
import org.apache.hadoop.hdfs.server.datanode.BlockTransferThrottler;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.FSDataset;
import org.apache.hadoop.hdfs.server.datanode.FSDatasetInterface;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.DataChecksum;
import org.apache.hadoop.util.StringUtils;

class BlockReceiver
implements Closeable,
FSConstants {
    public static final Log LOG = DataNode.LOG;
    static final Log ClientTraceLog = DataNode.ClientTraceLog;
    private Block block;
    protected boolean finalized;
    private DataInputStream in = null;
    private DataChecksum checksum;
    private OutputStream out = null;
    private DataOutputStream checksumOut = null;
    private int bytesPerChecksum;
    private int checksumSize;
    private ByteBuffer buf;
    private int bufRead;
    private int maxPacketReadLen;
    protected long offsetInBlock;
    protected final String inAddr;
    protected final String myAddr;
    private String mirrorAddr;
    private DataOutputStream mirrorOut;
    private Daemon responder = null;
    private BlockTransferThrottler throttler;
    private FSDatasetInterface.BlockWriteStreams streams;
    private boolean isRecovery = false;
    private String clientName;
    DatanodeInfo srcDataNode = null;
    private Checksum partialCrc = null;
    private DataNode datanode = null;

    BlockReceiver(Block block, DataInputStream in, String inAddr, String myAddr, boolean isRecovery, String clientName, DatanodeInfo srcDataNode, DataNode datanode) throws IOException {
        try {
            this.block = block;
            this.in = in;
            this.inAddr = inAddr;
            this.myAddr = myAddr;
            this.isRecovery = isRecovery;
            this.clientName = clientName;
            this.offsetInBlock = 0L;
            this.srcDataNode = srcDataNode;
            this.datanode = datanode;
            this.checksum = DataChecksum.newDataChecksum(in);
            this.bytesPerChecksum = this.checksum.getBytesPerChecksum();
            this.checksumSize = this.checksum.getChecksumSize();
            this.streams = datanode.data.writeToBlock(block, isRecovery);
            this.finalized = datanode.data.isValidBlock(block);
            if (this.streams != null) {
                this.out = this.streams.dataOut;
                this.checksumOut = new DataOutputStream(new BufferedOutputStream(this.streams.checksumOut, SMALL_BUFFER_SIZE));
                if (datanode.blockScanner != null && isRecovery) {
                    datanode.blockScanner.deleteBlock(block);
                }
            }
        }
        catch (BlockAlreadyExistsException bae) {
            throw bae;
        }
        catch (IOException ioe) {
            IOUtils.closeStream(this);
            this.cleanupBlock();
            IOException cause = FSDataset.getCauseIfDiskError(ioe);
            if (cause != null) {
                ioe = cause;
                datanode.checkDiskError(ioe);
            }
            throw ioe;
        }
    }

    @Override
    public void close() throws IOException {
        IOException ioe = null;
        try {
            if (this.checksumOut != null) {
                this.checksumOut.flush();
                this.checksumOut.close();
                this.checksumOut = null;
            }
        }
        catch (IOException e) {
            ioe = e;
        }
        try {
            if (this.out != null) {
                this.out.flush();
                this.out.close();
                this.out = null;
            }
        }
        catch (IOException e) {
            ioe = e;
        }
        if (ioe != null) {
            this.datanode.checkDiskError(ioe);
            throw ioe;
        }
    }

    void flush() throws IOException {
        if (this.checksumOut != null) {
            this.checksumOut.flush();
        }
        if (this.out != null) {
            this.out.flush();
        }
    }

    private void handleMirrorOutError(IOException ioe) throws IOException {
        LOG.info((Object)(this.datanode.dnRegistration + ":Exception writing block " + this.block + " to mirror " + this.mirrorAddr + "\n" + StringUtils.stringifyException(ioe)));
        this.mirrorOut = null;
        if (this.clientName.length() > 0) {
            throw ioe;
        }
    }

    private void verifyChunks(byte[] dataBuf, int dataOff, int len, byte[] checksumBuf, int checksumOff) throws IOException {
        while (len > 0) {
            int chunkLen = Math.min(len, this.bytesPerChecksum);
            this.checksum.update(dataBuf, dataOff, chunkLen);
            if (!this.checksum.compare(checksumBuf, checksumOff)) {
                if (this.srcDataNode != null) {
                    try {
                        LOG.info((Object)("report corrupt block " + this.block + " from datanode " + this.srcDataNode + " to namenode"));
                        LocatedBlock lb = new LocatedBlock(this.block, new DatanodeInfo[]{this.srcDataNode});
                        this.datanode.namenode.reportBadBlocks(new LocatedBlock[]{lb});
                    }
                    catch (IOException e) {
                        LOG.warn((Object)("Failed to report bad block " + this.block + " from datanode " + this.srcDataNode + " to namenode"));
                    }
                }
                throw new IOException("Unexpected checksum mismatch while writing " + this.block + " from " + this.inAddr);
            }
            this.checksum.reset();
            dataOff += chunkLen;
            checksumOff += this.checksumSize;
            len -= chunkLen;
        }
    }

    private void shiftBufData() {
        if (this.bufRead != this.buf.limit()) {
            throw new IllegalStateException("bufRead should be same as buf.limit()");
        }
        if (this.buf.position() > 0) {
            int dataLeft = this.buf.remaining();
            if (dataLeft > 0) {
                byte[] b = this.buf.array();
                System.arraycopy(b, this.buf.position(), b, 0, dataLeft);
            }
            this.buf.position(0);
            this.bufRead = dataLeft;
            this.buf.limit(this.bufRead);
        }
    }

    private int readToBuf(int toRead) throws IOException {
        int nRead;
        if (toRead < 0) {
            toRead = (this.maxPacketReadLen > 0 ? this.maxPacketReadLen : this.buf.capacity()) - this.buf.limit();
        }
        if ((nRead = this.in.read(this.buf.array(), this.buf.limit(), toRead)) < 0) {
            throw new EOFException("while trying to read " + toRead + " bytes");
        }
        this.bufRead = this.buf.limit() + nRead;
        this.buf.limit(this.bufRead);
        return nRead;
    }

    private int readNextPacket() throws IOException {
        if (this.buf == null) {
            int chunkSize = this.bytesPerChecksum + this.checksumSize;
            int chunksPerPacket = (this.datanode.writePacketSize - 21 - 4 + chunkSize - 1) / chunkSize;
            this.buf = ByteBuffer.allocate(25 + Math.max(chunksPerPacket, 1) * chunkSize);
            this.buf.limit(0);
        }
        if (this.bufRead > this.buf.limit()) {
            this.buf.limit(this.bufRead);
        }
        while (this.buf.remaining() < 4) {
            if (this.buf.position() > 0) {
                this.shiftBufData();
            }
            this.readToBuf(-1);
        }
        this.buf.mark();
        int payloadLen = this.buf.getInt();
        this.buf.reset();
        if (payloadLen == 0) {
            this.buf.limit(this.buf.position() + 4);
            return 0;
        }
        if (payloadLen < 0 || payloadLen > 0x6400000) {
            throw new IOException("Incorrect value for packet payload : " + payloadLen);
        }
        int pktSize = payloadLen + 21;
        if (this.buf.remaining() < pktSize) {
            int spaceLeft;
            int toRead = pktSize - this.buf.remaining();
            if (toRead > (spaceLeft = this.buf.capacity() - this.buf.limit()) && this.buf.position() > 0) {
                this.shiftBufData();
                spaceLeft = this.buf.capacity() - this.buf.limit();
            }
            if (toRead > spaceLeft) {
                byte[] oldBuf = this.buf.array();
                int toCopy = this.buf.limit();
                this.buf = ByteBuffer.allocate(toCopy + toRead);
                System.arraycopy(oldBuf, 0, this.buf.array(), 0, toCopy);
                this.buf.limit(toCopy);
            }
            while (toRead > 0) {
                toRead -= this.readToBuf(toRead);
            }
        }
        if (this.buf.remaining() > pktSize) {
            this.buf.limit(this.buf.position() + pktSize);
        }
        if (pktSize > this.maxPacketReadLen) {
            this.maxPacketReadLen = pktSize;
        }
        return payloadLen;
    }

    private int receivePacket() throws IOException {
        int payloadLen = this.readNextPacket();
        if (payloadLen <= 0) {
            return payloadLen;
        }
        this.buf.mark();
        this.buf.getInt();
        this.offsetInBlock = this.buf.getLong();
        long seqno = this.buf.getLong();
        boolean lastPacketInBlock = this.buf.get() != 0;
        int endOfHeader = this.buf.position();
        this.buf.reset();
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Receiving one packet for block " + this.block + " of length " + payloadLen + " seqno " + seqno + " offsetInBlock " + this.offsetInBlock + " lastPacketInBlock " + lastPacketInBlock));
        }
        this.setBlockPosition(this.offsetInBlock);
        if (this.mirrorOut != null) {
            try {
                this.mirrorOut.write(this.buf.array(), this.buf.position(), this.buf.remaining());
                this.mirrorOut.flush();
            }
            catch (IOException e) {
                this.handleMirrorOutError(e);
            }
        }
        this.buf.position(endOfHeader);
        int len = this.buf.getInt();
        if (len < 0) {
            throw new IOException("Got wrong length during writeBlock(" + this.block + ") from " + this.inAddr + " at offset " + this.offsetInBlock + ": " + len);
        }
        if (len == 0) {
            LOG.debug((Object)("Receiving empty packet for block " + this.block));
        } else {
            this.offsetInBlock += (long)len;
            int checksumLen = (len + this.bytesPerChecksum - 1) / this.bytesPerChecksum * this.checksumSize;
            if (this.buf.remaining() != checksumLen + len) {
                throw new IOException("Data remaining in packet does not match sum of checksumLen and dataLen");
            }
            int checksumOff = this.buf.position();
            int dataOff = checksumOff + checksumLen;
            byte[] pktBuf = this.buf.array();
            this.buf.position(this.buf.limit());
            if (this.mirrorOut == null || this.clientName.length() == 0) {
                this.verifyChunks(pktBuf, dataOff, len, pktBuf, checksumOff);
            }
            try {
                if (!this.finalized) {
                    this.out.write(pktBuf, dataOff, len);
                    if (this.partialCrc != null) {
                        if (len > this.bytesPerChecksum) {
                            throw new IOException("Got wrong length during writeBlock(" + this.block + ") from " + this.inAddr + " " + "A packet can have only one partial chunk." + " len = " + len + " bytesPerChecksum " + this.bytesPerChecksum);
                        }
                        this.partialCrc.update(pktBuf, dataOff, len);
                        byte[] buf = FSOutputSummer.convertToByteStream(this.partialCrc, this.checksumSize);
                        this.checksumOut.write(buf);
                        LOG.debug((Object)("Writing out partial crc for data len " + len));
                        this.partialCrc = null;
                    } else {
                        this.checksumOut.write(pktBuf, checksumOff, checksumLen);
                    }
                    this.datanode.myMetrics.bytesWritten.inc(len);
                }
            }
            catch (IOException iex) {
                this.datanode.checkDiskError(iex);
                throw iex;
            }
        }
        this.flush();
        if (this.responder != null) {
            ((PacketResponder)this.responder.getRunnable()).enqueue(seqno, lastPacketInBlock);
        }
        if (this.throttler != null) {
            this.throttler.throttle(payloadLen);
        }
        return payloadLen;
    }

    void writeChecksumHeader(DataOutputStream mirrorOut) throws IOException {
        this.checksum.writeHeader(mirrorOut);
    }

    void receiveBlock(DataOutputStream mirrOut, DataInputStream mirrIn, DataOutputStream replyOut, String mirrAddr, BlockTransferThrottler throttlerArg, int numTargets) throws IOException {
        this.mirrorOut = mirrOut;
        this.mirrorAddr = mirrAddr;
        this.throttler = throttlerArg;
        try {
            if (!this.finalized) {
                BlockMetadataHeader.writeHeader(this.checksumOut, this.checksum);
            }
            if (this.clientName.length() > 0) {
                this.responder = new Daemon(this.datanode.threadGroup, new PacketResponder(this, this.block, mirrIn, replyOut, numTargets));
                this.responder.start();
            }
            while (this.receivePacket() > 0) {
            }
            if (this.mirrorOut != null) {
                try {
                    this.mirrorOut.writeInt(0);
                    this.mirrorOut.flush();
                }
                catch (IOException e) {
                    this.handleMirrorOutError(e);
                }
            }
            if (this.responder != null) {
                ((PacketResponder)this.responder.getRunnable()).close();
            }
            if (this.clientName.length() == 0) {
                this.close();
                this.block.setNumBytes(this.offsetInBlock);
                this.datanode.data.finalizeBlock(this.block);
                this.datanode.myMetrics.blocksWritten.inc();
            }
        }
        catch (IOException ioe) {
            LOG.info((Object)("Exception in receiveBlock for block " + this.block + " " + ioe));
            IOUtils.closeStream(this);
            if (this.responder != null) {
                this.responder.interrupt();
            }
            this.cleanupBlock();
            throw ioe;
        }
        finally {
            if (this.responder != null) {
                try {
                    this.responder.join();
                }
                catch (InterruptedException e) {
                    throw new IOException("Interrupted receiveBlock");
                }
                this.responder = null;
            }
        }
    }

    private void cleanupBlock() throws IOException {
        if (this.clientName.length() == 0) {
            this.datanode.data.unfinalizeBlock(this.block);
        }
    }

    private void setBlockPosition(long offsetInBlock) throws IOException {
        if (this.finalized) {
            if (!this.isRecovery) {
                throw new IOException("Write to offset " + offsetInBlock + " of block " + this.block + " that is already finalized.");
            }
            if (offsetInBlock > this.datanode.data.getLength(this.block)) {
                throw new IOException("Write to offset " + offsetInBlock + " of block " + this.block + " that is already finalized and is of size " + this.datanode.data.getLength(this.block));
            }
            return;
        }
        if (this.datanode.data.getChannelPosition(this.block, this.streams) == offsetInBlock) {
            return;
        }
        long offsetInChecksum = (long)BlockMetadataHeader.getHeaderSize() + offsetInBlock / (long)this.bytesPerChecksum * (long)this.checksumSize;
        if (this.out != null) {
            this.out.flush();
        }
        if (this.checksumOut != null) {
            this.checksumOut.flush();
        }
        if (offsetInBlock % (long)this.bytesPerChecksum != 0L) {
            LOG.info((Object)("setBlockPosition trying to set position to " + offsetInBlock + " for block " + this.block + " which is not a multiple of bytesPerChecksum " + this.bytesPerChecksum));
            this.computePartialChunkCrc(offsetInBlock, offsetInChecksum, this.bytesPerChecksum);
        }
        LOG.info((Object)("Changing block file offset of block " + this.block + " from " + this.datanode.data.getChannelPosition(this.block, this.streams) + " to " + offsetInBlock + " meta file offset to " + offsetInChecksum));
        this.datanode.data.setChannelPosition(this.block, this.streams, offsetInBlock, offsetInChecksum);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void computePartialChunkCrc(long blkoff, long ckoff, int bytesPerChecksum) throws IOException {
        int sizePartialChunk = (int)(blkoff % (long)bytesPerChecksum);
        int checksumSize = this.checksum.getChecksumSize();
        LOG.info((Object)("computePartialChunkCrc sizePartialChunk " + sizePartialChunk + " block " + this.block + " offset in block " + (blkoff -= (long)sizePartialChunk) + " offset in metafile " + ckoff));
        byte[] buf = new byte[sizePartialChunk];
        byte[] crcbuf = new byte[checksumSize];
        FSDatasetInterface.BlockInputStreams instr = null;
        try {
            instr = this.datanode.data.getTmpInputStreams(this.block, blkoff, ckoff);
            IOUtils.readFully(instr.dataIn, buf, 0, sizePartialChunk);
            IOUtils.readFully(instr.checksumIn, crcbuf, 0, crcbuf.length);
        }
        catch (Throwable throwable) {
            IOUtils.closeStream(instr);
            throw throwable;
        }
        IOUtils.closeStream(instr);
        this.partialCrc = new CRC32();
        this.partialCrc.update(buf, 0, sizePartialChunk);
        LOG.info((Object)("Read in partial CRC chunk from disk for block " + this.block));
        if (this.partialCrc.getValue() != FSInputChecker.checksum2long(crcbuf)) {
            String msg = "Partial CRC " + this.partialCrc.getValue() + " does not match value computed the " + " last time file was closed " + FSInputChecker.checksum2long(crcbuf);
            throw new IOException(msg);
        }
    }

    private static class Packet {
        long seqno;
        boolean lastPacketInBlock;

        Packet(long seqno, boolean lastPacketInBlock) {
            this.seqno = seqno;
            this.lastPacketInBlock = lastPacketInBlock;
        }
    }

    class PacketResponder
    implements Runnable,
    FSConstants {
        private LinkedList<Packet> ackQueue = new LinkedList();
        private volatile boolean running = true;
        private Block block;
        DataInputStream mirrorIn;
        DataOutputStream replyOut;
        private int numTargets;
        private BlockReceiver receiver;

        public String toString() {
            return "PacketResponder " + this.numTargets + " for Block " + this.block;
        }

        PacketResponder(BlockReceiver receiver, Block b, DataInputStream in, DataOutputStream out, int numTargets) {
            this.receiver = receiver;
            this.block = b;
            this.mirrorIn = in;
            this.replyOut = out;
            this.numTargets = numTargets;
        }

        synchronized void enqueue(long seqno, boolean lastPacketInBlock) {
            if (this.running) {
                LOG.debug((Object)("PacketResponder " + this.numTargets + " adding seqno " + seqno + " to ack queue."));
                this.ackQueue.addLast(new Packet(seqno, lastPacketInBlock));
                this.notifyAll();
            }
        }

        synchronized void close() {
            while (this.running && this.ackQueue.size() != 0 && ((BlockReceiver)BlockReceiver.this).datanode.shouldRun) {
                try {
                    this.wait();
                }
                catch (InterruptedException e) {
                    this.running = false;
                }
            }
            LOG.debug((Object)("PacketResponder " + this.numTargets + " for block " + this.block + " Closing down."));
            this.running = false;
            this.notifyAll();
        }

        private synchronized void lastDataNodeRun() {
            long lastHeartbeat = System.currentTimeMillis();
            boolean lastPacket = false;
            while (this.running && ((BlockReceiver)BlockReceiver.this).datanode.shouldRun && !lastPacket) {
                long now = System.currentTimeMillis();
                try {
                    while (this.running && ((BlockReceiver)BlockReceiver.this).datanode.shouldRun && this.ackQueue.size() == 0) {
                        long idle = now - lastHeartbeat;
                        long timeout = (long)(((BlockReceiver)BlockReceiver.this).datanode.socketTimeout / 2) - idle;
                        if (timeout <= 0L) {
                            timeout = 1000L;
                        }
                        try {
                            this.wait(timeout);
                        }
                        catch (InterruptedException e) {
                            if (!this.running) break;
                            LOG.info((Object)("PacketResponder " + this.numTargets + " for block " + this.block + " Interrupted."));
                            this.running = false;
                            break;
                        }
                        now = System.currentTimeMillis();
                        if (now - lastHeartbeat <= (long)(((BlockReceiver)BlockReceiver.this).datanode.socketTimeout / 2)) continue;
                        this.replyOut.writeLong(-1L);
                        this.replyOut.flush();
                        lastHeartbeat = now;
                    }
                    if (!this.running || !((BlockReceiver)BlockReceiver.this).datanode.shouldRun) break;
                    Packet pkt = this.ackQueue.removeFirst();
                    long expected = pkt.seqno;
                    this.notifyAll();
                    LOG.debug((Object)("PacketResponder " + this.numTargets + " for block " + this.block + " acking for packet " + expected));
                    if (pkt.lastPacketInBlock) {
                        if (!this.receiver.finalized) {
                            this.receiver.close();
                            this.block.setNumBytes(this.receiver.offsetInBlock);
                            ((BlockReceiver)BlockReceiver.this).datanode.data.finalizeBlock(this.block);
                            ((BlockReceiver)BlockReceiver.this).datanode.myMetrics.blocksWritten.inc();
                            BlockReceiver.this.datanode.notifyNamenodeReceivedBlock(this.block, "");
                            if (ClientTraceLog.isInfoEnabled() && this.receiver.clientName.length() > 0) {
                                ClientTraceLog.info((Object)String.format("src: %s, dest: %s, bytes: %s, op: %s, cliID: %s, srvID: %s, blockid: %s", this.receiver.inAddr, this.receiver.myAddr, this.block.getNumBytes(), "HDFS_WRITE", this.receiver.clientName, ((BlockReceiver)BlockReceiver.this).datanode.dnRegistration.getStorageID(), this.block));
                            } else {
                                LOG.info((Object)("Received block " + this.block + " of size " + this.block.getNumBytes() + " from " + this.receiver.inAddr));
                            }
                        }
                        lastPacket = true;
                    }
                    this.replyOut.writeLong(expected);
                    this.replyOut.writeShort(0);
                    this.replyOut.flush();
                }
                catch (Exception e) {
                    if (!this.running) continue;
                    LOG.info((Object)("PacketResponder " + this.block + " " + this.numTargets + " Exception " + StringUtils.stringifyException(e)));
                    this.running = false;
                }
            }
            LOG.info((Object)("PacketResponder " + this.numTargets + " for block " + this.block + " terminating"));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            if (this.numTargets == 0) {
                this.lastDataNodeRun();
                return;
            }
            boolean lastPacketInBlock = false;
            while (this.running && ((BlockReceiver)BlockReceiver.this).datanode.shouldRun && !lastPacketInBlock) {
                try {
                    long expected;
                    boolean didRead;
                    short op;
                    block25: {
                        op = 0;
                        didRead = false;
                        expected = -2L;
                        try {
                            long seqno = this.mirrorIn.readLong();
                            didRead = true;
                            if (seqno == -1L) {
                                this.replyOut.writeLong(-1L);
                                this.replyOut.flush();
                                LOG.debug((Object)("PacketResponder " + this.numTargets + " got -1"));
                                continue;
                            }
                            if (seqno == -2L) {
                                LOG.debug((Object)("PacketResponder " + this.numTargets + " got -2"));
                                break block25;
                            }
                            LOG.debug((Object)("PacketResponder " + this.numTargets + " got seqno = " + seqno));
                            Packet pkt = null;
                            PacketResponder packetResponder = this;
                            synchronized (packetResponder) {
                                while (this.running && ((BlockReceiver)BlockReceiver.this).datanode.shouldRun && this.ackQueue.size() == 0) {
                                    if (LOG.isDebugEnabled()) {
                                        LOG.debug((Object)("PacketResponder " + this.numTargets + " seqno = " + seqno + " for block " + this.block + " waiting for local datanode to finish write."));
                                    }
                                    this.wait();
                                }
                                pkt = this.ackQueue.removeFirst();
                                expected = pkt.seqno;
                                this.notifyAll();
                                LOG.debug((Object)("PacketResponder " + this.numTargets + " seqno = " + seqno));
                                if (seqno != expected) {
                                    throw new IOException("PacketResponder " + this.numTargets + " for block " + this.block + " expected seqno:" + expected + " received:" + seqno);
                                }
                                lastPacketInBlock = pkt.lastPacketInBlock;
                            }
                        }
                        catch (Throwable e) {
                            if (!this.running) break block25;
                            LOG.info((Object)("PacketResponder " + this.block + " " + this.numTargets + " Exception " + StringUtils.stringifyException(e)));
                            this.running = false;
                        }
                    }
                    if (Thread.interrupted()) {
                        LOG.info((Object)("PacketResponder " + this.block + " " + this.numTargets + " : Thread is interrupted."));
                        this.running = false;
                        continue;
                    }
                    if (!didRead) {
                        op = 1;
                    }
                    if (lastPacketInBlock && !this.receiver.finalized) {
                        this.receiver.close();
                        this.block.setNumBytes(this.receiver.offsetInBlock);
                        ((BlockReceiver)BlockReceiver.this).datanode.data.finalizeBlock(this.block);
                        ((BlockReceiver)BlockReceiver.this).datanode.myMetrics.blocksWritten.inc();
                        BlockReceiver.this.datanode.notifyNamenodeReceivedBlock(this.block, "");
                        if (ClientTraceLog.isInfoEnabled() && this.receiver.clientName.length() > 0) {
                            ClientTraceLog.info((Object)String.format("src: %s, dest: %s, bytes: %s, op: %s, cliID: %s, srvID: %s, blockid: %s", this.receiver.inAddr, this.receiver.myAddr, this.block.getNumBytes(), "HDFS_WRITE", this.receiver.clientName, ((BlockReceiver)BlockReceiver.this).datanode.dnRegistration.getStorageID(), this.block));
                        } else {
                            LOG.info((Object)("Received block " + this.block + " of size " + this.block.getNumBytes() + " from " + this.receiver.inAddr));
                        }
                    }
                    this.replyOut.writeLong(expected);
                    this.replyOut.writeShort(0);
                    LOG.debug((Object)("PacketResponder " + this.numTargets + " for block " + this.block + " responded my status " + " for seqno " + expected));
                    for (int i = 0; i < this.numTargets && ((BlockReceiver)BlockReceiver.this).datanode.shouldRun; ++i) {
                        try {
                            if (op == 0 && (op = this.mirrorIn.readShort()) != 0) {
                                LOG.debug((Object)("PacketResponder for block " + this.block + ": error code received from downstream " + " datanode[" + i + "] " + op));
                            }
                        }
                        catch (Throwable e) {
                            op = 1;
                        }
                        this.replyOut.writeShort(op);
                    }
                    this.replyOut.flush();
                    LOG.debug((Object)("PacketResponder " + this.block + " " + this.numTargets + " responded other status " + " for seqno " + expected));
                    if (expected == -2L) {
                        this.running = false;
                    }
                    if (op != 1 || this.receiver.clientName.length() <= 0) continue;
                    this.running = false;
                }
                catch (IOException e) {
                    if (!this.running) continue;
                    LOG.info((Object)("PacketResponder " + this.block + " " + this.numTargets + " Exception " + StringUtils.stringifyException(e)));
                    this.running = false;
                }
                catch (RuntimeException e) {
                    if (!this.running) continue;
                    LOG.info((Object)("PacketResponder " + this.block + " " + this.numTargets + " Exception " + StringUtils.stringifyException(e)));
                    this.running = false;
                }
            }
            LOG.info((Object)("PacketResponder " + this.numTargets + " for block " + this.block + " terminating"));
        }
    }
}

