<!-- 

Copyright 2009, XBRL International Inc.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

This stylesheet, extractXBRL.xsl, converts an Inline XBRL Document to an XBRL 2.1 Target Document.

     NOTE: This stylesheet must be used with a wrapper script or on pre-processed input. For details
     on how to use this stylesheet, please see the README file.

     It will work with any XSLT 1.0 or 2.0 processor.

     Template rules are indented as a convenience for reading the code, showing the 
     implicit hierarchical relationships and groupings of the rules.

     ASSUMPTION: The input is valid Inline XBRL, as constrained by the Inline XBRL specification.

     2009-03-19 - Direct questions to evan@lenzconsulting.com

-->
<!DOCTYPE xsl:stylesheet [

<!-- The following entity declarations are re-used in multiple places by the stylesheet. -->

<!-- Facts are uniquely identified by a combination of target (not shown below) and the following
     values, corresponding respectively to the XPath expressions in the entity declaration below:

       1) the local part of the fact's QName, e.g., "bar" in name="foo:bar" or name="bar"
       2) the namespace URI referenced by the prefix (or default namespace) of the fact's QName
       3) the type of fact (either "fraction", "nonFraction", "nonNumeric", or "tuple")
       4) the contextRef value (always empty for tuples)
       5) the unitRef value (always empty for tuples and nonNumeric facts)
       6) the tuple it belongs to (empty if N/A)
       7) its order within the tuple (empty if N/A)
       8) its string-value
       9) its numerator value (empty if not fraction)
       10) its denominator value (empty if not fraction)
       11) the node ID, if this fact is a tuple; in other words, never de-duplicate tuples

     Any two facts which have the same target and values for the above are assumed to be duplicates
     of each other. Duplicates will be not appear in the result.
     
     Duplicate removal is optional for Inline XBRL procesors and can be done selectively, if at all.
     Here we are removing duplicates non-exhaustively; it's possible that some duplicates will not
     be removed. In particular, XBRL 2.1 says that two items can be duplicates and have different
     values. However, a consequence of the Inline XBRL mapping rules is that certain duplicate items
     in the result, however erroneous, can't be prevented (e.g., when two items have the same name,
     context, unit, and target, but different values).  Also, since reliable identification of tuple
     duplicates requires access to the DTS, we've opted to never de-duplicate tuples.
-->
<!ENTITY composite-key-for-facts "' ', concat(substring-after(@name[contains(.,':')], ':'),
                                              @name[not(contains(.,':'))]),
                                  ' ', namespace::*[name() = substring-before(../@name, ':')],
                                  ' ', local-name(.),
                                  ' ', @contextRef,
                                  ' ', @unitRef,
                                  ' ', @tupleRef,
                                  ' ', @order,
                                  ' ', .,
                                  ' ', ix:numerator,
                                  ' ', ix:denominator,
                                  ' ', generate-id(self::ix:tuple)">

<!-- References are uniquely identified by a combination of target and the following
     values:

       1) the type of reference (either "schemaRef" or "linkbaseRef")
       2) the value of @xlink:href

     Any two references which have the same target and values for the above are assumed
     to be duplicates.
-->
<!ENTITY composite-key-for-references "' ', local-name(.),
                                       ' ', @xlink:href"> 

<!-- Namespaces which will be excluded from the result -->
<!ENTITY ix    "http://www.xbrl.org/2008/inlineXBRL">
<!ENTITY temp  "http://www.xbrl.org/2008/inlineXBRL/processor-internal-use-only">
<!ENTITY xhtml "http://www.w3.org/1999/xhtml">
]>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:ix="&ix;"
  xmlns:xbrli="http://www.xbrl.org/2003/instance"
  xmlns:link="http://www.xbrl.org/2003/linkbase"
  xmlns:xlink="http://www.w3.org/1999/xlink"
  xmlns:temp="&temp;"
  xmlns:x="&xhtml;"
  exclude-result-prefixes="ix xbrli link xlink temp x">

  <!-- For escaping inline markup when @escape is present on ix:nonNumeric -->
  <xsl:import href="xml-to-string.xsl"/>

  <!-- Helpful for HTML compatibility (though not required by the spec):
       render escaped <br/> and <hr/> as "<br />" and "<hr />" -->
  <xsl:variable name="empty-tag-end" select="' />'"/>

  <!-- Configure xml-to-string.xsl to exclude these namespace declarations -->
  <xsl:variable name="namespaces-to-exclude" select="document('')/*/temp:namespaces-to-exclude/ns"/>
  <temp:namespaces-to-exclude>
    <ns>&ix;</ns>
    <ns>&temp;</ns>
  </temp:namespaces-to-exclude>

  <!-- To override xml-to-string.xsl -->
  <xsl:output omit-xml-declaration="no"/>


  <!-- Includes the "resolve-uri" library template for resolving relative URIs -->
  <xsl:include href="uri.xsl"/>


  <!-- Inline XBRL Document Set root (overridden by optional controller stylesheet) -->
  <xsl:variable name="IXDS" select="/"/>

  <!-- All facts -->
  <xsl:variable name="all-facts" select="$IXDS//ix:fraction
                                       | $IXDS//ix:nonFraction
                                       | $IXDS//ix:nonNumeric
                                       | $IXDS//ix:tuple"/>
  <!-- All references -->
  <xsl:variable name="all-references" select="$IXDS//link:schemaRef
                                            | $IXDS//link:linkbaseRef"/>

  <!-- Index of facts -->
  <xsl:key name="facts" match="ix:fraction
                             | ix:nonFraction
                             | ix:nonNumeric
                             | ix:tuple" use="concat(@target,
                                                     &composite-key-for-facts;)"/>

  <!-- Index of references -->
  <xsl:key name="references" match="link:schemaRef
                                  | link:linkbaseRef" use="concat(../@target,
                                                                  &composite-key-for-references;)"/>

  <!-- Index of targets -->
  <xsl:key name="targets" match="ix:fraction   /@target
                               | ix:nonFraction/@target
                               | ix:nonNumeric /@target
                               | ix:references /@target
                               | ix:tuple      /@target" use="."/>

  <!-- Indices for roleRef, arcroleRef, context, unit, and footnote elements -->
  <xsl:key name="roleRefs"    match="link:roleRef"    use="@roleURI"/>
  <xsl:key name="arcroleRefs" match="link:arcroleRef" use="@arcroleURI"/>
  <xsl:key name="contexts"    match="xbrli:context"   use="@id"/>
  <xsl:key name="units"       match="xbrli:unit"      use="@id"/>
  <xsl:key name="footnotes"   match="ix:footnote"     use="@footnoteID"/>


  <!-- Overrides the root rule in xml-to-string.xsl when extractXBRL.xsl is invoked by itself;
       the Main* wrapper stylesheets override this rule. -->
  <xsl:template match="/">
    <xsl:apply-templates select="*"/>
  </xsl:template>

  <xsl:template match="/*">
    <xsl:variable name="all-target-atts" select="//*[ self::ix:fraction
                                                    | self::ix:nonFraction
                                                    | self::ix:nonNumeric
                                                    | self::ix:references
                                                    | self::ix:tuple
                                                    ]
                                                    /@target"/>
    <!-- Filter duplicate target values, using the Muenchian Method -->
    <xsl:variable name="unique-target-values"
                  select="$all-target-atts[generate-id(.) = generate-id(key('targets', .)[1])]"/>
    <!-- If there are any facts or references without a target,
         then that means they belong to the default target document.
    -->
    <xsl:variable name="first-fact-slated-for-default-target"
                  select="($all-facts | //ix:references)[not(@target)][1]"/>
    <TargetDocuments>
      <!-- Decide whether to include a default target document among them -->
      <xsl:if test="$first-fact-slated-for-default-target">
        <TargetDocument default="yes">
          <xsl:call-template name="xbrl-doc"/>
        </TargetDocument>
      </xsl:if>
      <!-- Create a <TargetDocument> for each unique target value -->
      <xsl:for-each select="$unique-target-values">
        <TargetDocument target="{.}">
          <xsl:call-template name="xbrl-doc">
            <xsl:with-param name="target" select="string(.)"/>
          </xsl:call-template>
        </TargetDocument>
      </xsl:for-each>
    </TargetDocuments>
  </xsl:template>


  <!-- ###################################### -->
  <!-- Construct the XBRL 2.1 target document -->
  <!-- ###################################### -->
  <xsl:template name="xbrl-doc">
    <xsl:param name="target" select="''"/> <!-- empty string indicates the default target -->

    <!-- All references for this target, excluding duplicates using the Muenchian Method -->
    <xsl:variable name="references"
                  select="$all-references[generate-id(.) =
                                          generate-id(key('references',
                                                          concat($target, &composite-key-for-references;)
                                                         )[1])]"/>

    <!-- All facts for this target, excluding duplicates -->
    <xsl:variable name="facts"
                  select="$all-facts[generate-id(.) =
                                     generate-id(key('facts',
                                                     concat($target, &composite-key-for-facts;)
                                                    )[1])]"/>

    <!-- All top-level facts for this target, i.e., that aren't tuple children
         (whether by reference or by being embedded) -->
    <xsl:variable name="top-level-facts" select="$facts[not(@tupleRef or ancestor::ix:*[1][self::ix:tuple])]"/>
                                                                      <!-- (if nearest ix ancestor is tuple) -->

    <!-- All footnotes referenced by facts for this target -->
    <xsl:variable name="footnotes" select="key('footnotes', $facts/temp:footnoteRef/@ref)"/>

    <!-- All roleRefs referenced by footnotes for this target, with duplicates removed -->
    <xsl:variable name="roleRefs"
                  select="key('roleRefs', ( $footnotes/@footnoteRole
                                          | $footnotes/@footnoteLinkRole))[generate-id(.) =
                                                                           generate-id(key('roleRefs',
                                                                                           @roleURI)[1])]"/>

    <!-- All arcroleRefs referenced by footnotes for this target, with duplicates removed -->
    <xsl:variable name="arcroleRefs"
                  select="key('arcroleRefs', $footnotes/@arcrole)[generate-id(.) =
                                                                  generate-id(key('arcroleRefs',
                                                                                  @arcroleURI)[1])]"/>

    <!-- ASSUMPTION: Valid input contains no duplicate contexts and no duplicate units -->
    <!-- All contexts referenced by facts for this target -->
    <xsl:variable name="contexts" select="key('contexts', $facts/@contextRef)"/>

    <!-- All units referenced by facts for this target -->
    <xsl:variable name="units" select="key('units', $facts/@unitRef)"/>

    <xbrli:xbrl>
      <!-- Copy all the namespaces except the InlineXBRL and XHTML namespaces -->
      <xsl:copy-of select="$IXDS//x:html/namespace::*[not(. = '&ix;' or
                                                          . = '&xhtml;')]"/>

      <!-- Copy references and resources -->
      <!-- Enforce the order mandated by the XBRL 2.1 schema -->
      <xsl:apply-templates mode="replicate" select="$references[self::link:schemaRef]"/>
      <xsl:apply-templates mode="replicate" select="$references[self::link:linkbaseRef]"/>
      <xsl:apply-templates mode="replicate" select="$roleRefs"/>
      <xsl:apply-templates mode="replicate" select="$arcroleRefs"/>

      <!-- Convert just the top-level facts -->
      <xsl:apply-templates select="$top-level-facts">
        <xsl:with-param name="facts-for-target" select="$facts"/>
      </xsl:apply-templates>

      <!-- Create an extended link for each footnote -->
      <xsl:apply-templates mode="footnote-link" select="$footnotes">
        <xsl:with-param name="facts-for-target" select="$facts"/>
      </xsl:apply-templates>

      <!-- Copy the contexts and units -->
      <xsl:apply-templates mode="replicate" select="$contexts | $units"/>

    </xbrli:xbrl>
  </xsl:template>

          <!-- Copy attributes, text, comments, & PIs, as is -->
          <xsl:template mode="replicate" match="@* | node()">
            <xsl:copy/>
          </xsl:template>

          <!-- But "replicate" elements rather than copying them; that way,
               we prevent unwanted namespaces from appearing in the result. -->
          <xsl:template mode="replicate" match="*" priority="1">
            <xsl:element name="{name()}" namespace="{namespace-uri()}">
              <xsl:apply-templates mode="replicate" select="@* | node()"/>
            </xsl:element>
          </xsl:template>


  <!-- ##################################### -->
  <!-- Convert iXBRL facts to XBRL 2.1 facts -->
  <!-- ##################################### -->

  <!-- Convert the generically named iXBRL elements into actual XBRL fact elements -->
  <xsl:template match="ix:fraction
                     | ix:nonFraction
                     | ix:nonNumeric
                     | ix:tuple">
    <!-- For avoiding ID collisions -->
    <xsl:param name="facts-for-target"/>
                                   <!-- Use the in-scope namespace URI that corresponds to the given QName's prefix -->
                                   <!-- This will also work for the default namespace (no prefix) -->
    <xsl:element name="{@name}" namespace="{namespace::*[name() = substring-before(../@name, ':')]}">
      <xsl:apply-templates select="@*"/>
      <!-- Output an "id" attribute if one is already present,
           or if the fact will need to participate in a footnote link -->
      <xsl:if test="@id or @footnoteRefs">
        <xsl:attribute name="id">
          <xsl:apply-templates mode="safe-fact-id" select=".">
            <xsl:with-param name="facts-for-target" select="$facts-for-target"/>
          </xsl:apply-templates>
        </xsl:attribute>
      </xsl:if>
      <!-- Generate the XBRL fact's content -->
      <xsl:apply-templates mode="content" select=".">
        <!-- Keep these around for avoiding ID collisions when generating tuple content -->
        <xsl:with-param name="facts-for-target" select="$facts-for-target"/>
      </xsl:apply-templates>
    </xsl:element>
  </xsl:template>

          <xsl:template mode="safe-fact-id" match="*">
            <xsl:param name="facts-for-target"/>
            <xsl:choose>
              <!-- We're safe to use the given ID if it's the first
                   (or only) instance of that value for this target -->
              <xsl:when test="generate-id(.) =
                              generate-id($facts-for-target[@id = current()/@id][1])">
                <xsl:value-of select="@id"/>
              </xsl:when>
              <!-- Otherwise, we need to generate a new one to avoid collisions -->
              <xsl:otherwise>
                <xsl:apply-templates mode="new-fact-id" select="."/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:template>

                  <!-- Generate a new fact id value to avoid collisions -->
                  <xsl:template mode="new-fact-id" match="*">
                                                 <!-- Add prefix; some processors may
                                                      not generate valid standalone IDs -->
                    <xsl:value-of select="concat('FACT_',generate-id())"/>
                  </xsl:template>


          <!-- By default, don't copy the iXBRL attributes (not in a namespace); most are for iXBRL use only -->
          <xsl:template match="@*"/>

          <!-- But do copy these through -->
          <xsl:template match="@contextRef
                             | @unitRef
                             | @decimals
                             | @precision">
            <xsl:copy/>
          </xsl:template>

          <!-- Also, copy all attributes that are in a namespace (e.g., user-defined, extension attributes, or @xml:*) -->
          <xsl:template match="@*[string(namespace-uri(.))]">
            <xsl:copy/>
          </xsl:template>


  <!-- ########################################################################### -->
  <!-- The mode="content" rules create the content of each resulting XBRL 2.1 fact -->
  <!-- ########################################################################### -->

  <!-- For tuple content, find all the tuple's children and convert them to XBRL facts -->
  <xsl:template mode="content" match="ix:tuple">
    <xsl:param name="facts-for-target"/>
    <!-- Get all descendant facts whose nearest ix ancestor is the current fact -->
    <xsl:variable name="embedded-children" select="( .//ix:fraction
                                                   | .//ix:nonFraction
                                                   | .//ix:nonNumeric
                                                   | .//ix:tuple
                                                   )
                                                   [generate-id(ancestor::ix:*[1]) =
                                                    generate-id(current())]"/>

    <!-- Get all facts that are included by reference -->
    <xsl:variable name="referenced-children" select="$all-facts[@tupleRef = current()/@tupleID]"/>

    <!-- Process the union of the embedded and referenced children... -->
    <xsl:apply-templates select="$embedded-children | $referenced-children">
      <!-- ...sorting them by the "order" attribute -->
      <xsl:sort select="@order" data-type="number"/>
      <!-- Still need to keep track of these to avoid ID collisions -->
      <xsl:with-param name="facts-for-target" select="$facts-for-target"/>
    </xsl:apply-templates>
  </xsl:template>


  <!-- Render nonNumeric content -->
  <xsl:template mode="content" match="ix:nonNumeric">
    <!-- Get the effective value of the element (minus exclusions and optionally escaped) -->
    <xsl:variable name="effective-value">
      <xsl:apply-templates mode="nonNumeric-value" select="."/>
    </xsl:variable>
    <!-- Apply any applicable transformation rules to ix:nonNumeric's value -->
    <xsl:apply-templates mode="ixt-rule" select=".">
      <xsl:with-param name="value" select="string($effective-value)"/>
    </xsl:apply-templates>
  </xsl:template>

          <!-- Normally, just concatenate text nodes... -->
          <xsl:template mode="nonNumeric-value" match="ix:nonNumeric">
            <!-- Concatenate all descendant text nodes, except for the ones that are descendants of
                 an <ix:exclude> element whose nearest ix ancestor is the current fact -->
            <xsl:copy-of select=".//text()[not(ancestor::ix:exclude[generate-id(ancestor::ix:*[1]) =
                                                                    generate-id(current())]
                                              )]"/>
          </xsl:template>

          <!-- But when escape="true", then convert elements to escaped markup -->
          <xsl:template mode="nonNumeric-value" match="ix:nonNumeric[normalize-space(@escape) = '1'
                                                                  or normalize-space(@escape) = 'true']">
            <xsl:call-template name="xml-to-string">
              <xsl:with-param name="node-set" select="node()"/>
            </xsl:call-template>
          </xsl:template>

                  <!-- Strip out <ix:exclude> (and any temp:* elements) -->
                  <xsl:template mode="xml-to-string" match="ix:exclude | temp:*"/>

                  <!-- Strip out the start & end tags of any other ix:* elements -->
                  <xsl:template mode="xml-to-string" match="ix:*">
                    <xsl:param name="depth"/>
                    <xsl:apply-templates mode="xml-to-string">
                      <!-- Babysit the xml-to-string code's way of keeping track of namespace declarations -->
                      <xsl:with-param name="depth" select="$depth + 1"/>
                    </xsl:apply-templates>
                  </xsl:template>

                  <!-- Attempt to resolve relative URIs -->
                  <xsl:template mode="xml-to-string" match="@href | @src | @background | @longdesc | @usemap | @cite | @action | @profile | @codebase">
                    <!-- In XSLT 1.0, for getting the base URI, we're limited to querying the XHTML <base> element -->
                    <xsl:variable name="base-uri" select="string(ancestor::x:html/x:head/x:base/@href)"/>
                    <xsl:choose>
                      <!-- If the base URI isn't present, then we can't improve on things -->
                      <xsl:when test="not($base-uri)">
                        <!-- Just use the normal rule for serializing attributes -->
                        <xsl:apply-imports/>
                      </xsl:when>
                      <!-- But if we *do* have the base URI, then let's resolve the current
                           URI reference using it -->
                      <xsl:otherwise>
                        <xsl:variable name="resolved-uri">
                          <xsl:call-template name="uri:resolve-uri" xmlns:uri="http://xsltsl.org/uri">
                            <xsl:with-param name="reference" select="string(.)"/>
                            <xsl:with-param name="base"      select="$base-uri"/>
                          </xsl:call-template>
                        </xsl:variable>
                        <!-- Serialize the attribute but override its value -->
                        <xsl:call-template name="serialize-attribute">
                          <xsl:with-param name="att-value" select="string($resolved-uri)"/>
                        </xsl:call-template>
                      </xsl:otherwise>
                    </xsl:choose>
                  </xsl:template>


  <!-- Convert <ix:numerator> and <ix:denominator> to their XBRL counterparts -->
  <xsl:template mode="content" match="ix:fraction">
    <xsl:apply-templates select="ix:numerator | ix:denominator"/>
  </xsl:template>

          <xsl:template match="ix:numerator | ix:denominator">
            <xsl:element name="xbrli:{local-name()}">
              <xsl:apply-templates mode="content" select="."/>
            </xsl:element>
          </xsl:template>


  <!-- Transform numeric values, according to the values of "sign", "format", and "scale" -->
  <xsl:template mode="content" match="ix:nonFraction
                                    | ix:numerator
                                    | ix:denominator">
    <!-- Add minus sign if @sign is present -->
    <!-- ASSUMPTION: Input is valid; an actual minus sign, if present, will appear outside the element content -->
    <xsl:if test="@sign and string(.)"> <!-- make sure the content isn't empty, as when xsi:nil="true" -->
      <xsl:text>-</xsl:text>
    </xsl:if>

    <!-- Get the result of applying the ixt transformation rule (only if applicable) -->
    <!-- ASSUMPTION: the input is valid: a value with the "format" property supplied will *not* contain 
                     extraneous whitespace. -->
    <xsl:variable name="possibly-transformed-value">
      <xsl:apply-templates mode="ixt-rule" select="."/>
    </xsl:variable>

    <!-- Get a whitespace-normalized version (with leading and trailing whitespace removed) -->
    <xsl:variable name="normalized-value" select="normalize-space($possibly-transformed-value)"/>


    <!-- Apply @scale, if present -->
    <!-- This approach uses lexical rather than numeric manipulation to apply the
         scaling factor. There are a couple of benefits to this approach:
         
           1. It avoids accuracy/precision problems with numeric computations
              (e.g., processors outputting scientific notation or garbage data).
           2. Less important but nice to have: it preserves lexical details of
              the source, specifically the number of trailing zeroes that are present.
    -->
    <xsl:choose>
      <!-- If @scale isn't present, then just output the (possibly-transformed) value as is -->
      <xsl:when test="not(@scale)">
        <xsl:choose>
          <!-- If we've just output a minus sign, then we'd better
               make sure it's not followed by any whitespace -->
          <xsl:when test="@sign">
            <xsl:value-of select="$normalized-value"/>
          </xsl:when>
          <!-- If not, then don't make any further changes to the whitespace; leave it unchanged-->
          <xsl:otherwise>
            <xsl:value-of select="$possibly-transformed-value"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <!-- Otherwise, move the decimal point accordingly -->
      <xsl:otherwise>

        <!-- To get the mantissa... -->
        <xsl:variable name="mantissa">
          <!-- First, remove the decimal point -->
          <xsl:variable name="decimal-stripped" select="translate($normalized-value, '.', '')"/>
          <!-- Then, remove leading zeroes -->
          <xsl:call-template name="strip-leading-zeroes">
            <xsl:with-param name="string" select="$decimal-stripped"/>
          </xsl:call-template>
        </xsl:variable>
        <!-- The initial exponent is always 0 or below -->
        <xsl:variable name="initial-exponent">
          <xsl:choose>
            <!-- If there's no decimal point, then the initial exponent is 0 -->
            <xsl:when test="not(contains($normalized-value, '.'))">0</xsl:when>
            <!-- Otherwise, the initial exponent is (the complement of)
                 how many decimal places there are in the given value -->
            <xsl:otherwise>
              <xsl:value-of select="-(string-length(substring-after($normalized-value, '.')))"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <!-- Add scale to get the new exponent -->
        <xsl:variable name="new-exponent" select="$initial-exponent + @scale"/>

        <!-- Finally, output the result of applying the new exponent to our mantissa -->
        <xsl:choose>
          <!-- If we're scaling up (using a positive power of ten) -->
          <xsl:when test="$new-exponent >= 0">
            <!-- Output the mantissa... -->
            <xsl:value-of select="$mantissa"/>
            <!-- followed by 0 or more zeroes. -->
            <xsl:call-template name="zeroes">
              <xsl:with-param name="how-many" select="$new-exponent"/>
            </xsl:call-template>
          </xsl:when>
          <!-- Otherwise, we need to add a decimal point (because the exponent is negative) -->
          <xsl:otherwise>
            <xsl:variable name="decimal-places"  select="-($new-exponent)"/>
            <xsl:variable name="mantissa-length" select="string-length($mantissa)"/>
            <xsl:choose>
              <!-- If the decimal point appears inside the mantissa, then split it up -->
              <xsl:when test="$mantissa-length > $decimal-places">
                <xsl:variable name="left-side-digits" select="$mantissa-length - $decimal-places"/>
                <!-- Left-side digits -->
                <xsl:value-of select="substring($mantissa, 1, $left-side-digits)"/>
                <xsl:text>.</xsl:text>
                <!-- Right-side digits -->
                <xsl:value-of select="substring($mantissa, 1 + $left-side-digits)"/>
              </xsl:when>
              <!-- Otherwise, lead with the decimal point -->
              <xsl:otherwise>
                <xsl:text>0.</xsl:text>
                <!-- Pad with as many zeroes as necessary -->
                <xsl:call-template name="zeroes">
                  <xsl:with-param name="how-many" select="$decimal-places - $mantissa-length"/>
                </xsl:call-template>
                <!-- The mantissa itself -->
                <xsl:value-of select="$mantissa"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

          <!-- Recursively remove all leading zeroes from the given string -->
          <xsl:template name="strip-leading-zeroes">
            <xsl:param name="string"/>
            <xsl:choose>
              <xsl:when test="starts-with($string, '0')">
                <xsl:call-template name="strip-leading-zeroes">
                  <xsl:with-param name="string" select="substring($string,2)"/>
                </xsl:call-template>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="$string"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:template>

          <!-- Output the given number of zeroes -->
          <xsl:template name="zeroes">
            <xsl:param name="how-many"/>
            <xsl:if test="$how-many >= 1">
              <xsl:text>0</xsl:text>
              <xsl:call-template name="zeroes">
                <xsl:with-param name="how-many" select="$how-many - 1"/>
              </xsl:call-template>
            </xsl:if>
          </xsl:template>


  <!-- ####################################################################################### -->
  <!-- The mode="ixt-rule" rules convert numbers and dates according to the "format" property -->
  <!-- ####################################################################################### -->

          <!-- By default, don't apply any format conversions -->
          <xsl:template mode="ixt-rule" match="*">
            <xsl:param name="value" select="string(.)"/>
            <xsl:value-of select="$value"/>
          </xsl:template>

          <!-- ASSUMPTION: The supplied format QName values use the ixt namespace.
               These rules will work regardless of what prefix is used in the QName. -->

          <xsl:template mode="ixt-rule" match="*[substring-after(@format,':') = 'numcomma']">
            <xsl:param name="value" select="string(.)"/>
            <!-- Convert commas to periods -->
            <xsl:value-of select="translate($value,',','.')"/>
          </xsl:template>

          <xsl:template mode="ixt-rule" match="*[substring-after(@format,':') = 'numcommadot']">
            <xsl:param name="value" select="string(.)"/>
            <!-- Remove commas -->
            <xsl:value-of select="translate($value,',','')"/>
          </xsl:template>

          <xsl:template mode="ixt-rule" match="*[substring-after(@format,':') = 'numdotcomma']">
            <xsl:param name="value" select="string(.)"/>
            <!-- Remove periods and convert commas to periods -->
            <xsl:value-of select="translate($value,',.','.')"/>
          </xsl:template>

          <xsl:template mode="ixt-rule" match="*[substring-after(@format,':') = 'numspacecomma']">
            <xsl:param name="value" select="string(.)"/>
            <!-- Convert commas to periods and remove spaces -->
            <xsl:value-of select="translate($value,', ','.')"/>
          </xsl:template>

          <xsl:template mode="ixt-rule" match="*[substring-after(@format,':') = 'numspacedot']">
            <xsl:param name="value" select="string(.)"/>
            <!-- Remove spaces -->
            <xsl:value-of select="translate($value,' ','')"/>
          </xsl:template>


          <!-- Convert to ISO date format -->
          <xsl:template mode="ixt-rule" match="*[starts-with(substring-after(@format,':'), 'date')]">
            <xsl:param name="value" select="string(.)"/>
            <xsl:variable name="date" select="normalize-space($value)"/>
            <!-- Year -->
            <xsl:apply-templates mode="year" select=".">
              <xsl:with-param name="date"
                              select="$date"/>
            </xsl:apply-templates>
            <xsl:text>-</xsl:text>
            <!-- Month -->
            <xsl:apply-templates mode="month" select=".">
              <xsl:with-param name="date"
                              select="$date"/>
            </xsl:apply-templates>
            <xsl:text>-</xsl:text>
            <!-- Day -->
            <xsl:apply-templates mode="day" select=".">
              <xsl:with-param name="date"
                              select="$date"/>
            </xsl:apply-templates>
          </xsl:template>

                  <xsl:template mode="year" match="*">
                    <xsl:param name="date"/>
                    <xsl:variable name="third-to-last-character" select="substring($date, string-length($date) - 2, 1)"/>
                    <!-- If it's a number, then the full year is present -->
                    <xsl:variable name="contains-full-year" select="string(number($third-to-last-character)) != 'NaN'"/>
                    <xsl:choose>
                      <xsl:when test="$contains-full-year">
                        <!-- last four characters -->
                        <xsl:value-of select="substring($date, string-length($date) - 3)"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <!-- Assume the 21st century -->
                        <xsl:text>20</xsl:text>
                        <!-- last two characters -->
                        <xsl:value-of select="substring($date, string-length($date) - 1)"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </xsl:template>


                  <!-- When the day comes after the month number -->
                  <xsl:template mode="day" match="*[substring-after(@format,':') = 'dateslashus' or
                                                    substring-after(@format,':') = 'datedotus']">
                    <xsl:param name="date"/>
                    <xsl:value-of select="substring($date, 4, 2)"/>
                  </xsl:template>

                  <!-- When the day comes after the month *name* -->
                  <xsl:template mode="day" match="*[substring-after(@format,':') = 'datelongus' or
                                                    substring-after(@format,':') = 'dateshortus']">
                    <xsl:param name="date"/>
                    <xsl:value-of select="substring(substring-after($date,' '), 1, 2)"/>
                  </xsl:template>

                  <!-- In every other case, the day is the first two characters -->
                  <xsl:template mode="day" match="*">
                    <xsl:param name="date"/>
                    <xsl:value-of select="substring($date, 1, 2)"/>
                  </xsl:template>


                  <!-- When the month is the first two characters -->
                  <xsl:template mode="month" match="*[substring-after(@format,':') = 'dateslashus' or
                                                      substring-after(@format,':') = 'datedotus']">
                    <xsl:param name="date"/>
                    <xsl:value-of select="substring($date, 1, 2)"/>
                  </xsl:template>

                  <!-- When the month comes after the day -->
                  <xsl:template mode="month" match="*[substring-after(@format,':') = 'dateslasheu' or
                                                      substring-after(@format,':') = 'datedoteu']">
                    <xsl:param name="date"/>
                    <xsl:value-of select="substring($date, 4, 2)"/>
                  </xsl:template>

                  <!-- When the month name (or abbreviation) comes first -->
                  <xsl:template mode="month" match="*[substring-after(@format,':') = 'datelongus'
                                                   or substring-after(@format,':') = 'dateshortus']">
                    <xsl:param name="date"/>
                    <xsl:call-template name="month-number">
                      <xsl:with-param name="month-name" select="substring-before($date, ' ')"/>
                    </xsl:call-template>
                  </xsl:template>

                  <!-- When the month name (or abbreviation) comes after the day -->
                  <xsl:template mode="month" match="*[substring-after(@format,':') = 'datelonguk'
                                                   or substring-after(@format,':') = 'dateshortuk']">
                    <xsl:param name="date"/>
                    <xsl:call-template name="month-number">
                      <xsl:with-param name="month-name" select="substring-before(substring($date,4), ' ')"/>
                    </xsl:call-template>
                  </xsl:template>

                          <xsl:template name="month-number">
                            <xsl:param name="month-name"/>
                            <xsl:choose>
                              <xsl:when test="starts-with($month-name,'Jan')">01</xsl:when>
                              <xsl:when test="starts-with($month-name,'Feb')">02</xsl:when>
                              <xsl:when test="starts-with($month-name,'Mar')">03</xsl:when>
                              <xsl:when test="starts-with($month-name,'Apr')">04</xsl:when>
                              <xsl:when test="starts-with($month-name,'May')">05</xsl:when>
                              <xsl:when test="starts-with($month-name,'Jun')">06</xsl:when>
                              <xsl:when test="starts-with($month-name,'Jul')">07</xsl:when>
                              <xsl:when test="starts-with($month-name,'Aug')">08</xsl:when>
                              <xsl:when test="starts-with($month-name,'Sep')">09</xsl:when>
                              <xsl:when test="starts-with($month-name,'Oct')">10</xsl:when>
                              <xsl:when test="starts-with($month-name,'Nov')">11</xsl:when>
                              <xsl:when test="starts-with($month-name,'Dec')">12</xsl:when>
                            </xsl:choose>
                          </xsl:template>


  <!-- ########################################## -->
  <!-- Create an extended link for each footnote. -->
  <!-- ########################################## -->

  <!-- Construct each <link:footnoteLink> element -->
  <xsl:template mode="footnote-link" match="ix:footnote">
    <xsl:param name="facts-for-target"/>
    <xsl:variable name="link-role">
      <xsl:apply-templates mode="link-role" select="."/>
    </xsl:variable>
    <xsl:variable name="arcrole">
      <xsl:apply-templates mode="arcrole" select="."/>
    </xsl:variable>
    <xsl:variable name="footnote-role">
      <xsl:apply-templates mode="footnote-role" select="."/>
    </xsl:variable>
    <!-- The value of @xml:lang that's in scope for this element;
         in valid input, this will always be present. -->
    <xsl:variable name="lang">
      <xsl:value-of select="(ancestor-or-self::*/@xml:lang)[last()]"/>
    </xsl:variable>
    <link:footnoteLink xlink:type="extended" xlink:role="{$link-role}">
      <!-- Copy the nearest xml:base value that's in effect, if applicable. -->
      <xsl:copy-of select="(ancestor-or-self::*/@xml:base)[last()]"/>
      <!-- Add a locator for each fact that this footnote pertains to -->
      <xsl:for-each select="$facts-for-target[temp:footnoteRef/@ref = current()/@footnoteID]">
        <!-- Get the fact ID value -->
        <xsl:variable name="fact-id">
          <xsl:apply-templates mode="safe-fact-id" select=".">
            <xsl:with-param name="facts-for-target" select="$facts-for-target"/>
          </xsl:apply-templates>
        </xsl:variable>
        <link:loc xlink:type="locator" xlink:href="#{$fact-id}"
                                       xlink:label="fact"/>
      </xsl:for-each>

      <link:footnoteArc xlink:type="arc"      xlink:from="fact"
                                              xlink:to="footnote"
                                              xlink:arcrole="{$arcrole}"/>

      <link:footnote    xlink:type="resource" xlink:label="footnote"
                                              xlink:role="{$footnote-role}"
                                              xml:lang="{$lang}">
        <!-- Copy all @xml:* attributes; the more verbose expression is a workaround for a libxslt bug -->
        <xsl:copy-of select="@*[namespace-uri() = 'http://www.w3.org/XML/1998/namespace']"/>
        <xsl:apply-templates mode="replicate"/>
      </link:footnote>
    </link:footnoteLink>
  </xsl:template>
                                                   <!-- default value for <link:footnoteLink>'s xlink:role -->
          <xsl:template mode="link-role" match="*">http://www.xbrl.org/2003/role/link</xsl:template>
          <xsl:template mode="link-role" match="*[@footnoteLinkRole]">
            <xsl:value-of select="@footnoteLinkRole"/>
          </xsl:template>

                                                 <!-- default value for <link:footnoteArc>'s xlink:arcrole -->
          <xsl:template mode="arcrole" match="*">http://www.xbrl.org/2003/arcrole/fact-footnote</xsl:template>
          <xsl:template mode="arcrole" match="*[@arcrole]">
            <xsl:value-of select="@arcrole"/>
          </xsl:template>

                                                       <!-- default value for <link:footnote>'s xlink:role -->
          <xsl:template mode="footnote-role" match="*">http://www.xbrl.org/2003/role/footnote</xsl:template>
          <xsl:template mode="footnote-role" match="*[@footnoteRole]">
            <xsl:value-of select="@footnoteRole"/>
          </xsl:template>

</xsl:stylesheet>
