package jp.sourceforge.tamanegisoul.sa.widget;

import jp.sourceforge.tamanegisoul.sa.util.PreferenceUtils;
import android.content.Context;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.net.Uri;
import android.preference.DialogPreference;
import android.util.AttributeSet;
import android.view.View;
import android.widget.LinearLayout;
import android.widget.LinearLayout.LayoutParams;
import android.widget.SeekBar;
import android.widget.Toast;

/**
 * ボリューム設定。 SeekBarを動かしている間だけプレビュー音を鳴らす。 設定対象はシステムのアラーム音。
 * 
 * @author TamanegiSoul
 */
public class VolumePreference extends DialogPreference implements SeekBar.OnSeekBarChangeListener {

    private Context mContext;
    private SeekBar mSeekBar;
    private AudioManager mAudioManager;
    private MediaPlayer mRingtone;
    private int mCurrentVolume;

    public VolumePreference(Context context, AttributeSet attrs) {
        super(context, attrs);
        mContext = context;
        mAudioManager = (AudioManager) mContext.getSystemService(Context.AUDIO_SERVICE);
    }

    public VolumePreference(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        mContext = context;
        mAudioManager = (AudioManager) mContext.getSystemService(Context.AUDIO_SERVICE);
    }

    @Override
    protected View onCreateDialogView() {
        // 現在保存されているボリューム
        mCurrentVolume = mAudioManager.getStreamVolume(AudioManager.STREAM_ALARM);
        // ダイアログ開いた時に生成、閉じた時にrelease。
        mRingtone = new MediaPlayer();
        // レイアウト
        LinearLayout layout = new LinearLayout(mContext);
        layout.setPadding(50, 50, 50, 50);
        mSeekBar = new SeekBar(mContext);
        mSeekBar.setOnSeekBarChangeListener(this);
        mSeekBar.setMax(mAudioManager.getStreamMaxVolume(AudioManager.STREAM_ALARM));
        mSeekBar.setProgress(mCurrentVolume);
        layout.addView(mSeekBar, new LinearLayout.LayoutParams(LayoutParams.FILL_PARENT, LayoutParams.WRAP_CONTENT));
        return layout;
    }

    @Override
    protected void onDialogClosed(boolean positiveResult) {
        if (!positiveResult) {
            // 元に戻す
            mAudioManager.setStreamVolume(AudioManager.STREAM_ALARM, mCurrentVolume, 0);
        }
        mRingtone.release();
        setSummary(getSummary());
    }

    @Override
    public CharSequence getSummary() {
        int value = mAudioManager.getStreamVolume(AudioManager.STREAM_ALARM) * 100 / mAudioManager.getStreamMaxVolume(AudioManager.STREAM_ALARM);
        return value + "%";
    }

    @Override
    public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
        mAudioManager.setStreamVolume(AudioManager.STREAM_ALARM, progress, 0);
    }

    @Override
    public void onStartTrackingTouch(SeekBar seekBar) {
        // アラーム音
        Uri ringtoneUri = PreferenceUtils.getRingtone(mContext);
        if (ringtoneUri != null) {
            mRingtone.reset();
            try {
                mRingtone.setDataSource(mContext, ringtoneUri);
                // ↓これでアラームの音量設定が使われる
                mRingtone.setAudioStreamType(AudioManager.STREAM_ALARM);
                mRingtone.setLooping(true);
                mRingtone.prepare();
                mRingtone.start();
            } catch (Exception e) {
                Toast.makeText(mContext, "Failed to ring", Toast.LENGTH_LONG).show();
                e.printStackTrace();
            }
        }
    }

    @Override
    public void onStopTrackingTouch(SeekBar seekBar) {
        mRingtone.stop();
    }

}
