﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "SendToCMD.h"

// private headers
#include "Console.h"
#include "ConsoleInputArray.h"
#include "EffectiveBehavior.h"
#include "MessageBox.h"
#include "Path.h"
#include "PrevilegeChecker.h"
#include "Process.h"
#include "SettingRepository.h"
#include "ShellApi.h"
#include "Shortcut.h"
#include "SystemInfo.h"


namespace bearmini
{
    ///
    ///  「コマンドプロンプトに送る」のプログラムを実行します。
    ///  
    ///  @param args コマンドラインで指定された引数
    ///
    void SendToCMD::Run(const wstring_vector& args)
    {
        validateArguments(args);

        const std::wstring commandLine = generateCommandLine();

        FOREACH (wstring_vector::const_iterator, path, args)
        {
            std::wstring unquotedPath = Path::Unquote(*path);
            std::wstring workDir = getWorkDirectoryFor(unquotedPath);

            Process process(commandLine, workDir);
            process.Run();
            process.WaitForConsoleAllocated();

            Console console(process.GetId());
            console.SetTitle(getConsoleTitle(PrevilegeChecker::IsAdminProcess(process.GetHandle())));

            // ファイル名があるときは、コンソールに送る
            if (!Path::IsDirectory(unquotedPath))
            {
                ConsoleInputArray inputs;
                arrangeConsoleInputArray(unquotedPath, &inputs);
                console.Input(inputs);
            }
        }
    }


    ///
    ///  プログラムのセットアップ（インストールもしくはアンインストール）を行います。
    ///
    void SendToCMD::Setup()
    {
        int result = MessageBox::Show(this, L"インストールしますか？", MessageBoxType_YesNo);
        if (result == DialogResult_Yes)
        {
            install();
            return;
        }

        result = MessageBox::Show(this, L"それではアンインストールしますか？", MessageBoxType_YesNo);
        if (result == DialogResult_Yes)
        {
            uninstall();
            return;
        }
    }


    ///
    ///  コマンドラインで渡されてきた引数が CMD.EXE の実行にふさわしいものかどうか、
    ///  検証します。
    ///  実行にふさわしいかどうかの検証とは、ファイル数が多すぎないか、とか
    ///  CMD.EXE で取り扱うことのできないネットワーク上のファイルでないか、
    ///  といったような点を調べます。
    ///  実行にふさわしくないと判断された場合、例外を送出します。
    ///
    ///  @param[in] args コマンドラインで渡されてきた引数
    ///
    void SendToCMD::validateArguments(const wstring_vector& args)
    {
        static const unsigned int MAX_SIMULTANEOUS_EXEC_COUNT = SettingRepository::GetMaxSimultaneous();

        if (args.size() > MAX_SIMULTANEOUS_EXEC_COUNT)
        {
            throw std::domain_error("Too many files. \r\n\r\nIn order to change the number of PowerShell can be started, please modify the value of MAX_SIMULTANEOUS in SendToPS.ini file.");
        }

        // UNC ディレクトリには移動できないのでチェック（args の先頭だけチェックすればユースケース上十分だろうと思われる）
        if (Path::IsUNC(args.front()))
        {
            throw std::domain_error("A UNC (path starts with '\\\\') is not supported. \r\n\r\nFiles or directories on a network cannot be sent to CMD.EXE.");
        }

    }


    ///
    ///  CMD.EXE を実行するためのコマンドライン文字列を生成します。
    ///
    ///  @return    CMD.EXE を実行するためのコマンドライン文字列。"C:\Windows\cmd.exe /F" など
    std::wstring SendToCMD::generateCommandLine()
    {
        std::wstringstream s;
        s << getCmdExePath() << L" " << getAdditionalParam();

        return s.str();
    }


    ///
    ///  CMD.EXE のパスを取得します
    ///
    std::wstring SendToCMD::getCmdExePath()
    {
        std::wstringstream path;

        wchar_t sys_dir[BUFSIZ];
        int result = ::GetSystemDirectoryW(sys_dir, BUFSIZ);
        if (result == 0)
        {
            throw std::domain_error("GetSystemDirectory() failed.");
        }
        path << sys_dir << L"\\cmd.exe";

        return path.str();
    }


    ///
    ///  CMD.EXE に渡す追加のパラメータをファイルから読み取ります。
    ///
    ///  @return    CMD.EXE に渡す追加のパラメータ。"/F" など
    ///
    std::wstring SendToCMD::getAdditionalParam()
    {
        return SettingRepository::GetCmdExeOption();
    }


    ///
    ///  path で指定されたパスのファイルまたはディレクトリをコマンドプロンプトに送る際の
    ///  作業ディレクトリとなるディレクトリを取得します。
    ///
    std::wstring SendToCMD::getWorkDirectoryFor(const std::wstring& path)
    {
        if (Path::IsDirectory(path))
        {
            return path;
        }
        else
        {
            return Path::GetDirectory(path);
        }
    }


    ///
    ///  path で指定されたファイルのパスに応じて、プロンプトに送り込むべきキー入力の配列を取得します。
    ///
    ///  @param[in]     path                 ユーザから指定されたファイルのパス
    ///  @param[in/out] pConsoleInputArray   キー入力の配列を受け取るための ConsoleInputArray クラスのインスタンスへのポインタ
    ///
    void SendToCMD::arrangeConsoleInputArray(const std::wstring& path, ConsoleInputArray* pConsoleInputArray)
    {
        BehaviorSettingsCollection behaviorSettings = SettingRepository::GetBehaviorSettings();
        BehaviorSetting bs = behaviorSettings.GetBehaviorSettingFor(path);
        EffectiveBehavior eb = EffectiveBehavior::CreateFrom(bs, path);

        pConsoleInputArray->Append(eb.CommandLine());

        // いったんコマンドラインの先頭にカーソルを移動してから...
        pConsoleInputArray->AppendVirtualKeyCode(VK_LEFT, static_cast<unsigned int>(eb.CommandLine().length()));

        // カーソルを指定の位置に移動
        pConsoleInputArray->AppendVirtualKeyCode(VK_RIGHT, eb.CursorPosition());

        // 自動実行の場合は ENTER を入力する
        if (bs.auto_exec)
        {
            //pConsoleInputArray->AppendVirtualKeyCode(VK_RETURN);  <- なぜかうまくいかない
            pConsoleInputArray->Append(L"\r\n");
        }
    }


    ///
    ///  このプログラム（SendToCMD.exe）のフルパスを取得します。
    ///
    ///  @return    このプログラムのフルパス
    ///
    std::wstring SendToCMD::getSendToCmdExeFullPath()
    {
        wchar_t lpszModuleName[_MAX_PATH];
        ::GetModuleFileNameW(0, lpszModuleName, _MAX_PATH);

        return lpszModuleName;
    }


    ///
    ///  このプログラム（SendToCMD.exe）がインストールされているディレクトリを取得します。
    ///
    ///  @return    このプログラムがインストールされているディレクトリ
    ///
    std::wstring SendToCMD::getSendToCmdExeDirectory()
    {
        return Path::GetDirectory(getSendToCmdExeFullPath());
    }


    ///
    ///  コンソールのタイトルとして設定する文字列を取得します。
    ///
    std::wstring SendToCMD::getConsoleTitle(bool isAdminProcess)
    {
        if (isAdminProcess)
        {
            return L"管理者: コマンド プロンプト";
        }
        else
        {
            return L"コマンド プロンプト";
        }
    }


    ///
    ///  インストールを行います。
    ///
    void SendToCMD::install()
    {
        // ショートカットを作成（通常実行）
        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"コマンドプロンプトに送る.lnk");
            const std::wstring sendToCmdExePath = getSendToCmdExeFullPath();
            const std::wstring sendToCmdExeDir = getSendToCmdExeDirectory();
            const std::wstring description = L"ファイルやディレクトリを指定してコマンドプロンプトを起動します.";

            try
            {
                Shortcut::Create(linkFilePath, sendToCmdExePath, description, L"", sendToCmdExeDir, L"");
            }
            catch (std::domain_error&)
            {
                throw std::domain_error("ショートカットの作成に失敗しました.");
            }
        }

        // ショートカットを作成（管理者として実行）
        if (SystemInfo::GetOsMajorVersion() >= SystemInfo::OsMajorVersion_Vista) 
        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"コマンドプロンプトに送る（管理者として実行...）.lnk");
            const std::wstring sendToCmdExePath = getSendToCmdExeFullPath();
            const std::wstring sendToCmdExeDir = getSendToCmdExeDirectory();
            const std::wstring description = L"ファイルやディレクトリを指定してコマンドプロンプトを起動します.";

            try
            {
                Shortcut::Create(linkFilePath, sendToCmdExePath, description, L"", sendToCmdExeDir, L"", 0, 1, true);
            }
            catch (std::domain_error&)
            {
                throw std::domain_error("ショートカットの作成に失敗しました.");
            }
        }

        MessageBox::Show(this, L"インストールが完了しました.");
    }


    ///
    ///  アンインストールを行います。
    ///
    void SendToCMD::uninstall()
    {
        // ショートカットファイルを削除
        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"コマンドプロンプトに送る.lnk");
            ::DeleteFileW(linkFilePath.c_str());
        }
        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"コマンドプロンプトに送る（管理者として実行...）.lnk");
            ::DeleteFileW(linkFilePath.c_str());
        }

        MessageBox::Show(this, L"アンインストールが完了しました.\r\nご利用ありがとうございました.");
    }

}
