﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "Util.h"

// C++ headers
#include <locale>  // for std::tolower()
#include <sstream>


namespace bearmini
{
    ///
    ///  str で指定された文字列を小文字にして返します。
    ///
    ///  @param[in] str 小文字に変換したい文字列
    ///
    ///  @return   str を小文字に変換した文字列
    ///
    std::wstring make_string_to_lower(const std::wstring& str) 
    {
        //std::transform(str.begin(), str.end(), result.begin(), std::tolower);  <- うまくいかない orz
        std::wstringstream result;
        for (std::size_t i = 0; i < str.length(); i++)
        {
            result << std::tolower(str.at(i), std::locale());
        }
        return result.str();
    }


    ///
    ///  ワイルドカードを考慮して文字列を比較します。
    ///  wild で指定された文字列（ワイルドカード含む）と str で指定された文字列がマッチするかどうかを判定します。
    ///  使用可能なワイルドカード文字列は * と ? です。
    ///  wildcmp("ABC*", "ABCDEF") はマッチするので 1 を返します。
    ///  wildcmp("A?C", "ABBC") はマッチしないので 0 を返します。
    ///  
    ///  @param[in] wild  ワイルドカードを含む文字列です。
    ///  @param[in] str   wild にマッチしているかどうかを判定したい文字列です。こちらにはワイルドカードを使用できません。
    ///
    ///  @return  マッチする場合は 1、マッチしない場合は 0 を返します。
    ///
    int wildcmp(const wchar_t* wild, const wchar_t* str)
    {
        const wchar_t* cp = 0;
        const wchar_t* mp = 0;

        while ((*str) && (*wild != '*'))
        {
            if ((*wild != *str) && (*wild != '?'))
            {
                return 0;
            }
            wild++;
            str++;
        }

        while (*str)
        {
            if (*wild == '*')
            {
                if (!*++wild)
                {
                    return 1;
                }
                mp = wild;
                cp = str+1;
            }
            else if ((*wild == *str) || (*wild == '?'))
            {
                wild++;
                str++;
            }
            else
            {
                wild = mp;
                str = cp++;
            }
        }

        while (*wild == '*')
        {
            wild++;
        }

        return !*wild;
    }


    ///
    ///  文字列 src の中に含まれる文字列 sa を文字列 sb に置き換えます。
    ///
    ///  @param[in] src   置き換えたい文字列を含む文字列
    ///  @param[in] sa    置き換え元の文字列
    ///  @param[in] sb    置き換え後の文字列
    ///
    ///  @return    文字列 src の中に含まれる文字列 sa が文字列 sb に置き換えられたもの
    ///
    std::wstring ReplaceString(const std::wstring& src, const std::wstring& sa, const std::wstring& sb) 
    {
        std::wstring result = src;
        std::wstring::size_type n, nb = 0;
        while ((n = result.find(sa, nb)) != std::wstring::npos)
        {
            result.replace(n, sa.length(), sb);
            nb = n + sb.length();
        }
        return result;
    }


    ///
    ///  str で指定された文字列を " " で囲んだ文字列を取得します。
    ///
    ///  @param[in]  str   " " で囲みたい文字列
    ///
    ///  @return   " " で囲まれた文字列
    ///
    std::wstring Quote(const std::wstring& str)
    {
        std::wstringstream wss;
        wss << "\"" << str << "\"";
        return wss.str();
    }

}
