﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "Shortcut.h"

// Windows headers
#include <shlobj.h>


namespace bearmini
{
    //
    //  ショートカットを作成します。
    //
    ::HRESULT Shortcut::Create(
        const std::wstring& pathLinkFile,   // ショートカットの絶対パス
        const std::wstring& pathTargetFile, // ターゲットファイル
        const std::wstring& description,    // 説明
        const std::wstring& args,           // 引数
        const std::wstring& workingDir,     // 作業ディレクトリ
        const std::wstring& pathIcon,       // アイコンの場所
        int iIcon,                          // アイコンのインデックス
        int iShowCmd,                       // ウィンドウスタイル
        bool runasAdmin                     // 互換性タブの「管理者として実行」をチェックするかどうか
    )
    {
        ::HRESULT hr;
        ::IShellLinkW* psl = 0;
        ::IShellLinkDataList* pdl = 0;
        ::IPersistFile* ppf = 0;

        // COM ライブラリを初期化する
        hr = ::CoInitialize(0);
        if (FAILED(hr))
        {
            throw std::exception("CoInitialize() failed.");
        }

        // IShellLink オブジェクトを作成しポインタを取得する
        hr = ::CoCreateInstance(::CLSID_ShellLink, 0, ::CLSCTX_INPROC_SERVER, ::IID_IShellLinkW, (void**)&psl);
        if (FAILED(hr))
        {
            goto epilogue;
        }

        // ショートカットを二次記憶装置に保存するため IPersistFile インターフェイスの問い合わせをおこなう
        hr = psl->QueryInterface(::IID_IPersistFile, (void**)&ppf);
        if (FAILED(hr))
        {
            goto epilogue;
        }

        hr = psl->SetPath(pathTargetFile.c_str());           // ターゲットファイル
        if (FAILED(hr))
        {
            goto epilogue;
        }
        hr = psl->SetDescription(description.c_str());       // 説明
        if (FAILED(hr))
        {
            goto epilogue;
        }
        hr = psl->SetArguments(args.c_str());                // 引数
        if (FAILED(hr))
        {
            goto epilogue;
        }
        hr = psl->SetWorkingDirectory(workingDir.c_str());   // 作業ディレクトリ
        if (FAILED(hr))
        {
            goto epilogue;
        }
        hr = psl->SetIconLocation(pathIcon.c_str(), iIcon);  // アイコン
        if (FAILED(hr))
        {
            goto epilogue;
        }
        hr = psl->SetShowCmd(iShowCmd);                      // ウィンドウスタイル
        if (FAILED(hr))
        {
            goto epilogue;
        }

        if (runasAdmin)
        {
            hr = psl->QueryInterface(::IID_IShellLinkDataList, (void**)&pdl);
            if (FAILED(hr))
            {
                goto epilogue;
            }

            DWORD dwFlags = 0;
            hr = pdl->GetFlags(&dwFlags);
            if (FAILED(hr))
            {
                goto epilogue;
            }

            if ((dwFlags & SLDF_RUNAS_USER) == 0)
            {
                hr = pdl->SetFlags(dwFlags | SLDF_RUNAS_USER);
                if (FAILED(hr))
                {
                    goto epilogue;
                }
            }
        }

        // ショートカットを保存する
        hr = ppf->Save(pathLinkFile.c_str(), TRUE);
        if (FAILED(hr))
        {
            goto epilogue;
        }

        hr = ppf->SaveCompleted(pathLinkFile.c_str());
        if (FAILED(hr))
        {
            goto epilogue;
        }

    epilogue:
        // IPersistFile へのポインタを開放する
        if (ppf != 0) {
            ppf->Release();
        }

        if (pdl != 0)
        {
            pdl->Release();
        }

        // IShellLinkへのポインタを開放する
        if (psl != 0) {
            psl->Release();
        }

        // COM ライブラリを解放する
        ::CoUninitialize();

        if (FAILED(hr))
        {
            throw std::exception("ショートカットの作成に失敗しました.");
        }
        return hr;
    }


}