const { remote, ipcRenderer, shell } = require('electron');
const path = require('path');
const fs = require('fs');

const package = require(`${remote.app.getAppPath()}/package.json`);

const Config = require('electron-store');
const config = new Config();

// ファイルタイプの列挙体（のつもり）
const FileType = {
    File: 'file',
    Directory: 'directory',
    Unknown: 'unknown'
}

/**
 * ファイルの種類を取得する
 * @param {string} path パス
 * @return {FileType} ファイルの種類
 */
const getFileType = path => {
    try {
        const stat = fs.statSync(path);

        switch (true) {
            case stat.isFile():
                return FileType.File;

            case stat.isDirectory():
                return FileType.Directory;

            default:
                return FileType.Unknown;
        }

    } catch (e) {
        return FileType.Unknown;
    }
}

/**
 * 指定したディレクトリ配下のすべてのファイルをリストアップする
 * @param {string} dirPath 検索するディレクトリのパス
 * @return {Array<string>} ファイルのパスのリスト
 */
const listFiles = dirPath => {
    const ret = [];
    const paths = fs.readdirSync(dirPath);

    paths.forEach(a => {
        const path = `${dirPath}/${a}`;

        switch (getFileType(path)) {
            case FileType.File:
                ret.push(path);
                break;

            case FileType.Directory:
                ret.push(...listFiles(path));
                break;

            default:
            /* noop */
        }
    })

    return ret;
};

/*
// ====================================================================== //
// ====================================================================== //
// ====================================================================== //
*/

global.getConfigPath = () => {
    if (location.protocol != 'my:') return;

    return config.path;
}

global.getFiles = (pathName) => {
    if (location.protocol != 'my:') return;

    return listFiles(path.resolve(__dirname, pathName));
}

global.getFile = (path, json = false) => {
    if (location.protocol != 'my:') return;

    if (json) {
        return require(path);
    } else {
        const text = fs.readFileSync(path, 'utf8');
        return text;
    }
}

/*
// ====================================================================== //
// ====================================================================== //
// ====================================================================== //
*/

global.getAppName = (b) => {
    if (location.protocol != 'my:') return;

    return package.name;
}

global.getAppDescription = (b) => {
    if (location.protocol != 'my:') return;

    return package.description;
}

global.getAppVersion = (b) => {
    if (location.protocol != 'my:') return;

    return package.version;
}

/*
// ====================================================================== //
// ====================================================================== //
// ====================================================================== //
*/

global.clearBrowserData = (b) => {
    if (location.protocol != 'my:') return;

    b && ipcRenderer.send('clear-browsing-data', {});
}

/*
// ====================================================================== //
// ====================================================================== //
// ====================================================================== //
*/

global.updateFilters = (b) => {
    if (location.protocol != 'my:') return;

    ipcRenderer.send('update-filters', {});
}

/*
// ====================================================================== //
// ====================================================================== //
// ====================================================================== //
*/

global.getHistorys = () => new Promise((resolve) => {
    if (location.protocol != 'my:') return;
    ipcRenderer.send('data-history-get', {});

    ipcRenderer.on('data-history-get', (e, args) => {
        resolve(args.historys);
    });
});

global.clearHistory = (b) => {
    if (location.protocol != 'my:') return;

    b && ipcRenderer.send('data-history-clear', {});
}

global.getBookmarks = (isPrivate) => new Promise((resolve) => {
    if (location.protocol != 'my:') return;
    ipcRenderer.send('data-bookmark-get', { isPrivate });

    ipcRenderer.on('data-bookmark-get', (e, args) => {
        resolve(args.bookmarks);
    });
});

global.clearBookmark = (b) => {
    if (location.protocol != 'my:') return;

    b && ipcRenderer.send('data-bookmark-clear', {});
}

/*
// ====================================================================== //
// ====================================================================== //
// ====================================================================== //
*/

global.getHomeButton = () => {
    if (location.protocol != 'my:') return;

    return config.get('design.homeButton');
}

global.setHomeButton = (b) => {
    if (location.protocol != 'my:') return;

    return config.set('design.homeButton', b);
}

global.getDarkTheme = () => {
    if (location.protocol != 'my:') return;

    return config.get('design.darkTheme');
}

global.setDarkTheme = (b) => {
    if (location.protocol != 'my:') return;

    return config.set('design.darkTheme', b);
}

global.getStartPage = () => {
    if (location.protocol != 'my:') return;

    return config.get('homePage.defaultPage');
}

global.setStartPage = (url) => {
    if (location.protocol != 'my:') return;

    return config.set('homePage.defaultPage', url);
}

global.getSearchEngines = () => {
    if (location.protocol != 'my:') return;

    return config.get('homePage.searchEngines');
}

global.getSearchEngine = () => {
    if (location.protocol != 'my:') return;

    for (var i = 0; i < config.get('homePage.searchEngines').length; i++) {
        if (config.get('homePage.searchEngines')[i].name == config.get('homePage.defaultEngine')) {
            return config.get('homePage.searchEngines')[i];
        }
    }
}

global.setSearchEngine = (name) => {
    if (location.protocol != 'my:') return;

    getSearchEngines().some(function (item, i) {
        if (item.name && item.name === name)
            config.set('homePage.defaultEngine', name);
    });
}

global.getAdBlocker = () => {
    if (location.protocol != 'my:') return;

    return config.get('adBlocker');
}

global.setAdBlocker = (b) => {
    if (location.protocol != 'my:') return;

    return config.set('adBlocker', b);
}

global.getCustomTitlebar = () => {
    if (location.protocol != 'my:') return;

    return config.get('window.isCustomTitlebar');
}

global.setCustomTitlebar = (b) => {
    if (location.protocol != 'my:') return;

    return config.set('window.isCustomTitlebar', b);
}

global.restart = () => {
    if (location.protocol != 'my:') return;

    remote.app.relaunch();
    remote.app.exit(0);
}

global.isURL = (input) => {
    const pattern = /^((?:\w+:)?\/\/([^\s.]+\.\S{2}|localhost[:?\d]*)|my:\/\/\S.*|file:\/\/\S.*)\S*$/;

    if (pattern.test(input)) {
        return true;
    }
    return pattern.test(`http://${input}`);
};

onload = function () {
    // console.log(document.documentElement.innerText);
    if (location.protocol == 'my:') return;

    delete global.getConfigPath;
    delete global.getFiles;
    delete global.getFile;
    delete global.clearBrowserData;
    delete global.updateFilters;
    delete global.getHistorys;
    delete global.clearHistory;
    delete global.getBookmarks;
    delete global.clearBookmark;
    delete global.getHomeButton;
    delete global.setHomeButton;
    delete global.getDarkTheme;
    delete global.setDarkTheme;
    delete global.getStartPage;
    delete global.setStartPage;
    delete global.getSearchEngines;
    delete global.getSearchEngine;
    delete global.setSearchEngine;
    delete global.getAdBlocker;
    delete global.setAdBlocker;
    delete global.getCustomTitlebar;
    delete global.setCustomTitlebar;
    delete global.restart;
    delete global.isURL;
}

onmousedown = (e) => {
    if (remote.getCurrentWindow().getBrowserView() == undefined) return;
    const view = remote.getCurrentWindow().getBrowserView();

    if (e.button == 3) {
        if (view.webContents.canGoBack())
            view.webContents.goBack();
    } else if (e.button == 4) {
        if (view.webContents.canGoForward())
            view.webContents.goForward();
    }
}