/*
 * Copyright 2006-2007 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.codegen.impl;

import java.io.File;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.sql.DataSource;

import org.seasar.codegen.ImportCodeData;
import org.seasar.codegen.dbms.Dbms;
import org.seasar.codegen.element.DataType;
import org.seasar.codegen.element.Field;
import org.seasar.codegen.element.FieldSetting;
import org.seasar.codegen.element.LinkTable;
import org.seasar.codegen.element.PrimaryKey;
import org.seasar.codegen.element.Table;
import org.seasar.codegen.util.CreateTableTypeToTypeUtil;
import org.seasar.codegen.util.ImportUtil;
import org.seasar.codegen.util.LinkUtil;
import org.seasar.extension.jdbc.util.ConnectionUtil;
import org.seasar.extension.jdbc.util.DataSourceUtil;
import org.seasar.extension.jdbc.util.DatabaseMetaDataUtil;
import org.seasar.framework.exception.SQLRuntimeException;
import org.seasar.framework.util.ResultSetUtil;

/**
 * 
 * @author azusa
 */
public class DatabaseImportCodeData implements ImportCodeData {

    protected Dbms dataTypeSelectUtil;

    private Set<String> tables = new HashSet<String>();

    private DataSource dataSource;

    protected String schemaName = null;

    public DatabaseImportCodeData(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    public Map readCodeData(File srcFile) {
        Connection con = null;
        try {
            con = DataSourceUtil.getConnection(dataSource);
            return createMap(con);
        } finally {
            if (con != null) {
                try {
                    con.close();
                } catch (Exception ignore) {
                }
            }
        }

    }

    protected Map createMap(Connection connection) {

        Map<String, Table> tableMap = new HashMap<String, Table>();
        try {
            DatabaseMetaData dbmd = ConnectionUtil.getMetaData(connection);
            ResultSet rsTable = dbmd.getTables(null, schemaName, "%", null);
            try {
                while (rsTable.next()) {
                    Table table = new Table();
                    String tableName = rsTable.getString("TABLE_NAME");
                    if (!tables.isEmpty()) {
                        if (!tables.contains(tableName)) {
                            continue;
                        }
                    }
                    table.setTableName(tableName);
                    String[] primaryKeys = DatabaseMetaDataUtil.getPrimaryKeys(
                            dbmd, tableName);
                    ResultSet rsColumn = dbmd.getColumns(null, schemaName,
                            tableName, "%");
                    try {
                        while (rsColumn.next()) {
                            Field field = getField(rsColumn);
                            table.addTableField(field);
                            setUpPrimaryKey(table, field, primaryKeys);
                        }
                        tableMap.put(table.getTableName(), table);
                    } finally {
                        ResultSetUtil.close(rsColumn);
                    }
                }
                ResultSet rsExKey = dbmd
                        .getExportedKeys(null, schemaName, null);
                try {
                    while (rsExKey.next()) {
                        LinkTable parentLink = getParentLink(rsExKey);
                        Table childTable = tableMap.get(rsExKey
                                .getString("FKTABLE_NAME"));
                        if (childTable != null) {
                            childTable.addLinkTable(childTable.getTableName(),
                                    parentLink);
                        }
                    }
                } finally {
                    ResultSetUtil.close(rsExKey);
                }

                LinkUtil.setupChildLinks(tableMap);
                ImportUtil.setUpImportList(tableMap);
                return tableMap;
            } finally {
                ResultSetUtil.close(rsTable);
            }
        } catch (SQLException e) {
            throw new SQLRuntimeException(e);
        }
    }

    private LinkTable getParentLink(ResultSet rsExKey) throws SQLException {
        LinkTable link = new LinkTable();
        link.setTableName(rsExKey.getString("PKTABLE_NAME"));
        link.setParentFieldName(rsExKey.getString("PKCOLUMN_NAME"));
        link.setChildFieldName(rsExKey.getString("FKCOLUMN_NAME"));
        return link;
    }

    protected Field getField(ResultSet rsColumn) throws SQLException {
        String columnName = rsColumn.getString(4);
        String typeName = rsColumn.getString(6);
        String strNotNull = getNotNull(rsColumn.getInt(11));
        int iColumnSize = rsColumn.getInt(7);
        int decimalDegit = rsColumn.getInt(9);
        String columnSize = null;
        if (decimalDegit != 0) {
            columnSize = "(" + iColumnSize + "," + decimalDegit + ")";
        } else {
            columnSize = "(" + iColumnSize + ")";
        }

        Field field = new Field();
        field.setFieldName(columnName);
        field.setFieldAttributeName(columnName);

        String columnDef = rsColumn.getString(13);
        if (columnDef == null) {
            columnDef = "";
        }

        field.setDataType(getDataType(typeName, columnSize, strNotNull,
                columnDef));
        return field;
    }

    protected void setUpPrimaryKey(Table table, Field field,
            String[] primaryKeys) {
        for (int i = 0; i < primaryKeys.length; i++) {
            // プライマリーキーと列名で大文字小文字が食い違う場合があるのでequalsIgnoreCaseで比較する。
            if (primaryKeys[i].equalsIgnoreCase(field.getFieldName())) {
                field.setSequence(field.getFieldName());
                PrimaryKey primaryKey = new PrimaryKey();
                primaryKey.setField(field);
                table.addPrimaryKey(primaryKey);
            }
        }
    }

    protected DataType getDataType(String dataType, String columnSize,
            String notNull, String defaultValue) {
        FieldSetting fieldSetting = new FieldSetting();
        String langType = dataTypeSelectUtil.convDBTypeToDataType(dataType);
        fieldSetting.setTypeName(langType);
        String length = CreateTableTypeToTypeUtil.getLength(dataType
                + columnSize, dataType);
        int precision = CreateTableTypeToTypeUtil.getFullLength(length);
        fieldSetting.setColmnSize(precision);
        int scale = CreateTableTypeToTypeUtil.getPointNumberLength(length);
        fieldSetting.setPointNumber(scale);
        boolean isNotNull = "NOT NULL".equalsIgnoreCase(notNull);
        fieldSetting.setNotNull(isNotNull);
        fieldSetting.setFieldDefault(defaultValue);
        DataType type = dataTypeSelectUtil.selectBestDataType(fieldSetting);
        return type;
    }

    protected String getNotNull(int notNull) {
        if (notNull == DatabaseMetaData.columnNoNulls) {
            return "NOT NULL";
        } else {
            return "";
        }
    }

    public void setSchemaName(String schemaName) {
        this.schemaName = schemaName;
    }

    public void setDataTypeSelectUtil(Dbms dataTypeSelectUtil) {
        this.dataTypeSelectUtil = dataTypeSelectUtil;
    }

    public void addTable(String tableName) {
        tables.add(tableName);
    }
}
