package jp.co.epson.watch.plaWasabi.service;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import jp.co.epson.watch.plaWasabi.commons.WasabiRuntimeException;
import jp.co.epson.watch.plaWasabi.dto.formConfig.Action;
import jp.co.epson.watch.plaWasabi.dto.formConfig.EntityMapping;
import jp.co.epson.watch.plaWasabi.service.schema.CacheManager;
import jp.co.epson.watch.plaWasabi.service.schema.DbColumnDef;
import jp.co.epson.watch.plaWasabi.service.schema.DbTableDef;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.extension.jdbc.JdbcManager;
import org.seasar.framework.container.annotation.tiger.Binding;

/**
 * 入力フォームと D/B を連携するサービスクラス。
 * 
 * @author K.Ono
 * 
 */
public class DbService {

  /**
   * Commons Logging instance.
   */
  private static Log log = LogFactory.getLog(DbService.class);

  @Binding
  public JdbcManager jdbcManager;

  @Binding("schema_oracle_cacheManagerService")
  private CacheManager cacheManagerService;

  /**
   * SQLインジェクション対策のため文字列内の全ての "'" を "''" に置換する。
   * 
   * @param s
   *          置換対象の文字列
   * @return 置換された文字列
   */
  private static String sanitize(String s) {
    return s.replaceAll("'", "''");
  }

  /**
   * D/B の更新または挿入に必要な全ての情報を受け取り、動的にSQL文を組み立てて発行するサービスメソッド。
   * 
   * @param action
   *          Actionに関する設定情報。
   * @param keyMap
   *          D/B更新の際にレコードを特定するためにWHERE句などに指定される主キーのフィールド名と値のマップ。 キー名は小文字で指定する。
   * @param valueMap
   *          D/B更新対象となるフィールド名と値のマップ。 キー名は小文字で指定する。
   * @throws WasabiRuntimeException
   *           不正なD/B更新件数(1以外)が発生したケース。
   */
  public void persist(final Action action, Map<String, String> keyMap,
      final Map<String, String> valueMap) {
    EntityMapping em = action.getEntityMapping();

    for (String entityName : em.getEntities()) // 永続化対象の全てのエンティティについて
    {
      DbTableDef table = cacheManagerService.getTable(entityName); // キャッシュよりテーブル定義情報を取得
      if (null != table) {
        Map<String, String> targetKeyMap = this.extractMap(table.getPrimaryKey(), keyMap, false);
        Map<String, String> targetValueMap = this.extractMap(table.getColumns(), valueMap, true);
        if (targetKeyMap.size() > 0 || targetValueMap.size() > 0) {
          if (0 == targetKeyMap.size()) // INSERT
          {
            int inserted = this.insert(table, keyMap, valueMap);
            if (1 != inserted) {
              throw new WasabiRuntimeException("Insert Failed : count=" + inserted);
            }
          } else if (table.getPrimaryKey().size() == targetKeyMap.size()) // INSERT
          // or
          // UPDATE
          {
            long count = this.count(table, targetKeyMap); // UPDATE件数が1件以下になるための予防
            if (1 == count) // UPDATE
            {
              int updated = this.update(table, targetKeyMap, targetValueMap);
              if (1 < updated) // update件数が 0 か 1 以外は不正
              {
                throw new WasabiRuntimeException("Update Failed : count=" + updated);
              }
            } else // ERROR : UPDATE対象のレコード数が2つ以上ある
            {
              log.error("Cancel updating. '" + entityName + "'");
              throw new WasabiRuntimeException("Cancel updating. '" + entityName + "'");
            }
          } else // ERROR : 不完全な複合キーが指定されている。更新対象外とする。
          {
            log.error("Primary key is incomplete (" + targetKeyMap.size() + "). '" + entityName
                + "'");
            throw new WasabiRuntimeException("Primary key is incomplete (" + targetKeyMap.size()
                + "). '" + entityName + "'");
          }
        }
      } else // ERROR : テーブル定義情報の取得に失敗
      {
        log.error("Unable to get table definition for '" + entityName + "'");
        throw new WasabiRuntimeException("Unable to get table definition for '" + entityName + "'");
      }
    }
  }

  /**
   * 指定されたテーブルに対応するシーケンスからIDを発番する。
   * 
   * @param tableName
   *          シーケンス
   * @return
   */
  private long nextval(final String tableName) {

    return this.jdbcManager.selectBySql(Long.class,
        "SELECT " + tableName + "_SEQ.NEXTVAL FROM DUAL").getSingleResult();
  }

  /**
   * レコード件数を取得する。
   * 
   * @param table
   *          テーブル名
   * @param targetKeyMap
   *          対象テーブルの主キーのみ格納したMAP
   * @return レコード件数
   */
  private long count(final DbTableDef table, final Map<String, String> targetKeyMap) {
    StringBuilder sql = new StringBuilder();
    sql.append("SELECT * FROM ");
    sql.append(table.getName());
    sql.append(this.where(table, targetKeyMap));
    return this.jdbcManager.getCountBySql(sql.toString());
  }

  /**
   * 主キーであるカラムにはIDを自動セットして新規挿入する。
   * 
   * @param table
   *          挿入対象テーブル名
   * @param keyMap
   *          自動発番したIDをセットするMAP。
   * @param valueMap
   *          新規挿入する値が格納されたMAP
   * @return 挿入が成功した件数
   * @throws WasabiRuntimeException
   *           NULLを許容しない外部参照カラムの値が見つからない場合。 テーブル定義情報に複数の外部参照を持つカラムが存在した場合。
   */
  private int insert(final DbTableDef table, Map<String, String> keyMap,
      final Map<String, String> valueMap) {
    StringBuilder sql1 = new StringBuilder();
    StringBuilder sql2 = new StringBuilder();
    sql1.append("INSERT INTO " + table.getName() + " (");
    sql2.append(" VALUES(");

    int i = 0;
    for (DbColumnDef col : table.getColumns()) {
      if (col.isForeignKey()) {
        Map<String, DbColumnDef> FK = col.getForeingKey();
        if (1 == FK.size()) {
          String FKval = keyMap.get(FK.entrySet().iterator().next().getValue().getName());
          if (StringUtils.isNotEmpty(FKval)) {
            if (0 < i) {
              sql1.append(",");
              sql2.append(",");
            }
            sql1.append(col.getName());
            sql2.append("'" + sanitize(FKval) + "'");
          } else if (!col.isNullable()) // NULLを許容しない外部参照のカラムの値がない場合
          {
            throw new WasabiRuntimeException("EMPTY value is not acceptable : '"
                + col.getTableName() + "." + col.getName() + "' is required foreign key.");
          }
        } else // 複数の外部参照を持つカラムはサポートしない
        {
          throw new WasabiRuntimeException("Not supported : Multiple foreign keys '"
              + col.getTableName() + "." + col.getName() + "'");
        }
      } else if (col.isPrimaryKey()) {
        long id = this.nextval(table.getName()); // ID発番
        if (0 < i) {
          sql1.append(",");
          sql2.append(",");
        }
        sql1.append(col.getName());
        sql2.append("'" + id + "'");
        keyMap.put(col.getName(), String.valueOf(id)); // 発番したIDがフォームで利用できるようMAPにセット
      } else {
        if (valueMap.containsKey(col.getName())) {
          if (0 < i) {
            sql1.append(",");
            sql2.append(",");
          }
          sql1.append(col.getName());
          sql2.append("'" + sanitize(valueMap.get(col.getName())) + "'");
        }
      }
      i++;
    }
    sql1.append(")");
    sql2.append(")");

    return this.jdbcManager.updateBySql(sql1.toString() + sql2.toString()).execute();
  }

  /**
   * レコードの更新を行う。
   * 
   * @param table
   *          更新対象テーブル名
   * @param targetKeyMap
   *          対象テーブルの主キーのみ格納したMAP
   * @param targetValueMap
   *          対象テーブルの更新値のみ格納したMAP
   * @return 更新件数
   */
  private int update(final DbTableDef table, final Map<String, String> targetKeyMap,
      final Map<String, String> targetValueMap) {
    if (0 < targetValueMap.size()) // UPDATE対象の項目がある
    {
      StringBuilder sql = new StringBuilder();
      sql.append("UPDATE ");
      sql.append(table.getName());
      sql.append(" SET ");

      Set<String> keySet = targetValueMap.keySet();
      Iterator<String> iterator = keySet.iterator();
      int i = 0;
      while (iterator.hasNext()) {
        String key = iterator.next();
        String val = targetValueMap.get(key);
        if (0 < i)
          sql.append(",");
        sql.append(key.toUpperCase());
        sql.append("=");
        sql.append("'" + sanitize(val) + "'");
        i++;
      }
      sql.append(this.where(table, targetKeyMap));
      return this.jdbcManager.updateBySql(sql.toString()).execute();
    } else // Skip
    {
      return 0;
    }
  }

  /**
   * 指定のカラム一覧に存在するキーと値をMAPから取り出す。
   * 
   * @param baseList
   *          比較元となるカラム一覧
   * @param compareMap
   *          比較対象となるMAP
   * @param allowNullValue
   *          MAPの値が空の場合も取り出すか否か。
   * @return compareMapから抽出できた順序を持つMAP
   */
  private Map<String, String> extractMap(final List<DbColumnDef> baseList,
      final Map<String, String> compareMap, final boolean allowEmptyValue) {
    Map<String, String> resultMap = new TreeMap<String, String>();

    for (DbColumnDef col : baseList) {
      if (compareMap.containsKey(col.getName())) {
        if (allowEmptyValue || StringUtils.isNotEmpty(compareMap.get(col.getName()))) {
          resultMap.put(col.getName(), compareMap.get(col.getName()));
        }
      }
    }
    return resultMap;
  }

  /**
   * WHERE句を生成する。
   * 
   * @param table
   *          テーブル名
   * @param targetKeyMap
   *          対象テーブルの主キーのみ格納したMAP
   * @return WHERE句
   */
  private String where(final DbTableDef table, final Map<String, String> targetKeyMap) {
    StringBuilder where = new StringBuilder();
    where.append(" WHERE ");
    int i = 0;
    for (DbColumnDef col : table.getPrimaryKey()) {
      if (0 < i++)
        where.append(" AND ");
      where.append(col.getName() + "='" + sanitize(targetKeyMap.get(col.getName())) + "'");
    }
    return where.toString();
  }

  @SuppressWarnings("unchecked")
  public List<Map> selectBySql(String sql, String[] paramArray) {
    List<Map> resList = this.jdbcManager.selectBySql(Map.class, sql, paramArray).getResultList();

    return resList;
  }

  @SuppressWarnings("unchecked")
  public Map selectUniqueBySql(String sql, String[] paramArray) {
    Map res = this.jdbcManager.selectBySql(Map.class, sql, paramArray).getSingleResult();

    return res;
  }

  public int deleteBySql(String sql, String id) {
    int res = this.jdbcManager.updateBySql(sql, String.class).params(id).execute();

    return res;
  }

}
