/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jdbcacsess2.main;

import java.util.ArrayList;
import java.util.Vector;

import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.HighlighterFactory;

import com.ezware.oxbow.swingbits.table.filter.TableRowFilterSupport;

/**
 * @author sima
 * 
 */
public class JmyTable extends JXTable {

	private static final long serialVersionUID = -7507194043135028576L;

	public JmyTable() {
		super();
		initialize();
	}

	/**
	 * @param dm
	 */
	public JmyTable(TableModel dm) {
		super(dm);
		initialize();
	}

	/**
	 * @param dm
	 * @param cm
	 */
	public JmyTable(TableModel dm, TableColumnModel cm) {
		super(dm, cm);
		initialize();
	}

	/**
	 * @param numRows
	 * @param numColumns
	 */
	public JmyTable(int numRows, int numColumns) {
		super(numRows, numColumns);
		initialize();
	}

	/**
	 * @param rowData
	 * @param columnNames
	 */
	public JmyTable(Vector<?> rowData, Vector<?> columnNames) {
		super(rowData, columnNames);
		initialize();
	}

	/**
	 * @param rowData
	 * @param columnNames
	 */
	public JmyTable(Object[][] rowData, Object[] columnNames) {
		super(rowData, columnNames);
		initialize();
	}

	/**
	 * @param dm
	 * @param cm
	 * @param sm
	 */
	public JmyTable(TableModel dm, TableColumnModel cm, ListSelectionModel sm) {
		super(dm, cm, sm);
		initialize();
	}

	private void initialize() {
		setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		setShowVerticalLines(false);
		setShowHorizontalLines(false);
		setHighlighters(HighlighterFactory.createSimpleStriping());
		// setCellSelectionEnabled(true);
		setRowSelectionAllowed(true);
		setAutoCreateRowSorter(true);
		setEditable(false);
		FontZoom.setMouseWheel(this);
	}

	@Override
	public void setModel(TableModel tableModel) {
		super.setModel(tableModel);
		new TableColumnFit(this).setEnableAutoFit(true);
		TableRowFilterSupport.forTable(this).searchable(true).apply();
		tableModel.addTableModelListener(tableModelListener);
		setSortsOnUpdates(true);
	}

	/**
	 * @return
	 */
	public int getSelectedModelRow() {
		int row = getSelectedRow();
		if (row < 0) {
			return -1;
		}
		return convertRowIndexToModel(row);
	}

	/**
	 * @return
	 */
	public int getSelectedModelColumn() {
		int column = getSelectedColumn();
		if (column < 0) {
			return -1;
		}
		return convertColumnIndexToModel(column);
	}

	/**
	 * @return
	 */
	public ArrayList<Integer> getSelectedModelRows() {
		int[] rows = getSelectedRows();
		ArrayList<Integer> list = new ArrayList<Integer>(rows.length);
		for (int i = 0; i < rows.length; i++) {
			list.add(convertRowIndexToModel(rows[i]));
		}
		return list;
	}

	/**
	 * @return
	 */
	public ArrayList<Integer> getSelectedModelColumns() {
		int[] columns = getSelectedColumns();
		ArrayList<Integer> list = new ArrayList<Integer>(columns.length);
		for (int i = 0; i < columns.length; i++) {
			list.add(convertColumnIndexToModel(columns[i]));
		}
		return list;
	}

	/**
	 * 追加変更削除が行われた行を選択する。
	 */
	transient private final TableModelListener tableModelListener = new TableModelListener() {
		@Override
		public void tableChanged(TableModelEvent event) {

			switch (event.getType()) {
			case TableModelEvent.UPDATE:
				int updateRow = convertRowIndexToView(event.getFirstRow());
				changeSelection(updateRow, updateRow, false, false);
				break;

			case TableModelEvent.INSERT:
				final int insertRow = event.getFirstRow();
				// モデルの変更後に、追加行のGUI表示が行われる為、ここで追加イベントを拾ってもまだ画面表示は行われていない。
				// なので、EDTにキューイング？して、画面書き換え後に選択させる。
				SwingUtilities.invokeLater(new Runnable() {
					@Override
					public void run() {
						// 行追加後のview状態で、view座標に変換。反映前だと変換テーブルがみつからないらしい
						int row = convertRowIndexToView(insertRow);
						changeSelection(row, row, false, false);
					}
				});
				break;

			case TableModelEvent.DELETE:
				if (getModel().getRowCount() == 0) {
					break;
				}

				// 行削除前のview状態で、view座標に変換。反映後だと変換テーブルが見つからないらしい
				final int deletedViewRow = convertRowIndexToView(event.getFirstRow());
				SwingUtilities.invokeLater(new Runnable() {

					@Override
					public void run() {
						int row = deletedViewRow;
						if (getRowSorter().getViewRowCount() <= row) {
							row = getRowSorter().getViewRowCount() - 1;
						}
						changeSelection(row, row, false, false);
					}
				});
				break;

			default:
				break;
			}
		}
	};

}
