﻿using System;
using System.Collections.Generic;
using System.Text;
using SlothLib.LinearAlgebra.FeatureVector;

namespace SlothLib.LinearAlgebra.FeatureVector
{

	/// <summary>
	/// クラスタ保持クラス
	/// </summary>
	public class Cluster<T>
	{
		IVector<T>[] vectors;
		int[] indices;
		IVector<T> centroidVector;



		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="vectors"></param>
		/// <param name="indices"></param>
		/// <param name="centroidVector"></param>
		public Cluster(IVector<T>[] vectors, int[] indices, IVector<T> centroidVector)
		{
			this.vectors = vectors;
			this.indices = indices;
			this.centroidVector = centroidVector;
		}

		/// <summary>
		/// このクラスタに含まれるアイテムのベクトル群を取得する
		/// </summary>
		/// <returns></returns>
		public IVector<T>[] GetVectors()
		{
			return this.vectors;
		}


		/// <summary>
		/// クラスタリングされるときに与えられたベクトル配列のインデックスのうち、このクラスタに含まれるものを取得する
		/// </summary>
		/// <returns></returns>
		public int[] GetIndices()
		{
			return this.indices;
		}


		/// <summary>
		/// このクラスタにあるアイテムの数
		/// </summary>
		public int ItemCount
		{
			get
			{
				return this.vectors.Length;
			}
		}


		/// <summary>
		/// 指定されたベクトルのアイテムがこのクラスタにあるかどうか
		/// </summary>
		/// <param name="vector"></param>
		/// <returns></returns>
		public bool Contains(IVector<T> vector)
		{
			foreach (IVector<T> v in this.vectors)
			{
				if (v == vector)
				{
					return true;
				}
			}
			return false;
		}

		/// <summary>
		/// 指定されたインデックスのアイテムがこのクラスタにあるかどうか
		/// </summary>
		/// <param name="index"></param>
		/// <returns></returns>
		public bool Contains(int index)
		{
			foreach (int i in this.indices)
			{
				if (i == index)
				{
					return true;
				}
			}
			return false;
		}


		/// <summary>
		/// 重心のベクトルを取得する。
		/// セントロイド法の場合はキャッシュされているので取得が速い。
		/// Ward法が実装された場合は、Ward法による重心が返される。
		/// </summary>
		/// <returns></returns>
		public IVector<T> GetCentroidVector()
		{
			if (this.centroidVector == null)
			{
				// セントロイド法的重心を求める。
				IVector<T> v = new MultiplyVector<T>(1.0 / this.ItemCount, new AddVector<T>(this.vectors));
			}
			return this.centroidVector;
		}


	}

}
