package slothLib.linearAlgebra.featureVector;

import java.util.ArrayList;
import java.util.List;

/**
 * 階層型クラスタリングを実際に行うクラス
 * [TODO]デバッグ用に例外を投げるところが残っている。
 */

public abstract class HierarchicalClusteringProcess<T> implements IHierarchicalClusteringProcess<T>{
	/**
	 * 距離の最長値
	 * </sIHierarchicalClusteringProcess<T>ummary>
	 */
	public final static double INFINITE_DISTANCE = Double.MAX_VALUE;


	/**
	 * クラスタリングされるアイテム群
	 */
	protected IVector<T>[] vectors;

	/**
	 * 距離テーブル
	 * 類似度の場合は類似度に-1かけたもの
	 */
	private double[][] distanceTable;

	/**
	 * 総アイテム数
	 */
	protected int itemCount;

	/**
	 * 距離テーブル作成に使う計算器
	 */
	protected ICalculatorScalarFromTwoVectors<T> calculator;

	/**
	 * 距離テーブルの計算器タイプ
	 */
	protected ClusteringDistanceType dType;

	/**
	 * 距離テーブルの計算器が距離ならば1、類似度ならば-1を保持する。
	 */
	protected double ds;

	/**
	 * 距離テーブルのアイテムが、現在どのクラスタに属しているか
	 */
	protected int[] clusterID;

	/**
	 * コンストラクタ
	 * @param vectors 
	 * @param calculator 
	 * @param dType 
	 */
	public HierarchicalClusteringProcess(IVector<T>[] vectors, ICalculatorScalarFromTwoVectors<T> calculator, ClusteringDistanceType dType)
	{

		this.vectors = vectors;
		this.itemCount = vectors.length;
		this.calculator = calculator;

		this.dType = dType;
		// 類似度の場合は、距離テーブルで-1を掛けた値を保持する。
		this.ds = 1;
		if (dType == ClusteringDistanceType.Similarity)
		{
			this.ds = -1;
		}

		// 距離テーブル
		this.distanceTable = createDistanceTable(vectors);

		this.clusterID = new int[itemCount];
		for (int i = 0; i < this.itemCount; i++)
		{
			// はじめは、要素数個のクラスタがある
			// クラスタになるときには、若い方のアイテムIDがクラスタ番号になる。
			this.clusterID[i] = i;
		}
	}


	/**
	 * クラスタを結合する
	 * @param cid1 
	 * @param cid2 
	 */
	protected abstract void union(int cid1, int cid2);

	/**
	 * クラスタが1つになるまでクラスタリングを行う
	 * @return 
	 */
	public HierarchicalClusteringResult<T> doClustering()
	{
		return this.doClustering(1, INFINITE_DISTANCE);
	}

	/**
	 * クラスタが指定された個数になるまでクラスタリングを行う
	 * @param thresholdClusterCount 
	 * @return 
	 */
	public HierarchicalClusteringResult<T> doClustering(int thresholdClusterCount)
	{
		return this.doClustering(thresholdClusterCount, INFINITE_DISTANCE);
	}

	/**
	 * クラスタリングを行う
	 * @param thresholdDistanceOrSimilarity クラスタリングを止める閾値となる距離。類似度の場合は類似度に-1かけたもの
	 * @return 
	 */
	public HierarchicalClusteringResult<T> doClustering(double thresholdDistanceOrSimilarity)
	{
		return this.doClustering(1, thresholdDistanceOrSimilarity * this.ds);
	}

	/**
	 * クラスタリングを行う
	 * @param thresholdClusterCount クラスタリングを止める閾値となるクラスタ数
	 * @param thresholdDistanceOrSimilarity クラスタリングを止める閾値となる距離。類似度の場合は類似度に-1かけたもの
	 * @return 
	 */
	public HierarchicalClusteringResult<T> doClustering(int thresholdClusterCount, double thresholdDistanceOrSimilarity)
	{
		// 最終個数は1以上ですよ。
		if (thresholdClusterCount < 1)
		{
			throw new IllegalArgumentException("クラスタの最終個数は1以上を指定してください。" + "thresholdClusterCount");
		}

		// 距離または類似度の閾値
		double thresholdDs = thresholdDistanceOrSimilarity;

		// 各アイテムの現状のクラスタID
		for (int i = 0; i < this.itemCount; i++)
		{
			// はじめは、要素数個のクラスタがある
			// クラスタになるときには、若い方のアイテムIDがクラスタ番号になる。
			this.clusterID[i] = i;
		}

		// 結果をおさめるやつ
		List<IDendrogramNode<T>> nodeList = new ArrayList<IDendrogramNode<T>>();

		// 現在のクラスタ個数
		int currentClusterCount = this.itemCount;

		// クラスタリング開始

		// クラスタがlimitNumより多い場合は繰り返し。
		while (currentClusterCount > thresholdClusterCount)
		{
			int cid1 = -1;
			int cid2 = -1;
			double minDistance = HierarchicalClusteringProcess.INFINITE_DISTANCE;

			// 現在のクラスタで最も近い奴を求める
			for (int i = 0; i < this.itemCount - 1; i++)
			{
				if (this.clusterID[i] != i)
				{
					// クラスタの代表ではないなら次へ
					continue;
				}
				for (int j = i + 1; j < this.itemCount; j++)
				{
					if (this.clusterID[j] != j)
					{
						// クラスタの代表ではないなら次へ
						continue;
					}

					if (this.distanceTable[i][j] < minDistance)
					{
						minDistance = this.distanceTable[i][j];
						cid1 = i;
						cid2 = j;
					}
				}
			}

			// 停止条件に引っかかっているか、結合可能なクラスタが存在しない場合はクラスタリングをやめる。
			if (minDistance > thresholdDs || (cid1 == -1 || cid2 == -1))
			{
				break;
			}

			// 結合候補を結合する。
			union(cid1, cid2);
			// 結果のノード一つを保持
			//nodeList.add(new DendrogramNode(cid1, cid2, minDistance));
			nodeList.add(getDendrogramNode(cid1, cid2, minDistance));
			currentClusterCount--;
		}

		// 結果。
		HierarchicalClusteringResult<T> result = new HierarchicalClusteringResult<T>((IDendrogramNode<T>[])(nodeList.toArray()), vectors, dType);
		return result;

	}

	/**
	 * デンドログラムのノードを取得する
	 * @param cid1 
	 * @param cid2 
	 * @param minDistance 
	 * @return 
	 */
	protected IDendrogramNode<T> getDendrogramNode(int cid1, int cid2, double minDistance)
	{
		return new DendrogramNode<T>(cid1, cid2, minDistance);
	}



	/**
	 * クラスタ間の距離を取得する
	 * @param cid1 
	 * @param cid2 
	 * @return 
	 */
	protected double getDistance(int cid1, int cid2)
	{
		if (cid1 < cid2)
		{
			return this.distanceTable[cid1][cid2];
		}
		else if (cid1 > cid2)
		{
			return this.distanceTable[cid2][cid1];
		}
		else
		{
			throw new RuntimeException("ここにアクセスしないような設計に変えてください。");
			// 到達できないのはわざとです。最終的には上記例外スルーは消します。
			// uncomment below 
			// return HierarchicalClusteringProcess.INFINITE_DISTANCE;
		}
	}

	/**
	 * クラスタ間の距離を設定する
	 * @param cid1 
	 * @param cid2 
	 * @param distance 
	 */
	protected void setDistance(int cid1, int cid2, double distance)
	{
		if (cid1 < cid2)
		{
			this.distanceTable[cid1][cid2] = distance;
		}
		else if (cid1 > cid2)
		{
			this.distanceTable[cid2][cid1] = distance;
		}
		else
		{
			throw new RuntimeException("ここにアクセスしないような設計に変えてください。");
			// 到達できないのはわざとです。最終的には上記例外スルーは消します。
			// uncomment below
			// this.distanceTable[cid1][cid2] = distance; 
		}
	}



	/**
	 * 距離テーブルを計算する
	 * @return 
	 */
	private double[][] createDistanceTable(IVector<T>[] vectors)
	{
		/*
		// 類似度の場合は、距離テーブルで-1を掛けた値を保持する。
		double ds = 1;
		if (this.dType == ClusteringDistanceType.Similarity)
		{
			ds = -1;
		}
		*/

		// アイテム数
		int count = vectors.length;

		// 距離テーブル
		double[][] distanceTable = new double[count][count];


		// 距離テーブル計算
		for (int i = 0; i < count - 1; i++)
		{
			for (int j = i + 1; j < count; j++)
			{
				distanceTable[i][j] = this.calculator.doCalculate(vectors[i], vectors[j]) * this.ds;
			}
		}

		return distanceTable;
	}


}
