package slothLib.linearAlgebra.vector;

public class VectorOperator {
	private static void CheckDimension(IVector vector1, IVector vector2)
	{
		if (vector1.dimension() != vector2.dimension())
		{
			throw new IllegalArgumentException("2つのベクトルの次元は同じでなくてはいけません。");
		}
	}

	// vertor -> scalar 型演算
	// 最大値ノルム（スープノルム）

	/**
	 * ベクトルの要素のうち、絶対値の最大値を返す。
	 */

	public static double getMaxNorm(IVector vector)
	{
		double d = 0.0;
		for (int i = 0; i < vector.dimension(); i++)
		{
			if (d < Math.abs(vector.get(i)))
			{
				d = Math.abs(vector.get(i));
			}
		}
		return d;
	}

	// p-ノルム

	/**
	 * ベクトルの全ての要素をp乗したものの和を、1/p乗した値を返す。
	 * @param vector 計算対象のベクトル
	 * @param p パラメータ。1以上でなければならない。
	 * @return 
	 */

	public static double getPNorm(IVector vector, double p)
	{
		if (p < 1)
			throw(new IllegalArgumentException("pは1より大きい必要があります。"));

		double d = 0.0;
		for (int i = 0; i < vector.dimension(); i++)
		{
			d += Math.pow(vector.get(i), p);
		}

		return Math.pow(d, 1/p);
	}

	// ユークリッド距離（単項）

	/**
	 * 原点からのユークリッド距離を計算する。
	 * @param vector 
	 * @return 
	 */
	public static double getEuclideanDistance(IVector vector)
	{
		double d = 0.0;
		for (int i = 0; i < vector.dimension(); i++)
		{
			d = d + Math.pow(vector.get(i), 2.0);
		}
		return Math.sqrt(d);
	}

	// vector * vector -> scalar 型演算
	// ユークリッド距離（二項）

	/**
	 * 2つのベクトル間のユークリッド距離を計算する。
	 * @param vector1 
	 * @param vector2 
	 * @return 
	 */
	public static double getEuclideanDistance(IVector vector1, IVector vector2)
	{
		CheckDimension(vector1, vector2);
		double d = 0.0;
		for (int i = 0; i < vector1.dimension(); i++)
		{
			d = d + Math.pow(vector1.get(i) - vector2.get(i), 2.0);
		}
		return Math.sqrt(d);
	}

	// マンハッタン距離

	/**
	 * マンハッタン距離（格子距離）を返す。
	 * すなわち、各要素の差の絶対値の総和。
	 */

	public static double getManhattanDistance(IVector vector1, IVector vector2)
	{
		CheckDimension(vector1, vector2);
		double d = 0.0;
		for (int i = 0; i < vector1.dimension(); i++)
		{
			d = d + Math.abs(vector1.get(i) - vector2.get(i));
		}
		return d;
	}


	// 内積

	/**
	 * 2つのベクトル間の内積を計算する。
	 * @param vector1 
	 * @param vector2 
	 * @return 
	 */
	public static double getInnerProduct(IVector vector1, IVector vector2)
	{
		CheckDimension(vector1, vector2);
		double ip = 0.0;
		for (int i = 0; i < vector1.dimension(); i++)
		{
			ip += vector1.get(i) * vector2.get(i);
		}
		return ip;
	}

	
	// コサイン

	/**
	 * 2つのベクトル間のコサイン値を計算する。
	 * @param vector1 
	 * @param vector2 
	 * @return 
	 */
	public static double getCosine(IVector vector1, IVector vector2)
	{
		CheckDimension(vector1, vector2);
		double ip = getInnerProduct(vector1, vector2);
		double cos = ip / (getEuclideanDistance(vector1) * getEuclideanDistance(vector2));
		return cos;
	}

	
	// Jaccard係数

	/**
	 * Jaccard係数を返す。具体的には、次の式により計算される。
	 * Jaccard = 内積(vector1,vector2) / (|vector1|^2 + |vector2|^2 - 内積(vector1,vector2)) 
	 */

	public static double getJaccardCoefficient(IVector vector1, IVector vector2)
	{
		double v1 = 0, v2 = 0;
		double inner = getInnerProduct(vector1, vector2);

		CheckDimension(vector1, vector2);

		for (int i = 0; i < vector1.dimension(); i++)
		{
			v1 += vector1.get(i) * vector1.get(i);
			v2 += vector2.get(i) * vector2.get(i);
		}

		return inner / (v1 + v2 - inner);
	}


	// Dice係数

	/**
	 * Dice係数を返す。具体的には、次の式により計算される。
	 * Dice = 2 * 内積(vector1,vector2) / (|vector1|^2) / (|vector2|^2)
	 */

	public static double getDiceCoefficient(IVector vector1, IVector vector2)
	{
		double v1 = 0, v2 = 0;

		CheckDimension(vector1, vector2);

		for (int i = 0; i < vector1.dimension(); i++)
		{
			v1 += vector1.get(i) * vector1.get(i);
			v2 += vector2.get(i) * vector2.get(i);
		}

		return 2 * getInnerProduct(vector1, vector2) / v1 / v2;
	}



	// vector * vector -> vector 型演算
	// 和

	/**
	 * ベクトルの加算
	 * @param vector1 
	 * @param vector2 
	 * @return 
	 */
	public static IVector add(IVector vector1, IVector vector2)
	{
		CheckDimension(vector1, vector2);

		IVector r = (IVector)vector1.clone();

		for (int i = 0; i < vector2.dimension(); i++)
		{
			r.set(i, r.get(i) + vector2.get(i));
		}

		return r;
	}

	
	// 差

	/**
	 * ベクトルの減算
	 * @param vector1 
	 * @param vector2 
	 * @return 
	 */
	public static IVector subtract(IVector vector1, IVector vector2)
	{
		CheckDimension(vector1, vector2);

		return add(vector1, negate(vector2));
	}

	// vector -> vector 型演算
	//負

	/**
	 * ベクトルの負
	 * @param vector 
	 * @return 
	 */
	public static IVector negate(IVector vector)
	{
		AbstractVector r = (AbstractVector)vector.clone();

		for (int i = 0; i < vector.dimension(); i++)
		{
			r.set(i, -vector.get(i));
		}

		return r;
	}

	// scalar * vector -> vector 型演算
	// 定数ｘベクトル

	/**
	 * ベクトルの定数倍
	 * @param scalar 
	 * @param vector 
	 * @return 
	 */
	static public IVector multiply(double scalar, IVector vector) {
		IVector r = (IVector)vector.clone();

		for (int i = 0; i < vector.dimension(); i++)
		{
			r.set(i, scalar * r.get(i));
		}

		return r;
	}

}
