package slothLib.web.search;

import java.net.Proxy;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import slothLib.portUtil.PortUtil;
/**
 * Yahoo!動画検索Webサービス
 * http://developer.yahoo.co.jp/search/video/V1/videoSearch.html
 */
public class YahooJpVideoSearch implements IVideoSearch
{
	
	// プライベートフィールド
	
	/**
	 * アプリケーションID
	 */
	private String applicationID;
	
	private SearchType type;
	private SearchFormat format;
	private boolean adultOk;
	private String[] site;
	private Proxy proxy;
	
	//結果が繰り返さないようにするために必要
	Map<String, Boolean> alreadyGotURL = new HashMap<String, Boolean>();

	

		// コンストラクタ

		// / <summary>
		// / コンストラクタ アプリケーションIDを指定する
		// / </summary>
		// / <param name="applicationID">プログラムで用いるアプリケーションID</param>

		public YahooJpVideoSearch(String applicationID)
		{
			this.applicationID = applicationID;

			this.type = SearchType.all;
			this.format = SearchFormat.any;
			this.adultOk = false;
			this.site = null;
		}




		// public メソッド

		// doSearch

		/**
		 * Yahoo!動画検索を実行する
		 * @param query 検索クエリ
		 * @param resultNum 検索結果取得数
		 * @return YahooVideoSearchResultの検索結果
		 */
		public YahooJpVideoSearchResult doSearch(String query, int resultNum) throws WebSearchException
		{
            return doSearch(query, resultNum, 1);
		}

		/**
		 * Yahoo!動画検索を実行する
		 * @param query 検索クエリ
		 * @param resultNum 検索結果取得数
		 * @param start 検索開始の先頭位置　デフォルトは1
         * @return YahooVideoSearchResultの検索結果
		 */
		public YahooJpVideoSearchResult doSearch(String query, int resultNum, int start) throws WebSearchException
		{
            return doSearchOver(query, resultNum, start);
		}



		/**
		 * 検索HIT数のみを取得するメソッド
		 * @param query 検索クエリ
		 * @return HIT数
		 */
        public long getTotalNumber(String query) throws WebSearchException
        {
            return doSearch(query, 10).getTotalResultsAvailable();
        }



		// private メソッド

		/**
		 * 50件以内の検索結果を行う
		 * @param query 検索クエリ
		 * @param results 検索結果取得数
		 * @param start 検索開始の先頭位置
		 * @return YahooVideoSearchResult型の検索結果
		 */
		private YahooJpVideoSearchResult doSearchOriginal(String query, int results, int start) throws WebSearchException
		{
			String requestURL = MakeRequestURL(query, this.type, results, start, this.format, this.adultOk, this.site);
			//System.Diagnostics.Debug.WriteLine(requestURL);

			Document xmlDoc = PortUtil.getDocumentFromURL(requestURL, this.proxy);

			// ルートの要素を取得
			Element xmlRoot = xmlDoc.getDocumentElement();

			
			// <Result>要素のtotalResultsAvailableの属性値を取得
			long totalResultsAvailable = Long.parseLong(xmlRoot.getAttribute("totalResultsAvailable"));
			int totalResultsReturned = Integer.parseInt(xmlRoot.getAttribute("totalResultsReturned"));
			int firstResultPosition = Integer.parseInt(xmlRoot.getAttribute("firstResultPosition"));

			// Resultを入れていく
			List<YahooJpVideoElement> resultElementList = new ArrayList<YahooJpVideoElement>();

			NodeList xmlResultList = xmlRoot.getElementsByTagName("Result");
			int rank = 1;

            String firstURL = getElementString(((Element)xmlResultList.item(0)).getElementsByTagName("Url"));
            if (alreadyGotURL.containsKey(firstURL))
            {
                return new YahooJpVideoSearchResult(query, 0, 0, -1, new YahooJpVideoElement[0]);
            }
            else
            {
                alreadyGotURL.put(firstURL, true);
            }

			for (int i = 0; i < xmlResultList.getLength(); i++)
			{
				Element xmlResult = (Element)xmlResultList.item(i);
				String title = getElementString(xmlResult.getElementsByTagName("Title"));
				String summary = getElementString(xmlResult.getElementsByTagName("Summary"));
				String url = getElementString(xmlResult.getElementsByTagName("Url"));
				String clickUrl = getElementString(xmlResult.getElementsByTagName("ClickUrl"));
				String refererUrl = getElementString(xmlResult.getElementsByTagName("RefererUrl"));
				String fileSize = getElementString(xmlResult.getElementsByTagName("FileSize"));
				String fileFormat = getElementString(xmlResult.getElementsByTagName("FileFormat"));
				String height = getElementString(xmlResult.getElementsByTagName("Height"));
				String width = getElementString(xmlResult.getElementsByTagName("Width"));
				String duration = getElementString(xmlResult.getElementsByTagName("Duration"));
				String streaming = getElementString(xmlResult.getElementsByTagName("Streaming"));
				String channels = getElementString(xmlResult.getElementsByTagName("Channels"));
				String restrictions = getElementString(xmlResult.getElementsByTagName("Restrictions"));
				String thumbnailUrl = "";
				String thumbnailHeight = "";
				String thumbnailWidth = "";
				String publisher = getElementString(xmlResult.getElementsByTagName("Publisher"));
				String copyright = getElementString(xmlResult.getElementsByTagName("Copyright"));

				NodeList xmlCacheNode = xmlResult.getElementsByTagName("Thumbnail");
				if (xmlCacheNode != null)
				{
					for (int j = 0; j < xmlCacheNode.getLength(); j++)
					{
						Element xmlCacheElement = (Element)xmlCacheNode.item(j);
						thumbnailUrl = getElementString(xmlCacheElement.getElementsByTagName("Url"));
						thumbnailHeight = getElementString(xmlCacheElement.getElementsByTagName("Height"));
						thumbnailWidth = getElementString(xmlCacheElement.getElementsByTagName("Width"));
					}
				}
				YahooJpVideoElement result
					= new YahooJpVideoElement(rank, title, summary, url, clickUrl, refererUrl, fileSize, fileFormat,
					height, width, duration, channels, streaming, thumbnailUrl, thumbnailHeight, thumbnailWidth, publisher
					, restrictions, copyright);
				resultElementList.add(result);
				rank++;

			}

			return new YahooJpVideoSearchResult(query, totalResultsAvailable, totalResultsReturned, firstResultPosition, resultElementList.toArray(new YahooJpVideoElement[0]));
		}

		/**
		 * 50件以上の検索を行う
		 * @param query 検索クエリ
		 * @param results 検索結果取得数
		 * @param start 検索開始の先頭位置
		 * @return YahooJpVideoSearchResult型の検索結果
		 */
		private YahooJpVideoSearchResult doSearchOver(String query,int results, int start) throws WebSearchException
		{
			int i;
			int loop = (results - 1) / 50;
            //List<ISearchResultElement> result = new List<ISearchResultElement>();
            List<YahooJpVideoElement> result = new ArrayList<YahooJpVideoElement>();
            long totalResultsAvailable = 0;
			YahooJpVideoSearchResult r;

			for (i = 0; i < loop; i++)
			{
				r = doSearchOriginal(query, 50, i * 50 + 1);
                if (r.getFirstResultPosition() == -1) { break; }
                totalResultsAvailable = r.getTotalResultsAvailable();
                result.addAll(PortUtil.toList(r.getResultElements()));
			}
			r = doSearchOriginal(query, results - (loop * 50), loop * 50 + 1);
            //result.AddRange(r);
            result.addAll(PortUtil.toList(r.getResultElements()));
            if (r.getFirstResultPosition() == -1)
            {
            }
            else
            {
                totalResultsAvailable = r.getTotalResultsAvailable();
            }

			return new YahooJpVideoSearchResult(query, totalResultsAvailable, result.size(), 1, result.toArray(new YahooJpVideoElement[0]));
		}




		// 雑用メソッド


		/**
		 * 例外を投げる
		 * @param errorCode 
		 * @param errorMessage 
		 */
//		private void ThrowException(HttpStatusCode errorCode, String errorMessage)
//		{
//			switch (errorCode)
//			{
//				case HttpStatusCode.BadRequest: // 400
//					throw new YahooJpSearchException(YahooJpSearchException.HttpCode.BadRequest, errorMessage);
//				case HttpStatusCode.Forbidden: // 403
//					throw new YahooJpSearchException(YahooJpSearchException.HttpCode.Forbidden, errorMessage);
//				case HttpStatusCode.ServiceUnavailable: // 503
//					throw new YahooJpSearchException(YahooJpSearchException.HttpCode.ServiceUnavailable, errorMessage);
//				default:
//					throw new Exception("YahooWebSearchで想定外のHTTPエラーが発生しました。（エラーコード: " + (int)errorCode + "）" + errorMessage);
//
//			}
//		}

		/**
		 * NodeListの初めのノードのテキストを取得する
		 * @param nodeList NodeList
		 * <returns>NodeListの初めのノードのInnerText
		 *          NodeListが空であれば空文字列を返す</returns>
		 */
		private String getElementString(NodeList nodeList)
		{
			if (nodeList.getLength() == 0)
			{
				return "";
			}
			else
			{
				return nodeList.item(0).getTextContent();
			}
		}

		/**
		 * リクエストURLを作成する
		 * @param query 検索クエリ
		 * @param type 指定検索の種類
		 * @param results 検索結果取得数
		 * @param start 検索開始の先頭位置
		 * @param format 検索するファイルの種類　デフォルトはany
		 * @param adultOk アダルトコンテンツの検索結果を含めるかどうか　デフォルトはfalse
		 * <param name="site">検索するドメイン（例えば www.yahoo.co.jp）を制限します。30ドメインまで指定することができます
		 * デフォルトはnull</param>
		 * @return URL
		 */
		private String MakeRequestURL(String query, SearchType type, int results, int start,
			SearchFormat format, boolean adultOk, String[] site)
		{
			String strType = "";
			String strFormat = "";
			String strAdult = "";
			String strSite = "";

			switch (type)
			{
				case all:
					break;
				case any:
					strType = "&type=any";
					break;
				case phrase:
					strType = "&type=phrase";
					break;
			}

			switch (format)
			{
				case any:
					break;
				case avi:
					strFormat = "&format=avi";
					break;
				case flash:
					strFormat = "&format=flash";
					break;
				case mpeg:
					strFormat = "&format=mpeg";
					break;
				case msmedia:
					strFormat = "&format=msmedia";
					break;
				case quicktime:
					strFormat = "&format=quicktime";
					break;
				case realmedia:
					strFormat = "&format=realmedia";
					break;
			}

			if (adultOk == true)
			{
				strAdult = "&adulat_ok=1";
			}

			if (site != null)
			{
				if (site.length > 30)
				{
					throw new IllegalArgumentException("siteに指定できるドメインは30個までです" + "site");
				}
				else
				{
					for (String s: site)
					{
						strSite += "&site=" + s;
					}
				}
			}

			String requestURL = "http://api.search.yahoo.co.jp/VideoSearchService/V1/videoSearch?"
			+ "appid=" + this.applicationID
			+ "&query=" + query
			+ strType
			+ "&results=" + results
			+ "&start=" + start
			+ strFormat
			+ strAdult
			+ strSite;
			
			return PortUtil.absoluteUri(requestURL);
		}




		// 引数で指定するための列挙型

		/**
		 * 指定検索の種類
		 */
		public enum SearchType
		{
			/**
			 * デフォルト
			 */
			all,
			/**
			 * 

			 */
			any,
			/**
			 * 

			 */
			phrase
		}

		/**
		 * 検索するファイルの種類 デフォルトはany
		 */
		public enum SearchFormat
		{
			/**
			 * デフォルト
			 */
			any,
			/**
			 * 

			 */
			avi,
			/**
			 * 

			 */
			flash,
			/**
			 * 

			 */
			mpeg,
			/**
			 * 

			 */
			msmedia,
			/**
			 * 

			 */
			quicktime,
			/**
			 * 

			 */
			realmedia
		}




		// プロパティ

		/**
		 * 指定検索の種類 デフォルトはall
		 */
		public void setType(SearchType value)
		{
			this.type = value;
		}

		/**
		 * 検索するファイルの種類　デフォルトはany
		 */
		public void setFormat(SearchFormat value) 
		{
			this.format = value;
		}

		/**
		 * アダルトコンテンツの検索結果を含めるかどうかを指定します。デフォルトはfalse
		 */
		public void setAdultOk(boolean value)
		{
			this.adultOk = value;
		}

		/**
		 * >検索するドメイン（例えば www.yahoo.co.jp）を制限します。30ドメインまで指定することができます。デフォルトはnull
		 */
		public void setSite(String[] value)
		{
			this.site = value;
		}

        /**
         * プロクシを取得・設定する。
         */
        public void setProxy(String value)
        {
        	if (PortUtil.isNullOrEmpty(value))
        	{
        		this.proxy = null;
        	}
        	else
        	{
        		this.proxy = PortUtil.createProxy(value);
        	}
        }
        public String getProxy()
        {
                return PortUtil.getProxyURL(this.proxy);
        }





		// IVideoSearch メンバ

//		IVideoSearchResult IVideoSearch.doSearch(String query, int resultNum)
		//{
//			return this.doSearch(query, resultNum);
		//}




		// ISearch メンバ

		//ISearchResult ISearch.doSearch(String query, int resultNum)
		//{
//			return this.doSearch(query, resultNum);
		//}



}
