package slothLib.web.search;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

/**
 * 検索結果のキャッシュに関するクラス
 */
public class CachedSearch implements ISearch
{
	// privateフィールド
	
	
	/**
	 * キャッシュを保存するディレクトリのパス
	 */
	String cacheDir;
	
	/** 
	 * 実際にたたく検索エンジン 
	 * */
	ISearch searchEngine;
	

	// コンストラクタ
	
	/**
	 * コンストラクタ
	 * @param cacheDir キャッシュファイルのディレクトリ
	 * @param searchEngine 実際の検索に用いる検索エンジンのインスタンス
	 */
	public CachedSearch(String cacheDir, ISearch searchEngine)
	{
		if (cacheDir.charAt(cacheDir.length() - 1) != '\\')
		{
			// cacheDirの最後には必ず\がつくように変更する。
			cacheDir = cacheDir + '\\';
		}
		
		File di = new File(cacheDir);
		if (!di.exists())
		{
			// 指定されたディレクトリが無いらしいので作る。
			di.mkdirs();
		}
		
		this.cacheDir = cacheDir;
		this.searchEngine = searchEngine;
	}
	
	/**
	 * コンストラクタ
	 * @param searchEngine 実際の検索に用いる検索エンジンのインスタンス
	 */
	public CachedSearch(ISearch searchEngine) 
	{
		this(System.getProperty("java.io.tmpdir") + "\\cache\\", searchEngine);
	}
	
//	#endregion
	
	
	
	/**
	 * 検索を行う。
	 * @param query 
	 * @param resultNum 
	 * @return 
	 */
	public ISearchResult doSearch(String query, int resultNum) throws WebSearchException
	{
		// ファイルが保存される場所
		String filePath = this.cacheDir + this.searchEngine.getClass().getName() + "_" + query.hashCode() + "_" + resultNum + ".dat";
		
		CachedSearchResult result = this.LoadResult(filePath);
		if (result == null)
		{
			ISearchResult realResult = this.searchEngine.doSearch(query, resultNum);
			result = new CachedSearchResult(realResult);
			this.saveResult(result, filePath);
		}
		return result;
	}


	/**
	 * 検索結果をキャッシュとしてセーブする。
	 * @param searchResult 
	 * @param filePath 
	 */
	private void saveResult(CachedSearchResult searchResult, String filePath)
	{
		ObjectOutputStream stream = null;
		try {
			stream = new ObjectOutputStream(new FileOutputStream(filePath));
			stream.writeObject(searchResult);

		} catch (IOException e){
//			例外は投げない。
		} finally{
			if (stream != null)
				try {
					stream.close();
				} catch (IOException e) {
					//ignore
				}
		}
	}


	/**
	 * キャッシュされた検索結果をロードする。
	 * @param filePath 
	 */
	private CachedSearchResult LoadResult(String filePath)
	{
		if ((new File(filePath)).exists())
		{
			// 例外処理適当
			ObjectInputStream stream = null;
			try{
				stream = new ObjectInputStream(new FileInputStream(filePath));
				CachedSearchResult result = (CachedSearchResult)(stream.readObject()); 

				return result;
					
			} catch (IOException e){
				(new File(filePath)).delete();
				
				//				例外は投げない。
			} catch (ClassNotFoundException e) {
				(new File(filePath)).delete();
				//				例外は投げない。
			} finally{
				if (stream != null)
					try {
						stream.close();
					} catch (IOException e) {
						//ignore
					}
			}
		}
		// ファイルが存在しないときやロードに失敗した場合はnullを返す。
		return null;
	}
	
	// ISearFileInputStreamch メンバ
	
	/**
	 * キャッシュされたファイルの検索を行う
	 * @param query 検索クエリ
	 * @param resultNum 検索数
         * @return 検索結果
	 */
//		ISearchResult doSearch(string query, int resultNum)
//		{
//			return this.doSearch(query, resultNum);
//		}

//		#endregion
	

}
