/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: Dir.cpp,v 1.16 2004/05/15 11:17:50 randy Exp $
 */

#ifdef __WIN32__
#include <dir.h>
#endif
 
#ifndef __WIN32__

#include <unistd.h>

#endif /* __WIN32__ */

#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#ifdef _MBCS
#include <direct.h>
#else
#include <dirent.h>
#endif
#include "soopy.h"


SpValue& SpDir::toString()
{
    SpString* str;

    str = new SpString();
    *str = "#<dir>";
    //    static SpValue s;
    //    s.setNewObject(str);
    //    return s;
    return SpObjectResult(str);
}

SpValue& SpDir::onMessage(SpValue& rec, SpValue& msg)
{
    return DirMsgHandler(rec, msg);
}

// primitives

SpValue& SpDir::Chdir(SpValue&, SpValue& path)
{
    if(!path.isString()){
        throw SpException("not string (Dir.chdir)");
    }
#ifdef _MBCS
    if(_chdir(path.toCStringWithEncoder()) != 0){
#else
    if(chdir(path.toCStringWithEncoder()) != 0){
#endif
        throw SpException("can't change dir");
    }
    return TrueObject;
}

SpValue& SpDir::Mkdir(SpValue&, SpValue& path)
{
    if(!path.isString()){
        throw SpException("not string (Dir.mkdir)");
    }
#ifdef __WIN32__
    if(mkdir(path.toCStringWithEncoder()) != 0){
#else
    if(mkdir(path.toCStringWithEncoder(), 0777) != 0){
#endif
        if(errno == EACCES){
            throw SpException("can't make dir. permission denied.");
        }
        if(errno == ENOENT){
            throw SpException("can't make dir. no such a entry.");
        }
        throw SpException("can't make dir.");
    }
    return TrueObject;
}

SpValue& SpDir::Rmdir(SpValue&, SpValue& path)
{
    if(!path.isString()){
        throw SpException("not string (Dir.rmdir)");
    }
    if(rmdir(path.toCStringWithEncoder()) != 0){
        if(errno == EACCES){
            throw SpException("can't rmdir. permission denied.");
        }
        if(errno == ENOENT){
            throw SpException("can't rmdir. no such a directory.");
        }
        throw SpException("can't remove dir.");
    }
    return TrueObject;
}

SpValue& SpDir::List(SpValue&, SpValue& path)
{
    if(!path.isString()){
        throw SpException("not string (Dir.list)");
    }
    DIR* dir = opendir(path.toCStringWithEncoder());
    struct dirent* ent;
    if(dir == NULL){
        if(errno == ENOENT){
            throw SpException("can't open directory. no such a directory. (Dir.list)");
        }
        throw SpException("can't open dir. (Dir.list)");
    }
    ent = readdir(dir);
    if(ent == NULL){
        return NilObject;
    }
    SpValue temp(new SpString(ent->d_name));
    SpCons* head = new SpCons(temp);
    SpCons* ptr = head;
    try{
        while((ent = readdir(dir)) != NULL){
            temp.setNewObject(new SpString(ent->d_name));
            ptr->append(temp);
            ptr = (SpCons*)ptr->next();
        }
    }catch(...){
        closedir(dir);
        throw;
    }
    closedir(dir);

    //    static SpValue result;
    //    result.setNewObject(head);
    //    return result;
    return SpObjectResult(head);
}

SpValue& SpDir::Pwd(SpValue&)
{
    char buf[MAXPATH];
    if(getcwd(buf, MAXPATH) == NULL){
        throw SpException("can't get working directory name");
    }
    SpString* str = new SpString(buf);
    //    static SpValue result;
    //    result.setNewObject(str);
    //    return result;
    return SpObjectResult(str);
}

// Dir.glob
SpValue& SpDir::prim_glob(SpValue&, SpValue& p)
{
    if(!p.isString()){
        throw SpException("not string (Dir.glob)");
    }
    SpString* str = p.asString();
    SpTuple* tuple = str->split_path_file();
    SpValue taker(tuple);
    SpString* path = (*tuple)[0].asString();
    SpString* glob = (*tuple)[1].asString();
    DIR* dir = opendir(path->toCStringWithEncoder());
    struct dirent* ent;
    if(dir == NULL){
        if(errno == ENOENT){
            throw SpException("can't open directory. no such a directory. (Dir.glob)");
        }
        throw SpException("can't open dir. (Dir.glob)");
    }
    SpValue temp;
    SpCons* head = NULL;
    SpCons* ptr = NULL;
    try{
        while((ent = readdir(dir)) != NULL){
            SpString* fname;
            SpValue taker;
#ifdef EUC_JP
            fname = new SpString(ent->d_name, *eucjpin);
#else
            fname = new SpString(ent->d_name, *sjisin);
#endif
            taker.setNewObject(fname);
            if(fname->match_glob(glob)){
                if(head == NULL){
                    temp.setNewObject(new SpString(ent->d_name));
                    head = new SpCons(temp);
                    ptr = head;
                }else{
                    temp.setNewObject(new SpString(ent->d_name));
                    ptr->append(temp);
                    ptr = (SpCons*)ptr->next();
                }
            }
        }
    }catch(...){
        closedir(dir);
        throw;
    }
    closedir(dir);

    if(head == NULL){
        return NilObject;
    }
    //    static SpValue result;
    //    result.setNewObject(head);
    //    return result;
    return SpObjectResult(head);
}



// init Message Handler
void SpDir::init()
{
    SpValue ObjDir(new SpDir);
    PMainNameSpace->internConst(SymDir, ObjDir);

    SpValue SymChdir(new SpSymbol("chdir"));
    SpValue PrimChdir(new SpPrim2(Chdir));
    DirMsgHandler.append(SymChdir, PrimChdir);

    SpValue SymMkdir(new SpSymbol("mkdir"));
    SpValue PrimMkdir(new SpPrim2(Mkdir));
    DirMsgHandler.append(SymMkdir, PrimMkdir);

    SpValue SymRmdir(new SpSymbol("rmdir"));
    SpValue PrimRmdir(new SpPrim2(Rmdir));
    DirMsgHandler.append(SymRmdir, PrimRmdir);

    SpValue SymList(new SpSymbol("list"));
    SpValue PrimList(new SpPrim2(List));
    DirMsgHandler.append(SymList, PrimList);

    SpValue SymPwd(new SpSymbol("pwd"));
    SpValue PrimPwd(new SpPrim1(Pwd));
    DirMsgHandler.append(SymPwd, PrimPwd);

    SpValue SymGlob(new SpSymbol("glob"));
    SpValue PrimGlob(new SpPrim2(prim_glob));
    DirMsgHandler.append(SymGlob, PrimGlob);
}

