/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002-2004 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: File.cpp,v 1.28 2004/05/18 12:47:17 randy Exp $
 */

#include <sys/stat.h>
#include <time.h>

#ifdef __WIN32__

#include <io.h>

#else

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>

#endif /* __WIN32__ */

#ifdef _MBCS
#include <direct.h>
#else
#include <dirent.h>
#endif

#include "soopy.h"


SpValue& SpFile::toString()
{
    return SpObjectResult(new SpString("#<file util>"));
}

SpValue& SpFile::onMessage(SpValue& rec, SpValue& msg)
{
    return FileMsgHandler(rec, msg);
}

// primitives

SpValue& SpFile::prim_exists(SpValue&, SpValue& path)
{
    if(!path.isString()){
        throw SpException("not string (File.exists)");
    }
    char* fname = (char*)path.toCStringWithEncoder();
    if(access(fname, 04) == 0){
        return TrueObject;
    }
    return FalseObject;
}

SpValue& SpFile::prim_delete(SpValue&, SpValue& path)
{
    if(!path.isString()){
        throw SpException("not string (File.delete)");
    }
    char* fname = (char*)path.toCStringWithEncoder();
    if(remove(fname) == 0){
        return TrueObject;
    }
    return FalseObject;
}

SpValue& SpFile::prim_size(SpValue&, SpValue& path)
{
    struct stat statbuf;

    if(!path.isString()){
        throw SpException("not string (File.size)");
    }
    char* fname = (char*)path.toCStringWithEncoder();
    if(stat(fname, &statbuf) != 0){
        throw SpException("can't get file status");
    }

    return SpIntResult(statbuf.st_size);
}

SpValue& SpFile::prim_rename(SpValue&, SpValue& from, SpValue& to)
{
    if(!from.isString()){
        throw SpException("not string (File.rename)");
    }
    if(!to.isString()){
        throw SpException("not string (File.rename)");
    }
    char oldname[SP_NAME_MAX];
    strncpy(oldname, (char*)from.toCStringWithEncoder(), SP_NAME_MAX-2);
    oldname[SP_NAME_MAX-2] = '\0';
    oldname[SP_NAME_MAX-1] = '\0';
    char* newname = (char*)to.toCStringWithEncoder();
    if(rename(oldname, newname) == 0){
        return TrueObject;
    }
    return FalseObject;
}

SpValue& SpFile::prim_atime(SpValue&, SpValue& path)
{
    SpInt y, m, d, h, min, s;
    struct tm* area;
    struct stat statbuf;

    if(!path.isString()){
        throw SpException("not string (File.atime)");
    }
    char* fname = (char*)path.toCStringWithEncoder();
    if(stat(fname, &statbuf) != 0){
        throw SpException("can't get file status");
    }

    area = localtime(&statbuf.st_atime);
    y = area->tm_year + 1900;
    m = area->tm_mon+1;
    d = area->tm_mday;
    h = area->tm_hour;
    min = area->tm_min;
    s = area->tm_sec;

    return SpObjectResult(new SpDateTime(y, m, d, h, min, s));
}

SpValue& SpFile::prim_ctime(SpValue&, SpValue& path)
{
    SpInt y, m, d, h, min, s;
    struct tm* area;
    struct stat statbuf;

    if(!path.isString()){
        throw SpException("not string (File.ctime)");
    }
    char* fname = (char*)path.toCStringWithEncoder();
    if(stat(fname, &statbuf) != 0){
        throw SpException("can't get file status");
    }

    area = localtime(&statbuf.st_ctime);
    y = area->tm_year + 1900;
    m = area->tm_mon+1;
    d = area->tm_mday;
    h = area->tm_hour;
    min = area->tm_min;
    s = area->tm_sec;

    return SpObjectResult(new SpDateTime(y, m, d, h, min, s));
}

SpValue& SpFile::prim_mtime(SpValue&, SpValue& path)
{
    SpInt y, m, d, h, min, s;
    struct tm* area;
    struct stat statbuf;

    if(!path.isString()){
        throw SpException("not string (File.mtime)");
    }
    char* fname = (char*)path.toCStringWithEncoder();
    if(stat(fname, &statbuf) != 0){
        throw SpException("can't get file status");
    }

    area = localtime(&statbuf.st_mtime);
    y = area->tm_year + 1900;
    m = area->tm_mon+1;
    d = area->tm_mday;
    h = area->tm_hour;
    min = area->tm_min;
    s = area->tm_sec;

    return SpObjectResult(new SpDateTime(y, m, d, h, min, s));
}


// init Message Handler
void SpFile::init()
{
    SpValue SymFile(new SpSymbol("file"));
    SpValue ObjFile(new SpFile);
    PMainNameSpace->internConst(SymFile, ObjFile);

    SpValue SymExists(new SpSymbol("exist?"));
    SpValue PrimExists(new SpPrim2(prim_exists));
    FileMsgHandler.append(SymExists, PrimExists);

    SpValue SymDelete(new SpSymbol("delete"));
    SpValue PrimDelete(new SpPrim2(prim_delete));
    FileMsgHandler.append(SymDelete, PrimDelete);

    SpValue SymSize(new SpSymbol("size"));
    SpValue PrimSize(new SpPrim2(prim_size));
    FileMsgHandler.append(SymSize, PrimSize);

    SpValue SymRename(new SpSymbol("rename"));
    SpValue PrimRename(new SpPrim3(prim_rename));
    FileMsgHandler.append(SymRename, PrimRename);

    SpValue PrimAtime(new SpPrim2(prim_atime));
    FileMsgHandler.append(SymAtime, PrimAtime);

    SpValue PrimCtime(new SpPrim2(prim_ctime));
    FileMsgHandler.append(SymCtime, PrimCtime);

    SpValue PrimMtime(new SpPrim2(prim_mtime));
    FileMsgHandler.append(SymMtime, PrimMtime);
}

