/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002,2003 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: ISO2022JP.cpp,v 1.9 2004/05/22 05:22:34 randy Exp $
 */

#include "soopy.h"

/*
 * Class ISO2022JPReadEncoder
 */

SpChar ISO2022JPReadEncoder::readCh()
{
    SpChar c;
    int ch, ku, ten;

    if(count >= 0){
        return buf[count--];
    }
    if(eof()){ return (SpChar)NULL; }
    c = reader->ReadChar();
    if(isEOF(c)){ return (SpChar)NULL; }

    if(!EqualCharCode(c, CodeJIS)){
        return c;
    }

    ch = SpCharGetChar(c);
    ku = (ch >> 8);
    ten = (ch & 0xFF);
    if(ku == 0){
        if(out_now != kCodeASCII){
            out_now = kCodeASCII;
            buf[3] = 0x1B;
            buf[2] = 0x28;
            buf[1] = 0x42;
            buf[0] = ten;
            count = 3;
        }else{
            buf[0] = ten;
            count = 0;
        }
    }else{
        if(out_now != kCodeJISX0208){
            out_now = kCodeJISX0208;
            buf[4] = 0x1B;
            buf[3] = 0x24;
            buf[2] = 0x42;
            buf[1] = ku;
            buf[0] = ten;
            count = 4;
        }else{
            buf[1] = ku;
            buf[0] = ten;
            count = 1;
        }
    }
    return buf[count--];
}

/*
 * Class ISO2022JPReadDecoder
 */

SpChar ISO2022JPReadDecoder::readCh()
{
    SpChar c;
    int ch1, ch2, ch3;
    int ku, ten, base1, base2;
    if(eof()){ return (SpChar)NULL; }
    c = reader->ReadChar();
    if(isEOF(c)){ return (SpChar)NULL; }
    ch1 = c & 0xff;
    if(ch1 == 0x1B){
        c = reader->ReadChar();
        ch2 = c & 0xff;
        c = reader->ReadChar();
        ch3 = c & 0xff;
        if(ch2 == '(' &&
           (ch3 == 'B' || ch3 == 'J')){
            in_now = kCodeASCII;
        }else if(ch2 == '$' && ch3 == 'B'){
            in_now = kCodeJISX0208;
        }else{
            in_now = kCodeUNKNOWN;
        }
        c = reader->ReadChar();
        ch1 = c & 0xff;
    }
    if(in_now == kCodeASCII){ // ASCII( JIS X 0201 )
        return MakeSpChar(CodeJIS, ch1);
    }else if(in_now == kCodeJISX0208){
        SpChar ku = ch1;
        c = reader->ReadChar();
        SpChar ten = c & 0xff;
        return MakeSpChar(CodeJIS, (ku<<8) | ten);
    }
    return MakeSpChar(CodeJIS, c);
}

/*
 * Class ISO2022JPWriteEncoder
 */

void ISO2022JPWriteEncoder::WriteChar(SpChar c)
{
    int ch, ku, ten;

    if(!EqualCharCode(c, CodeJIS)){
        if(out_now != kCodeASCII){
            out_now = kCodeASCII;
            writer->WriteChar(0x1B);
            writer->WriteChar(0x28);
            writer->WriteChar(0x42);
        }
        writeTextTronCode(c);
        return;
    }
    ch = SpCharGetChar(c);
    ku = (ch >> 8);
    ten = (ch & 0xFF);
    if(ku == 0){
        if(out_now != kCodeASCII){
            out_now = kCodeASCII;
            writer->WriteChar(0x1B);
            writer->WriteChar(0x28);
            writer->WriteChar(0x42);
        }
        writer->WriteChar(ten);
    }else{
        if(out_now != kCodeJISX0208){
            out_now = kCodeJISX0208;
            writer->WriteChar(0x1B);
            writer->WriteChar(0x24);
            writer->WriteChar(0x42);
        }
        writer->WriteChar(ku);
        writer->WriteChar(ten);
    }
}

/*
 * Class ISO2022JPWriteDecoder
 *  ISO2022JP -> TronCode
 */

void ISO2022JPWriteDecoder::WriteChar(SpChar c)
{
    switch(count){
    case 0:
        if(ch1 == 0x1B){
            ch1 = c;
            in_now = kCodeESCAPE;
            count++;
        }else if(c > 0xFF){ // not jis
            writer->WriteChar(c);
        }else{
            if(in_now == kCodeASCII){
                writer->WriteChar(MakeSpChar(CodeJIS, c));
            }else{
                ch1 = c;
                count++;
            }
        }
        break;
    case 1:
        if(in_now == kCodeESCAPE){
            ch2 = c;
            count++;
        }else{ // kCodeJISX0208
            writer->WriteChar(MakeSpChar(CodeJIS, (ch1<<8) | c));
            count = 0;
        }
        break;
    case 2:
        // kCodeESCAPE only
        if(ch2 == '(' &&
           (c == 'B' || c == 'J')){
            in_now = kCodeASCII;
        }else if(ch2 == '$' && c == 'B'){
            in_now = kCodeJISX0208;
        }else{
            in_now = kCodeUNKNOWN;
        }
        count = 0;
        break;
    }
}


/*
 * ISO2022JPFile
 *   openIn, openOut
 */

SpValue& iso2022jp_openIn(SpValue& v)
{
    SpValue name = v.eval();
    if(!name.isString()){
        throw SpException("filename is not string (ISO2022JP.openIn)");
    }
    SpString* str = name.asString();
    char* filename = (char*)str->toCStringWithEncoder();
    ifstream* fin = new ifstream;
#ifdef __WIN32__
    fin->open(filename, ios::binary);
#else
    fin->open(filename);
#endif
    if(fin->fail()){
        delete fin;
        throw SpException("can't open file");
    }
    FileStreamReader* sr = new FileStreamReader(fin, filename);
    ReadEncoder* encoder = new ISO2022JPReadDecoder(sr);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& iso2022jp_openOut(SpValue& v)
{
    SpValue name = v.eval();
    if(!name.isString()){
        throw SpException("filename is not string (ISO2022JP.openOut)");
    }
    SpString* str = name.asString();
    char* filename = (char*)str->toCStringWithEncoder();
    ofstream* fout = new ofstream;
#ifdef __WIN32__
    fout->open(filename, ios::binary);
#else
    fout->open(filename);
#endif
    if(fout->fail()){
        delete fout;
        throw SpException("can't open file");
    }
    FileStreamWriter* sw = new FileStreamWriter(fout, filename);
    WriteEncoder* encoder = new ISO2022JPWriteEncoder(sw);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& iso2022jp_openAppend(SpValue& v)
{
    SpValue name = v.eval();
    if(!name.isString()){
        throw SpException("filename is not string (ISO2022JP.openAppend)");
    }
    SpString* str = name.asString();
    char* filename = (char*)str->toCStringWithEncoder();
    ofstream* fout = new ofstream;
#ifdef __WIN32__
    fout->open(filename, ios::app | ios::binary);
#else
    fout->open(filename, ios::app);
#endif
    if(fout->fail()){
        delete fout;
        throw SpException("can't open file");
    }
    FileStreamWriter* sw = new FileStreamWriter(fout, filename);
    WriteEncoder* encoder = new ISO2022JPWriteEncoder(sw);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& iso2022jp_encoderIn(SpValue& v)
{
    SpValue r = v.eval();
    if(!r.isReader()){
        throw SpException("not reader (ISO2022JP.encoderIn)");
    }
    Reader* reader = r.asReader();
    ReadEncoder* encoder = new ISO2022JPReadEncoder(reader);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& iso2022jp_decoderIn(SpValue& v)
{
    SpValue r = v.eval();
    if(!r.isReader()){
        throw SpException("not reader (ISO2022JP.decoderIn)");
    }
    Reader* reader = r.asReader();
    ReadEncoder* encoder = new ISO2022JPReadDecoder(reader);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& iso2022jp_encoderOut(SpValue& v)
{
    SpValue w = v.eval();
    if(!w.isWriter()){
        throw SpException("not writer (ISO2022JP.encoderOut)");
    }
    Writer* writer = w.asWriter();
    WriteEncoder* encoder = new ISO2022JPWriteEncoder(writer);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& iso2022jp_decoderOut(SpValue& v)
{
    SpValue w = v.eval();
    if(!w.isWriter()){
        throw SpException("not writer (ISO2022JP.decoderOut)");
    }
    Writer* writer = w.asWriter();
    WriteEncoder* encoder = new ISO2022JPWriteDecoder(writer);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

// init
void initISO2022JPFile()
{
    SpValue SymISO2022JP(new SpSymbol("iso2022jp"));
    SpNameSpace* ns = new SpNameSpace;
    SpValue NSISO2022JP(ns);
    PMainNameSpace->internConst(SymISO2022JP, NSISO2022JP);

    SpValue PrimOpenIn(new SpPrim1(&iso2022jp_openIn));
    ns->internConst(SymOpenIn, PrimOpenIn);
    SpValue PrimEncoderIn(new SpPrim1(&iso2022jp_encoderIn));
    ns->internConst(SymEncoderIn, PrimEncoderIn);
    SpValue PrimDecoderIn(new SpPrim1(&iso2022jp_decoderIn));
    ns->internConst(SymDecoderIn, PrimDecoderIn);

    SpValue PrimOpenOut(new SpPrim1(&iso2022jp_openOut));
    ns->internConst(SymOpenOut, PrimOpenOut);
    SpValue PrimOpenAppend(new SpPrim1(&iso2022jp_openAppend));
    ns->internConst(SymOpenAppend, PrimOpenAppend);
    SpValue PrimEncoderOut(new SpPrim1(&iso2022jp_encoderOut));
    ns->internConst(SymEncoderOut, PrimEncoderOut);
    SpValue PrimDecoderOut(new SpPrim1(&iso2022jp_decoderOut));
    ns->internConst(SymDecoderOut, PrimDecoderOut);
}

