/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: Socket.cpp,v 1.15 2007/01/30 05:00:48 randy Exp $
 */

#ifdef __WIN32__
#include <dos.h>
#else
#include <unistd.h>
#endif

#include <netdb.h>

#include "soopy.h"

#ifdef USE_SOCKET

#ifndef INVALID_SOCKET
#define INVALID_SOCKET -1
#endif

#ifndef SOCKET_ERROR
#define SOCKET_ERROR -1
#endif

/*
 * Socket
 *   -close
 *   -eof?
 *   -write string     -- send string
 *   -writeline string -- write one line
 *   -readline         -- read one line
 *   -receive n        -- receive n byte data
 */

/*
 * ClientSocket
 *   +open(adr, port)  -- open socket
 */

/*
 * ServerSocket
 *   +open port        -- open server socket
 *   -close
 *   -accept           -- return Socket
 */


//
// class BaseSocket
//
void BaseSocket::close()
{
    shutdown(sock, 2);
#ifdef __WIN32__
    closesocket(sock);
#else
    ::close(sock);
#endif
    isOpen = false;
}

//
// class Socket
//
SpValue& Socket::toString()
{
    SpString* str = new SpString("<socket>");
    //    static SpValue s;
    //    s.setNewObject(str);
    //    return s;
    return SpObjectResult(str);
}

void Socket::open()
{
    unsigned long serveraddr;
    struct hostent* hent;
    struct sockaddr_in sockadd;

    /* create socket */
    if((sock = socket(AF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET){
        throw SpException("can't open socket");
    }
    serveraddr = inet_addr(hostname);
    if(serveraddr != INADDR_NONE){
        sockadd.sin_addr.s_addr = serveraddr;
    }else{
      if((hent = gethostbyname(hostname)) == NULL){
          throw SpException("can't get hostbyname");
      }
      memcpy(&sockadd.sin_addr, hent->h_addr, hent->h_length);
    }
    sockadd.sin_family = AF_INET;
    sockadd.sin_port = htons(port);

    /* connect */
    for(int loop=0;
        0 != connect(sock, (struct sockaddr*)&sockadd, sizeof(sockadd));
        loop++)
    {
        if(10 < loop){
            throw SpException("can't connect socket");
        }
#ifdef __WIN32__
        _sleep(1);
#else
        sleep(1);
#endif
    }
    isOpen = true;
    receive_end = false;
}

void Socket::read_buffer()
{
    read_bytes = recv(sock, buff[which_buf], BUFFSIZE-1, 0);
    if(read_bytes == SOCKET_ERROR){
        throw SpException("socket error");
    }
    if(read_bytes == 0){
        receive_end = true;
    }
    if(read_bytes < 0){
        throw SpException("can't read socket");
    }
    buff[which_buf][read_bytes] = '\0';
    already_read[which_buf] = true;
    read_point = 0;
}

SpChar Socket::readCh()
{
    if(eof()){
        return (SpChar)NULL;
    }
    if(read_point >= read_bytes){
        next_buf();
    }
    if(!already_read[which_buf]){
        read_buffer();
    }
    if(receive_end){
        return (SpChar)NULL;
//        throw SpException("end of socket");
    }
    return buff[which_buf][read_point++];
}

SpValue& Socket::receive(SpInt n, unsigned char* buffer)
{
    read_bytes = recv(sock, buffer, n, 0);

    if(read_bytes == SOCKET_ERROR){
        throw SpException("socket error: can't receive");
    }
    if(read_bytes == 0){
        receive_end = true;
    }
    if(read_bytes < 0){
        throw SpException("can't read socket");
    }

    //    static SpValue result;
    //    result.setInt(read_bytes);
    //    return result;
    return SpIntResult(read_bytes);
}

SpValue& Socket::send_buffer(SpInt n, unsigned char* buffer)
{
    int i = send(sock, buffer, n, 0);

    if(i == SOCKET_ERROR){
        throw SpException("socket error: can't write");
    }

    //    static SpValue result;
    //    result.setInt(i);
    //    return result;
    return SpIntResult(i);
}

void Socket::flush()
{
    if(write_point <= 0){ return; }
    int result = send(sock, wbuff, write_point, 0);
    if(result == SOCKET_ERROR){
        throw SpException("socket error: can't write");
    }
    write_point = 0;
}

void Socket::writeCh(unsigned char ch)
{
    wbuff[write_point] = ch;
    write_point++;
    if(write_point >= BUFFSIZE){
        flush();
    }
}

bool Socket::eof()
{
    if(receive_end){
        if(read_point < (read_bytes - 1)){
            return false;
        }else{
            return true;
        }
    }
    return false;
}

void Socket::write(char* str)
{
    flush();
    int result = send(sock, str, strlen(str), 0);
    if(result == SOCKET_ERROR){
        throw SpException("socket error: can't write");
    }
}

/*
 * primitives
 */

SpValue& Socket::prim_getname(SpValue& adr){
    if(!adr.isString()){
        throw SpException("not string. (socket.open)");
    }
    SpString* address = adr.asString();

    char hbuf[NI_MAXHOST];
    struct sockaddr_in sa;    /* input */
    memset(&sa, 0, sizeof(sa));
    sa.sin_family = AF_INET;
    sa.sin_port = htons(80); // dummy
    inet_aton((char*)address->toCStringWithEncoder(), &sa.sin_addr);

//    if (getnameinfo(sa, sa->sa_len, hbuf, sizeof(hbuf),
    if (getnameinfo((sockaddr*)&sa, sizeof(sa), hbuf, sizeof(hbuf),
		    NULL, 0, NI_NAMEREQD))
    {
	return NilObject;
    }else{
	return SpObjectResult(new SpString(hbuf));
    }
}

// client Socket open
SpValue& Socket::prim_open(SpValue& adr, SpValue& port)
{
    if(!adr.isString()){
        throw SpException("not string. (socket.open)");
    }
    SpString* address = adr.asString();
    if(!port.isInt()){
        throw SpException("not int. (socket.open)");
    }
    SpInt port_no = port.getInt();

    Socket* sock = new Socket(port_no, (char*)address->toCStringWithEncoder());
    sock->open();
    //    static SpValue result;
    //    result.setNewObject(sock);
    //    return result;
    return SpObjectResult(sock);
}

SpValue& Socket::prim_close(SpValue& self)
{
    Socket* sock = self.asSocket();
    sock->close();
    return TrueObject;
}

SpValue& Socket::prim_eof(SpValue& self)
{
    Socket* sock = self.asSocket();
    if(sock->eof()){
        return TrueObject;
    }
    return FalseObject;
}

SpValue& Socket::prim_write(SpValue& self, SpValue& str)
{
    Socket* sock = self.asSocket();
    if(!str.isString()){
        throw SpException("not string (socket.write)");
    }
    SpString* s = str.asString();
    sock->write((char*)s->toCStringWithEncoder());
    return TrueObject;
}

SpValue& Socket::prim_writeline(SpValue& self, SpValue& str)
{
    Socket* sock = self.asSocket();
    if(!str.isString()){
        throw SpException("not string (socket.write)");
    }
    SpString* s = str.asString();
    SpString s2(*s);
    s2 += "\r\n";
    sock->write((char*)s2.toCStringWithEncoder());
    sock->flush();
    return TrueObject;
}

SpValue& Socket::prim_receive(SpValue& self, SpValue& n, SpValue& byte_vector)
{
    if(!n.isInt()){
        throw SpException("not integer(socket.receive)");
    }
    if(!byte_vector.isByteArray()){
        throw SpException("not byte array(socket.receive)");
    }
    Socket* sock = self.asSocket();
    SpInt i = n.getInt();
    SpByteArray* ary = byte_vector.asByteArray();
    if(i > ary->length()){
        throw SpException("overflow (socket.receive)");
    }
    unsigned char* buffer = ary->getBuffer();
    return sock->receive(i, buffer);
}

SpValue& Socket::prim_send(SpValue& self, SpValue& n, SpValue& byte_vector)
{
    if(!n.isInt()){
        throw SpException("not integer(socket.send)");
    }
    if(!byte_vector.isByteArray()){
        throw SpException("not byte array(socket.send)");
    }
    Socket* sock = self.asSocket();
    SpInt i = n.getInt();
    SpByteArray* ary = byte_vector.asByteArray();
    if(i > ary->length()){
        throw SpException("overflow (socket.send)");
    }
    unsigned char* buffer = ary->getBuffer();
    return sock->send_buffer(i, buffer);
}


// init Message Handler
SpValue& Socket::onMessage(SpValue& rec, SpValue& msg)
{
    return SocketMsgHandler(rec, msg);
}

void Socket::init()
{
    /* Client Socket */
    // set namespace to symbol 'socket'.
    SpNameSpace* PSocketNameSpace = new SpNameSpace;
    SpValue SocketNS;
    SocketNS.setNewObject(PSocketNameSpace);
    SpValue SymSocket(new SpSymbol("socket"));
    PMainNameSpace->internConst(SymSocket, SocketNS);
    // Socket open adr port
    SpValue PrimOpen(new SpPrim2(prim_open));
    PSocketNameSpace->internFunc(SymOpen, PrimOpen);
    // getname
    SpValue PrimGetName(new SpPrim1(prim_getname));
    SpValue SymGetName(new SpSymbol("getname"));
    PSocketNameSpace->internFunc(SymGetName, PrimGetName);

    // setup primitives
    SpValue PrimClose(new SpPrim1(prim_close));
    SocketMsgHandler.append(SymClose, PrimClose);

    SpValue PrimEof(new SpPrim1(prim_eof));
    SocketMsgHandler.append(SymIsEOF, PrimEof);

    SpValue PrimWrite(new SpPrim2(prim_write));
    SocketMsgHandler.append(SymWrite, PrimWrite);

    SpValue PrimWriteLine(new SpPrim2(prim_writeline));
    SocketMsgHandler.append(SymWriteLine, PrimWriteLine);

    SpValue PrimReadLine(new SpPrim1(prim_readLine));
    SocketMsgHandler.append(SymReadLine, PrimReadLine);

    SpValue SymReceive(new SpSymbol("receive"));
    SpValue PrimReceive(new SpPrim3(prim_receive));
    SocketMsgHandler.append(SymReceive, PrimReceive);

    SpValue SymSend(new SpSymbol("send"));
    SpValue PrimSend(new SpPrim3(prim_send));
    SocketMsgHandler.append(SymSend, PrimSend);

    ServerSocket::init();

#ifdef __WIN32__
    WORD wVersionRequested = MAKEWORD(1, 1);
    WSADATA wsaData;
    int err = WSAStartup(wVersionRequested, &wsaData);
    if(atexit((void(*)(void))(WSACleanup))){
        throw SpException("atexit(WSACleanup)");
    }
    if(err != 0){
        throw SpException("can't initialize WinSock");
    }
#endif

}

/*
 * Server Socket
 */

SpValue& ServerSocket::toString()
{
    SpString* str = new SpString("<server socket>");
    //    static SpValue s;
    //    s.setNewObject(str);
    //    return s;
    return SpObjectResult(str);
}

const int BACKLOG = 5;

void ServerSocket::open()
{
    /* create socket */
    if((sock = socket(AF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET){
        throw SpException("can't open ServerSocket");
    }
    sockadd.sin_port = htons(port);
    sockadd.sin_family = AF_INET;
    memset(&sockadd.sin_addr, 0, sizeof(sockadd.sin_addr));

    /* bind */
    if(bind(sock, (sockaddr*)&sockadd, sizeof(sockadd)) != 0){
        throw SpException("can't bind ServerSocket");
    }
    /* listen */
    if(listen(sock, BACKLOG) != 0){
        throw SpException("can't listen ServerSocket");
    }

    isOpen = true;
}

SpValue& ServerSocket::accept()
{
    /* accept */
    SOCKET s = ::accept(sock, NULL, NULL);
    if(s == INVALID_SOCKET){
        throw SpException("can't accept ServerSocket");
    }
    Socket* ptr = new Socket(s);

    //    static SpValue result;
    //    result.setNewObject(ptr);
    //    return result;
    return SpObjectResult(ptr);
}

// server Socket open
SpValue& ServerSocket::prim_open(SpValue& port)
{
    if(!port.isInt()){
        throw SpException("not int. (socket.open)");
    }
    SpInt port_no = port.getInt();

    ServerSocket* sock = new ServerSocket(port_no);
    sock->open();
    //    static SpValue result;
    //    result.setNewObject(sock);
    //    return result;
    return SpObjectResult(sock);
}

SpValue& ServerSocket::prim_accept(SpValue& self)
{
    ServerSocket* sock = self.asServerSocket();
    return sock->accept();
}

SpValue& ServerSocket::prim_close(SpValue& self)
{
    ServerSocket* sock = self.asServerSocket();
    sock->close();
    return TrueObject;
}

SpValue& ServerSocket::onMessage(SpValue& rec, SpValue& msg)
{
    return ServerSocketMsgHandler(rec, msg);
}

void ServerSocket::init()
{
    /* Server Socket */
    // set namespace to symbol 'serversocket'.
    SpNameSpace* PSSocketNameSpace = new SpNameSpace;
    SpValue SSocketNS;
    SSocketNS.setNewObject(PSSocketNameSpace);
    SpValue SymSSocket(new SpSymbol("serversocket"));
    PMainNameSpace->internConst(SymSSocket, SSocketNS);
    // Socket open adr port
    SpValue PrimSOpen(new SpPrim1(prim_open));
    PSSocketNameSpace->internFunc(SymOpen, PrimSOpen);

    // setup primitives
    SpValue PrimClose(new SpPrim1(prim_close));
    ServerSocketMsgHandler.append(SymClose, PrimClose);

    SpValue SymAccept(new SpSymbol("accept"));
    SpValue PrimAccept(new SpPrim1(prim_accept));
    ServerSocketMsgHandler.append(SymAccept, PrimAccept);
}


#endif /* USE_SOCKET */

