/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 *
 * Copyright (C) 2002,2003 SUZUKI Jun
 *
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 *
 *
 * $Id: soopy.h,v 1.228 2007/01/30 05:00:48 randy Exp $
 */

#ifndef __SOOPY_H
#define __SOOPY_H

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <iostream>
#include <iomanip>
#include <sstream>
#include <fstream>
#include <string>
#include <vector>
#include <map>
#include <exception>
#include <typeinfo>
#include <stdio.h>

#ifdef USE_SOCKET

#include <sys/types.h>
#ifdef __WIN32__
#include <dir.h>
#include "winsock.h"
#else
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#endif

#endif /* USE_SOCKET */

#ifdef USE_PTHREAD

#include <pthread.h>

#ifndef THREAD
#define THREAD
#endif

#endif /* USE_PTHREAD */

#ifdef USE_W32THREAD

#include <process.h>
#include <windows.h>

#ifndef THREAD
#define THREAD
#endif /* USE_W32THREAD */

#endif

#ifdef __WIN32__
#define PATH_DELIMIT ';'
#endif

#define CR 0x0d
#define LF 0x0a

using namespace std;

typedef bool SpBool;
typedef long int SpInt;
typedef unsigned long int SpUInt;
typedef double SpReal;
typedef unsigned long int SpChar; // high16bit = CodeType,
// low16bit  = CharCode

#ifndef MAXPATH
#define MAXPATH 2048
#endif

#ifdef __WIN32__
const int SP_NAME_MAX = MAXPATH+4;
#else
#define SP_NAME_MAX (MAXPATH+4)
#endif

extern unsigned int lineno;
extern char Filename[SP_NAME_MAX];
inline void setFilename(char* s)
{
  strncpy(Filename, s, SP_NAME_MAX-2);
  Filename[SP_NAME_MAX-2] = '\0';
  Filename[SP_NAME_MAX-1] = '\0';
}

const int MaxLibPath = 20;
const int MaxLibPathBuffer = 1024 * 4;
extern int NumberOfLibPath;
extern char* LibPaths[];

inline void warning(char* str)
{
  cerr << "Warning: " << str << endl;
}

inline SpChar SpCharGetCode(SpChar ch){
  return ch & 0xFFFF0000;
}
inline SpChar SpCharGetChar(SpChar ch){
  return ch & 0x0000FFFF;
}
inline bool EqualCharCode(SpChar ch, SpChar code){
  return SpCharGetCode(ch) == code;
}
/*
inline SpChar MakeSpChar(SpChar code, SpChar ch){
    return (SpChar)(code | ch);
}
 */
#define MakeSpChar(code, ch)  ((SpChar)(code | ch))


// Code number from TronCode
const SpChar CodeJIS       = 0x00210000;
const SpChar CodeGT1       = 0x00220000;
const SpChar CodeGT2       = 0x00230000;
const SpChar CodeCNS       = 0x00260000;
const SpChar CodeDAIKANWA1 = 0x00280000;
const SpChar CodeDAIKANWA2 = 0x00290000;
const SpChar CodeTONPA     = 0x002a0000;
const SpChar CodeUNICODE1  = 0x00300000;
const SpChar CodeUNICODE2  = 0x00310000;

/*
 * Define Soopy's Types
 */
enum SpType {
  TypeNil,
  TypeBool,
  TypeInt,
  TypeChar,
  TypeReal,
  TypeString,
  TypeSymbol,
  TypeList,
  TypeTuple,
  TypeNameSpace,
  TypeExpr,
  TypeArray,
  TypeDType,
  TypeObject,
  TypePersistent
};

class SpValue;
class Writer;
class WriteEncoder;
class Reader;
class ReadEncoder;
class SpTuple;
class SpNameSpace;
class SpDataType;
class SpConstructor;
class SpArg;
class SpDType;
class NSKey;
class SpList;
class SpString;
class SpFileOut;
class SpFunc;
class SpArray;
class SpByteArray;
class SpSymbol;
class SpDir;
class SpAssign;
#ifdef USE_CLX
class SpComponent;
#endif
#ifdef AQUA
class SpComponent;
#endif
#ifdef USE_SOCKET
class Socket;
class ServerSocket;
#endif
class SpDate;
class SpTime;
class SpDateTime;
#ifdef USE_DATABASE
class SpDatabase;
#endif

extern WriteEncoder* spout;
extern ReadEncoder* spin;
extern SpValue NilObject;
extern SpValue TrueObject;
extern SpValue FalseObject;
extern SpValue NullString;
extern SpValue ObjectZero;
extern SpValue ObjectOne;
extern SpValue ObjectTwo;

//
// util routines
//
#define JIS(c) (CodeJIS | c)

const char* spCharToCString(SpChar ch, WriteEncoder& writer = *spout);
bool isWhiteSpace(SpChar c);
bool isAlpha(SpChar c);
bool isDIGIT(SpChar c);
bool isEOL(SpChar c);
bool isEOF(SpChar c);
int toINT(SpChar c);
SpChar toHankakuLower(SpChar c);
SpInt int_fromString(SpString* str);

inline bool isUpper(SpChar c)
{
  return (JIS('A') <= c) && (c <= JIS('Z'));
}

inline bool isLower(SpChar c)
{
  return (JIS('a') <= c) && (c <= JIS('z'));
}

inline SpChar SpChar2Lower(SpChar c)
{
  if(isUpper(c)){
    c = c - ('A' - 'a');
  }
  return c;
}

inline SpChar SpChar2Upper(SpChar c)
{
  if(isLower(c)){
    c = c + ('A' - 'a');
  }
  return c;
}


/*
 * Soopy Base Exception Class
 */

class SpException {
protected:
  const char* str;
  char file[SP_NAME_MAX];
  unsigned int line;
public:
  SpException(const char* s){
    str = s;
    //file = Filename;
    strcpy(file, Filename);
    line = lineno;
  }
  virtual ~SpException(){}
  virtual const char* what() const throw();
};


/*
 * Soopy Base Class
 */
class SpObject {
private:
  unsigned int links;
#ifdef USE_PTHREAD
  pthread_mutex_t obj_mutex;
#endif
#ifdef USE_W32THREAD
  HANDLE obj_mutex;
#endif

public:
  SpObject(){
    links = 0;
#ifdef USE_PTHREAD
    pthread_mutex_init(&obj_mutex, NULL);
#endif /* USE_PTHREAD */
#ifdef USE_W32THREAD
    obj_mutex = CreateMutex(NULL, FALSE, NULL);
    if(obj_mutex == NULL){
      throw SpException("can't create mutex for Object");
    }
#endif /* USE_W32THREAD */
  }
  virtual ~SpObject(){
#ifdef USE_PTHREAD
    pthread_mutex_destroy(&obj_mutex);
#endif /* USE_PTHREAD */
#ifdef USE_W32THREAD
    CloseHandle(obj_mutex);
#endif /* USE_W32THREAD */
  }

  static void init();
  //  static void finalize();

  //virtual const char* toCString(); // only SpString should override
  virtual const char* toCString(WriteEncoder& writer=*spout); // only SpString should override
  virtual const char* toCStringWithEncoder(WriteEncoder& writer=*spout); // only SpString should override
  virtual const char* typeString() = 0;
  virtual SpValue& toString() = 0;

  virtual bool operator==(SpObject& obj){ return false; }
  virtual bool operator<(SpObject& obj);

  virtual SpValue& uminus(SpValue& e);
  virtual SpValue& plus(SpValue& e1, SpValue& e2);
  virtual SpValue& minus(SpValue& e1, SpValue& e2);
  virtual SpValue& times(SpValue& e1, SpValue& e2);
  virtual SpValue& div(SpValue& e1, SpValue& e2);
  virtual SpValue& mod(SpValue& e1, SpValue& e2);
  virtual SpValue& boolNot(SpValue& e1);
  virtual SpValue& boolAnd(SpValue& e1, SpValue& e2);
  virtual SpValue& boolOr(SpValue& e1, SpValue& e2);
  virtual SpValue& eq(SpValue& e1, SpValue& e2);
  virtual SpValue& ne(SpValue& e1, SpValue& e2);
  virtual SpValue& gt(SpValue& e1, SpValue& e2);
  virtual SpValue& ge(SpValue& e1, SpValue& e2);
  virtual SpValue& lt(SpValue& e1, SpValue& e2);
  virtual SpValue& le(SpValue& e1, SpValue& e2);

  virtual bool isString(){ return false; }
  virtual bool isSymbol(){ return false; }
  virtual bool isTuple(){ return false; }
  virtual bool isList(){ return false; }
  virtual bool isCons(){ return false; }
  virtual bool isNameSpace(){ return false; }
  virtual bool isSendMsg(){ return false; }
  virtual bool isArray(){ return false; }
  virtual bool isByteArray(){ return false; }
  virtual bool isConstructor(){ return false; }
  virtual bool isDataType(){ return false; }
  virtual bool isArg(){ return false; }
  virtual bool isDType(){ return false; }
  virtual bool isNSKey(){ return false; }
  virtual bool isFunc(){ return false; }
  //virtual bool isFileIn(){ return false; }
  virtual bool isReader(){ return false; }
  //virtual bool isFileOut(){ return false; }
  virtual bool isWriter(){ return false; }
  virtual bool isDir(){ return false; }
  virtual bool isAssign(){ return false; }
#ifdef USE_CLX
  virtual bool isComponent(){ return false; }
#endif
#ifdef AQUA
  virtual bool isComponent(){ return false; }
#endif
#ifdef USE_SOCKET
  virtual bool isSocket(){ return false; }
  virtual bool isServerSocket(){ return false; }
#endif
  virtual bool isDate(){ return false; }
  virtual bool isTime(){ return false; }
  virtual bool isDateTime(){ return false; }
#ifdef USE_DATABASE
  virtual bool isDatabase(){ return false; }
#endif

  virtual SpValue& eval(){ return NilObject; }
  virtual bool canEval(){ return false; }
  virtual SpType getType(){ return TypeObject; }
  virtual SpValue& onMessage(SpValue& rec, SpValue& msg);
  virtual bool match(SpValue& self, SpValue& val, SpNameSpace* ns){ return false; }

  // primitives
  static SpValue& prim_toString(SpValue& self);

  friend class SpValue;
  friend class ReadEncoder;
  friend class WriteEncoder;
};

#ifndef THREAD
#define LOCK(obj)
#define UNLOCK(obj)
#endif /* not THREAD */

#ifdef USE_W32THREAD
#define LOCK(obj)   (WaitForSingleObject(obj->obj_mutex, INFINITE))
#define UNLOCK(obj) (ReleaseMutex(obj->obj_mutex))
#endif /* USE_W32THREAD */

#ifdef USE_PTHREAD
#define LOCK(obj)   (pthread_mutex_lock(&obj->obj_mutex))
#define UNLOCK(obj) (pthread_mutex_unlock(&obj->obj_mutex))
#endif /* USE_PTHREAD */

/*
 * Soopy Class for shared pointer
 */
class SpValue {
private:
  SpType typ;
  union {
    SpBool Bool;
    SpInt Int;
    SpChar Char;
    SpReal Real;
    SpObject* pObject;
  };

  void RetainObject(SpObject* obj){
    pObject = obj;
    if(pObject != NULL){
      LOCK(pObject);
      pObject->links++;
      UNLOCK(pObject);
    }
  }

  void ReleaseObject(){
    if(pObject != NULL){
      LOCK(pObject);
      pObject->links--;
      int links = pObject->links;
      UNLOCK(pObject);
      if(links == 0){
        delete pObject;
      }
    }
  }

  void clearObject(){
    if(typ == TypeObject){
      ReleaseObject();
    }
  }

public:
  SpValue(){ typ = TypeNil; pObject = NULL; } // create Nil
  SpValue(bool p){ typ = TypeBool; Bool = p; }
  SpValue(int i){ typ = TypeInt; Int = i; }
  SpValue(char c, ReadEncoder& reader=*spin);
  /*
#ifdef EUC_JP
    SpValue(char c, ReadEncoder& reader=(ReadEncoder&)*eucjpin);
#else
    SpValue(char c, ReadEncoder& reader=(ReadEncoder&)*sjisin);
#endif
   */
  SpValue(double d){ typ = TypeReal; Real = d; }
  SpValue(const SpValue& val);
  SpValue(SpObject* obj){
    typ = TypeObject;
    RetainObject(obj);
  }
  ~SpValue(){ clearObject(); }

  void Clear(){ clearObject(); typ = TypeNil; }

  SpValue& operator=(const SpValue& val);
  /*
    SpValue& operator=(SpValue* val){ operator=(*val); return *this; };
    SpValue& operator=(SpObject* obj){ // BUGθ
        cout << "operator=, BUG!!" << endl;
        throw SpException("illegal use SpValue& operator=(SpObject*)");
        return *this;
    }
    SpValue& operator=(const bool p);
    SpValue& operator=(const int i);
    SpValue& operator=(const char c);
    SpValue& operator=(const char* str);
    SpValue& operator=(const double d);
   */
  void setBool(SpBool p){
    clearObject();
    typ = TypeBool;
    Bool = p;
  }
  void setSpChar(SpChar c){
    clearObject();
    typ = TypeChar;
    Char = c;
  }
  void setInt(SpInt i){
    clearObject();
    typ = TypeInt;
    Int = i;
  }
  void setReal(SpReal r){
    clearObject();
    typ = TypeReal;
    Real = r;
  }
  void setNewObject(SpObject* obj){
    if(isObject() && (pObject == obj)){
      return;
    }
    clearObject();
    typ = TypeObject;
    RetainObject(obj);
  }
  void setObject(SpObject* obj){
    if(isObject() && (pObject == obj)){
      return;
    }
    clearObject();
    typ = TypeObject;
    RetainObject(obj);
  }
  SpBool getBool(){ return Bool; }
  SpChar getChar(){ return Char; }
  SpInt  getInt(){ return Int; }
  SpReal getReal(){ return Real; }
  SpObject* getObject();

  SpInt asInt();
  SpTuple* asTuple();
  SpNameSpace* asNameSpace();
  SpDataType* asDataType();
  SpConstructor* asConstructor();
  SpArg* asArg();
  SpDType* asDType();
  NSKey* asNSKey();
  SpList* asList();
  SpString* asString();
  //SpFileIn* asFileIn();
  Reader* asReader();
  //SpFileOut* asFileOut();
  Writer* asWriter();
  SpFunc* asFunc();
  SpArray* asArray();
  SpByteArray* asByteArray();
  SpSymbol* asSymbol();
  SpDir* asDir();
  SpAssign* asAssign();
#ifdef USE_CLX
  SpComponent* asComponent();
#endif
#ifdef AQUA
  SpComponent* asComponent();
#endif
#ifdef USE_SOCKET
  Socket* asSocket();
  ServerSocket* asServerSocket();
#endif
  SpDate* asDate();
  SpTime* asTime();
  SpDateTime* asDateTime();
#ifdef USE_DATABASE
  SpDatabase* asDatabase();
#endif

  SpValue& uminus();
  SpValue& plus(SpValue& e1);
  SpValue& minus(SpValue& e1);
  SpValue& times(SpValue& e1);
  SpValue& div(SpValue& e1);
  SpValue& mod(SpValue& e1);
  SpValue& boolNot();
  SpValue& boolAnd(SpValue& e1);
  SpValue& boolOr(SpValue& e1);
  SpValue& eq(SpValue& e1);
  SpValue& ne(SpValue& e1);
  SpValue& gt(SpValue& e1);
  SpValue& ge(SpValue& e1);
  SpValue& lt(SpValue& e1);
  SpValue& le(SpValue& e1);
  SpValue& wcolon(SpValue& e1);
  SpValue& at(SpValue& e1);

  bool operator==(SpValue& key);
  bool operator!=(SpValue& key){ return ! operator==(key); }
  bool operator<(SpValue& key);

  bool isNil(){ return typ == TypeNil; }
  bool isBool(){ return typ == TypeBool; }
  bool isInt(){ return typ == TypeInt; }
  bool isReal(){ return typ == TypeReal; }
  bool isChar(){ return typ == TypeChar; }
  bool isObject(){ return typ == TypeObject; }
  bool isTrue(){
    if(typ == TypeBool){
      return Bool;
    }
    return false;
  }
  bool isFalse(){
    if(typ == TypeBool){
      return !Bool;
    }
    return false;
  }
  bool isString(){
    if(typ == TypeObject){
      return getObject()->isString();
    }
    return false;
  }
  bool isSymbol(){
    if(typ == TypeObject){
      return getObject()->isSymbol();
    }
    return false;
  }
  bool isTuple(){
    if(typ == TypeObject){
      return getObject()->isTuple();
    }
    return false;
  }
  bool isList(){
    if(typ == TypeObject){
      return getObject()->isList();
    }
    return false;
  }
  bool isCons(){
    if(typ == TypeObject){
      return getObject()->isCons();
    }
    return false;
  }
  bool isNameSpace(){
    if(typ == TypeObject){
      return getObject()->isNameSpace();
    }
    return false;
  }
  bool isSendMsg(){
    if(typ == TypeObject){
      return getObject()->isSendMsg();
    }
    return false;
  }
  bool isArray(){
    if(typ == TypeObject){
      return getObject()->isArray();
    }
    return false;
  }
  bool isByteArray(){
    if(typ == TypeObject){
      return getObject()->isByteArray();
    }
    return false;
  }
  bool isConstructor(){
    if(typ == TypeObject){
      return getObject()->isConstructor();
    }
    return false;
  }
  bool isDataType(){
    if(typ == TypeObject){
      return getObject()->isDataType();
    }
    return false;
  }
  bool isArg(){
    if(typ == TypeObject){
      return getObject()->isArg();
    }
    return false;
  }
  bool isDType(){
    if(typ == TypeObject){
      return getObject()->isDType();
    }
    return false;
  }
  bool isNSKey(){
    if(typ == TypeObject){
      return getObject()->isNSKey();
    }
    return false;
  }
  bool isFunc(){
    if(typ == TypeObject){
      return getObject()->isFunc();
    }
    return false;
  }
  bool isReader(){
    if(typ == TypeObject){
      return getObject()->isReader();
    }
    return false;
  }
  bool isWriter(){
    if(typ == TypeObject){
      return getObject()->isWriter();
    }
    return false;
  }
  bool isDir(){
    if(typ == TypeObject){
      return getObject()->isDir();
    }
    return false;
  }
  bool isAssign(){
    if(typ == TypeObject){
      return getObject()->isAssign();
    }
    return false;
  }
#ifdef USE_CLX
  bool isComponent(){
    if(typ == TypeObject){
      return getObject()->isComponent();
    }
    return false;
  }
#endif
#ifdef AQUA
  bool isComponent(){
    if(typ == TypeObject){
      return getObject()->isComponent();
    }
    return false;
  }
#endif
#ifdef USE_SOCKET
  bool isSocket(){
    if(typ == TypeObject){
      return getObject()->isSocket();
    }
    return false;
  }
  bool isServerSocket(){
    if(typ == TypeObject){
      return getObject()->isServerSocket();
    }
    return false;
  }
#endif
  bool isDate(){
    if(typ == TypeObject){
      return getObject()->isDate();
    }
    return false;
  }
  bool isTime(){
    if(typ == TypeObject){
      return getObject()->isTime();
    }
    return false;
  }
  bool isDateTime(){
    if(typ == TypeObject){
      return getObject()->isDateTime();
    }
    return false;
  }
#ifdef USE_DATABASE
  bool isDatabase(){
    if(typ == TypeObject){
      return getObject()->isDatabase();
    }
    return false;
  }
#endif

  SpValue& toString();
  const char* toCString();
  const char* toCStringWithEncoder(WriteEncoder& writer=*spout);
  const char* typeString();
  SpValue& eval();
  SpType getType();
  SpValue& onMessage(SpValue& msg);
  bool match(SpValue& val, SpNameSpace* ns);

  friend class WriteEncoder;
  friend class RawReadEncoder;
  friend int yyparse();
  friend void SpInit(int,char**);
  friend SpValue& char_toInt(SpValue& v);
  friend SpValue& int_toChar(SpValue& v);
};
/*
inline ostream& operator<<(ostream& out, SpValue& spVal)
{
  if(spVal.isList()){
    SpList* list = spVal.asList();
    out << list;
  }else{
    out << spVal.toCString();
  }
  return out;
};
 */


extern SpValue NilObject;

/*
 * Soopy Exception Class
 */

class SpIllegalMessageException : public SpException {
private:
  SpValue msg;

public:
  SpIllegalMessageException(SpValue& v)
       : SpException("illegal message"), msg(v) {}
  const char* what() const throw();
};

class SpSystemError : public SpException {
private:
  char* filename;
  int lineno;
public:
  SpSystemError(char* f, int l)
       : SpException("system error"), filename(f), lineno(l) {}
  const char* what() const throw();
};

class SpFileException : public SpException {
public:
  SpFileException(const char* s)
       : SpException(s) {}
};

class SpAccessException : public SpException {
public:
  SpAccessException(const char* s)
       : SpException(s) {}
};

class SpKeyException : public SpException {
private:
  const char* key;
public:
  SpKeyException(const char* s, const char* k)
       : SpException(s) , key(k) {}
  const char* what() const throw();
};

class SpTypeException : public SpException {
private:
  const char* rtype; // require type
  const char* vtype; // value type
public:
  SpTypeException(const char* s, const char* r, const char* v)
       : SpException(s) , rtype(r), vtype(v) {}
  const char* what() const throw();
};

class SpRequireException : public SpException {
public:
  SpRequireException(const char* s)
       : SpException(s) {}
};

class SpEnsureException : public SpException {
public:
  SpEnsureException(const char* s)
       : SpException(s) {}
};

class SpRetryException : public SpException {
public:
  SpRetryException(const char* s)
       : SpException(s) {}
};

class SpLoopVariantException : public SpException {
public:
  SpLoopVariantException(const char* s)
       : SpException(s) {}
};

class SpLoopInvariantException : public SpException {
public:
  SpLoopInvariantException(const char* s)
       : SpException(s) {}
};

class SpNoMethodException : public SpException {
private:
  char* receiver;
  char* message;
public:
  SpNoMethodException(const char* s,
                      char* rec,
                      char* mes)
       : SpException(s), receiver(rec), message(mes) {}
  const char* what() const throw();
};

class SpExitException : public SpException {
private:
  SpValue symbol, val;
public:
  SpExitException(SpValue& sym, SpValue& v, const char* s)
       : SpException(s), symbol(sym), val(v) {}
  friend class SpLoop;
  friend class SpUsrFunc;
};

class SpNextException : public SpException {
private:
  SpValue symbol;
public:
  SpNextException(SpValue& sym, const char* s)
       : SpException(s), symbol(sym) {}
  friend class SpLoop;
};

class SpNoLengthException : public SpException {
public:
  SpNoLengthException(const char* s)
       : SpException(s) {}
};

/*
class SpNullListCreator : public SpException {
public:
  SpNullListCreator(const char* s) : SpException(s) {}
};
 */

/*
 * Thread
 */

struct ThreadContext {
  SpValue gResult;
  vector<SpValue> Frame;
};

#ifdef USE_PTHREAD

extern pthread_key_t pthread_key;

class SpThread : public SpObject {
private:
  pthread_t thread;
  SpValue   assoc;
  SpValue   body;
  static    std::vector<SpValue> all_threads;  // GCʤ褦˥åɤݻƤ

  static void removeContext(ThreadContext* con);
  static void start(SpThread* th);
  void run();
  // primitives
  static SpValue& prim_start(SpValue& body);
  static SpValue& prim_join(SpValue& thread);
  static SpValue& prim_quit(SpValue& thread);

public:
  SpThread(SpValue& ns, SpValue& b) : assoc(ns), body(b) {}
  ~SpThread();

  bool isThread(){ return true; }
  const char* typeString(){ return "thread"; }
  SpValue& toString();
  SpValue& onMessage(SpValue& rec, SpValue& msg);

  static void init();
  static void init2();
  static void finalize();
};

inline SpValue* getCurrentResult(){
  ThreadContext* context;
  context = (ThreadContext*)pthread_getspecific(pthread_key);
  return &(context->gResult);
}

class SpMutex : public SpObject {
private:
  pthread_mutex_t* mutex;

  static SpValue& prim_synchronize(SpValue& mutex, SpValue& func);
  static SpValue& prim_new();

public:
  SpMutex(){
    mutex = (pthread_mutex_t*)malloc(sizeof(pthread_mutex_t));
    pthread_mutex_init(mutex, NULL);
  }
  ~SpMutex(){
    if(mutex != NULL){
      pthread_mutex_destroy(mutex);
      free(mutex);
    }
  }

  void lock(){
    pthread_mutex_lock(mutex);
  }
  void unlock(){
    pthread_mutex_unlock(mutex);
  }

  bool isMutex(){ return true; }
  const char* typeString(){ return "mutex"; }
  SpValue& toString();
  SpValue& onMessage(SpValue& rec, SpValue& msg);

  static void init();
};

#endif  /* USE_PTHREAD */

#ifdef USE_W32THREAD

extern DWORD tls_key;

class SpThread : public SpObject {
private:
  HANDLE  hThread;
  unsigned id;
  SpValue assoc;
  SpValue body;
  static  std::vector<SpValue> all_threads;  // GCʤ褦˥åɤݤäƤ

  static void removeContext(ThreadContext* con);
  static unsigned __stdcall start(SpThread* th);
  void run();
  // primitives
  static SpValue& prim_start(SpValue& body);
  static SpValue& prim_join(SpValue& thread);
  static SpValue& prim_quit(SpValue& thread);

public:
  SpThread(SpValue& ns, SpValue& b) : assoc(ns), body(b) {}
  ~SpThread();

  bool isThread(){ return true; }
  const char* typeString(){ return "thread"; }
  SpValue& toString();
  SpValue& onMessage(SpValue& rec, SpValue& msg);

  static void init();
  static void init2();
  static void finalize();
};

inline SpValue* getCurrentResult(){
  ThreadContext* context;
  //  context = (ThreadContext*)pthread_getspecific(pthread_key);
  context = (ThreadContext*)TlsGetValue(tls_key);
  return &(context->gResult);
}

class SpMutex : public SpObject {
private:
  HANDLE mutex;

  static SpValue& prim_synchronize(SpValue& mutex, SpValue& func);
  static SpValue& prim_new();

public:
  SpMutex(){
    mutex = CreateMutex(NULL, FALSE, NULL);
    if(mutex == NULL){
      throw SpException("can't create mutex");
    }
  }
  ~SpMutex(){
    CloseHandle(mutex);
  }

  void lock(){
    WaitForSingleObject(mutex, INFINITE);
  }
  void unlock(){
    ReleaseMutex(mutex);
  }

  bool isMutex(){ return true; }
  const char* typeString(){ return "mutex"; }
  SpValue& toString();
  SpValue& onMessage(SpValue& rec, SpValue& msg);

  static void init();
};

#endif  /* USE_W32THREAD */

#ifndef THREAD

extern SpValue gResult;

inline SpValue* getCurrentResult(){
  return &gResult;
}

#endif /* not THREAD */

inline SpValue& SpBoolResult(SpBool p){
  SpValue* res = getCurrentResult();
  res->setBool(p);
  return *res;
}
inline SpValue& SpIntResult(SpInt i){
  SpValue* res = getCurrentResult();
  res->setInt(i);
  return *res;
}
inline SpValue& SpCharResult(SpChar c){
  SpValue* res = getCurrentResult();
  res->setSpChar(c);
  return *res;
}
inline SpValue& SpRealResult(SpReal r){
  SpValue* res = getCurrentResult();
  res->setReal(r);
  return *res;
}
inline SpValue& SpObjectResult(SpObject* obj){
  SpValue* res = getCurrentResult();
  res->setObject(obj);
  return *res;
}
inline SpValue& SpValueResult(SpValue& val){
  SpValue* res = getCurrentResult();
  *res = val;
  return *res;
}

/*
 * Readers for SOOPY
 */

class Reader : public virtual SpObject {
private:
  bool unread;
  SpChar unreadChar;

protected:
  virtual char* getFilename(){ return "a reader"; }

public:
  Reader(){ unread=false; }

  virtual SpChar fromChar(char c)
    {
      return MakeSpChar(CodeJIS, c);
    }
  virtual SpChar readCh() = 0;
  virtual bool eof() = 0;
  virtual bool is_open() = 0;
  virtual void close(){}
  virtual void setReader(Reader* r){
    throw SpException("reader does not have a reader (setReader)");
  }
  virtual Reader* getReader(){
    throw SpException("reader does not have a reader (getReader)");
    return NULL;
  }
  SpChar ReadChar(){
    if(!unread){
      unreadChar = readCh();
    }
    unread = false;
    return unreadChar;
  }
  void UnreadChar(SpChar ch){
    unread = true;
    if(unreadChar != ch){
      throw SpException("not match unreadChar");
    }
  }
  virtual SpValue& Read(SpInt n){
    throw SpException("reader does not have a function 'Read'");
    return NilObject;
  }
  virtual SpValue& ReadByte(SpInt n, unsigned char* buf){
    throw SpException("reader does not have a function 'ReadByte'");
    return NilObject;
  }
  SpValue& ReadLine();
  SpValue& ReadLineS();
  virtual SpValue& ScanByte(SpString*, Reader*){
    throw SpException("reader does not have a function 'ScanByte'");
    return NilObject;
  }
  virtual SpValue& SkipByte(SpInt n){
    throw SpException("reader does not have a function 'SkipByte'");
    return NilObject;
  }
  virtual SpValue& ScanByteStr(int count){
    throw SpException("reader does not have a function 'ScanByte'");
    return NilObject;
  }

  SpChar ReadAsLowerChar(){
    SpChar c = ReadChar();
    return toHankakuLower(c);
  }
  void UnreadAsLowerChar(SpChar ch){
    unread = true;
  }

  bool isReader(){ return true; }
  SpValue& toString();
  const char* typeString(){ return "reader"; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  static void init();

  // primitives
  static SpValue& prim_close(SpValue& fin);
  static SpValue& prim_closed(SpValue& fin);
  static SpValue& prim_readChar(SpValue& fin);
  static SpValue& prim_unreadChar(SpValue& fin, SpValue& ch);
  static SpValue& prim_peekChar(SpValue& fin);
  static SpValue& prim_readLine(SpValue& fin);
  static SpValue& prim_readLineS(SpValue& fin);
  static SpValue& prim_read(SpValue& fin, SpValue& n);
  static SpValue& prim_readbyte(SpValue& fin, SpValue& n, SpValue& byte_vector);
  static SpValue& prim_eof(SpValue& fin);
  static SpValue& prim_skipbyte(SpValue& fin, SpValue& n);
  static SpValue& prim_scanbyte(SpValue& fin, SpValue& str);

  friend class ReadEncoder;
};

class StreamReader : public Reader {
protected:
  istream* stream;
  SpValue& ScanByteStr(int count);

public:
  StreamReader(istream* s) : stream(s) {}

  void setStream(istream* s){ stream = s; }
  istream* getStream(){ return stream; }

#ifdef OSX
  bool eof(){ return stream->eof(); }
#else
  bool eof(){
    unsigned char c;
    stream->read((char*)&c, 1);
    bool p = stream->eof();
    stream->unget();
    return p;
  }
#endif
  bool is_open(){ return true; }
  SpChar readCh(){
    unsigned char c;
    stream->read((char*)&c, 1);
    return (SpChar)c;
  }
  SpValue& Read(SpInt n);
  SpValue& ReadByte(SpInt n, unsigned char* buf);
  SpValue& SkipByte(SpInt n);
  SpValue& ScanByte(SpString* format, Reader* self);
};

class FileStreamReader : public StreamReader {
private:
  char* filename;
  char* getFilename(){ return filename; }
public:
  FileStreamReader(ifstream* s, char* n)
       : StreamReader(s),
         filename(n)
    {}

  bool is_open(){ return ((ifstream*)stream)->is_open(); }
  void close(){ ((ifstream*)stream)->close(); }
};

class ReadEncoder : public Reader {
protected:
  SpValue taker;
  Reader* reader;
  char* getFilename(){ return reader->getFilename(); }
  SpValue& ScanByteStr(int count);
public:
  ReadEncoder(Reader* r) : reader(r) { taker.setObject(r); }

  virtual void setReader(Reader* r){
    reader = r;
  }
  Reader* getReader(){ return reader; }

  bool eof(){ return reader->eof(); }
  void close(){ reader->close(); }
  bool is_open(){ return reader->is_open(); }
  SpChar fromChar(char c){ return reader->fromChar(c); }
  SpValue& Read(SpInt n){ return reader->Read(n); }
  SpValue& ReadByte(SpInt n, unsigned char* buf){ return reader->ReadByte(n, buf); }
  SpValue& SkipByte(SpInt n){ return reader->SkipByte(n); }
  SpValue& ScanByte(SpString* format, Reader*){
      return reader->ScanByte(format, this);
  }
};

class RawReadEncoder : public ReadEncoder {
private:
  SpChar readCh(){ return reader->ReadChar(); }
public:
  RawReadEncoder(Reader* r) : ReadEncoder(r) {}
};

class SjisReadEncoder : public ReadEncoder {
private:
  SpChar buf[1];
  int count;
  SpChar readCh();
public:
  SjisReadEncoder(Reader* r) : ReadEncoder(r) { count=-1; }
  bool eof(){ return (count < 0) && reader->eof(); }

  friend class JapaneseReadEncoder;
};

class SjisReadDecoder : public ReadEncoder {
private:
  SpChar readCh();
public:
  SjisReadDecoder(Reader* r) : ReadEncoder(r) {}

  friend class JapaneseReadEncoder;
};

class EucJPReadEncoder : public ReadEncoder {
private:
  SpChar buf[1];
  int count;
  SpChar readCh();
public:
  EucJPReadEncoder(Reader* r) : ReadEncoder(r) { count=-1; }
  bool eof(){ return (count < 0) && reader->eof(); }

  friend class JapaneseReadEncoder;
};

class EucJPReadDecoder : public ReadEncoder {
private:
  SpChar readCh();
public:
  EucJPReadDecoder(Reader* r) : ReadEncoder(r) {}

  friend class JapaneseReadEncoder;
};

const int kCodeASCII    = 0;
const int kCodeJISX0208 = 1;
const int kCodeUNKNOWN  = 2;
const int kCodeESCAPE   = 3; // escape sequence

class ISO2022JPReadEncoder : public ReadEncoder {
private:
  SpChar buf[5];
  int count;
  int out_now;
  SpChar readCh();
public:
  ISO2022JPReadEncoder(Reader* r) : ReadEncoder(r) {
    out_now  = kCodeASCII;
    count = -1;
  }
  bool eof(){ return (count < 0) && reader->eof(); }
};

class ISO2022JPReadDecoder : public ReadEncoder {
private:
  int in_now;
  SpChar readCh();
public:
  ISO2022JPReadDecoder(Reader* r) : ReadEncoder(r) {
    in_now  = kCodeASCII;
  }
};

class JapaneseReadDecoder : public ReadEncoder {
private:
  SpValue EncTaker;
  ReadEncoder* encoder;

  SpChar readCh();
public:
  JapaneseReadDecoder(Reader* r) : ReadEncoder(r) {
    encoder = NULL;
  }
  void setEncoder(ReadEncoder* e){
    encoder = e;
    EncTaker.setObject(encoder);
  }
  virtual void setReader(Reader* r){
    reader = r;
    if(encoder != NULL){
      encoder->setReader(r);
    }
  }
};

class CRLF2LFReadEncoder : public ReadEncoder {
private:
  SpChar readCh();
public:
  CRLF2LFReadEncoder(Reader* r) : ReadEncoder(r) {}
};

class CRLF2CRReadEncoder : public ReadEncoder {
private:
  SpChar readCh();
public:
  CRLF2CRReadEncoder(Reader* r) : ReadEncoder(r) {}
};

class LF2CRLFReadEncoder : public ReadEncoder {
private:
  bool prevLF;
  SpChar readCh();
public:
  LF2CRLFReadEncoder(Reader* r) : ReadEncoder(r) { prevLF = false; }
};

class CR2CRLFReadEncoder : public ReadEncoder {
private:
  bool prevCR;
  SpChar readCh();
public:
  CR2CRLFReadEncoder(Reader* r) : ReadEncoder(r) { prevCR = false; }
};

class CR2LFReadEncoder : public ReadEncoder {
private:
  SpChar readCh();
public:
  CR2LFReadEncoder(Reader* r) : ReadEncoder(r) {}
};

class LF2CRReadEncoder : public ReadEncoder {
private:
  SpChar readCh();
public:
  LF2CRReadEncoder(Reader* r) : ReadEncoder(r) {}
};

class Base64ReadEncoder : public ReadEncoder {
private:
  int count;
  SpChar results[4];

  SpChar readCh();
public:
  Base64ReadEncoder(Reader* r) : ReadEncoder(r) {
    count = 0;
  }
};

class Base64ReadDecoder : public ReadEncoder {
private:
  int count;
  SpChar results[3];

  SpChar readCh();
public:
  Base64ReadDecoder(Reader* r) : ReadEncoder(r) {
    count = 0;
  }
};

class URLReadEncoder : public ReadEncoder {
private:
  int count;
  SpChar results[6];

  SpChar readCh();
public:
  URLReadEncoder(Reader* r) : ReadEncoder(r) {
    count = 0;
    results[0] = MakeSpChar(CodeJIS, '%');
    results[3] = MakeSpChar(CodeJIS, '%');
  }
  bool eof(){ return (count == 0) && reader->eof(); }
};

class URLReadDecoder : public ReadEncoder {
private:
  SpChar readCh();
public:
  URLReadDecoder(Reader* r) : ReadEncoder(r) {}
};

/*
 * Writers for SOOPY
 */

class Writer : public virtual SpObject {
protected:
  void write(SpObject& obj);
  void write(SpList* list);
  virtual char* getFilename(){ return "a writer"; }
  virtual void writeCh(unsigned char ch) = 0;
public:
  virtual void setWriter(Writer* w){
    throw SpException("writer don't have a writer (setWriter)");
  }
  virtual Writer* getWriter(){
    throw SpException("writer don't have a writer (getWriter)");
    return NULL; // not reached
  }
  //    virtual void WriteChar(SpChar c) = 0;
  virtual void WriteChar(SpChar c){
    writeCh(c);
  }
  virtual bool is_open() = 0;
  virtual void close(){}
  virtual void flush(){}
  SpValue& Write(SpValue& val);
  SpValue& WriteLine(SpValue& val);
  SpValue& WriteLineS(SpValue& list);
  SpValue& Terpri();
  virtual SpValue& WriteByte(SpInt n, unsigned char* buf){
    throw SpException("writer does not have a function 'WriteByte'");
    return NilObject;
  }

  Writer& operator<<(const char* str);
  Writer& operator<<(const SpInt i);
  Writer& operator<<(const SpReal r);
  void writeTextTronCode(SpChar c);

  bool isWriter(){ return true; }
  SpValue& toString();
  const char* typeString(){ return "writer"; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  static void init();

  // primitives
  static SpValue& prim_close(SpValue& fout);
  static SpValue& prim_closed(SpValue& fout);
  static SpValue& prim_writeChar(SpValue& fout, SpValue& ch);
  static SpValue& prim_write(SpValue& fout, SpValue& val);
  static SpValue& prim_writeLine(SpValue& fout, SpValue& val);
  static SpValue& prim_writeLineS(SpValue& fout, SpValue& val);
  static SpValue& prim_flush(SpValue& fout);
  static SpValue& prim_terpri(SpValue& fout);
  static SpValue& prim_writebyte(SpValue& fout, SpValue& n, SpValue& byte_vector);

  friend class WriteEncoder;
};

class StreamWriter : public Writer {
protected:
  ostream* stream;
public:
  StreamWriter(ostream* s) : stream(s) {}

  void setStream(ostream* s){ stream = s; }
  ostream* getStream(){ return stream; }
  bool is_open(){ return true; };
  void close(){}
  //    void WriteChar(SpChar ch);
  void writeCh(unsigned char ch);
  //void writeCh(SpChar ch);
  SpValue& WriteByte(SpInt n, unsigned char* buf);
};

class FileStreamWriter : public StreamWriter {
private:
  char* filename;
  char* getFilename(){ return filename; }
public:
  FileStreamWriter(ofstream* s, char* n)
       : StreamWriter(s),
         filename(n)
    {}
  ~FileStreamWriter(){
    if(is_open()){
      close();
    }
  }

  bool is_open(){ return ((ofstream*)stream)->is_open(); }
  void close(){ ((ofstream*)stream)->close(); }
  void flush(){ ((ofstream*)stream)->flush(); }
};

class WriteEncoder : public Writer {
protected:
  Writer* writer;
  void writeCh(unsigned char ch){ writer->writeCh(ch); }
  char* getFilename(){ return writer->getFilename(); }
public:
  WriteEncoder(Writer* w) : writer(w) { writer->links++; }
  ~WriteEncoder(){ writer->links--; }

  void WriteChar(SpChar ch){ writer->WriteChar(ch); }
  SpValue& WriteByte(SpInt n, unsigned char* buf){ return writer->WriteByte(n, buf); }
  virtual void setWriter(Writer* w){
    writer = w;
  }
  virtual Writer* getWriter(){
    return writer;
  }
  bool is_open(){ return writer->is_open(); }
  void close(){ flush(); writer->close(); }
  void flush(){ writer->flush(); }

  WriteEncoder& operator<<(const char* str){
    *writer << str;
    return *this;
  }
  WriteEncoder& operator<<(const SpInt i){
    *writer << i;
    return *this;
  }
  WriteEncoder& operator<<(const SpReal r){
    *writer << r;
    return *this;
  }
  WriteEncoder& operator<<(SpObject& obj){
    write(obj);
    return *this;
  }
  WriteEncoder& operator<<(SpValue& v);
};

class RawWriteEncoder : public WriteEncoder {
public:
  RawWriteEncoder(Writer* w) : WriteEncoder(w) {}
};

class SjisWriteEncoder : public WriteEncoder {
private:
  void WriteChar(SpChar c);
public:
  SjisWriteEncoder(Writer* w) : WriteEncoder(w) {}
};

class SjisWriteDecoder : public WriteEncoder {
private:
  int count;
  SpChar ch1;
  void WriteChar(SpChar c);
public:
  SjisWriteDecoder(Writer* w) : WriteEncoder(w) { count=0; }
};

class EucJPWriteEncoder : public WriteEncoder {
private:
  void WriteChar(SpChar c);
public:
  EucJPWriteEncoder(Writer* w) : WriteEncoder(w) {}
};

class EucJPWriteDecoder : public WriteEncoder {
private:
  int count;
  SpChar ch1;
  void WriteChar(SpChar c);
public:
  EucJPWriteDecoder(Writer* w) : WriteEncoder(w) { count=0; }
};

class ISO2022JPWriteEncoder : public WriteEncoder {
private:
  int out_now;

  void WriteChar(SpChar c);
public:
  ISO2022JPWriteEncoder(Writer* w) : WriteEncoder(w) {
    out_now = kCodeASCII;
  }
};

class ISO2022JPWriteDecoder : public WriteEncoder {
private:
  int in_now;
  int count;
  SpChar ch1, ch2;
  void WriteChar(SpChar c);
public:
  ISO2022JPWriteDecoder(Writer* w) : WriteEncoder(w) {
    in_now = kCodeASCII;
    count = 0;
  }
};

class CRLF2LFWriteEncoder : public WriteEncoder {
private:
  bool prevCR;
  void WriteChar(SpChar c);
public:
  CRLF2LFWriteEncoder(Writer* w) : WriteEncoder(w) { prevCR = false; }
};

class CRLF2CRWriteEncoder : public WriteEncoder {
private:
  bool prevCR;
  void WriteChar(SpChar c);
public:
  CRLF2CRWriteEncoder(Writer* w) : WriteEncoder(w) { prevCR = false; }
};

class LF2CRLFWriteEncoder : public WriteEncoder {
private:
  void WriteChar(SpChar c);
public:
  LF2CRLFWriteEncoder(Writer* w) : WriteEncoder(w) {}
};

class CR2CRLFWriteEncoder : public WriteEncoder {
private:
  void WriteChar(SpChar c);
public:
  CR2CRLFWriteEncoder(Writer* w) : WriteEncoder(w) {}
};

class CR2LFWriteEncoder : public WriteEncoder {
private:
  void WriteChar(SpChar c);
public:
  CR2LFWriteEncoder(Writer* w) : WriteEncoder(w) {}
};

class LF2CRWriteEncoder : public WriteEncoder {
private:
  void WriteChar(SpChar c);
public:
  LF2CRWriteEncoder(Writer* w) : WriteEncoder(w) {}
};

class Base64WriteEncoder : public WriteEncoder {
private:
  int src, count;

  //void writeCh(unsigned char ch);
public:
  Base64WriteEncoder(Writer* w) : WriteEncoder(w) {
    count = 0;
  }
  void WriteChar(SpChar ch);
  void flush();
};

class Base64WriteDecoder : public WriteEncoder {
private:
  int src, count;
  bool eof;

  //void writeCh(unsigned char ch);
public:
  Base64WriteDecoder(Writer* w) : WriteEncoder(w) {
    count = 0;
    eof = false;
  }
  void WriteChar(SpChar ch);
  void flush();
};

class URLWriteEncoder : public WriteEncoder {
private:
  //void writeCh(unsigned char ch);
public:
  URLWriteEncoder(Writer* w) : WriteEncoder(w) {}
  void WriteChar(SpChar ch);
};

class URLWriteDecoder : public WriteEncoder {
private:
  int count;
  unsigned char src;

  //void writeCh(unsigned char ch);
public:
  URLWriteDecoder(Writer* w) : WriteEncoder(w) {
    count = 0;
  }
  void WriteChar(SpChar ch);
};

class IO : public Reader, public Writer {
public:
  SpValue& toString() = 0;
  const char* typeString() = 0;
  SpValue& onMessage(SpValue& rec, SpValue& msg) = 0;
};

/*
 * Soopy String Class
 */

typedef vector<SpChar> SpCharVector;

class SpString : public SpObject {
protected:
  SpCharVector str;

  void append(const char* s, ReadEncoder& reader = *spin);
public:
  SpString(){};
  //SpString(const char* s){ append(s); }
  SpString(const char* s, ReadEncoder& enc=*spin){ append(s, enc); }
  SpString(SpCharVector& vec){ str = vec; }
  SpString(SpString& s2){ str = s2.str; }
  ~SpString();

  SpValue& toString();
  void clear(){ str.clear(); }
  SpCharVector::iterator begin(){ return str.begin(); }
  SpCharVector::iterator end(){ return str.end(); }
  //SpCharVector::iterator find(SpString* s);
  const char* toCStringWithEncoder(WriteEncoder& writer=*spout);
  const char* toCString(WriteEncoder& writer=*spout){
    return toCStringWithEncoder(writer);
  }
  const char* typeString(){ return "string"; }
  bool isString(){ return true; }
  int length(){ return str.size(); }
  SpType getType(){ return TypeString; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  SpValue& plus(SpValue& e1, SpValue& e2);
  bool match(SpValue& self, SpValue& val, SpNameSpace* ns);
  SpValue& eq(SpValue& e1, SpValue& e2);
  SpValue& gt(SpValue& e1, SpValue& e2);
  SpValue& lt(SpValue& e1, SpValue& e2);
  SpTuple* split_path_file();
  bool match_glob(SpString* glob);
  SpValue& split(SpChar c);

  void addSpChar(SpChar c){
    str.push_back(c);
  }

  void insertSpChar(SpChar c){
    str.insert(str.begin(), c);
  }

  SpString& operator=(const char* s)
    {
      str.clear();
      append(s);
      return *this;
    }

  SpString& operator+=(const char* s)
    {
      append(s);
      return *this;
    }
  SpString& operator+=(SpValue& s)
    {
      append(s.toCString());
      return *this;
    }
  SpString& operator+=(SpString& s);

  bool operator==(SpObject& obj);
  bool operator<(SpObject& obj);
  SpChar operator[](unsigned int index){ return str[index]; }

  static void init();
  // primitives
  static SpValue& prim_nth(SpValue& self, SpValue& index);
  static SpValue& prim_sub(SpValue& self, SpValue& from, SpValue& len);
  static SpValue& prim_split(SpValue& self, SpValue& ch);
  static SpValue& prim_split1(SpValue& self, SpValue& ch);
  static SpValue& prim_tr(SpValue& self, SpValue& ch1, SpValue& ch2);
  static SpValue& prim_reader(SpValue& self);
  static SpValue& prim_upper(SpValue& self);
  static SpValue& prim_lower(SpValue& self);
  static SpValue& prim_replace(SpValue& self, SpValue& s1, SpValue& s2);
  static SpValue& prim_replace1(SpValue& self, SpValue& s1, SpValue& s2);
  static SpValue& prim_toInt(SpValue& self);
  static SpValue& prim_find(SpValue& self, SpValue& s, SpValue& from);
  static SpValue& prim_format(SpValue& self);

  friend class Writer;
  friend class SpStringCmp;
};

class SpStringCmp {
public:
  bool operator()(SpString* s1, SpString* s2) const {
    return s1->str < s2->str;
  }
};

class StringReader : public Reader {
private:
  SpCharVector::iterator it;
  SpCharVector::iterator end;
  SpValue str;
public:
  StringReader(SpString* s){
    str.setNewObject(s);
    it = s->begin();
    end = s->end();
  }

  bool eof(){
    return !(it < end);
  }
  bool is_open(){ return true; }
  SpChar readCh()
    {
      if(eof()){
        return (SpChar)NULL;
      }
      SpChar c = *it;
      it++;
      return c;
    }
  /*
    void unreadCh(SpChar c){
        it--;
    }
   */
};

/*
 * Soopy Tuple Class
 */

typedef vector<SpValue> SpValueVector;

class SpTuple : public SpObject {
private:
  SpValueVector vec;
public:
  SpTuple(SpValueVector& v) : vec(v) {}
  SpTuple(SpValue& v){ append(v); }
  ~SpTuple();
  static void init();
  bool canEval(){ return true; }
  SpValue& eval();
  SpValue& toString();
  const char* typeString(){ return "tuple"; }
  bool isTuple(){ return true; }
  SpType getType(){ return TypeTuple; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  bool match(SpValue& self, SpValue& val, SpNameSpace* ns);

  SpInt length(){ return vec.size(); }
  SpValueVector::iterator begin(){ return vec.begin(); }
  SpValueVector::iterator end(){ return vec.end(); }
  SpValueVector::reverse_iterator rbegin(){ return vec.rbegin(); }
  SpValueVector::reverse_iterator rend(){ return vec.rend(); }
  void append(SpValue& val){ vec.push_back(val); }

  bool operator==(SpObject& obj);
  bool operator<(SpObject& obj);
  SpValue& operator[](unsigned int index){ return vec[index]; }

  // primitives
  static SpValue& get_length(SpValue& self);
  static SpValue& first(SpValue& self);
  static SpValue& second(SpValue& self);
  static SpValue& third(SpValue& self);
  static SpValue& fourth(SpValue& self);
  static SpValue& fifth(SpValue& self);
  static SpValue& nth(SpValue& self, SpValue& index);

  friend int yyparse();
  friend class SpClosureN;
  friend class SpConstructor;
};

/*
 * Soopy List Class
 */

class SpList : public SpObject {
protected:
  virtual SpList* next() = 0;
  static SpValue& sub_foldr(SpFunc* f, SpValue& v1, SpList* list);
  SpList* nth(int i);

public:
  static void init();
  virtual SpValue& value() = 0;
  virtual SpValue& nextList();
  virtual int length(){ throw SpException("can't calc length"); return 0; }
  bool isList(){ return true; }
  SpValue& toString();
  const char* typeString(){ return "list"; }
  SpType getType(){ return TypeList; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  bool match(SpValue& self, SpValue& val, SpNameSpace* ns);
  SpValue& eq(SpValue& e1, SpValue& e2);

  // primitives
  static SpValue& prim_head(SpValue& list);
  static SpValue& prim_tail(SpValue& list);
  static SpValue& prim_nth(SpValue& self, SpValue& index);
  static SpValue& prim_each(SpValue& self, SpValue& func);
  static SpValue& prim_map(SpValue& self, SpValue& func);
  static SpValue& prim_foldl(SpValue& self, SpValue& func);
  static SpValue& prim_foldr(SpValue& self, SpValue& func);
  static SpValue& prim_member(SpValue& self, SpValue& value);
  static SpValue& prim_sort(SpValue& self);
  static SpValue& prim_filter(SpValue& self, SpValue& func);
  static SpValue& prim_length(SpValue& self);

  friend class SpContainer;
  friend class Writer;
  friend class SpAssignList;
  friend class SpListCompre;
  friend class SpAppendedList;
  friend class SpListCompreDiag;
};

class SpCons : public SpList {
private:
  SpValue car;
  SpValue cdr;

  SpList* next()
    {
      if(cdr.isNil()){
        return NULL;
      }
      SpObject* obj = cdr.getObject();
      if(obj != NULL){
        return dynamic_cast<SpList*>(obj);
      }
      return NULL;
    }
public:
  SpCons(SpValue& head, SpValue& tail = NilObject){
    car = head;
    cdr = tail;
  }
  ~SpCons();

  SpValue& value(){ return car; }
  SpValue& nextList(){ return cdr; }
  void append(SpValue& val);
  void setNext(SpValue& next){ cdr = next; } // next: SpList
  int length();
  bool isCons(){ return true; }
  SpCons& operator+=(SpValue& val){ append(val); return *this; }

  bool operator==(SpObject& obj);
  bool operator<(SpObject& obj);

  friend class Reader;
  //friend class ReadEncoder;
  friend class SpList;
  friend class SpDir;
};

class SpRange : public SpList {
private:
  int start, end, step;
  bool infinity;

  SpList* next();
public:
  SpRange(int s, int e, int st = 1, bool p = false){
    start = s;
    end = e;
    step = st;
    infinity = p;
  }
  ~SpRange();
  SpValue& value();
  SpValue& toString();
  int length();
};

class SpMakeRange : public SpObject {
private:
  SpValue vstart;
  SpValue vend;
  SpValue vstep;
  bool infinity;
  bool hasStep;

  const char* typeString(){ return "*range maker*"; }
  SpValue& toString();

public:
  SpMakeRange(SpValue& s, SpValue& e, SpValue& st, bool p = false)
       : vstart(s), vend(e), vstep(st), infinity(p)
    {
      hasStep = true;
    }
  SpMakeRange(SpValue& s, SpValue& e, bool p = false)
       : vstart(s), vend(e), infinity(p)
    {
      hasStep = false;
    }

  bool canEval(){ return true; }
  SpValue& eval();
};

// class SpAppendedList
//   first @ second
class SpAppendedList : public SpList {
private:
  SpValue first;
  SpValue second;
  SpValue environ;   // SpNameSpace

  SpList* next();

public:
  SpAppendedList(SpValue& f, SpValue& s, SpValue& env) : first(f), second(s), environ(env) {}

  SpValue& value();
};

//
// list comprehension
//   ex. [x | x <- [1..10], x % 2 != 0]
//

class SpListCreator : public SpObject {
private:
  SpValue symbol;
  SpValue expr;
  SpValue list;

public:
  SpListCreator(SpValue& sym, SpValue& e){
    symbol = sym;
    expr   = e;
  }

  const char* typeString(){ return "*list creator*"; }
  SpValue& toString();

  friend class SpListCompreBase;
  friend class SpListCompre;
  friend class SpListCompreDiag;
};

class SpListCompreBase : public SpList {
protected:
  SpValue expr;
  SpValue creators;
  SpValue filters;
  SpValue environ;   // SpNameSpace
  SpValue current_value;

  SpListCompreBase(SpValue& cur_val, SpValue& e, SpValue& c, SpValue& f, SpValue& env){
    expr     = e;
    creators = c;
    filters  = f;
    environ  = env;
    // set current_value
    current_value = cur_val;
  }

public:
  SpValue& value(){
    return current_value;
  }
};

class SpListCompre : public SpListCompreBase {
private:
  SpValueVector cur_creators;

  SpList* next();

  static void copy_tuple_to_valVec(SpValueVector* dst, SpValue& src);
  static void copy_valVec_to_valVec(SpValueVector* dst, SpValueVector* src);
  static bool calc_next_param(SpValue& c, SpValueVector* v);
  static bool calc_value(SpValue& result, SpValue& e, SpValue& f, SpValueVector* v, SpValue& env);
  static SpListCompre* calc_next_list(SpValue& e, SpValue& c, SpValue& f, SpValue& env, SpValueVector* v=NULL);

  SpListCompre(SpValue& cur_val, SpValue& e, SpValue& c, SpValue& f, SpValueVector* v, SpValue& env);

public:
  static SpListCompre* Create(SpValue& e, SpValue& c, SpValue& f);

  //  int length();

  friend class SpListCompreCreator;
};

// class SpListCompreCreator
class SpListCompreCreator : public SpObject {
private:
  SpValue expr;
  SpValue creators;
  SpValue filters;

  SpValue& toString();
  const char* typeString(){ return "*list comprehension creator*"; }

public:
  SpListCompreCreator(SpValue& e, SpValue& c, SpValue& f){
    expr     = e;
    creators = c;
    filters  = f;
  }

  bool canEval(){ return true; }
  SpValue& eval();
};

//
// list comprehension (diagonalising)
//   ex. [(x, y) // x <- [1..], y <- [1..]]
//
typedef std::vector<int> IntVector;

class SpListCompreDiag : public SpListCompreBase {
private:
  int  max;
  int  depth;
  int* dim;
  //  int  len;
  IntVector lengths;
  IntVector rvalue;   // x1 + x2 + .. + xN = rvalue
  IntVector base_pos;
  IntVector move_pos;
  int  index;

  SpList* next();

  static bool dec_idx(int  max,
                      int& idx,
                      IntVector& rval,
                      IntVector& base,
                      IntVector& move,
                      int depth,
                      int* dimension);
  static bool calc_value(SpValue& result, SpValue& e, SpValue& c, SpValue& f, SpValue& env, int* dimension);
  static bool calc_next_param(IntVector& lengths, int max, int& idx, IntVector& rval, IntVector& base, IntVector& move, int depth, int* dimension);
  static bool next_rval(int  max, int& idx, IntVector& rval, IntVector& base, IntVector& move, int depth, int* dimension);
  static SpListCompreDiag* calc_next_list(IntVector& lengths, SpValue& e, SpValue& c, SpValue& f, SpValue& env, int max, int& idx, IntVector& rval, IntVector& base, IntVector& move, int depth, int* dimension=NULL);

  static void clearIntVector(IntVector& dst, int len);
  void copyIntVector(IntVector& dst, IntVector& src);
  void calcLength();

  SpListCompreDiag(SpValue& cur_val, SpValue& e, SpValue& c, SpValue& f, SpValue& env, int mx, int idx, IntVector& rval, IntVector& base, IntVector& move, int size, int* p)
       : SpListCompreBase(cur_val, e, c, f,env)
    {
      max      = mx;
      copyIntVector(rvalue, rval);
      copyIntVector(base_pos, base);
      copyIntVector(move_pos, move);
      index = idx;
      depth    = size;
      dim      = p;
      calcLength();
    }

  ~SpListCompreDiag(){
    if(dim != NULL){
      delete[] dim;
    }
  }

public:
  static SpListCompreDiag* Create(SpValue& e, SpValue& c, SpValue& f);

  //  int length(){ return len; }

  friend class SpListCompreDiagCreator;
};

// class SpListCompreDiagCreator
class SpListCompreDiagCreator : public SpObject {
private:
  SpValue expr;
  SpValue creators;
  SpValue filters;

  SpValue& toString();
  const char* typeString(){ return "*list comprehension creator*"; }

public:
  SpListCompreDiagCreator(SpValue& e, SpValue& c, SpValue& f){
    expr     = e;
    creators = c;
    filters  = f;
  }

  bool canEval(){ return true; }
  SpValue& eval();
};

/*
 * Soopy Symbol Object
 */

class SpSymbol : public SpString {
public:
  //SpSymbol(const char* s) : SpString(s) {}
  SpSymbol(const char* s){
    //        appendWithEncoder(s);
    append(s);
  }
  SpSymbol(SpCharVector& vec) : SpString(vec) {}
  SpSymbol(SpString& s) : SpString(s) {}
  ~SpSymbol(){
#ifdef TEST
    cout << ":destroy SpSymbol" << endl;
#endif
  }
  static void init();
  bool canEval(){ return true; }
  SpValue& eval();
  const char* typeString(){ return "symbol"; }
  SpType getType(){ return TypeSymbol; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  SpValue& toString();
  bool match(SpValue& self, SpValue& val, SpNameSpace* ns);
  SpValue& eq(SpValue& e1, SpValue& e2);

  bool isString(){ return false; }
  bool isSymbol(){ return true; }

  bool operator==(SpObject& obj);
  bool operator<(SpObject& obj);
};

/*
 * Soopy NameSpace Class
 *  & KeyClass NSKey : abstract class
                   NSVar
                   NSConst
                   NSProperty
                   NSFunc
                   NSDataType
 *  & ValueClass SpValue
 */

class NSKey : public SpObject {
public:
  SpValue val;
  bool isPublic;
  virtual bool isNSConst(){ return true; }
  virtual bool isNSProperty(){ return false; }
  virtual bool isNSPrimProperty(){ return false; }
  virtual bool isNSFunc(){ return false; }
  virtual bool isNSDataType(){ return false; }
  bool isNSKey(){ return true; }

  NSKey(SpValue& v, bool p=true): val(v), isPublic(p) {}
  virtual SpValue& toString(){ return val.toString(); }
  const char* typeString(){ return "ns-key"; }

  bool operator<(SpObject& k){
    NSKey& key = dynamic_cast<NSKey&>(k);
    return val < key.val;
  }
  bool operator!=(NSKey& key) { return val != key.val; }
};

class NSVar : public NSKey {
public:
  NSVar(SpValue& v, bool p=true) : NSKey(v,p) {}
  ~NSVar() {}
  SpValue& toString();
  bool isNSConst(){ return false; }
};

class NSConst : public NSKey {
private:
  SpType typ;
public:
  NSConst(SpValue& v, bool p=true, SpType t=TypeNil)
       : NSKey(v,p), typ(t) {}
  ~NSConst() {}
  SpValue& toString();
};

class NSProperty : public NSKey {
private:
  SpValue getter, setter;
public:
  NSProperty(SpValue& key, SpValue& r, SpValue& w, bool p=true)
       : NSKey(key,p), getter(r), setter(w) {}
  ~NSProperty() {}
  SpValue& toString();
  bool isNSProperty(){ return true; }

  friend class SpNameSpace;
};

class NSPrimProperty : public NSKey {
private:
  SpValue getter, setter; // SpPrim
public:
  NSPrimProperty(SpValue& key, SpValue& r, SpValue& w, bool p=true)
       : NSKey(key,p), getter(r), setter(w) {}
  ~NSPrimProperty() {}
  SpValue& toString();
  bool isNSPrimProperty(){ return true; }

  friend class SpNameSpace;
};

class NSFunc : public NSKey {
public:
  NSFunc(SpValue& v, bool p=true) : NSKey(v,p) {}
  ~NSFunc() {}
  SpValue& toString() { return val.toString();}
  bool isNSFunc(){ return true; }
};

class NSDataType : public NSKey {
public:
  NSDataType(SpValue& symbol, bool p=true) : NSKey(symbol, p) {}
  SpValue& toString() { return val.toString();}
  bool isNSDataType(){ return true; }
};

class VVCmp {
public:
  //    bool operator()(SpValue key1, SpValue key2) const;
  bool operator()(const SpValue& key1, const SpValue& key2) const;
};
/*
class NSKeyCmp {
public:
    bool operator()(SpValue key1, SpValue key2);
};
 */
//typedef map<NSKey*, SpValue, NSKeyCmp> NSMap;
//typedef pair<NSKey*, SpValue> NSPair;

typedef map<SpValue, SpValue, VVCmp> NSMap;
//typedef map<SpValue, SpValue, NSKeyCmp> NSMap;
typedef pair<SpValue, SpValue> NSPair;

// for make SpNameSpace in yyparse
class NSPairAdaptor : public SpObject {
public:
  NSPair* pair;

  NSPairAdaptor(NSPair* p){ pair = p; }
  ~NSPairAdaptor(){ delete pair; }
  SpValue& toString(){ return NilObject; }
  const char* typeString(){ return "pair-adaptor"; }
};

// for make SpNameSpace in yyparse
class NSMapAdaptor : public SpObject {
public:
  NSMap* map;

  NSMapAdaptor(NSMap* m){ map = m; }
  ~NSMapAdaptor(){ delete map; }
  SpValue& toString(){ return NilObject; }
  const char* typeString(){ return "map-adaptor"; }
};

// for SpNameSpace
class SpPatternWColon : public SpObject {
private:
  SpValue head, tail;
public:
  SpPatternWColon(SpValue& v1, SpValue& v2)
       : head(v1), tail(v2) {}
  SpValue& toString();
  const char* typeString(){ return "pattern::"; }
  bool match(SpValue& self, SpValue& val, SpNameSpace* ns);
};

// for SpNameSpace
class SpPatternAt : public SpObject {
private:
  SpValue list1, list2;
public:
  SpPatternAt(SpValue& v1, SpValue& v2)
       : list1(v1), list2(v2) {}
  SpValue& toString();
  const char* typeString(){ return "pattern@"; }
  bool match(SpValue& self, SpValue& val, SpNameSpace* ns);
};

// for SpNameSpace
class SpPatternCon : public SpObject {
private:
  SpValue symbol; // symbol
  SpValue tuple;  // tuple
public:
  SpPatternCon(SpValue& sym, SpValue& t)
       : symbol(sym), tuple(t) {}
  SpValue& toString();
  const char* typeString(){ return "constructor pattern"; }
  bool match(SpValue& self, SpValue& val, SpNameSpace* ns);
};

///////////////////////////////////////
////////////// NameSpace //////////////
///////////////////////////////////////

class SpNameSpace : public SpObject {
private:
  NSMap aMap;
  SpValue parentNS;
  bool reachable(SpNameSpace* ns);
  void intern(SpValue& key, SpValue& val);
  void setParent(SpValue& p){
    parentNS = p;
  }
  SpValue& getValue(NSKey* key,
                    SpNameSpace* searcher,
                    SpNameSpace* cur_ns=NULL,
                    bool search_parent=true);
public:
  SpNameSpace(NSMap& m, SpValue& p=NilObject);
  SpNameSpace(SpNameSpace* ns1, SpNameSpace* ns2, SpValue& p=NilObject);
  SpNameSpace(SpValue& p=NilObject);
  ~SpNameSpace();

  NSMap& getMap(){ return aMap; }
  static void init();
  SpValue& toString();
  const char* typeString(){ return "namespace"; }
  bool isNameSpace(){ return true; }
  SpType getType(){ return TypeNameSpace; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  SpValue& plus(SpValue& e1, SpValue& e2);

  void clear(){ aMap.clear(); }
  SpInt size(){ return aMap.size(); }
  NSMap::iterator begin(){ return aMap.begin(); }
  NSMap::iterator end(){ return aMap.end(); }

  SpValue& lookup(SpValue& v, SpNameSpace* searcher=NULL);
  SpValue& lookup1(SpValue& v, SpNameSpace* searcher=NULL);
  void setValue(SpValue& key, SpValue& val, SpNameSpace* searcher, SpNameSpace* cur_ns=NULL);
  void internVar(SpValue& v1, SpValue& v2, bool p=true){
    NSVar* var = new NSVar(v1,p);
    SpValue key(var);
    intern(key, v2);
  }
  void internConst(SpValue& v1, SpValue& v2, bool p=true, SpType t=TypeNil){
    NSConst* con = new NSConst(v1, p, t);
    SpValue key(con);
    intern(key, v2);
  }
  void internProperty(SpValue& v1, SpValue& v2, SpValue& r, SpValue& w, bool p=true){
    NSProperty* pro = new NSProperty(v1, r, w, p);
    SpValue key(pro);
    intern(key, v2);
  }
  void internPrimProperty(SpValue& v1, SpValue& v2, SpValue& r, SpValue& w, bool p=true){
    NSPrimProperty* pro = new NSPrimProperty(v1, r, w, p);
    SpValue key(pro);
    intern(key, v2);
  }
  void internFunc(SpValue& v1, SpValue& v2, bool p=true){
    NSFunc* func = new NSFunc(v1);
    SpValue key(func);
    intern(key, v2);
  }
  void internDataType(SpValue& v1, SpValue& v2, bool p=true){
    NSDataType* dt = new NSDataType(v1);
    SpValue key(dt);
    intern(key, v2);
  }

  bool operator==(SpObject& obj);
  bool operator<(SpObject& obj);

  SpNameSpace& operator+=(SpNameSpace& ns);
  SpNameSpace& operator+=(NSPair& p)
    {
      // insert data
      setValue(p.first, p.second, this);
      return *this;
    }


  // primitives
  static SpValue& rename(SpValue& self, SpValue& renames);
  static SpValue& make_func(SpValue& self);
  static SpValue& prim_append(SpValue& self, SpValue& key, SpValue& value);
  static SpValue& prim_keys(SpValue& self);
  static SpValue& prim_delete(SpValue& self, SpValue& key);
  static SpValue& prim_lookup(SpValue& self, SpValue& key);

#ifdef USE_CLX
  friend class SpContainer;
#endif
#ifdef AQUA
  friend class SpContainer;
#endif
  friend class MakeNameSpace;
  friend class SpPatternCon;
  friend class SpThread;
  friend int yyparse();
};

class MakeNameSpace : public SpObject {
private:
  SpValue ns; // SpNameSpace
public:
  MakeNameSpace(SpValue& names) : ns(names) {}
  bool canEval(){ return true; }
  SpValue& eval();
  SpValue& toString();
  const char* typeString(){ return "make namespace"; }
};

#ifdef USE_PTHREAD

inline SpValue& getCurrentNS()
{
  ThreadContext* context;
  context = (ThreadContext*)pthread_getspecific(pthread_key);
  return context->Frame.back();
}
inline SpValue& getPrevNS()
{
  ThreadContext* context;
  context = (ThreadContext*)pthread_getspecific(pthread_key);
  int size = context->Frame.size();
  if(size >= 2){
    return context->Frame[size-2];
  }
  return NilObject;
}
inline void pushCurrentNS(SpValue& ns)
{
  ThreadContext* context;
  context = (ThreadContext*)pthread_getspecific(pthread_key);
  context->Frame.push_back(ns);
}
inline void popCurrentNS()
{
  ThreadContext* context;
  context = (ThreadContext*)pthread_getspecific(pthread_key);
  context->Frame.pop_back();
}

#endif /* USE_PTHREAD */

#ifdef USE_W32THREAD

#ifndef THREAD
#define THREAD
#endif

inline SpValue& getCurrentNS()
{
  ThreadContext* context;
  context = (ThreadContext*)TlsGetValue(tls_key);
  if(context == 0){
    throw SpException("can't get tls value");
  }
  return context->Frame.back();
}
inline SpValue& getPrevNS()
{
  ThreadContext* context;
  context = (ThreadContext*)TlsGetValue(tls_key);
  if(context == 0){
    throw SpException("can't get tls value");
  }
  int size = context->Frame.size();
  if(size >= 2){
    return context->Frame[size-2];
  }
  return NilObject;
}
inline void pushCurrentNS(SpValue& ns)
{
  ThreadContext* context;
  context = (ThreadContext*)TlsGetValue(tls_key);
  if(context == 0){
    throw SpException("can't get tls value");
  }
  context->Frame.push_back(ns);
}
inline void popCurrentNS()
{
  ThreadContext* context;
  context = (ThreadContext*)TlsGetValue(tls_key);
  if(context == 0){
    throw SpException("can't get tls value");
  }
  context->Frame.pop_back();
}

#endif /* USE_W32THREAD */


#ifndef THREAD

extern vector<SpValue> Frame;
inline SpValue& getCurrentNS()
{
  return Frame.back();
}
inline SpValue& getPrevNS()
{
  int size = Frame.size();
  if(size >= 2){
    return Frame[size-2];
  }
  return NilObject;
}
inline void pushCurrentNS(SpValue& ns)
{
  Frame.push_back(ns);
}
inline void popCurrentNS()
{
  Frame.pop_back();
}

#endif /* not THREAD */

/*
 * Soopy Assign Class
 */

class SpAssign : public SpObject {
private:
  SpValue place;
  SpValue val;
public:
  SpAssign(SpValue& p, SpValue& v)
       : place(p), val(v) {}
  ~SpAssign();

  bool isAssign(){ return true; }
  bool canEval(){ return true; }
  SpValue& eval();
  SpValue& toString();
  const char* typeString(){ return "="; }

  bool operator==(SpObject& obj);
  bool operator<(SpObject& obj);

  friend class SpContainer;
};

class SpAssignS : public SpObject {
private:
  SpValueVector places;
  SpValue val;
public:
  SpAssignS(SpValueVector& p, SpValue& v)
       : places(p), val(v) {}
  ~SpAssignS();

  bool canEval(){ return true; }
  SpValue& eval();
  SpValue& toString();
  const char* typeString(){ return "let = "; }

  bool operator==(SpObject& obj);
  bool operator<(SpObject& obj);
};

class SpAssignSM : public SpObject {
private:
  SpValue place;
  SpValue val;
public:
  SpAssignSM(SpValue& p, SpValue& v)
       : place(p), val(v) {}
  ~SpAssignSM();

  bool canEval(){ return true; }
  SpValue& eval();
  SpValue& toString();
  const char* typeString(){ return "obj msg = "; }

  bool operator==(SpObject& obj);
  bool operator<(SpObject& obj);
};

class SpAssignList : public SpObject {
private:
  SpValueVector places;
  SpValue val;
public:
  SpAssignList(SpValueVector& p, SpValue& v)
       : places(p), val(v) {}
  ~SpAssignList();

  bool canEval(){ return true; }
  SpValue& eval();
  SpValue& toString();
  const char* typeString(){ return "let [...] = "; }

  bool operator==(SpObject& obj);
  bool operator<(SpObject& obj);
};

class SpAssignHT : public SpObject {
private:
  SpValue head;
  SpValue tail;
  SpValue val;
public:
  SpAssignHT(SpValue& h, SpValue& t, SpValue& v)
       : head(h), tail(t), val(v) {}
  ~SpAssignHT();

  bool canEval(){ return true; }
  SpValue& eval();
  SpValue& toString();
  const char* typeString(){ return "let head::tail = "; }

  bool operator==(SpObject& obj);
  bool operator<(SpObject& obj);
};


/*
 * SpSendMsg Class
 */

class SpSendMsg : public SpObject {
private:
  SpValue receiver, message;
public:
  SpSendMsg(SpValue& r, SpValue& m)
       : receiver(r), message(m) {}
  ~SpSendMsg();

  bool canEval(){ return true; }
  SpValue& eval();
  SpValue& toString();
  const char* typeString(){ return "obj msg"; }
  bool isSendMsg(){ return true; }

  bool operator==(SpObject& obj);
  bool operator<(SpObject& obj);

  friend class SpAssignSM;
};


/*
 *
 * Function class
 *
 */

typedef SpValue& (*PRIM0)();
typedef SpValue& (*PRIM1)(SpValue&);
typedef SpValue& (*PRIM2)(SpValue&, SpValue&);
typedef SpValue& (*PRIM3)(SpValue&, SpValue&, SpValue&);

class SpFunc : public SpObject {
protected:
  unsigned int argLen;
public:
  bool isFunc(){ return true; }
  virtual SpValue& operator()(SpValue& arg) = 0;
  SpValue& onMessage(SpValue& rec, SpValue& msg)
    {
      return (*this)(msg);
    }
  const char* typeString(){ return "function"; }

  friend class SpClosure;
  friend class SpClosureN;
};

//
// Primitive class
//

class SpPrim0 : public SpFunc {
  PRIM0 func;
public:
  SpPrim0(PRIM0 f) : func(f) { argLen = 0; }

  SpValue& operator()(SpValue& arg);
  SpValue& toString();
  const char* typeString(){ return "primitive0"; }
};

class SpPrim1 : public SpFunc {
  PRIM1 func;
public:
  SpPrim1(PRIM1 f) : func(f) { argLen = 1; }

  SpValue& operator()(SpValue& arg);
  SpValue& toString();
  const char* typeString(){ return "primitive1"; }
};

class SpPrim2 : public SpFunc {
  PRIM2 func;
public:
  SpPrim2(PRIM2 f) : func(f) { argLen = 2; }

  SpValue& operator()(SpValue& arg);
  SpValue& operator()(SpValue& arg1, SpValue& arg2);
  SpValue& toString();
  const char* typeString(){ return "primitive2"; }
};

class SpPrim3 : public SpFunc {
  PRIM3 func;
public:
  SpPrim3(PRIM3 f) : func(f) { argLen = 3; }

  SpValue& operator()(SpValue& arg);
  SpValue& operator()(SpValue& arg1, SpValue& arg2);
  SpValue& operator()(SpValue& arg1, SpValue& arg2, SpValue& arg3);
  SpValue& toString();
  const char* typeString(){ return "primitive3"; }
};

//
// class SpUsrFunc && class SpArg
//

class SpArg : public SpObject {
private:
  SpValue name; // SpSymbol
  SpValue typ;  // SpSymbol or NilObject
public:
  SpArg(SpValue& n, SpValue& t)
       : name(n), typ(t) {}
  SpValue& toString();
  const char* typeString(){ return "arg"; }
  SpValue& getVarName(){ return name; }
  SpValue& getVarType(){ return typ; }
  bool isArg(){ return true; }

  friend class SpUsrFunc;
  friend class SpConstructor;
};

class SpUsrFunc : public SpFunc {
private:
  SpValue name;    // SpSymbol
  SpValue args;    // SpTuple or NilObject
  SpValue vars;    // SpList or NilObject
  SpValue requir;  // SpList or NilObject
  SpValue body;    // SpNameSpace
  SpValue ensure;  // SpList or NilObject
  SpValue rescue;  // SpList or NilObject

  void checkRequire();
  void checkEnsure();
  bool doRescue();
public:
  SpUsrFunc(SpValue& n,
            SpValue& a,
            SpValue& v,
            SpValue& r,
            SpValue& b,
            SpValue& e,
            SpValue& res)
       : name(n), args(a), vars(v), requir(r), body(b), ensure(e), rescue(res)
    {
      if(a.isNil()){
        argLen = 0;
      }else{
        SpTuple* ptr = dynamic_cast<SpTuple*>(a.getObject());
        argLen = ptr->length();
      }
    }
  static SpUsrFunc* fromNameSpace(SpNameSpace* ns);

  SpValue& doBody();

  SpValue& operator()(SpValue& arg);
  SpValue& toString();
  SpValue& getName(){ return name; }
};

class SpRetry : public SpObject {
public:
  SpRetry(){}

  bool canEval(){ return true; }
  SpValue& eval(){ throw SpRetryException("uncaught retry"); return NilObject; }
  SpValue& toString();
  const char* typeString(){ return "retry"; }
};

//
// class SpClosure
//

class SpClosure : public SpFunc {
private:
  SpValue func; // SpFunc
  SpValue ns;   // SpNameSpace
public:
  SpClosure(SpValue& f, SpValue& aNS);

  SpValue& operator()(SpValue& a);
  SpValue& toString();
  const char* typeString(){ return "closure"; }
};

class SpClosureP2 : public SpFunc { // for Prim2 only
private:
  SpValue func; // Prim2
  SpValue arg;
  SpValue ns; // NameSpace
public:
  SpClosureP2(SpValue& f, SpValue& a, SpValue& aNS)
       : func(f), arg(a), ns(aNS) { argLen = 1; }

  SpValue& operator()(SpValue& a);
  SpValue& toString();
};

class SpClosureP3_1 : public SpFunc { // for Prim3 only
private:
  SpValue func; // Prim3
  SpValue arg;
  SpValue ns; // NameSpace
public:
  SpClosureP3_1(SpValue& f, SpValue& a, SpValue& aNS)
       : func(f), arg(a), ns(aNS) { argLen = 1; }

  SpValue& operator()(SpValue& a);
  SpValue& toString();
};

class SpClosureP3_2 : public SpFunc { // for Prim3 only
private:
  SpValue func; // Prim3
  SpValue arg1;
  SpValue arg2;
  SpValue ns; // NameSpace
public:
  SpClosureP3_2(SpValue& f, SpValue& a1, SpValue& a2, SpValue& aNS)
       : func(f), arg1(a1), arg2(a2), ns(aNS) { argLen = 1; }

  SpValue& operator()(SpValue& a);
  SpValue& toString();
};

class SpClosureN : public SpFunc {
private:
  SpValue func; // SpFunc
  SpValue args; // SpTuple
  SpValue ns;   // NameSpace
public:
  SpClosureN(SpValue& f, SpValue& a, SpValue& aNS);

  SpValue& operator()(SpValue& a);
  SpValue& toString();
};

typedef map<SpValue, SpValue, VVCmp> VVMap;

class MsgHandler {
  VVMap aMap;
  MsgHandler* parent;
public:
  MsgHandler(MsgHandler* p=NULL) : parent(p) {}
  ~MsgHandler();
  void append(SpValue& message, SpValue& handler);
  SpValue& operator()(SpValue& rec, SpValue& msg);
  bool hasMessage(SpValue& message);
};

/*
 * Exprs
 */

class SpExpr : public SpObject {
public:
  bool canEval(){ return true; }
  const char* typeString(){ return "expr"; }
};

class ExprUMinus : public SpExpr {
private:
  SpValue expr;
public:
  ExprUMinus(SpValue& e) : expr(e) {}
  SpValue& eval();
  SpValue& toString();
};

/*
 * SpQuote class
 */

class SpQuote : public SpExpr {
private:
  SpValue expr;
public:
  SpQuote(SpValue& v): expr(v) {}
  SpValue& eval(){ return expr; }
  SpValue& toString();
};

class ExprPlus : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprPlus(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprMinus : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprMinus(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprTimes : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprTimes(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprDiv : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprDiv(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprMod : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprMod(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprNOT : public SpExpr {
private:
  SpValue expr;
public:
  ExprNOT(SpValue& e1)
       : expr(e1) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprAND : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprAND(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprOR : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprOR(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprEQ : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprEQ(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprNE : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprNE(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprGT : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprGT(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprGE : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprGE(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprLT : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprLT(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprLE : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprLE(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprWColon : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprWColon(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

class ExprAT : public SpExpr {
private:
  SpValue expr1, expr2;
public:
  ExprAT(SpValue& e1, SpValue& e2)
       : expr1(e1), expr2(e2) {}
  SpValue& eval();
  SpValue& toString();
};

/*
 * If statement
 */

class SpIf : public SpExpr {
private:
  SpValue cond;  // condition
  SpValue assoc; // NSPairAdaptor Vector(SpTuple)
public:
  SpIf(SpValue& c, SpValue& a)
       : cond(c), assoc(a) {}
  SpValue& eval();
  SpValue& toString();
};

/*
 * Loop statement
 */

class SpLoop: public SpExpr {
private:
  SpValue symbol;
  SpValue from;       // from
  SpValue step;       // step
  SpValue body;       // do
  SpValue while_cond; // while
  SpValue until_cond; // until
  SpValue variant;    // variant
  SpValue invariant;  // invariant
public:
  SpLoop(SpValue& sym=NilObject,
         SpValue& fr=NilObject,
         SpValue& st=NilObject,
         SpValue& bd=NilObject,
         SpValue& wcond=NilObject,
         SpValue& ucond=NilObject,
         SpValue& vari=NilObject,
         SpValue& invari=NilObject)
       : symbol(sym),
         from(fr),
         step(st),
         body(bd),
         while_cond(wcond),
         until_cond(ucond),
         variant(vari),
         invariant(invari) {}
  SpValue& eval();
  SpValue& toString();
};

class SpExit : public SpExpr {
private:
  SpValue symbol, expr;
public:
  SpExit(SpValue& sym = NilObject, SpValue& ex  = NilObject)
       : symbol(sym), expr(ex) {}
  SpValue& eval();
  SpValue& toString();
};

class SpNext : public SpExpr {
private:
  SpValue symbol, expr;
public:
  SpNext(SpValue& sym = NilObject, SpValue& ex  = NilObject)
       : symbol(sym), expr(ex) {}
  SpValue& eval();
  SpValue& toString();
};

/*
 * Array
 */

typedef vector<unsigned int> uintvec;

class SpBaseArray : public SpObject {
protected:
  unsigned int depth;
  uintvec dimension;

public:
  SpBaseArray(){} // for TableOfTableView(soopyg)
  SpBaseArray(uintvec& v);

  const char* typeString(){ return "array"; }
  bool isArray(){ return true; }
  SpType getType(){ return TypeArray; }
  virtual SpValue& getFeatureType() = 0;

  static void init();
  // primitives
  static SpValue& getDepth(SpValue& self);
  static SpValue& getDimension(SpValue& self);
};

class SpArray : public SpBaseArray {
public:
  SpArray(){} // for TableOfTableView(soopyg)
  SpArray(uintvec& v) : SpBaseArray(v){}
  virtual SpValue& operator[](SpValue& tuple) = 0;
  virtual void set(SpValue& tuple, SpValue& value) = 0;
};

class SpObjectArray : public SpArray {
private:
  SpValue typ;       // SpSymbol or NilObject
  SpValueVector* pvec;
public:
  SpObjectArray(SpValue& t, uintvec& v);
  ~SpObjectArray(){
    if(pvec != NULL){
      delete pvec;
    }
  }
  SpValue& toString();
  SpValue& operator[](SpValue& tuple);
  void set(SpValue& tuple, SpValue& value);
  SpValue& getFeatureType(){ return typ; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);
};

class SpByteArray : public SpBaseArray {
private:
  unsigned char* buf;

public:
  SpByteArray(uintvec& v);
  ~SpByteArray(){
    if(buf != NULL){
      //            delete buf;
      free(buf);
    }
    buf = NULL;
  }

  unsigned char* getBuffer(){
    if(buf == NULL){
      throw SpException("null buffer (bytearray)");
    }
    return buf;
  }
  bool isByteArray(){ return true; }
  SpInt length(){ return (SpInt)dimension[0]; }
  SpValue& toString();
  unsigned char& operator[](SpValue& tuple);
  SpValue& getFeatureType();
  SpValue& onMessage(SpValue& rec, SpValue& msg);
};

class MakeArray : public SpExpr {
private:
  SpValue typ;    // SpSymbol or NilObject
  SpValue vec;    // SpTuple
public:
  MakeArray(SpValue& t, SpValue& v): typ(t), vec(v) {}
  SpValue& eval();
  SpValue& toString();
};


/*
 * DataType
 */

class SpDataType : public SpObject {
private:
  SpValue symbol;    // SpSymbol
  SpValueVector vec; // SpConstructor vector
public:
  SpDataType(SpValue& sym, SpValueVector& v)
       : symbol(sym), vec(v) {}
  SpValue& toString();
  const char* typeString(){ return "datatype"; }
  bool isDataType(){ return true; }
  void mapping2ns(SpValue& self, SpNameSpace* ns);

  friend int yyparse();
  friend void CheckType(SpValue& featureType, SpValue& value);
};

class SpConstructor : public SpFunc {
private:
  SpDataType* datatype;
  SpValue symbol; // SpSymbol
  SpValue args;   // Nil or SpTuple
  SpValue ns;     // SpNameSpace
  void setDataType(SpDataType* dt){ datatype = dt; }
public:
  SpConstructor(SpValue& sym, SpValue& a, SpValue& n)
       : symbol(sym), args(a), ns(n)
    {
      if(args.isNil()){
        argLen = 0;
      }else{
        SpTuple* t = args.asTuple();
        argLen = t->length();
      }
    }

  bool isConstructor(){ return true; }
  SpValue& operator()(SpValue& a);
  SpValue& toString();
  const char* typeString(){ return "constructor"; }
  bool canEval(){ return true; }
  SpValue& eval();

  friend class SpDataType;
  friend class SpDType;
  friend class SpPatternCon;
  friend int yyparse();
  friend void CheckType(SpValue& featureType, SpValue& value);
};

class SpDType : public SpObject {
private:
  SpConstructor* con;
  SpValue& symbol;
  SpValue ns;
public:
  SpDType(SpConstructor* c, SpValue& sym, SpValue& n)
       : con(c), symbol(sym), ns(n) {}
  SpValue& toString();
  const char* typeString(){ return symbol.toCString(); }
  SpType getType(){ return TypeDType; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  SpValue& eq(SpValue& e1, SpValue& e2);
  //bool match(SpValue& self, SpValue& val, SpNameSpace* ns);
  bool isDType(){ return true; }

  friend class SpPatternCon;
  friend void CheckType(SpValue& featureType, SpValue& value);
};


class SpDir : public SpObject {
public:
  SpDir(){};
  bool isDir(){ return true; }
  SpValue& toString();
  const char* typeString(){ return "dir"; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  static void init();
  // primitives
  static SpValue& Chdir(SpValue& dir, SpValue& path);
  static SpValue& Mkdir(SpValue& dir, SpValue& path);
  static SpValue& Rmdir(SpValue& dir, SpValue& path);
  static SpValue& List(SpValue& dir, SpValue& path);
  static SpValue& Pwd(SpValue& dir);
  static SpValue& prim_glob(SpValue&, SpValue& path);
};

class SpFile : public SpObject {
public:
  SpFile(){};
  bool isFile(){ return true; }
  SpValue& toString();
  const char* typeString(){ return "fileutil"; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  static void init();
  // primitives
  static SpValue& prim_exists(SpValue& dir, SpValue& path);
  static SpValue& prim_delete(SpValue&, SpValue& path);
  static SpValue& prim_size(SpValue&, SpValue& path);
  static SpValue& prim_rename(SpValue&, SpValue& from, SpValue& to);
  static SpValue& prim_atime(SpValue&, SpValue& path);
  static SpValue& prim_ctime(SpValue&, SpValue& path);
  static SpValue& prim_mtime(SpValue&, SpValue& path);
};

#ifdef USE_SOCKET

const int BUFFSIZE = 1024;

#ifndef SOCKET
#define SOCKET int
#endif

//class BaseSocket : public Reader, public Writer {
class BaseSocket : public IO {
protected:
  SOCKET sock;
  int port;
  bool isOpen;
public:
  BaseSocket(int p) : port(p){
    isOpen = false;
  }
  BaseSocket(){
    isOpen = false;
  }
  ~BaseSocket(){
    if(isOpen){
      close();
    }
  }

  bool is_open(){
    return isOpen;
  }
  void close();  // close socket
};

class Socket : public BaseSocket {
private:
  char* hostname;
  // for read
  char buff[2][BUFFSIZE]; // read buffers
  bool already_read[2]; // buff was already read flag
  int which_buf;
  int read_bytes;  // read buff bytes
  int read_point;
  bool receive_end;
  // for write
  char wbuff[BUFFSIZE]; // write buffer
  int write_point;

  void next_buf(){
    already_read[which_buf] = false;
    which_buf = (which_buf + 1) & 1;
  }
  void read_buffer();
public:
  Socket(int p, char* hname) : BaseSocket(p), hostname(hname) {
    which_buf = 0;
    read_bytes = 0;
    read_point = 0;
    receive_end = false;
    already_read[0] = false;
    already_read[1] = false;
    write_point = 0;
  }
  Socket(SOCKET s){
    sock = s;
    which_buf = 0;
    read_bytes = 0;
    read_point = 0;
    receive_end = false;
    already_read[0] = false;
    already_read[1] = false;
    write_point = 0;
  }
  bool isSocket(){ return true; }

  SpChar readCh();
  /*
    void unreadCh(SpChar){
        throw SpException("can't unread socket");
    }
   */
  void writeCh(unsigned char ch);
  //void WriteChar(SpChar ch);
  void open();   // open client socket
  bool eof();
  void flush();

  void write(char* str);
  SpValue& receive(SpInt n, unsigned char* buffer);
  SpValue& send_buffer(SpInt n, unsigned char* buffer);

  SpValue& toString();
  const char* typeString(){ return "socket"; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);

  static void init();
  // primitives
  static SpValue& prim_getname(SpValue& adr);
  static SpValue& prim_open(SpValue& adr, SpValue& port);
  static SpValue& prim_close(SpValue& self);  // close socket
  static SpValue& prim_eof(SpValue& self);
  static SpValue& prim_write(SpValue& self, SpValue& str);
  static SpValue& prim_writeline(SpValue& self, SpValue& str);
  //    static SpValue& prim_readline(SpValue& self);
  static SpValue& prim_receive(SpValue& self, SpValue& n, SpValue& byte_vector);
  static SpValue& prim_send(SpValue& self, SpValue& n, SpValue& byte_vector);
};

class ServerSocket : public BaseSocket {
private:
  struct sockaddr_in sockadd;

public:
  ServerSocket(int p) : BaseSocket(p) {}
  bool isServerSocket(){ return true; }
  SpValue& toString();
  const char* typeString(){ return "server_socket"; }
  SpValue& onMessage(SpValue& rec, SpValue& msg);

  void open();
  SpValue& accept();

  SpChar readCh(){
    throw SpException("illegal call of ServerSocket::readCh()");
    return 0;
  }
  /*
    void unreadCh(SpChar){
        throw SpException("can't unread server socket");
    }
   */
  void writeCh(unsigned char ch){
    throw SpException("illegal call of ServerSocket::writeCh()");
  }
  void WriteChar(SpChar ch){
    throw SpException("illegal call of ServerSocket::WriteChar()");
  }
  bool eof(){
    throw SpException("illegal call of ServerSocket::eof()");
    return true;
  }

  static void init();
  // primitives
  static SpValue& prim_open(SpValue& port);
  static SpValue& prim_accept(SpValue& self);
  static SpValue& prim_close(SpValue& self);
};


#endif /* USE_SOCKET */

class SpDate : public SpObject {
private:
  SpInt year;
  SpInt month;
  SpInt day;

  void regular();
public:
  SpDate(int y, int m, int d) : year(y), month(m), day(d) {}

  bool isDate(){ return true; }
  const char* typeString(){ return "date"; }
  SpValue& toString();
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  void assign(SpValue& feature, SpValue& value);
  SpValue& plus(SpValue& e1, SpValue& e2);
  SpValue& minus(SpValue& e1, SpValue& e2);
  SpValue& eq(SpValue& e1, SpValue& e2);
  SpValue& ne(SpValue& e1, SpValue& e2);
  SpValue& gt(SpValue& e1, SpValue& e2);
  SpValue& ge(SpValue& e1, SpValue& e2);
  SpValue& lt(SpValue& e1, SpValue& e2);
  SpValue& le(SpValue& e1, SpValue& e2);

  static void init();
  // primitives
  static SpValue& prim_today();
  static SpValue& prim_new(SpValue& v1, SpValue& v2, SpValue& v3);
  static SpValue& prim_succ(SpValue& self);
  static SpValue& prim_pred(SpValue& self);
  static SpValue& prim_getYear(SpValue& self);
  static SpValue& prim_getMonth(SpValue& self);
  static SpValue& prim_getDay(SpValue& self);
  static SpValue& prim_fromString(SpValue& date_str);
  static SpValue& prim_toString(SpValue& self);

  friend class SpDateTime;
};

class SpTime : public SpObject {
private:
  SpInt hour, minute, second;

  void regular();
public:
  SpTime(int h, int m, int s) : hour(h), minute(m), second(s) {}

  bool isTime(){ return true; }
  const char* typeString(){ return "time"; }
  SpValue& toString();
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  void assign(SpValue& feature, SpValue& value);
  SpValue& plus(SpValue& e1, SpValue& e2);
  SpValue& minus(SpValue& e1, SpValue& e2);
  SpValue& eq(SpValue& e1, SpValue& e2);
  SpValue& ne(SpValue& e1, SpValue& e2);
  SpValue& gt(SpValue& e1, SpValue& e2);
  SpValue& ge(SpValue& e1, SpValue& e2);
  SpValue& lt(SpValue& e1, SpValue& e2);
  SpValue& le(SpValue& e1, SpValue& e2);

  static void init();
  // primitives
  static SpValue& prim_now();
  static SpValue& prim_new(SpValue& v1, SpValue& v2, SpValue& v3);
  //static SpValue& prim_succ(SpValue& self);
  //static SpValue& prim_pred(SpValue& self);
  static SpValue& prim_getHour(SpValue& self);
  static SpValue& prim_getMinute(SpValue& self);
  static SpValue& prim_getSecond(SpValue& self);
  static SpValue& prim_fromString(SpValue& time_str);
  static SpValue& prim_toString(SpValue& self);

  friend class SpDateTime;
};

class SpDateTime : public SpObject {
private:
  SpValue date; // SpDate
  SpValue time; // SpTime

  void regular();
public:
  SpDateTime(int year, int mon, int day,
             int hour, int min, int sec)
    {
      date.setNewObject((SpObject*)new SpDate(year, mon, day));
      time.setNewObject((SpObject*)new SpTime(hour, min, sec));
    }

  bool isDateTime(){ return true; }
  const char* typeString(){ return "datetime"; }
  SpValue& toString();
  SpValue& onMessage(SpValue& rec, SpValue& msg);
  void assign(SpValue& feature, SpValue& value);
  SpValue& plus(SpValue& e1, SpValue& e2);
  SpValue& minus(SpValue& e1, SpValue& e2);
  SpValue& eq(SpValue& e1, SpValue& e2);
  SpValue& ne(SpValue& e1, SpValue& e2);
  SpValue& gt(SpValue& e1, SpValue& e2);
  SpValue& ge(SpValue& e1, SpValue& e2);
  SpValue& lt(SpValue& e1, SpValue& e2);
  SpValue& le(SpValue& e1, SpValue& e2);

  static void init();
  // primitives
  static SpValue& prim_now();
  static SpValue& prim_new(SpValue& date, SpValue& time);
  static SpValue& prim_getYear(SpValue& self);
  static SpValue& prim_getMonth(SpValue& self);
  static SpValue& prim_getDay(SpValue& self);
  static SpValue& prim_getHour(SpValue& self);
  static SpValue& prim_getMinute(SpValue& self);
  static SpValue& prim_getSecond(SpValue& self);
  static SpValue& prim_getDate(SpValue& self);
  static SpValue& prim_getTime(SpValue& self);
};

#ifdef USE_DATABASE
///////////////////////////
// Database
///////////////////////////

struct DBRoot {
  DBRoot* next;
  SpValue value;
  int pos;  // file position
};

class DB {
private:
  FILE* file;
  char filename[MAXPATH];
  bool closed;
  //bool locked;

  DBRoot* findRoot(SpValue& name);
  DBRoot* createRoot(SpValue& name);
  void setObject(DBRoot* root, SpValue& obj)

  public:
  DB(char* name){
    strncpy(filename, name, MAXPATH);
    filename[MAXPATH-1] = '\0';
    closed = true;
  }

  void open();
  void close();
  /*
    void lock();
    void unlock();
    bool isLocked(){
        return locked;
    }
   */
  void setRoot(SpValue& name, SpValue& obj);
  SpValue& getRoot(SpValue& name);
};

class SpDatabase : public SpObject {
private:
  DB* database;

  void open();
  void close();
public:
  SpDatabase(char* filename);
  ~SpDatabase();

  bool isDatabase(){ return true; }
  const char* typeString(){ return "database"; }
  SpValue& toString();
  SpValue& onMessage(SpValue& rec, SpValue& msg);

  void setRoot(SpValue& name, SpValue& obj);
  SpValue& getRoot(SpValue& name);

  static void init();

  // primitives
  static SpValue& prim_open(SpValue& filename);
  static SpValue& prim_close(SpValue& db);
  // NameSpaceȶ̤Υץߥƥ
  /*
    static SpValue& rename(SpValue& self, SpValue& renames);
    static SpValue& make_func(SpValue& self);
    static SpValue& prim_append(SpValue& self, SpValue& key, SpValue& value);
    static SpValue& prim_keys(SpValue& self);
    static SpValue& prim_delete(SpValue& self, SpValue& key);
    static SpValue& prim_lookup(SpValue& self, SpValue& key);
   */
};


class DBNamespace : public SpObject {
};


class DBArray : public SpObject {
};

#endif

/*
 * Lexer
 */

const int MaxBuf = 1000;

typedef map<SpString*, int, SpStringCmp> KeywordMap;

class Lexer {
private:
  ReadEncoder* reader;
  KeywordMap reserved;
  static Lexer* lexer;
  char buf[MaxBuf+1]; // for lex number
  int index;  // index of buf
  int integer; // for lex number

  SpChar skipWhiteSpace();
  int lexZeroNumber();
  int lexNumber();
  int lexBIN();
  int lexOCT();
  int lexHEX();
  int lexReal();
  int lexString();
  int lexChar();
  SpChar lexBackslashChar();
  void skipOneLine();
  void skipComment();
  int isReserved(SpCharVector& vec);
  bool isReservedKigou(SpChar c);
  bool wdot_read;

  Lexer(ReadEncoder& r=*spin);
  ~Lexer();
public:
  static Lexer* getLexer(ReadEncoder& r=*spin){
    if(lexer == NULL){
      lexer = new Lexer(r);
    }
    return lexer;
  }

  int lex();
  static void setReadEncoder(ReadEncoder* r){ getLexer()->reader = r; }
  static ReadEncoder* getReadEncoder(){ return getLexer()->reader; }
};


/*
 * globals, etc
 */

// globals
extern MsgHandler NilMsgHandler;
extern MsgHandler BoolMsgHandler;
extern MsgHandler CharMsgHandler;
extern MsgHandler IntMsgHandler;
extern MsgHandler RealMsgHandler;
extern MsgHandler ObjectMsgHandler;
extern MsgHandler ListMsgHandler;
extern MsgHandler StringMsgHandler;
extern MsgHandler SymbolMsgHandler;
extern MsgHandler TupleMsgHandler;
extern MsgHandler NameSpaceMsgHandler;
extern MsgHandler ArrayMsgHandler;
extern MsgHandler FileInMsgHandler;
extern MsgHandler FileOutMsgHandler;
extern MsgHandler DirMsgHandler;
extern MsgHandler FileMsgHandler; /* file util */
#ifdef USE_SOCKET
extern MsgHandler SocketMsgHandler;
extern MsgHandler ServerSocketMsgHandler;
#endif
extern MsgHandler DateMsgHandler;
extern MsgHandler TimeMsgHandler;
extern MsgHandler DateTimeMsgHandler;
#ifdef USE_DATABASE
extern MsgHandler DatabaseMsgHandler;
#endif
#ifdef THREAD
extern MsgHandler ThreadMsgHandler;
#endif

extern SpValue SymIsInt;
extern SpValue SymIsChar;
extern SpValue SymIsBool;
extern SpValue SymIsReal;
extern SpValue SymIsList;
extern SpValue SymIsNil;
extern SpValue SymIsTuple;
extern SpValue SymIsString;
extern SpValue SymIsSymbol;
extern SpValue SymHead;
extern SpValue SymTail;
extern SpValue SymLength;
extern SpValue SymVar;
extern SpValue SymRequire;
extern SpValue SymDo;
extern SpValue SymEnsure;
extern SpValue SymRescue;
extern SpValue SymFrom;
extern SpValue SymStep;
extern SpValue SymWhile;
extern SpValue SymUntil;
extern SpValue SymVariant;
extern SpValue SymInvariant;
extern SpValue SymSoopy;
extern SpValue SymEnv;
extern SpValue SymClose;
extern SpValue SymRename;
extern SpValue SymArg;
extern SpValue SymName;
extern SpValue SymToInt;
extern SpValue SymToString;
extern SpValue SymNth;
extern SpValue SymEach;
extern SpValue SymFirst;
extern SpValue SymSecond;
extern SpValue SymThird;
extern SpValue SymFourth;
extern SpValue SymFifth;
extern SpValue SymEucjpOut;
extern SpValue SymEucjpIn;
extern SpValue SymSjisOut;
extern SpValue SymSjisIn;
/*
nil, bool, int, char, real, string, function, expr,
        tuple, list, array, namespace, datatype_name,
        stream, file, socket
 */
extern SpValue SymBool;
extern SpValue SymInt;
extern SpValue SymByte;
extern SpValue SymWord;
extern SpValue SymChar;
extern SpValue SymReal;
extern SpValue SymString;
extern SpValue SymSymbol;
extern SpValue SymFunction;
extern SpValue SymTuple;
extern SpValue SymList;
extern SpValue SymArray;
extern SpValue SymNameSpace;
extern SpValue SymReader;
extern SpValue SymWriter;
extern SpValue SymEncoderIn;
extern SpValue SymEncoderOut;
extern SpValue SymDecoderIn;
extern SpValue SymDecoderOut;
extern SpValue SymDir;
extern SpValue SymMember;
extern SpValue SymOpen;
extern SpValue SymOpenIn;
extern SpValue SymOpenOut;
extern SpValue SymOpenAppend;
extern SpValue SymSelf;
extern SpValue SymWriteChar;
extern SpValue SymWrite;
extern SpValue SymWriteLine;
extern SpValue SymReadLine;
extern SpValue SymFlush;
extern SpValue SymClosed;
extern SpValue SymIsEOF;
extern SpValue SymUpper;
extern SpValue SymLower;
extern SpValue SymAppend;
extern SpValue SymKeys;
extern SpValue SymDelete;
extern SpValue SymSplit;
extern SpValue SymSplit1;
extern SpValue SymSort;
extern SpValue SymMap;
extern SpValue SymFoldl;
extern SpValue SymFoldr;
extern SpValue SymFilter;
extern SpValue SymReplace;
extern SpValue SymReplace1;
extern SpValue SymFind;
extern SpValue SymYear;
extern SpValue SymMonth;
extern SpValue SymDay;
extern SpValue SymSucc;
extern SpValue SymPred;
extern SpValue SymHour;
extern SpValue SymMinute;
extern SpValue SymDate;
extern SpValue SymTime;
extern SpValue SymAtime;
extern SpValue SymCtime;
extern SpValue SymMtime;
#ifdef USE_DATABASE
extern SpValue SymDatabase;
#endif
extern SpValue SymGet;
extern SpValue SymSet;
extern SpValue SymCurrentThread;

extern SpValue PrimNil;
extern SpValue PrimNil2;
extern SpValue PrimNil3;
extern SpValue PrimTrue;
extern SpValue PrimFalse;
extern SpValue PrimZero;

extern SpNameSpace* PMainNameSpace;
extern SpValue MainNS; // main(global) NameSpace
extern SpNameSpace* PSoopyNameSpace;
extern SpValue SoopyNS;
extern SpNameSpace* PEnvNameSpace;
extern SpValue EnvNS;

extern SjisWriteEncoder* sjisout;
extern SjisReadDecoder* sjisin;
extern EucJPWriteEncoder* eucjpout;
extern EucJPReadDecoder* eucjpin;

void SpInit(int argc, char* argv[], char* env[]);
void SpFinalize();
void CheckType(SpValue& featureType, SpValue& value);

#define YYSTYPE SpValue
int yyerror(char* str);


#endif /* __SOOPY_H */
