﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using FDK;
using FDK.メディア;

namespace SST.設定
{
	/// <summary>
	///		SSTユーザの管理と選択を行う。
	/// </summary>
	class ユーザ管理 : Activity
	{
		public List<ユーザ> ユーザリスト
		{
			get;
		} = new List<ユーザ>();

		/// <summary>
		///		現在選択されているユーザの、ユーザリストにおけるインデックス番号（0～）。
		///		負数ならどのユーザも未選択である（か、またはユーザリストが空）。
		/// </summary>
		public int 選択されているユーザのインデックス番号
		{
			get;
			set;
		} = -1;

		/// <summary>
		///		未選択である場合は null。
		/// </summary>
		public ユーザ 選択されているユーザ
		{
			get
				=> ( 0 > this.選択されているユーザのインデックス番号 ) ?
					null :
					this.ユーザリスト[ this.選択されているユーザのインデックス番号 ];
		}

		/// <summary>
		///		いずれかのユーザがログインしている状態なら true。
		/// </summary>
		public bool ログイン中
		{
			get;
			protected set;
		} = false;


		public ユーザ管理()
		{
		}

		// 選択

		/// <remarks>
		///		指定されたユーザがユーザリストに存在しないなら、未選択状態になる。
		/// </remarks>
		public void ユーザを選択する( string ユーザ名 )
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				this.選択されているユーザのインデックス番号 = this.ユーザリスト.FindIndex( ( user ) => ( user.プロパティ.名前 == ユーザ名 ) ); // 見つからなければ -1 が返される。
				Log.Info( $"選択されたユーザのインデックス: {this.選択されているユーザのインデックス番号}" );
			}
		}

		/// <remarks>
		///		ユーザリストが空なら、未選択状態になる。
		/// </remarks>
		public void 最初のユーザを選択する()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				if( 0 < this.ユーザリスト.Count )
				{
					this.選択されているユーザのインデックス番号 = 0;
				}
				else
				{
					Log.WARNING( "ユーザリストが空です。" );
					this.選択されているユーザのインデックス番号 = -1;
				}
			}
		}

		/// <remarks>
		///		先頭のユーザが選択されているなら、末尾のユーザが選択される。
		///		ユーザリストが空なら、未選択状態になる。
		/// </remarks>
		public void ひとつ前のユーザを選択する()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				if( 0 <= this.選択されているユーザのインデックス番号 )
				{
					this.選択されているユーザのインデックス番号--;

					if( 0 > this.選択されているユーザのインデックス番号 )
						this.選択されているユーザのインデックス番号 = this.ユーザリスト.Count - 1;	// 末尾へ。

					Log.Info( $"ユーザリスト[{this.選択されているユーザのインデックス番号}], {this.選択されているユーザ.プロパティ.名前} を選択しました。" );
				}
				else
				{
					Log.WARNING( "現在、選択されているユーザはいません。" );
				}
			}
		}

		/// <remarks>
		///		末尾のユーザが選択されているなら、先頭のユーザが選択される。
		///		ユーザリストが空なら、未選択状態になる。
		/// </remarks>
		public void ひとつ後のユーザを選択する()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				if( 0 < this.ユーザリスト.Count )
				{
					if( 0 <= this.選択されているユーザのインデックス番号 )
					{
						if( ++this.選択されているユーザのインデックス番号 >= this.ユーザリスト.Count )
							this.選択されているユーザのインデックス番号 = 0;	// 先頭へ。

						Log.Info( $"ユーザリスト[{this.選択されているユーザのインデックス番号}], {this.選択されているユーザ.プロパティ.名前} を選択しました。" );
					}
				}
				else
				{
					Log.WARNING( "現在、選択されているユーザはいません。" );
					this.選択されているユーザのインデックス番号 = -1;
				}
			}
		}

		/// <returns>
		///		ユーザリストに存在していないなら -1 。
		/// </returns>
		public int ユーザのインデックス番号を返す( string ユーザ名 )
		{
			return this.ユーザリスト.FindIndex( ( user ) => ( user.プロパティ.名前 == ユーザ名 ) );
		}

		// ログイン・ログアウト

		/// <summary>
		///		現在選択されているユーザのログイン処理を行う。
		///		未選択なら何もしない。
		/// </summary>
		public void ログインする( グラフィックデバイス gd )
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				Debug.Assert( this.活性化している );

				if( this.ログイン中 )
					this.ログアウトする( gd );

				var user = App.ユーザ管理.選択されているユーザ;

				if( null == user )
				{
					Log.WARNING( "現在選択されているユーザはいません。" );
					return;
				}

				// 曲を検索して曲ツリーを構築する。（通常モード時のみ）
				if( App.ビュアーモードではない )
				{
					user.非活性化する( gd );
					user.曲ツリーを再構築する();
					user.活性化する( gd );
				}

				this.ログイン中 = true;
				Log.Info( $"ユーザ [{App.ユーザ管理.選択されているユーザ.プロパティ.名前}] がログインしました。" );
			}
		}

		/// <summary>
		///		現在ログイン中のユーザのログアウト処理を行う。
		///		現在誰もログインしていないなら何もしない。
		///	</summary>
		public void ログアウトする( グラフィックデバイス gd )
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				Debug.Assert( this.活性化している );

				if( !( this.ログイン中 ) )
				{
					Log.WARNING( "現在ログインしているユーザはいません。" );
					return;
				}

				var user = App.ユーザ管理.選択されているユーザ;

				user.保存する();

				// 曲ツリーをクリアする。ユーザは活性化状態のまま。
				if( null != user.曲ツリー )
				{
					user.曲ツリー.非活性化する( gd );
					user.曲ツリー.すべてのノードを削除する();
					user.曲ツリー.活性化する( gd );
				}

				this.ログイン中 = false;
				Log.Info( $"ユーザ [{user.プロパティ.名前}] をログアウトしました。" );
			}
		}

		// 追加・削除

		/// <param name="挿入位置">
		///		ユーザを追加したいユーザリストの位置。0から始まるインデックス番号。-1 なら末尾へ追加する。
		///	</param>
		public void ユーザを追加する( グラフィックデバイス gd, ユーザ user, int 挿入位置 = -1 )
		{
			Debug.Assert( this.活性化している );

			if( 0 <= this.ユーザのインデックス番号を返す( user.プロパティ.名前 ) )
				throw new InvalidOperationException( "すでに同名のユーザが存在しています。" );

			// ユーザリストに追加する。
			if( 0 > 挿入位置 )
				this.ユーザリスト.Add( user );
			else
				this.ユーザリスト.Insert( 挿入位置, user );	// 挿入位置の指定があればそこへ挿入。

			// 子Activity リストにも追加する。
			this.子リスト.Add( user );

			// 活性化する。
			user.活性化する( gd );
		}

		public void ユーザを追加する( グラフィックデバイス gd, string ユーザ名 )
		{
			this.ユーザを追加する( gd, new ユーザ( ユーザ名 ) );
		}

		public void ユーザを削除する( グラフィックデバイス gd, int index )
		{
			Debug.Assert( this.活性化している );

			if( index == this.選択されているユーザのインデックス番号 )
				this.ひとつ前のユーザを選択する();

			var user = this.ユーザリスト[ index ];

			// 非活性化する。
			user.非活性化する( gd );

			// 子Activityリストから削除。
			this.子リスト.Remove( user );

			// ユーザリストからも削除。
			this.ユーザリスト.RemoveAt( index );
		}

		public void ユーザを削除する( グラフィックデバイス gd, string ユーザ名 )
		{
			int index = this.ユーザのインデックス番号を返す( ユーザ名 );
			if( 0 > index )
				throw new InvalidOperationException( "指定されたユーザがユーザリストに存在しません。" );

			this.ユーザを削除する( gd, index );
		}

		// 保存・復元

		/// <summary>
		///		ユーザリストに存在するすべてのユーザを保存する。
		/// </summary>
		public void 保存する()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				int 保存成功件数 = 0;
				int 保存失敗件数 = 0;
				foreach( var user in this.ユーザリスト )
				{
					try
					{
						user.保存する();
						保存成功件数++;
					}
					catch( Exception e )
					{
						Log.ERROR( $"ユーザ「{user.プロパティ.名前}」のファイルへの保存に失敗しました。スキップします。[{e.Message}]" );
						保存失敗件数++;
					}
				}

				Log.Info( $"ユーザリスト内のユーザを保存しました。[{保存成功件数}件成功, {保存失敗件数}件失敗]" );
			}
		}

		/// <summary>
		///		ユーザフォルダを検索し、ユーザリストを構築（復元）する。
		/// </summary>
		/// <returns>検索されたユーザリストをもって新しく生成された<see cref="ユーザ管理"/>インスタンス。</returns>
		public static ユーザ管理 復元する( グラフィックデバイス gd )
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				var userManager = new ユーザ管理();

				// ユーザルートフォルダがなければ作成する。
				if( !( Directory.Exists( SST.IO.Folder.UserRoot ) ) )
					Directory.CreateDirectory( SST.IO.Folder.UserRoot );

				// 条件に適合するフォルダをユーザフォルダとみなし、そのパスを列挙する。
				var ユーザフォルダパスリスト =	
					from dir in Directory.GetDirectories( SST.IO.Folder.UserRoot )			 // (1) $(UserRoot) 配下のフォルダで、かつ、
					where File.Exists( Path.Combine( dir, ユーザ._Propertiesファイル名 ) )    // (2) 中にプロパティファイルが存在している。
					select dir;

				// 列挙されたそれぞれのパスについて、ユーザインスタンスの復元を試みる。
				int 復元成功件数 = 0;
				int 復元失敗件数 = 0;
				foreach( var ユーザフォルダパス in ユーザフォルダパスリスト )
				{
					try
					{
						var user = ユーザ.復元する( ユーザフォルダパス );

						// 同名のユーザがいれば削除する。
						int index = userManager.ユーザのインデックス番号を返す( user.プロパティ.名前 );
						if( 0 <= index )
							userManager.ユーザを削除する( gd, index );

						// ユーザリストに追加する。
						userManager.ユーザリスト.Add( user );

						// 子Activityリストにも追加する。
						userManager.子リスト.Add( user );

						復元成功件数++;
					}
					catch( Exception e )
					{
						Log.ERROR( $"ユーザの復元に失敗しました。スキップします。[{e.Message}]" );        // 失敗したら無視。
						復元失敗件数++;
					}
				}

				Log.Info( $"ユーザリストを復元しました。[成功{復元成功件数}件、失敗{復元失敗件数}件]" );

				// 活性化する。
				userManager.活性化する( gd );

				// AutoPlayer の登録。
				userManager._AutoPlayerの追加ならびに設定を行う( gd );

				// ファイルに反映されていないメンバはいつまでも反映されないので、ここで一度、明示的に保存することにする。
				userManager.保存する();

				return userManager;
			}
		}


		private void _AutoPlayerの追加ならびに設定を行う( グラフィックデバイス gd )
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				Debug.Assert( this.活性化している );

				ユーザ autoPlayer = null;

				// (1) AutoPlayer ユーザが存在しないなら、追加する。
				if( -1 == this.ユーザリスト.FindIndex( ( user ) => ( user.プロパティ.名前 == Properties.Resources.AUTOPLAYER ) ) )
				{
					autoPlayer = new ユーザ( Properties.Resources.AUTOPLAYER );
					
					// 既定の曲フォルダが追加されてないなら追加する。
					var 既定の曲フォルダ = @"$(Exe)\Songs";
					if( -1 == autoPlayer.プロパティ.曲検索フォルダパスリスト.FindIndex( ( path ) => ( path.Equals( 既定の曲フォルダ, StringComparison.OrdinalIgnoreCase ) ) ) )
						autoPlayer.プロパティ.曲検索フォルダパスリスト.Add( 既定の曲フォルダ );

					this.ユーザを追加する( gd, autoPlayer );	// ユーザはこの中で活性化される。
				}

				// (2) AutoPlayer の固定の設定を行う。
				autoPlayer = this.ユーザリスト.Find( ( user ) => ( user.プロパティ.名前 == Properties.Resources.AUTOPLAYER ) );
				autoPlayer.オプション設定.AutoPlayを一括設定する( true ); // 常に全オート。
			}
		}
	}
}
