/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2004 Masanao Izumo <iz@onicos.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

    The 8-bit uLaw to 16-bit PCM and the 13-bit-PCM to 8-bit uLaw
    tables were lifted from the rsynth-2.0 sources.  The README says:

    This is a text to speech system produced by integrating various pieces
    of code and tables of data, which are all (I believe) in the public domain.

    The bulk of the intergration was done by myself, that is Nick Ing-Simmons.
    I can be reached via my employer at nik@tiuk.ti.com.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#include <math.h>
#include "timidity.h"
#include "common.h"
#include "mt19937ar.h"
#include "tables.h"

int32 freq_table[128];
int32 freq_table_zapped[128];
int32 freq_table_tuning[128][128];
int32 freq_table_pytha[24][128];
int32 freq_table_meantone[48][128];
int32 freq_table_pureint[48][128];
int32 freq_table_user[4][48][128];

void init_freq_table(void)
{
	int i;
	
	for (i = 0; i < 128; i++) {
		freq_table[i] = 440 * pow(2.0, (i - 69) * DIV_12) * 1000 + 0.5;
		freq_table_zapped[i] = freq_table[i];
	}

#if 0 // test
	for (i = 0; i < 128; i++) {
		int32 test;
		int32 oct = i / 12;
		int32 key = i % 12;
		// 0:C 1:C# 2:D 3:D# 4:E 5:F 6:F# 7:G 8:G# 9:A 10:A# 11:B
		if(oct == 6 && key == 4 ){
			test = freq_table[i];
			continue;
		}
	}
#endif
}

void init_freq_table_tuning(void)
{
	int p, i;
	double f;
	
	for (i = 0; i < 128; i++)
			freq_table_tuning[0][i] = freq_table_zapped[i];
	for (i = 0; i < 128; i++) {
		f = 440 * pow(2.0, (i - 69) * DIV_12);
		for (p = 1; p < 128; p++)
			freq_table_tuning[p][i] = f * 1000 + 0.5;
	}
}

void init_freq_table_pytha(void)
{
	int i, j, k, l;
	double f;
	static const double major_ratio[] = {
		  1.0 /  1, 256.0 / 243,   9.0 /   8,  32.0 /  27,
		 81.0 / 64,   4.0 /   3, 729.0 / 512,   3.0 /   2,
		128.0 / 81,  27.0 /  16,  16.0 /   9, 243.0 / 128
	};
	static const double minor_ratio[] = {
		   1.0 /    1, 2187.0 / 2048,   9.0 /   8, 19683.0 / 16384,
		  81.0 /   64,    4.0 /    3, 729.0 / 512,     3.0 /     2,
		6561.0 / 4096,   27.0 /   16,  16.0 /   9,   243.0 /   128
	};
	
	for (i = 0; i < 12; i++)
		for (j = -1; j < 11; j++) {
			f = 440 * pow(2.0, (i - 9) * DIV_12 + j - 5);
			for (k = 0; k < 12; k++) {
				l = i + j * 12 + k;
				if (l < 0 || l >= 128)
					continue;
				freq_table_pytha[i][l] = f * major_ratio[k] * 1000 + 0.5;
				freq_table_pytha[i + 12][l] = f * minor_ratio[k] * 1000 + 0.5;
			}
		}
}

void init_freq_table_meantone(void)
{
	int i, j, k, l;
	double f;
	static double major_ratio[12], minor_ratio[12];
	static const double sc = 81.0 / 80;
	
	major_ratio[0] = 1;
	major_ratio[1] = 8 / pow(5.0, 5.0 / 4);
	major_ratio[2] = pow(5.0, 1.0 / 2) / 2;
	major_ratio[3] = 4 / pow(5.0, 3.0 / 4);
	major_ratio[4] = 5.0 / 4;
	major_ratio[5] = 2 / pow(5.0, 1.0 / 4);
	major_ratio[6] = pow(5.0, 3.0 / 2) / 8;
	major_ratio[7] = pow(5.0, 1.0 / 4);
	major_ratio[8] = 8.0 / 5;
	major_ratio[9] = pow(5.0, 3.0 / 4) / 2;
	major_ratio[10] = 4 / pow(5.0, 1.0 / 2);
	major_ratio[11] = pow(5.0, 5.0 / 4) / 4;
	minor_ratio[0] = 1;
	minor_ratio[1] = pow(10.0 / 3, 7.0 / 3) / 16;
	minor_ratio[2] = pow(10.0 / 3, 2.0 / 3) / 2;
	minor_ratio[3] = 125.0 / 108;
	minor_ratio[4] = pow(10.0 / 3, 4.0 / 3) / 4;
	minor_ratio[5] = 2 / pow(10.0 / 3, 1.0 / 3);
	minor_ratio[6] = 25.0 / 18;
	minor_ratio[7] = pow(10.0 / 3, 1.0 / 3);
	minor_ratio[8] = pow(10.0 / 3, 8.0 / 3) / 16;
	minor_ratio[9] = 5.0 / 3;
	minor_ratio[10] = 4 / pow(10.0 / 3, 2.0 / 3);
	minor_ratio[11] = pow(10.0 / 3, 5.0 / 3) / 4;
	for (i = 0; i < 12; i++)
		for (j = -1; j < 11; j++) {
			f = 440 * pow(2.0, (i - 9) * DIV_12 + j - 5);
			for (k = 0; k < 12; k++) {
				l = i + j * 12 + k;
				if (l < 0 || l >= 128)
					continue;
				freq_table_meantone[i][l] =
						f * major_ratio[k] * 1000 + 0.5;
				freq_table_meantone[i + 12][l] =
						f * minor_ratio[k] * sc * 1000 + 0.5;
				freq_table_meantone[i + 24][l] =
						f * minor_ratio[k] * 1000 + 0.5;
				freq_table_meantone[i + 36][l] =
						f * major_ratio[k] * sc * 1000 + 0.5;
			}
		}
}

void init_freq_table_pureint(void)
{
	int i, j, k, l;
	double f;
	static const double major_ratio[] = {
		 1.0 /  1, 16.0 / 15, 9.0 / 8, 6.0 / 5, 5.0 / 4,  4.0 / 3,
		45.0 / 32,  3.0 /  2, 8.0 / 5, 5.0 / 3, 9.0 / 5, 15.0 / 8
	};
	static const double minor_ratio[] = {
		 1.0 /  1, 25.0 / 24, 10.0 /  9, 75.0 / 64,  5.0 / 4,  4.0 / 3,
		25.0 / 18,  3.0 /  2, 25.0 / 16,  5.0 /  3, 16.0 / 9, 15.0 / 8
	};
	static const double sc = 81.0 / 80;
	
	for (i = 0; i < 12; i++)
		for (j = -1; j < 11; j++) {
			f = 440 * pow(2.0, (i - 9) * DIV_12 + j - 5);
			for (k = 0; k < 12; k++) {
				l = i + j * 12 + k;
				if (l < 0 || l >= 128)
					continue;
				freq_table_pureint[i][l] =
						f * major_ratio[k] * 1000 + 0.5;
				freq_table_pureint[i + 12][l] =
						f * minor_ratio[k] * sc * 1000 + 0.5;
				freq_table_pureint[i + 24][l] =
						f * minor_ratio[k] * 1000 + 0.5;
				freq_table_pureint[i + 36][l] =
						f * major_ratio[k] * sc * 1000 + 0.5;
			}
		}
}

void init_freq_table_user(void)
{
	int p, i, j, k, l;
	double f;
	
	for (p = 0; p < 4; p++)
		for (i = 0; i < 12; i++)
			for (j = -1; j < 11; j++) {
				f = 440 * pow(2.0, (i - 9) * DIV_12 + j - 5);
				for (k = 0; k < 12; k++) {
					l = i + j * 12 + k;
					if (l < 0 || l >= 128)
						continue;
					freq_table_user[p][i][l] = f * 1000 + 0.5;
					freq_table_user[p][i + 12][l] = f * 1000 + 0.5;
					freq_table_user[p][i + 24][l] = f * 1000 + 0.5;
					freq_table_user[p][i + 36][l] = f * 1000 + 0.5;
				}
			}
}

/* v=2.^((x/127-1) * 6) */
FLOAT_T def_vol_table[1025];

void init_def_vol_table(void)
{
	int i;
	
	for (i = 0; i < 1024; i++)
		def_vol_table[i] = pow(2.0f,((double)i / 1023.0 - 1) * 6);
	def_vol_table[1024] = 1.0;
}

/* v=2.^((x/127-1) * 8) */
FLOAT_T gs_vol_table[1025];

void init_gs_vol_table(void)
{
	int i;
	
	for (i = 0; i < 1024; i++)
		gs_vol_table[i] = pow(2.0f,((double)i / 1023.0 - 1) * 8);
	gs_vol_table[1024] = 1.0;
}

FLOAT_T *xg_vol_table = gs_vol_table;
FLOAT_T *vol_table = def_vol_table;

FLOAT_T bend_fine[256];
FLOAT_T bend_coarse[128];

void init_bend_fine(void)
{
	int i;
	
	for (i = 0; i < 256; i++)
		bend_fine[i] = pow(2.0, i * DIV_12 * DIV_256);
}

void init_bend_coarse(void)
{
	int i;
	
	for (i = 0; i < 128; i++)
		bend_coarse[i] = pow(2.0, i * DIV_12);
}

/*
 * midi_time_table(x + 16y) = midi_time_table(x) * (2^y)
 * midi_time_table(64) = 1
 * then,
 * midi_time_table(x) := (2^(x/16))/16
 */
FLOAT_T midi_time_table[128] =
{
    0.06250, 0.06527, 0.06816, 0.07117, 0.07433, 0.07762, 0.08105, 0.08464,
    0.08839, 0.09230, 0.09639, 0.10066, 0.10511, 0.10977, 0.11463, 0.11970,
    0.12500, 0.13053, 0.13631, 0.14235, 0.14865, 0.15523, 0.16210, 0.16928,
    0.17678, 0.18460, 0.19278, 0.20131, 0.21022, 0.21953, 0.22925, 0.23940,
    0.25000, 0.26107, 0.27263, 0.28470, 0.29730, 0.31046, 0.32421, 0.33856,
    0.35355, 0.36921, 0.38555, 0.40262, 0.42045, 0.43906, 0.45850, 0.47880,
    0.50000, 0.52214, 0.54525, 0.56939, 0.59460, 0.62093, 0.64842, 0.67713,
    0.70711, 0.73841, 0.77111, 0.80525, 0.84090, 0.87813, 0.91700, 0.95760,
    1.00000, 1.04427, 1.09051, 1.13879, 1.18921, 1.24186, 1.29684, 1.35426,
    1.41421, 1.47683, 1.54221, 1.61049, 1.68179, 1.75625, 1.83401, 1.91521,
    2.00000, 2.08855, 2.18102, 2.27758, 2.37841, 2.48372, 2.59368, 2.70851,
    2.82843, 2.95365, 3.08442, 3.22098, 3.36359, 3.51250, 3.66802, 3.83041,
    4.00000, 4.17710, 4.36203, 4.55515, 4.75683, 4.96743, 5.18736, 5.41702,
    5.65685, 5.90730, 6.16884, 6.44196, 6.72717, 7.02501, 7.33603, 7.66083,
    8.00000, 8.35419, 8.72406, 9.11031, 9.51366, 9.93486,10.37472,10.83404,
   11.31371,11.81461,12.33769,12.88392,13.45434,14.05002,14.67206,15.32165
};
/*
 * midi_time_table2(x) := 2^(x/16/128)  (for lsb tunning)
 */
FLOAT_T midi_time_table2[128] =
{
    1.00000, 1.00034, 1.00068, 1.00102, 1.00135, 1.00169, 1.00203, 1.00237,
    1.00271, 1.00305, 1.00339, 1.00373, 1.00407, 1.00441, 1.00475, 1.00509,
    1.00543, 1.00577, 1.00611, 1.00645, 1.00679, 1.00713, 1.00747, 1.00781,
    1.00816, 1.00850, 1.00884, 1.00918, 1.00952, 1.00986, 1.01021, 1.01055,
    1.01089, 1.01123, 1.01157, 1.01192, 1.01226, 1.01260, 1.01294, 1.01329,
    1.01363, 1.01397, 1.01432, 1.01466, 1.01500, 1.01535, 1.01569, 1.01603,
    1.01638, 1.01672, 1.01707, 1.01741, 1.01776, 1.01810, 1.01844, 1.01879,
    1.01913, 1.01948, 1.01982, 1.02017, 1.02051, 1.02086, 1.02121, 1.02155,
    1.02190, 1.02224, 1.02259, 1.02294, 1.02328, 1.02363, 1.02397, 1.02432,
    1.02467, 1.02501, 1.02536, 1.02571, 1.02606, 1.02640, 1.02675, 1.02710,
    1.02745, 1.02779, 1.02814, 1.02849, 1.02884, 1.02919, 1.02953, 1.02988,
    1.03023, 1.03058, 1.03093, 1.03128, 1.03163, 1.03198, 1.03233, 1.03268,
    1.03302, 1.03337, 1.03372, 1.03407, 1.03442, 1.03477, 1.03512, 1.03548,
    1.03583, 1.03618, 1.03653, 1.03688, 1.03723, 1.03758, 1.03793, 1.03828,
    1.03863, 1.03899, 1.03934, 1.03969, 1.04004, 1.04039, 1.04075, 1.04110,
    1.04145, 1.04180, 1.04216, 1.04251, 1.04286, 1.04321, 1.04357, 1.04392
};

#ifdef LOOKUP_SINE
static FLOAT_T sine_table[257]=
{
 0.0, 0.0061358846491544753, 0.012271538285719925, 0.01840672990580482,
 0.024541228522912288, 0.030674803176636626, 0.036807222941358832, 0.04293825693494082,
 0.049067674327418015, 0.055195244349689934, 0.061320736302208578, 0.067443919563664051,
 0.073564563599667426, 0.079682437971430126, 0.085797312344439894, 0.091908956497132724,
 0.098017140329560604, 0.10412163387205459, 0.11022220729388306, 0.11631863091190475,
 0.1224106751992162, 0.12849811079379317, 0.13458070850712617, 0.14065823933284921,
 0.14673047445536175, 0.15279718525844344, 0.15885814333386145, 0.16491312048996989,
 0.17096188876030122, 0.17700422041214875, 0.18303988795514095, 0.18906866414980619,
 0.19509032201612825, 0.2011046348420919, 0.20711137619221856, 0.21311031991609136,
 0.2191012401568698, 0.22508391135979283, 0.23105810828067111, 0.2370236059943672,
 0.24298017990326387, 0.24892760574572015, 0.25486565960451457, 0.26079411791527551,
 0.26671275747489837, 0.27262135544994898, 0.27851968938505306, 0.28440753721127188,
 0.29028467725446233, 0.29615088824362379, 0.30200594931922808, 0.30784964004153487,
 0.31368174039889152, 0.31950203081601569, 0.32531029216226293, 0.33110630575987643,
 0.33688985339222005, 0.34266071731199438, 0.34841868024943456, 0.35416352542049034,
 0.35989503653498811, 0.36561299780477385, 0.37131719395183754, 0.37700741021641826,
 0.38268343236508978, 0.38834504669882625, 0.3939920400610481, 0.39962419984564679,
 0.40524131400498986, 0.41084317105790391, 0.41642956009763715, 0.42200027079979968,
 0.42755509343028208, 0.43309381885315196, 0.43861623853852766, 0.4441221445704292,
 0.44961132965460654, 0.45508358712634384, 0.46053871095824001, 0.46597649576796618,
 0.47139673682599764, 0.47679923006332209, 0.48218377207912272, 0.487550160148436,
 0.49289819222978404, 0.49822766697278187, 0.50353838372571758, 0.50883014254310699,
 0.51410274419322166, 0.51935599016558964, 0.52458968267846895, 0.52980362468629461,
 0.53499761988709715, 0.54017147272989285, 0.54532498842204646, 0.55045797293660481,
 0.55557023301960218, 0.56066157619733603, 0.56573181078361312, 0.57078074588696726,
 0.57580819141784534, 0.58081395809576453, 0.58579785745643886, 0.59075970185887416,
 0.59569930449243336, 0.60061647938386897, 0.60551104140432555, 0.61038280627630948,
 0.61523159058062682, 0.6200572117632891, 0.62485948814238634, 0.62963823891492698,
 0.63439328416364549, 0.63912444486377573, 0.64383154288979139, 0.64851440102211244,
 0.65317284295377676, 0.65780669329707864, 0.66241577759017178, 0.66699992230363747,
 0.67155895484701833, 0.67609270357531592, 0.68060099779545302, 0.68508366777270036,
 0.68954054473706683, 0.693971460889654, 0.69837624940897292, 0.7027547444572253,
 0.70710678118654746, 0.71143219574521643, 0.71573082528381859, 0.72000250796138165,
 0.72424708295146689, 0.7284643904482252, 0.73265427167241282, 0.73681656887736979,
 0.74095112535495911, 0.74505778544146595, 0.74913639452345926, 0.75318679904361241,
 0.75720884650648446, 0.76120238548426178, 0.76516726562245896, 0.76910333764557959,
 0.77301045336273699, 0.77688846567323244, 0.78073722857209438, 0.78455659715557524,
 0.78834642762660623, 0.79210657730021239, 0.79583690460888346, 0.79953726910790501,
 0.80320753148064483, 0.80684755354379922, 0.81045719825259477, 0.8140363297059483,
 0.81758481315158371, 0.82110251499110465, 0.82458930278502529, 0.8280450452577558,
 0.83146961230254524, 0.83486287498638001, 0.83822470555483797, 0.84155497743689833,
 0.84485356524970701, 0.84812034480329712, 0.8513551931052652, 0.85455798836540053,
 0.85772861000027212, 0.86086693863776731, 0.8639728561215867, 0.86704624551569265,
 0.87008699110871135, 0.87309497841829009, 0.8760700941954066, 0.87901222642863341,
 0.88192126434835494, 0.88479709843093779, 0.88763962040285393, 0.89044872324475788,
 0.89322430119551532, 0.89596624975618511, 0.89867446569395382, 0.90134884704602203,
 0.90398929312344334, 0.90659570451491533, 0.90916798309052227, 0.91170603200542988,
 0.91420975570353069, 0.9166790599210427, 0.91911385169005777, 0.9215140393420419,
 0.92387953251128674, 0.92621024213831127, 0.92850608047321548, 0.93076696107898371,
 0.93299279883473885, 0.9351835099389475, 0.93733901191257496, 0.93945922360218992,
 0.94154406518302081, 0.94359345816196039, 0.94560732538052128, 0.94758559101774109,
 0.94952818059303667, 0.95143502096900834, 0.95330604035419375, 0.95514116830577067,
 0.95694033573220894, 0.9587034748958716, 0.96043051941556579, 0.96212140426904158,
 0.96377606579543984, 0.9653944416976894, 0.96697647104485207, 0.96852209427441727,
 0.97003125319454397, 0.97150389098625178, 0.97293995220556007, 0.97433938278557586,
 0.97570213003852857, 0.97702814265775439, 0.97831737071962765, 0.97956976568544052,
 0.98078528040323043, 0.98196386910955524, 0.98310548743121629, 0.98421009238692903,
 0.98527764238894122, 0.98630809724459867, 0.98730141815785843, 0.98825756773074946,
 0.98917650996478101, 0.99005821026229712, 0.99090263542778001, 0.99170975366909953,
 0.99247953459870997, 0.9932119492347945, 0.99390697000235606, 0.99456457073425542,
 0.99518472667219682, 0.99576741446765982, 0.996312612182778, 0.99682029929116567,
 0.99729045667869021, 0.99772306664419164, 0.99811811290014918, 0.99847558057329477,
 0.99879545620517241, 0.99907772775264536, 0.99932238458834954, 0.99952941750109314,
 0.99969881869620425, 0.9998305817958234, 0.9999247018391445, 0.99998117528260111,
 1.0
};

/*
   looks up sin(2 * Pi * x / 1024)
*/
FLOAT_T lookup_sine(int x)
{
  int xx = x & 0xFF;
  switch ((x>>8) & 0x03)
    {
    default: /* just to shut gcc up. */
    case 0:
      return sine_table[xx];
    case 1:
      return sine_table[0x100 - xx];
    case 2:
      return -sine_table[xx];
    case 3:
      return -sine_table[0x100 - xx];
    }
}

#endif /* LOOKUP_SINE */

static FLOAT_T triangular_table[257];

void init_triangular_table(void)
{
	int i;
	unsigned long init[4]={0x123, 0x234, 0x345, 0x456}, length=4;
    init_by_array(init, length);

	for (i = 0; i < 257; i++) {
		triangular_table[i] = (double)(i/* - (genrand_int32() % 1)*/) * DIV_256;
		if(triangular_table[i] < 0) {triangular_table[i] = 0;}
		else if(triangular_table[i] > 1.0) {triangular_table[i] = 1.0;}
	}
	triangular_table[0] = 0.0;
	triangular_table[256] = 1.0;
}

FLOAT_T lookup_triangular(int x)
{
  int xx = x & 0xFF;
  switch ((x>>8) & 0x03)
    {
    default:
    case 0:
      return triangular_table[xx];
    case 1:
      return triangular_table[0x100 - xx];
    case 2:
      return -triangular_table[xx];
    case 3:
      return -triangular_table[0x100 - xx];
    }
}

///r
static FLOAT_T saw_table[257];

void init_saw_table(void)
{
	int i;

	for (i = 0; i < 257; i++)
		saw_table[i] = (double)i * DIV_256 * DIV_2;
	saw_table[0] = 0.0;
	saw_table[256] = 0.5;
}

FLOAT_T lookup_saw1(int x)
{
  int xx = x & 0xFF;
  switch ((x>>8) & 0x03)
    {
    default:
    case 0:
      return saw_table[xx];
    case 1:
      return saw_table[xx] + 0.5;
    case 2:
      return -saw_table[xx];
    case 3:
      return -saw_table[xx] - 0.5;
    }
}

FLOAT_T lookup_saw2(int x)
{
  int xx = x & 0xFF;
  switch ((x>>8) & 0x03)
    {
    default:
    case 0:
      return saw_table[0x100 - xx] + 0.5;
    case 1:
      return saw_table[0x100 - xx];
    case 2:
      return -saw_table[0x100 - xx] - 0.5;
    case 3:
      return -saw_table[0x100 - xx];
    }
}

FLOAT_T lookup_square(int x)
{
  int xx = x & 0xFF;
  switch ((x>>8) & 0x03)
    {
    default:
    case 0:
      return 1.0;
    case 1:
      return 1.0;
    case 2:
      return -1.0;
    case 3:
      return -1.0;
    }
}

#ifdef LOOKUP_HACK
int16 _u2l[] =
{
 -32256, -31228, -30200, -29172, -28143, -27115, -26087, -25059,
 -24031, -23002, -21974, -20946, -19918, -18889, -17861, -16833,
 -16062, -15548, -15033, -14519, -14005, -13491, -12977, -12463,
 -11949, -11435, -10920, -10406, -9892, -9378, -8864, -8350,
 -7964, -7707, -7450, -7193, -6936, -6679, -6422, -6165,
 -5908, -5651, -5394, -5137, -4880, -4623, -4365, -4108,
 -3916, -3787, -3659, -3530, -3402, -3273, -3144, -3016,
 -2887, -2759, -2630, -2502, -2373, -2245, -2116, -1988,
 -1891, -1827, -1763, -1698, -1634, -1570, -1506, -1441,
 -1377, -1313, -1249, -1184, -1120, -1056, -992, -927,
 -879, -847, -815, -783, -751, -718, -686, -654,
 -622, -590, -558, -526, -494, -461, -429, -397,
 -373, -357, -341, -325, -309, -293, -277, -261,
 -245, -228, -212, -196, -180, -164, -148, -132,
 -120, -112, -104, -96, -88, -80, -72, -64,
 -56, -48, -40, -32, -24, -16, -8, 0,
 32256, 31228, 30200, 29172, 28143, 27115, 26087, 25059,
 24031, 23002, 21974, 20946, 19918, 18889, 17861, 16833,
 16062, 15548, 15033, 14519, 14005, 13491, 12977, 12463,
 11949, 11435, 10920, 10406, 9892, 9378, 8864, 8350,
 7964, 7707, 7450, 7193, 6936, 6679, 6422, 6165,
 5908, 5651, 5394, 5137, 4880, 4623, 4365, 4108,
 3916, 3787, 3659, 3530, 3402, 3273, 3144, 3016,
 2887, 2759, 2630, 2502, 2373, 2245, 2116, 1988,
 1891, 1827, 1763, 1698, 1634, 1570, 1506, 1441,
 1377, 1313, 1249, 1184, 1120, 1056, 992, 927,
 879, 847, 815, 783, 751, 718, 686, 654,
 622, 590, 558, 526, 494, 461, 429, 397,
 373, 357, 341, 325, 309, 293, 277, 261,
 245, 228, 212, 196, 180, 164, 148, 132,
 120, 112, 104, 96, 88, 80, 72, 64,
 56, 48, 40, 32, 24, 16, 8, 0
};

int32 *mixup;
#ifdef LOOKUP_INTERPOLATION
int8 *iplookup;
#endif

#endif


//// table2
// table size +1 for linear interpolation
const uint32 table2_size_mask = TABLE2_CYCLE_LENGTH - 1;
static FLOAT_T sine_table2[TABLE2_CYCLE_LENGTH + 1];
static FLOAT_T sine_table2_p[TABLE2_CYCLE_LENGTH + 1];

// lookup2
// in : 0.0 ~ 1.0
// out : -1.0 ~ 1.0 , p out : 0.0 ~ 1.0

inline FLOAT_T lookup2_sine(FLOAT_T in)
{
	return sine_table2[(int)(in * TABLE2_CYCLE_LENGTH) & table2_size_mask];
}

// NEW_LFO
inline FLOAT_T lookup2_sine_linear(FLOAT_T in)
{
	FLOAT_T fp1 = in * TABLE2_CYCLE_LENGTH;
	FLOAT_T fp2 = floor(fp1);
	int index = ((int)fp2) & table2_size_mask;
	FLOAT_T v1 = sine_table2[index];
	FLOAT_T v2 = sine_table2[index + 1];
	return v1 + (v2 - v1) * (fp1 - fp2);
}

inline FLOAT_T lookup2_sine_p(FLOAT_T in)
{
	FLOAT_T fp1 = in * TABLE2_CYCLE_LENGTH;
	FLOAT_T fp2 = floor(fp1);
	int index = ((int)fp2) & table2_size_mask;
	FLOAT_T v1 = sine_table2_p[index];
	FLOAT_T v2 = sine_table2_p[index + 1];
	return v1 + (v2 - v1) * (fp1 - fp2);
//	return sine_table2_p[(int)(in * TABLE2_CYCLE_LENGTH) & table2_size_mask];
}

static void init_table2(void)
{
	int i;
	const double div_len = 1.0 / (double)TABLE2_CYCLE_LENGTH;

	for(i = 0; i < (TABLE2_CYCLE_LENGTH + 1); i++){ // 0 ~ 4096
		FLOAT_T tmp, in = (FLOAT_T)i * div_len;
		// sine
		int32 tmpi = sin(in * M_PI2) * M_30BIT + 0.5;
		FLOAT_T tmpf = (FLOAT_T)tmpi * DIV_30BIT;
		sine_table2[i] = tmpf; // precision 30bit
		sine_table2_p[i] = sin((in + 0.75) * M_PI2) * DIV_2 + DIV_2;
	}
}


FLOAT_T filter_cb_p_table[9700];
FLOAT_T filter_cb_m_table[9700];

static void init_filter_coef(void)
{
	int i;

	for(i = 0; i < 9700; i++){
		filter_cb_p_table[i] = pow(10.0, (double)i * DIV_10 * DIV_40);
		filter_cb_m_table[i] = pow(10.0, (double)i * DIV_10 * -DIV_40);
	}
}


#define NOISE_TABLE_CYCLE 256 // 2^n
#define NOISE_TABLE_CYCLE_LENGTH 32 // 2^n
const int32 noise_table_mask = (NOISE_TABLE_CYCLE * NOISE_TABLE_CYCLE_LENGTH) - 1;
static float noise_lowbit_table[(NOISE_TABLE_CYCLE * NOISE_TABLE_CYCLE_LENGTH) + 1];

inline FLOAT_T lookup_noise_lowbit(FLOAT_T in, int32 cycle)
{
	return (FLOAT_T)noise_lowbit_table[(cycle * NOISE_TABLE_CYCLE_LENGTH + (int32)(in * NOISE_TABLE_CYCLE_LENGTH)) & noise_table_mask];
}

static void init_noise_lowbit_table(void)
{
	int i;

	for(i = 0; i < ((NOISE_TABLE_CYCLE * NOISE_TABLE_CYCLE_LENGTH) + 1); i++){
		noise_lowbit_table[i] = (double)((rand() & 0x1F) - M_4BIT) * DIV_5BIT; // 5bit mask
	}	
}


#define NRPN_PARAM_TABLE_LENGTH 128 // 2^n
static float nrpn_param_table[NRPN_PARAM_LIST_MAX][NRPN_PARAM_TABLE_LENGTH + 1];

static float nrpn_xg_attack_data[129] = {
	0, 1, 2, 3, 4, 5, 6, 7, 8, 9, //10
	10, 12, 15, 30, 45, 60, 85, 110, 135, 160, //20
	197, 234, 272, 310, 340, 370, 400, 430, 472, 514, //30
	557, 600, 700, 800, 900, 1000, 1105, 1210, 1315, 1420, //40
	1710, 1855, 2000, 2105, 2210, 2315, 2420, 2662, 2904, //50
	3147, 3390, 3755, 4120, 4485, 4850, 5057, 5264, 5472, 5680, //60
	6260, 6840, 7420, 8000, 8875, 9750, 10625, 11500, 12200, 12900, //70
	13600, 14456, 15313, 16170, 16170, 16170, 16170, 16170, 16170, 16170, //80
	16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, //90
	16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, //100
	16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, //110
	16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, //120
	16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, 16170, //128
};

static float nrpn_xg_release_data[129] = {
	5, 6, 7, 8, 9, 10, 11, 12, 13, 14, //10
	15, 16, 17, 18, 19, 20, 26, 32, 38, 45, //20
	47, 49, 51, 54, 55, 56, 57, 58,	59, 60, //30
	61, 62, 84, 106, 128, 150, 170, 190, 210, 230, //40
	260, 290, 320, 350, 375, 400, 425, 450, 505, 560, //50
	615, 670, 727, 784, 842, 900, 950, 1000, 1050, 1100, //60
	1217, 1334, 1452, 1570, 1717, 1864, 2012, 2160, 2375, 2590, //70
	2805, 3020, 3302, 3584, 3867, 4150, 4442, 4734, 5027, 5320, //80
	5590, 5860, 6130, 6400, 6857, 7314, 7772, 8230, 8920, 9610, //90
	10300, 10300, 10300, 10300, 10300, 10300, 10300, 10300, 10300, 10300, //100
	10300, 10300, 10300, 10300, 10300, 10300, 10300, 10300, 10300, 10300, //110
	10300, 10300, 10300, 10300, 10300, 10300, 10300, 10300, 10300, 10300, //120
	10300, 10300, 10300, 10300, 10300, 10300, 10300, 10300, 10300,//128
};

/*
nrpn_xg_attack_data , nrpn_xg_release_data
SYXGő肵Ă݂
A^bN^C 0ms`16170ms(0~74) , [X^C 5ms`10300ms(0~90)
	Z2ރvZbgŘAĂ悤Ȃ̂ ꂼ1̌Œle[uƎv
	r܂łȂ̂͏le[u̒lȏ(A^bN^C9000ms,[X^C1800ms)̃vZbgȂĤőłȂ
	ȏ͎p͈͊O낤邩̂ł̂܂܂ɂ
fBPC^C 5ms` nrpn=64:1.0, nrpn<64:2^((nrpn-64)/16), nrpn>64:10^((nrpn-64)/32)
	Z2ރvZbgőSAȂ nrpn64Ƃ̔䗦ߎ̂ł炭 lւ̔䗦Ǝv
	ω̂̓fBPC2炵 

GS̓\[XSClɂlւ̔䗦̃e[u͂邯
A^bN^C/[X^C͂ǂݒ肵ĂSMFԂŔj]邱Ƃ 䗦ł͂ȂŒle[uƎv fBPC͕s
ŎgȂȂ
XGfBPC䗦Ƃ GS䗦Ȃ sc_eg_decay_table͎g邩
*/

static inline int cnv_nrpn_param_table_num(FLOAT_T in, int mode)
{
	const int inc1 = NRPN_PARAM_TABLE_LENGTH/8;
	const int inc2 = NRPN_PARAM_TABLE_LENGTH/8/4;
	int i = 0;

	if(in <= nrpn_param_table[mode][0])
		return 0;
	if(in >= nrpn_param_table[mode][NRPN_PARAM_TABLE_LENGTH - 1])
		return NRPN_PARAM_TABLE_LENGTH - 1;
	while(i < NRPN_PARAM_TABLE_LENGTH && in >= nrpn_param_table[mode][i]){ i += inc1; } // max 8cycle
	i -= inc1;
	while(i < NRPN_PARAM_TABLE_LENGTH && in >= nrpn_param_table[mode][i]){ i += inc2; } // max 4cycle
	i -= inc2;
	while(i < NRPN_PARAM_TABLE_LENGTH && in >= nrpn_param_table[mode][i]){ i++; } // max 4cycle
	i--;
	return i;
}

FLOAT_T lookup_nrpn_param(int num, int mode)
{
	if(num <= 0)
		num = 0;
	else if(num >= (NRPN_PARAM_TABLE_LENGTH - 1))
		num = NRPN_PARAM_TABLE_LENGTH - 1;
	return nrpn_param_table[mode][num];
}

FLOAT_T calc_nrpn_param(FLOAT_T in, int add, int mode)
{
	int num = cnv_nrpn_param_table_num(in, mode);
	return lookup_nrpn_param(num + add, mode);
}

static void init_nrpn_param_table(void)
{
	const double div_len = (double)1.0 / (double)NRPN_PARAM_TABLE_LENGTH;
	int i;
	
	// NRPN_PARAM_GM_DELAY , NRPN_PARAM_GM_DELAY , NRPN_PARAM_GM_CUTOFF , NRPN_PARAM_GM_ATTACK , NRPN_PARAM_GM_DECAY , NRPN_PARAM_GM_RELEASE
	for(i = 0; i < NRPN_PARAM_TABLE_LENGTH; i++){
		nrpn_param_table[NRPN_PARAM_GM_DELAY][i] = 5.0 * (double)i * div_len; // 0.0 ~ 5.0
		nrpn_param_table[NRPN_PARAM_GM_RATE][i] = 10.0 * (double)i * div_len; // 0.0 ~ 10.0
		nrpn_param_table[NRPN_PARAM_GM_CUTOFF][i] = pow((double)10.0, 2.0 * (double)i * div_len) * 200.0; // 200 ~ 20000
		nrpn_param_table[NRPN_PARAM_GM_CUTOFF_HPF][i] = pow((double)10.0, 2.0 * (double)i * div_len) * 20.0; // 20 ~ 2000
		nrpn_param_table[NRPN_PARAM_GM_ATTACK][i] = nrpn_xg_attack_data[i];
		nrpn_param_table[NRPN_PARAM_GM_RELEASE][i] = nrpn_xg_release_data[i];
		nrpn_param_table[NRPN_PARAM_GM_DECAY][i] = 1.0 / sc_eg_decay_table[i];
	}	
	nrpn_param_table[NRPN_PARAM_GM_DELAY][NRPN_PARAM_TABLE_LENGTH] = 5.0;
	nrpn_param_table[NRPN_PARAM_GM_RATE][NRPN_PARAM_TABLE_LENGTH] = 10.0;
	nrpn_param_table[NRPN_PARAM_GM_CUTOFF][NRPN_PARAM_TABLE_LENGTH] = 20000.0;	
	nrpn_param_table[NRPN_PARAM_GM_CUTOFF_HPF][NRPN_PARAM_TABLE_LENGTH] = 2000.0;	
	nrpn_param_table[NRPN_PARAM_GM_ATTACK][NRPN_PARAM_TABLE_LENGTH] = nrpn_param_table[NRPN_PARAM_GM_ATTACK][127]; // ms
	nrpn_param_table[NRPN_PARAM_GM_RELEASE][NRPN_PARAM_TABLE_LENGTH] = nrpn_param_table[NRPN_PARAM_GM_RELEASE][127]; // ms
	nrpn_param_table[NRPN_PARAM_GM_DECAY][NRPN_PARAM_TABLE_LENGTH] = nrpn_param_table[NRPN_PARAM_GM_DECAY][127]; // ms
	
	// NRPN_PARAM_GS_DELAY
	// 00bA371bA522bA563bA594bA615bA639bt max10sec?
	for(i = 0; i < NRPN_PARAM_TABLE_LENGTH; i++){
		if(i <= 37*2)
			nrpn_param_table[NRPN_PARAM_GS_DELAY][i] = 1.0 * (double)i / (double)(37*2); // 0.0 ~ 1.0
		else if(i <= 52*2)
			nrpn_param_table[NRPN_PARAM_GS_DELAY][i] = 1.0 + 1.0 * ((double)i - 37*2) / (double)(52*2 - 37*2); // 1.0 ~ 2.0
		else if(i <= 56*2)
			nrpn_param_table[NRPN_PARAM_GS_DELAY][i] = 2.0 + 1.0 * ((double)i - 52*2) / (double)(56*2 - 52*2); // 2.0 ~ 3.0
		else if(i <= 59*2)
			nrpn_param_table[NRPN_PARAM_GS_DELAY][i] = 3.0 + 1.0 * ((double)i - 56*2) / (double)(59*2 - 56*2); // 3.0 ~ 4.0
		else if(i <= 61*2)
			nrpn_param_table[NRPN_PARAM_GS_DELAY][i] = 4.0 + 1.0 * ((double)i - 59*2) / (double)(61*2 - 59*2); // 4.0 ~ 5.0
		else
			nrpn_param_table[NRPN_PARAM_GS_DELAY][i] = 5.0 + 5.0 * ((double)i - 61*2) / (double)(NRPN_PARAM_TABLE_LENGTH - 61*2); // 5.0 ~ 10.0
	}	
	nrpn_param_table[NRPN_PARAM_GS_DELAY][NRPN_PARAM_TABLE_LENGTH] = 10.0;

	// NRPN_PARAM_GS_RATE
	// 01b6኱A-201b4A-401b2A-501b1A-552b1
	for(i = 0; i < NRPN_PARAM_TABLE_LENGTH; i++){	
		nrpn_param_table[NRPN_PARAM_GS_RATE][i] = rate1_table[i];
	}	
	nrpn_param_table[NRPN_PARAM_GS_RATE][NRPN_PARAM_TABLE_LENGTH] = 10.0;
	
	// NRPN_PARAM_GS_CUTOFF
	// cutoff_freq_table_gs[]  250, 315, 400, 500, 630, 800, 1000, 1250, 1600, 2000, 2500, 3150, 4000, 5000, 6300, 8000, (20000?)
	for(i = 0; i < NRPN_PARAM_TABLE_LENGTH; i++){	
		FLOAT_T v1 = cutoff_freq_table_gs[i];
		FLOAT_T v2 = ((i + 8) > 127) ? 20000.0 : cutoff_freq_table_gs[i + 8];
		nrpn_param_table[NRPN_PARAM_GS_CUTOFF][i] = v1 + (v2 - v1) * (double)(i & 0x7) * DIV_8;
	}	
	nrpn_param_table[NRPN_PARAM_GS_CUTOFF][NRPN_PARAM_TABLE_LENGTH] = 20000.0;
	
	// NRPN_PARAM_GS_CUTOFF_HPF , NRPN_PARAM_GS_ATTACK , NRPN_PARAM_GS_DECAY , NRPN_PARAM_GS_RELEASE
	// sȂ̂GM}[W
	for(i = 0; i <= NRPN_PARAM_TABLE_LENGTH; i++){	
		nrpn_param_table[NRPN_PARAM_GS_CUTOFF_HPF][i] = nrpn_param_table[NRPN_PARAM_GS_CUTOFF_HPF][i];
		nrpn_param_table[NRPN_PARAM_GS_ATTACK][i] = nrpn_param_table[NRPN_PARAM_GM_ATTACK][i];
		nrpn_param_table[NRPN_PARAM_GS_RELEASE][i] = nrpn_param_table[NRPN_PARAM_GM_RELEASE][i];
		nrpn_param_table[NRPN_PARAM_GS_DECAY][i] = nrpn_param_table[NRPN_PARAM_GM_DECAY][i];
	}

	// NRPN_PARAM_XG_DELAY
	// 00bA+401bA+502bA+563bA+614bt max5sec?
	for(i = 0; i < NRPN_PARAM_TABLE_LENGTH; i++){
		if(i <= 40*2)
			nrpn_param_table[NRPN_PARAM_XG_DELAY][i] = 1.0 * (double)i / (double)(40*2); // 0.0 ~ 1.0
		else if(i <= 50*2)
			nrpn_param_table[NRPN_PARAM_XG_DELAY][i] = 1.0 + 1.0 * ((double)i - 40*2) / (double)(50*2 - 40*2); // 1.0 ~ 2.0
		else if(i <= 56*2)
			nrpn_param_table[NRPN_PARAM_XG_DELAY][i] = 2.0 + 1.0 * ((double)i - 50*2) / (double)(56*2 - 50*2); // 2.0 ~ 3.0
		else if(i <= 61*2)
			nrpn_param_table[NRPN_PARAM_XG_DELAY][i] = 3.0 + 1.0 * ((double)i - 56*2) / (double)(61*2 - 56*2); // 3.0 ~ 4.0
		else
			nrpn_param_table[NRPN_PARAM_XG_DELAY][i] = 4.0 + 1.0 * ((double)i - 61*2) / (double)(NRPN_PARAM_TABLE_LENGTH - 61*2); // 4.0 ~ 5.0
	}	
	nrpn_param_table[NRPN_PARAM_XG_DELAY][NRPN_PARAM_TABLE_LENGTH] = 5.0;

	// NRPN_PARAM_XG_RATE
	// 1b5񋭁A-71b4A-191b3A-331b2A-491b1A-572b1t߂
	// x-57:0.5 x-49:1.0 x-33:2.0 x-19:1.0 x-7:4.0 x-0:5.5
	// x=72 ĂƁ[
	// 15       23       39        53      65      72
	for(i = 0; i < NRPN_PARAM_TABLE_LENGTH; i++){		
		if(i <= 15*1)
			nrpn_param_table[NRPN_PARAM_XG_RATE][i] = 0.5 * (double)i / (double)(15*1); // 0.0 ~ 0.5
		else if(i <= 23*1)
			nrpn_param_table[NRPN_PARAM_XG_RATE][i] = 0.5 + 0.5 * ((double)i - 15*1) / (double)(23*1 - 15*1); // 0.5 ~ 1.0
		else if(i <= 39*1)
			nrpn_param_table[NRPN_PARAM_XG_RATE][i] = 1.0 + 1.0 * ((double)i - 23*1) / (double)(39*1 - 23*1); // 1.0 ~ 2.0
		else if(i <= 53*1)
			nrpn_param_table[NRPN_PARAM_XG_RATE][i] = 2.0 + 1.0 * ((double)i - 39*1) / (double)(53*1 - 39*1); // 2.0 ~ 3.0
		else if(i <= 65*1)
			nrpn_param_table[NRPN_PARAM_XG_RATE][i] = 3.0 + 1.0 * ((double)i - 53*1) / (double)(65*1 - 53*1); // 3.0 ~ 4.0
		else if(i <= 72*1)
			nrpn_param_table[NRPN_PARAM_XG_RATE][i] = 4.0 + 1.5 * ((double)i - 65*1) / (double)(72*1 - 65*1); // 4.0 ~ 5.5
		else
			nrpn_param_table[NRPN_PARAM_XG_RATE][i] = 5.5 + 4.5 * ((double)i - 72*1) / (double)(NRPN_PARAM_TABLE_LENGTH - 72*1); // 5.5 ~ 10.0
	}	
	nrpn_param_table[NRPN_PARAM_XG_RATE][NRPN_PARAM_TABLE_LENGTH] = 10.0;
	
	// NRPN_PARAM_XG_CUTOFF , NRPN_PARAM_XG_CUTOFF_HPF , NRPN_PARAM_XG_ATTACK , NRPN_PARAM_XG_DECAY , NRPN_PARAM_XG_RELEASE
	// sȂ̂GM}[W
	for(i = 0; i <= NRPN_PARAM_TABLE_LENGTH; i++){	
		nrpn_param_table[NRPN_PARAM_XG_CUTOFF][i] = nrpn_param_table[NRPN_PARAM_GM_CUTOFF][i];
		nrpn_param_table[NRPN_PARAM_XG_CUTOFF_HPF][i] = nrpn_param_table[NRPN_PARAM_GM_CUTOFF_HPF][i];
		nrpn_param_table[NRPN_PARAM_XG_ATTACK][i] = nrpn_xg_attack_data[i];	
		nrpn_param_table[NRPN_PARAM_XG_RELEASE][i] = nrpn_xg_release_data[i];
		if(i == 64)
			nrpn_param_table[NRPN_PARAM_XG_DECAY][i] = 1.0;
		else if(i < 64)
			nrpn_param_table[NRPN_PARAM_XG_DECAY][i] = pow(2.0, (double)(i - 64) * DIV_16);
		else // if(i > 64)
			nrpn_param_table[NRPN_PARAM_XG_DECAY][i] = pow(10.0, (double)(i - 64) * DIV_32);
	}

}

FLOAT_T portament_time_table_xg[128];

static void init_portament_table(void)
{
	int i;
	FLOAT_T rate;
	const FLOAT_T xg[11] = {1000.0, 480.0, 213.0, 56.47, 28.24, 18.82, 10.0, 4.51, 2.0, 0.975, 0.125, };
/* 
XG portament time
0: 12000st/s instant
1: 1000st/s
8:8oct/0.2sec =480st/s
16:8oct/0.45sec =213st/s
32:8oct/1.7sec =56.47st/s
48:8oct/3.4sec =28.24st/s
64:8oct/5.1sec =18.82st/s
80:8oct/9.6sec =10.0st/s
96:8oct/21.3sec =4.51st/s
112:4oct/24sec = 2.0st/s
120:1oct/12.3sec = 0.975st/s
127:5st/40sec = 0.125st/s
*/
	
	for(i = 0; i < 128; i++){	
		if(i == 0){
			portament_time_table_xg[i] = 12000;
		}else if(i < 8){
			rate = (FLOAT_T)(i - 1) * DIV_7;
			portament_time_table_xg[i] = xg[0] * (1.0 - rate) + xg[1] * rate;
		}else if(i < 16){
			rate = (FLOAT_T)(i - 8) * DIV_8;
			portament_time_table_xg[i] = xg[1] * (1.0 - rate) + xg[2] * rate;
		}else if(i < 32){
			rate = (FLOAT_T)(i - 16) * DIV_16;
			portament_time_table_xg[i] = xg[2] * (1.0 - rate) + xg[3] * rate;
		}else if(i < 48){
			rate = (FLOAT_T)(i - 32) * DIV_16;
			portament_time_table_xg[i] = xg[3] * (1.0 - rate) + xg[4] * rate;
		}else if(i < 64){
			rate = (FLOAT_T)(i - 48) * DIV_16;
			portament_time_table_xg[i] = xg[4] * (1.0 - rate) + xg[5] * rate;
		}else if(i < 80){
			rate = (FLOAT_T)(i - 64) * DIV_16;
			portament_time_table_xg[i] = xg[5] * (1.0 - rate) + xg[6] * rate;
		}else if(i < 96){
			rate = (FLOAT_T)(i - 80) * DIV_16;
			portament_time_table_xg[i] = xg[6] * (1.0 - rate) + xg[7] * rate;
		}else if(i < 112){
			rate = (FLOAT_T)(i - 96) * DIV_16;
			portament_time_table_xg[i] = xg[7] * (1.0 - rate) + xg[8] * rate;
		}else if(i < 120){
			rate = (FLOAT_T)(i - 112) * DIV_8;
			portament_time_table_xg[i] = xg[8] * (1.0 - rate) + xg[9] * rate;
		}else if(i < 128){
			rate = (FLOAT_T)(i - 120) * DIV_7;
			portament_time_table_xg[i] = xg[9] * (1.0 - rate) + xg[10] * rate;
		}
	}
}

void init_tables(void)
{
#ifdef LOOKUP_HACK
  int i,j,v;
  mixup = (int32 *)safe_malloc(1<<(7+8+2)); /* Give your cache a workout! */

  for (i=0; i<128; i++)
    {
      v=_u2l[255-i];
      for (j=-128; j<128; j++)
	{
	  mixup[ ((i & 0x7F)<<8) | (j & 0xFF)] =
	    (v * j) << MIXUP_SHIFT;
	}
    }

#ifdef LOOKUP_INTERPOLATION
  iplookup = (int8 *)safe_malloc(1<<(9+5));
  for (i=-256; i<256; i++)
    for(j=0; j<32; j++)
      iplookup[((i<<5) & 0x3FE0) | j] = (i * j)>>5;
  /* I don't know. Quantum bits? Magick? */
#endif

#endif
	init_triangular_table();
///r
	init_saw_table();
	init_table2();
	init_filter_coef();
	init_noise_lowbit_table();
	init_nrpn_param_table();
	init_portament_table();
}

#ifdef LOOKUP_HACK
uint8 _l2u_[] =
{
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
 0x01, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
 0x02, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
 0x04, 0x04, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
 0x05, 0x05, 0x05, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
 0x06, 0x06, 0x06, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
 0x07, 0x07, 0x07, 0x07, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
 0x08, 0x08, 0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
 0x09, 0x09, 0x09, 0x09, 0x09, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E,
 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E,
 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E,
 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E,
 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E,
 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E,
 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E,
 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E,
 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12,
 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12,
 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12,
 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12,
 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13,
 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13,
 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13,
 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13,
 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14,
 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14,
 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14,
 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14,
 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15,
 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16,
 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16,
 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16,
 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16,
 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17,
 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17,
 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17,
 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17,
 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x19, 0x19, 0x19, 0x19, 0x19,
 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19,
 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19,
 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19,
 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x19, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A,
 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1A, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B,
 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B,
 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B,
 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B,
 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C,
 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C,
 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C,
 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C,
 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1D, 0x1D, 0x1D, 0x1D,
 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D,
 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D,
 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D,
 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1D, 0x1E, 0x1E, 0x1E, 0x1E,
 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E,
 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E,
 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E,
 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1F, 0x1F, 0x1F, 0x1F,
 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F,
 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F,
 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F,
 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x1F, 0x20, 0x20, 0x20, 0x20,
 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x21, 0x21, 0x21,
 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21,
 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x22, 0x22, 0x22,
 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x23, 0x23, 0x23,
 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23,
 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x23, 0x24, 0x24, 0x24,
 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24,
 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x25, 0x25, 0x25,
 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25,
 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x26, 0x26, 0x26,
 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26,
 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x27, 0x27, 0x27,
 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27,
 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x27, 0x28, 0x28,
 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28,
 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x29, 0x29,
 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29,
 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x29, 0x2A, 0x2A,
 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A,
 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x2B, 0x2B,
 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B,
 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2C, 0x2C,
 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C,
 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2C, 0x2D, 0x2D,
 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D,
 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2D, 0x2E, 0x2E,
 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E,
 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2E, 0x2F, 0x2F,
 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F,
 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x2F, 0x30,
 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31,
 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x32,
 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x33,
 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x34,
 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x34, 0x35,
 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x36,
 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37,
 0x37, 0x37, 0x37, 0x37, 0x37, 0x37, 0x37, 0x37, 0x37, 0x37, 0x37, 0x37, 0x37, 0x37, 0x37, 0x38,
 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x39,
 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x3A,
 0x3A, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A, 0x3B,
 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3C,
 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3D,
 0x3D, 0x3D, 0x3D, 0x3D, 0x3D, 0x3D, 0x3D, 0x3D, 0x3D, 0x3D, 0x3D, 0x3D, 0x3D, 0x3D, 0x3D, 0x3D,
 0x3E, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E,
 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x43, 0x43, 0x43, 0x43, 0x43, 0x43, 0x43, 0x43,
 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x45, 0x45, 0x45, 0x45, 0x45, 0x45, 0x45, 0x45,
 0x46, 0x46, 0x46, 0x46, 0x46, 0x46, 0x46, 0x46, 0x47, 0x47, 0x47, 0x47, 0x47, 0x47, 0x47, 0x47,
 0x48, 0x48, 0x48, 0x48, 0x48, 0x48, 0x48, 0x48, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49, 0x49,
 0x4A, 0x4A, 0x4A, 0x4A, 0x4A, 0x4A, 0x4A, 0x4A, 0x4B, 0x4B, 0x4B, 0x4B, 0x4B, 0x4B, 0x4B, 0x4B,
 0x4C, 0x4C, 0x4C, 0x4C, 0x4C, 0x4C, 0x4C, 0x4C, 0x4D, 0x4D, 0x4D, 0x4D, 0x4D, 0x4D, 0x4D, 0x4D,
 0x4E, 0x4E, 0x4E, 0x4E, 0x4E, 0x4E, 0x4E, 0x4E, 0x4F, 0x4F, 0x4F, 0x4F, 0x4F, 0x4F, 0x4F, 0x4F,
 0x50, 0x50, 0x50, 0x50, 0x51, 0x51, 0x51, 0x51, 0x52, 0x52, 0x52, 0x52, 0x53, 0x53, 0x53, 0x53,
 0x54, 0x54, 0x54, 0x54, 0x55, 0x55, 0x55, 0x55, 0x56, 0x56, 0x56, 0x56, 0x57, 0x57, 0x57, 0x57,
 0x58, 0x58, 0x58, 0x58, 0x59, 0x59, 0x59, 0x59, 0x5A, 0x5A, 0x5A, 0x5A, 0x5B, 0x5B, 0x5B, 0x5B,
 0x5C, 0x5C, 0x5C, 0x5C, 0x5D, 0x5D, 0x5D, 0x5D, 0x5E, 0x5E, 0x5E, 0x5E, 0x5F, 0x5F, 0x5F, 0x5F,
 0x60, 0x60, 0x61, 0x61, 0x62, 0x62, 0x63, 0x63, 0x64, 0x64, 0x65, 0x65, 0x66, 0x66, 0x67, 0x67,
 0x68, 0x68, 0x68, 0x69, 0x69, 0x6A, 0x6A, 0x6B, 0x6B, 0x6C, 0x6C, 0x6D, 0x6D, 0x6E, 0x6E, 0x6F,
 0x6F, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E,
 0xFF, 0xFE, 0xFD, 0xFC, 0xFB, 0xFA, 0xF9, 0xF8, 0xF7, 0xF6, 0xF5, 0xF4, 0xF3, 0xF2, 0xF1, 0xF0,
 0xEF, 0xEF, 0xEE, 0xEE, 0xED, 0xED, 0xEC, 0xEC, 0xEB, 0xEB, 0xEA, 0xEA, 0xE9, 0xE9, 0xE8, 0xE8,
 0xE7, 0xE7, 0xE6, 0xE6, 0xE5, 0xE5, 0xE4, 0xE4, 0xE3, 0xE3, 0xE2, 0xE2, 0xE1, 0xE1, 0xE0, 0xE0,
 0xDF, 0xDF, 0xDF, 0xDF, 0xDE, 0xDE, 0xDE, 0xDE, 0xDD, 0xDD, 0xDD, 0xDD, 0xDC, 0xDC, 0xDC, 0xDC,
 0xDB, 0xDB, 0xDB, 0xDB, 0xDA, 0xDA, 0xDA, 0xDA, 0xD9, 0xD9, 0xD9, 0xD9, 0xD8, 0xD8, 0xD8, 0xD8,
 0xD7, 0xD7, 0xD7, 0xD7, 0xD6, 0xD6, 0xD6, 0xD6, 0xD5, 0xD5, 0xD5, 0xD5, 0xD4, 0xD4, 0xD4, 0xD4,
 0xD3, 0xD3, 0xD3, 0xD3, 0xD2, 0xD2, 0xD2, 0xD2, 0xD1, 0xD1, 0xD1, 0xD1, 0xD0, 0xD0, 0xD0, 0xD0,
 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE,
 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA,
 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8,
 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6,
 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4,
 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2,
 0xC2, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0,
 0xC0, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF,
 0xBF, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE,
 0xBE, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD,
 0xBD, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC,
 0xBC, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
 0xBB, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA,
 0xBA, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9,
 0xB9, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8,
 0xB8, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7,
 0xB7, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6,
 0xB6, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5,
 0xB5, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4,
 0xB4, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3,
 0xB3, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2,
 0xB2, 0xB2, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1,
 0xB1, 0xB1, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0,
 0xB0, 0xB0, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF,
 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF,
 0xAF, 0xAF, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE,
 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE,
 0xAE, 0xAE, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD,
 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD,
 0xAD, 0xAD, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC,
 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC,
 0xAC, 0xAC, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB,
 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB,
 0xAB, 0xAB, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
 0xAA, 0xAA, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9,
 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9,
 0xA9, 0xA9, 0xA9, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8,
 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8,
 0xA8, 0xA8, 0xA8, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7,
 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7,
 0xA7, 0xA7, 0xA7, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6,
 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6,
 0xA6, 0xA6, 0xA6, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5,
 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5,
 0xA5, 0xA5, 0xA5, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4,
 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4,
 0xA4, 0xA4, 0xA4, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
 0xA3, 0xA3, 0xA3, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2,
 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2,
 0xA2, 0xA2, 0xA2, 0xA2, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1,
 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1,
 0xA1, 0xA1, 0xA1, 0xA1, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0,
 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0,
 0xA0, 0xA0, 0xA0, 0xA0, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F,
 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F,
 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F,
 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F,
 0x9F, 0x9F, 0x9F, 0x9F, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E,
 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E,
 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E,
 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E,
 0x9E, 0x9E, 0x9E, 0x9E, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D,
 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D,
 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D,
 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D,
 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C,
 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C,
 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C,
 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C,
 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B,
 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B,
 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B,
 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B,
 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A,
 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A,
 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A,
 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A,
 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98,
 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98,
 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98,
 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98,
 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97,
 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97,
 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97,
 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97,
 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96,
 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96,
 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96,
 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96,
 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95,
 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95,
 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95,
 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95,
 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93,
 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93,
 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93,
 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93,
 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92,
 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92,
 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92,
 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92,
 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91,
 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91,
 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91,
 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91,
 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F,
 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F,
 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F,
 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F,
 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F,
 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F,
 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F,
 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F,
 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E,
 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E,
 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E,
 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E,
 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E,
 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E,
 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E,
 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E,
 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x89, 0x89, 0x89, 0x89, 0x89,
 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89,
 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89,
 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89,
 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89,
 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89,
 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89,
 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89,
 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x88, 0x88, 0x88, 0x88,
 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x87, 0x87, 0x87, 0x87,
 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87,
 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87,
 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87,
 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87,
 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87,
 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87,
 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87,
 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x86, 0x86, 0x86,
 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86,
 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86,
 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86,
 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86,
 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86,
 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86,
 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86,
 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x85, 0x85, 0x85,
 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x84, 0x84,
 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84,
 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84,
 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84,
 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84,
 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84,
 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84,
 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84,
 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x83, 0x83,
 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83,
 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83,
 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83,
 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83,
 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83,
 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83,
 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83,
 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x82,
 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x81,
 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81,
 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81,
 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81,
 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81,
 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81,
 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81,
 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81,
 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81,
 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80
};
uint8 *_l2u = _l2u_ + 4096;
#endif /* LOOKUP_HACK */

uint8 reverb_macro_presets[] =
{	/* CHARACTER,PRE-LPF,LEVEL,TIME,DELAY FEEDBACK,PREDELAY TIME */
	0,3,64,80,0,0,		/* 00: Room1 */
	1,4,64,56,0,0,		/* 01: Room2 */
	2,0,64,64,0,0,		/* 02: Room3 */
	3,4,64,72,0,0,		/* 03: Hall1 */
	4,0,64,64,0,0,		/* 04: Hall2 */
	5,0,64,88,0,0,		/* 05: Plate */
	6,0,64,32,40,0,		/* 06: Delay */
	7,0,64,64,32,0,		/* 07: Panning Delay */
};

uint8 chorus_macro_presets[] =
{	/* PRE-LPF,LEVEL,FEEDBACK,DELAY,RATE,DEPTH,SEND TO REVERB,SEND TO DELAY */
	0,64,0,112,3,5,0,0,		/* 00: Chorus1 */
	0,64,5,80,9,19,0,0,		/* 01: Chorus2 */
	0,64,8,80,3,19,0,0,		/* 02: Chorus3 */
	0,64,16,64,9,16,0,0,	/* 03: Chorus4 */
	0,64,64,127,2,24,0,0,	/* 04: Feedback Chorus */
	0,64,112,127,1,5,0,0,	/* 05: Flanger */
	0,64,0,127,0,127,0,0,	/* 06: Short Delay */
	0,64,80,127,0,127,0,0,	/* 07: Short Delay(Feedback) */
};

uint8 delay_macro_presets[] = 
{	/* PRE-LPF,TIME(C),RATIO(L),RATIO(R),LEVEL(C),LEVEL(L),LEVEL(R),LEVEL,FEEDBACK,LEVEL TO REVERB */
	0,97,1,1,127,0,0,64,79,0,		/* 00: Delay1 */
	0,106,1,1,127,0,0,64,79,0,		/* 01: Delay2 */
	0,115,1,1,127,0,0,64,63,0,		/* 02: Delay3 */
	0,83,1,1,127,0,0,64,71,0,		/* 03: Delay4 */
	0,90,12,24,0,125,60,64,73,0,	/* 04: Pan Delay1 */
	0,109,12,24,0,125,60,64,70,0,	/* 05: Pan Delay2 */
	0,115,12,24,0,120,64,64,72,0,	/* 06: Pan Delay3 */
	0,93,12,24,0,120,64,64,63,0,	/* 07: Pan Delay4 */
	0,109,12,24,0,114,60,64,60,36,	/* 08: Delay to Reverb */
	0,110,21,31,97,127,67,64,39,0,	/* 09: Pan Repeat */
};




/* Derivation of Perceived Volume Curve Equation:
 *
 *    Given: delta dB = 20 * log10(amplitude_new / amplitude_old)
 *           delta dB of 10 == perceived volume change of 2x
 *
 *    10 = 20 * log10(?)
 *    0.5 = log10(?)
 *    10^0.5 = ?
 *
 *    therefore: 2x perceived volume == ~3.16x amplitude
 *               4x perceived volume ==    10x amplitude
 *
 *       Volume         Amplitude
 *       ------------   ---------------
 *       1              1
 *       0.25           0.1
 *       0.0625         0.01
 *       0.015625       0.001
 *       0.00390625     0.0001
 *       0.0009765625   0.00001
 *       0              0
 *
 *    Fit curve to table:
 *
 *       amplification = pow(volume, 1.66096404744)
 */
FLOAT_T perceived_vol_table[128];

void init_perceived_vol_table(void)
{
	int i;
	
	for (i = 0; i < 128; i++)
		perceived_vol_table[i] =
				127.0 * pow((double)i * DIV_127, 1.66096404744);
}

FLOAT_T gm2_vol_table[128];

void init_gm2_vol_table(void)
{
	int i;
	
	for(i = 0; i < 128; i++)
		gm2_vol_table[i] = (i * i) * DIV_127;
}

FLOAT_T user_vol_table[128];

void init_user_vol_table(FLOAT_T power)
{
	int i;
	double diff;

	if ((int)power >= 3) {
//		diff = power - 3.0 + 0.212;
		diff = power - 3.0;

		if ((int)power == 4) {
			for (i = 0; i < 128; i++) {
				user_vol_table[i] = 127.0 * (1.0 - cosf((double)i * DIV_127 * M_PI)) * 0.5;
			}
			return;
		}

		for (i = 0; i < 64; i++) {
//			user_vol_table[i] = 127.0 * ((1.0 - pow(1.0 - (double)i * DIV_64, 2.0)) * DIV_2);
//			user_vol_table[i] = 127.0 * (0.8 - (pow(1.0 - (double)i * DIV_64, 2.0) * 0.8) * DIV_2);
			user_vol_table[i] = ((1.0 - pow(1.0 - (double)i * DIV_64, 2.0)) / 1.8);
		}
		for (i = 64; i < 128; i++) {
//			user_vol_table[i] = 127.0 * ((1.0 + pow(((double)i - 60.0) / 67.0, 2.0)) * DIV_2);
//			user_vol_table[i] = 127.0 * ((0.4 + pow(((double)i - 53.0) / 74.0, 1.44)) * 0.6);
			user_vol_table[i] = ((1.0 + pow(((double)i - 62.0) * DIV_65, 2.0)) / 1.8);
		}
		for (i = 0; i < 128; i++) {
			user_vol_table[i] = 127.0 * (user_vol_table[i] * pow((double)i * DIV_127, diff));
		}
		return;
	}

	for (i = 0; i < 128; i++)
		user_vol_table[i] = 127.0 * pow((double)i * DIV_127, power);
}

/* measured value from SC-88STPro.
   approximate expression: y = (-0.3768x6 + 0.9528x5 - 0.8253x4 + 0.2665x3 + 0.9892x2 - 0.0059x + 0.001) * 127 */
FLOAT_T sc_vol_table[128] = 
{
    0.000000, 0.128905, 0.146482, 0.179815, 0.228982, 0.294049, 0.375078, 0.472120, 
    0.585221, 0.714419, 0.859746, 1.021229, 1.198887, 1.392736, 1.602785, 1.829039, 
    2.071501, 2.330166, 2.605028, 2.896078, 3.203301, 3.526682, 3.866202, 4.221841, 
    4.593575, 4.981382, 5.385233, 5.805103, 6.240963, 6.692783, 7.160536, 7.644189, 
    8.143714, 8.659080, 9.190256, 9.737215, 10.299925, 10.878360, 11.472491, 12.082292, 
    12.707738, 13.348803, 14.005465, 14.677704, 15.365497, 16.068829, 16.787681, 17.522039, 
    18.271890, 19.037223, 19.818029, 20.614301, 21.426034, 22.253225, 23.095873, 23.953980, 
    24.827548, 25.716584, 26.621094, 27.541088, 28.476578, 29.427576, 30.394100, 31.376165, 
    32.373791, 33.386998, 34.415810, 35.460249, 36.520342, 37.596115, 38.687597, 39.794815, 
    40.917801, 42.056586, 43.211200, 44.381677, 45.568048, 46.770346, 47.988605, 49.222856, 
    50.473131, 51.739464, 53.021883, 54.320420, 55.635102, 56.965957, 58.313010, 59.676284, 
    61.055802, 62.451580, 63.863636, 65.291982, 66.736627, 68.197578, 69.674835, 71.168397, 
    72.678255, 74.204399, 75.746811, 77.305466, 78.880337, 80.471388, 82.078576, 83.701851, 
    85.341158, 86.996429, 88.667594, 90.354568, 92.057260, 93.775571, 95.509387, 97.258589, 
    99.023042, 100.802603, 102.597116, 104.406412, 106.230309, 108.068613, 109.921115, 111.787592, 
    113.667805, 115.561500, 117.468408, 119.388243, 121.320699, 123.265458, 125.222177, 127.000000, 
};

/* measured value from SC-88STPro.
   approximate expression: y = (-1.5374x6 + 4.4002x5 - 4.8309x4 + 2.572x3 + 0.1487x2 + 0.2412x + 0.0044) * 127 */
FLOAT_T sc_vel_table[128] = 
{
    0.000000, 0.801328, 1.047122, 1.297056, 1.551953, 1.812583, 2.079668, 2.353885, 
    2.635863, 2.926190, 3.225412, 3.534034, 3.852525, 4.181317, 4.520806, 4.871357, 
    5.233303, 5.606946, 5.992560, 6.390392, 6.800666, 7.223577, 7.659301, 8.107993, 
    8.569785, 9.044792, 9.533111, 10.034824, 10.549997, 11.078680, 11.620912, 12.176722, 
    12.746124, 13.329126, 13.925725, 14.535911, 15.159666, 15.796968, 16.447787, 17.112090, 
    17.789842, 18.481003, 19.185529, 19.903380, 20.634509, 21.378873, 22.136426, 22.907127, 
    23.690932, 24.487801, 25.297696, 26.120582, 26.956425, 27.805197, 28.666872, 29.541428, 
    30.428848, 31.329119, 32.242232, 33.168184, 34.106974, 35.058609, 36.023099, 37.000459, 
    37.990710, 38.993877, 40.009989, 41.039080, 42.081190, 43.136361, 44.204639, 45.286075, 
    46.380723, 47.488640, 48.609885, 49.744520, 50.892609, 52.054217, 53.229410, 54.418254, 
    55.620816, 56.837160, 58.067351, 59.311450, 60.569517, 61.841607, 63.127770, 64.428053, 
    65.742497, 67.071133, 68.413988, 69.771079, 71.142412, 72.527985, 73.927782, 75.341776, 
    76.769925, 78.212172, 79.668444, 81.138652, 82.622687, 84.120420, 85.631702, 87.156361, 
    88.694200, 90.244998, 91.808507, 93.384452, 94.972526, 96.572391, 98.183679, 99.805985, 
    101.438869, 103.081852, 104.734417, 106.396007, 108.066018, 109.743805, 111.428675, 113.119886, 
    114.816648, 116.518116, 118.223392, 119.931522, 121.641492, 123.352230, 125.062599, 127.000000, 
};

FLOAT_T sc_pan_table[129] = 
{
	0.000000, 0.000000, 0.999479, 2.011744, 3.036530, 4.073569, 5.122593, 6.183332,
	7.255517, 8.338874, 9.433131, 10.538014, 11.653247, 12.778552, 13.913653, 15.058271,
	16.212123, 17.374930, 18.546409, 19.726275, 20.914243, 22.110027, 23.313339, 24.523890,
	25.741391, 26.965550, 28.196074, 29.432671, 30.675045, 31.922900, 33.175939, 34.433863,
	35.696373, 36.963168, 38.233946, 39.508403, 40.786235, 42.067137, 43.350800, 44.636918,
	45.925181, 47.215278, 48.506897, 49.799726, 51.093451, 52.387755, 53.682323, 54.976837,
	56.270977, 57.564424, 58.856855, 60.147950, 61.437382, 62.724829, 64.009963, 65.292456,
	66.571981, 67.848208, 69.120804, 70.389439, 71.653778, 72.913487, 74.168230, 75.417670,
	76.661468, 77.899286, 79.130781, 80.355614, 81.573439, 82.783913, 83.986691, 85.181425,
	86.367767, 87.545369, 88.713880, 89.872949, 91.022222, 92.161346, 93.289965, 94.407723,
	95.514263, 96.609225, 97.692249, 98.762975, 99.821039, 100.866079, 101.897729, 102.915623,
	103.919394, 104.908673, 105.883091, 106.842276, 107.785858, 108.713461, 109.624713, 110.519236,
	111.396655, 112.256590, 113.098663, 113.922493, 114.727699, 115.513896, 116.280702, 117.027730,
	117.754595, 118.460908, 119.146280, 119.810321, 120.452639, 121.072843, 121.670538, 122.245328,
	122.796819, 123.324612, 123.828308, 124.307509, 124.761812, 125.190815, 125.594115, 125.971307,
	126.321986, 126.645744, 126.942172, 127.210862, 127.451402, 127.663381, 127.846385, 128.000000,
	128.000000,
};

FLOAT_T gm2_pan_table[129];
FLOAT_T *pan_table = sc_pan_table;

void init_gm2_pan_table(void)
{
	int i;

	gm2_pan_table[0] = 0;
	for(i = 0; i < 127; i++)
		gm2_pan_table[i + 1] = sin(M_PI / 2 * i / 126) * 128;
								/* lookup_sine(i * SINE_CYCLE_LENGTH / 4 / 126) */
	gm2_pan_table[128] = 128.0;
}

FLOAT_T sc_drum_level_table[128] = 
{
    0.007874, 0.007874, 0.031496, 0.070866, 0.125984, 0.196850, 0.283465, 0.385827, 
    0.503937, 0.637795, 0.787402, 0.952756, 1.133858, 1.330709, 1.543307, 1.771654, 
    2.015748, 2.275591, 2.551181, 2.842520, 3.149606, 3.472441, 3.811024, 4.165354, 
    4.535433, 4.921260, 5.322835, 5.740157, 6.173228, 6.622047, 7.086614, 7.566929, 
    8.062992, 8.574803, 9.102362, 9.645669, 10.204724, 10.779528, 11.370079, 11.976378, 
    12.598425, 13.236220, 13.889764, 14.559055, 15.244094, 15.944882, 16.661417, 17.393701, 
    18.141732, 18.905512, 19.685039, 20.480315, 21.291339, 22.118110, 22.960630, 23.818898, 
    24.692913, 25.582677, 26.488189, 27.409449, 28.346457, 29.299213, 30.267717, 31.251969, 
    32.251969, 33.267717, 34.299213, 35.346457, 36.409449, 37.488189, 38.582677, 39.692913, 
    40.818898, 41.960630, 43.118110, 44.291339, 45.480315, 46.685039, 47.905512, 49.141732, 
    50.393701, 51.661417, 52.944882, 54.244094, 55.559055, 56.889764, 58.236220, 59.598425, 
    60.976378, 62.370079, 63.779528, 65.204724, 66.645669, 68.102362, 69.574803, 71.062992, 
    72.566929, 74.086614, 75.622047, 77.173228, 78.740157, 80.322835, 81.921260, 83.535433, 
    85.165354, 86.811024, 88.472441, 90.149606, 91.842520, 93.551181, 95.275591, 97.015748, 
    98.771654, 100.543307, 102.330709, 104.133858, 105.952756, 107.787402, 109.637795, 111.503937, 
    113.385827, 115.283465, 117.196850, 119.125984, 121.070866, 123.031496, 125.007874, 127.000000, 
};

FLOAT_T attack_vol_table[1025];

void init_attack_vol_table(void)
{
	int i;
	
	for (i = 0; i < 1024; i++)
		attack_vol_table[i] = (double)i / 1023.0;
	attack_vol_table[1024] = 1.0;

}

float sc_eg_decay_table[128] = 
{
    81.841218, 81.841218, 81.841218, 81.841218, 81.841218, 81.841218, 81.841218, 81.841218, 
    81.841218, 81.841218, 81.841218, 81.841218, 81.841218, 81.841218, 81.841218, 81.841218, 
    81.841218, 81.841218, 81.841218, 81.841218, 81.841218, 81.841218, 74.928977, 68.580327, 
    62.750584, 57.398521, 52.486110, 47.978275, 43.842671, 40.049475, 36.571192, 33.382477, 
    30.459975, 27.782161, 25.329207, 23.082845, 21.026252, 19.143935, 17.421629, 15.846202, 
    14.405568, 13.088604, 11.885077, 10.785570, 9.781425, 8.864676, 8.028000, 7.264663, 
    6.568475, 5.933745, 5.355241, 4.828153, 4.348058, 3.910885, 3.512890, 3.150618, 
    2.820877, 2.520709, 2.247348, 1.998183, 1.770681, 1.562261, 1.369978, 1.189386, 
    1.000000, 0.838459, 0.726301, 0.635581, 0.559656, 0.494986, 0.439286, 0.390934, 
    0.348712, 0.311669, 0.279045, 0.250221, 0.224684, 0.202006, 0.181825, 0.163831, 
    0.147761, 0.133387, 0.120513, 0.108967, 0.098600, 0.089282, 0.080897, 0.073346, 
    0.066540, 0.060399, 0.054854, 0.049845, 0.045315, 0.041217, 0.037506, 0.034144, 
    0.031097, 0.028333, 0.025826, 0.023549, 0.021480, 0.019601, 0.017892, 0.016337, 
    0.014923, 0.013635, 0.012462, 0.011394, 0.011394, 0.011394, 0.011394, 0.011394, 
    0.011394, 0.011394, 0.011394, 0.011394, 0.011394, 0.011394, 0.011394, 0.011394, 
    0.011394, 0.011394, 0.011394, 0.011394, 0.011394, 0.011394, 0.011394, 0.011394, 
};

float sc_eg_release_table[128] = 
{
    27.322002, 27.322002, 27.322002, 27.322002, 27.322002, 27.322002, 27.322002, 27.322002, 
    27.322002, 27.322002, 27.322002, 27.322002, 27.322002, 27.322002, 27.322002, 27.322002, 
    27.322002, 27.322002, 27.322002, 27.322002, 27.322002, 27.322002, 25.299110, 23.425992, 
    21.691556, 20.085537, 18.598425, 17.221418, 15.946363, 14.765711, 13.672474, 12.660179, 
    11.722833, 10.854887, 10.051203, 9.307023, 8.617941, 7.979878, 7.389056, 6.841978, 
    6.335406, 5.866339, 5.432002, 5.029822, 4.657419, 4.312589, 3.993290, 3.697631, 
    3.423862, 3.170363, 2.935633, 2.718282, 2.517023, 2.330665, 2.158106, 1.998322, 
    1.850368, 1.713369, 1.586513, 1.469049, 1.360282, 1.259569, 1.166311, 1.079959, 
    1.000000, 0.925961, 0.857404, 0.793923, 0.735141, 0.680712, 0.630313, 0.583645, 
    0.540433, 0.500420, 0.463369, 0.429062, 0.397295, 0.367879, 0.340642, 0.315421, 
    0.292068, 0.270443, 0.250420, 0.231879, 0.214711, 0.198814, 0.184094, 0.170464, 
    0.157843, 0.146157, 0.135335, 0.125315, 0.116037, 0.107446, 0.099491, 0.092124, 
    0.085304, 0.078988, 0.073140, 0.067724, 0.062710, 0.058067, 0.053768, 0.049787, 
    0.046101, 0.042688, 0.039527, 0.036601, 0.036601, 0.036601, 0.036601, 0.036601, 
    0.036601, 0.036601, 0.036601, 0.036601, 0.036601, 0.036601, 0.036601, 0.036601, 
    0.036601, 0.036601, 0.036601, 0.036601, 0.036601, 0.036601, 0.036601, 0.036601, 
};

float sc_eg_attack_table[128] = 
{
    82.756924, 82.756924, 82.756924, 82.756924, 82.756924, 82.756924, 82.756924, 82.756924, 
    82.756924, 82.756924, 82.756924, 82.756924, 82.756924, 82.756924, 82.756924, 82.756924, 
    82.756924, 82.756924, 82.756924, 82.756924, 82.756924, 82.756924, 75.473398, 68.815182, 
    62.729632, 57.168464, 52.087395, 47.445819, 43.206507, 39.335325, 35.800987, 32.574817, 
    29.630534, 26.944060, 24.493331, 22.258137, 20.219967, 18.361866, 16.668311, 15.125088, 
    13.719184, 12.438688, 11.272700, 10.211246, 9.245197, 8.366205, 7.566631, 6.839489, 
    6.178391, 5.577493, 5.031451, 4.535378, 4.084805, 3.675641, 3.304143, 2.966879, 
    2.660703, 2.382715, 2.130237, 1.900768, 1.691929, 1.501374, 1.326560, 1.163993, 
    1.000000, 0.859112, 0.753830, 0.666057, 0.591041, 0.526103, 0.469431, 0.419689, 
    0.375841, 0.337054, 0.302650, 0.272061, 0.244810, 0.220489, 0.198750, 0.179292, 
    0.161854, 0.146210, 0.132159, 0.119529, 0.108164, 0.097931, 0.088710, 0.080394, 
    0.072891, 0.066115, 0.059994, 0.054461, 0.049456, 0.044927, 0.040827, 0.037114, 
    0.033749, 0.030699, 0.027932, 0.025422, 0.023145, 0.021077, 0.019199, 0.017492, 
    0.015941, 0.014532, 0.013250, 0.012084, 0.012084, 0.012084, 0.012084, 0.012084, 
    0.012084, 0.012084, 0.012084, 0.012084, 0.012084, 0.012084, 0.012084, 0.012084, 
    0.012084, 0.012084, 0.012084, 0.012084, 0.012084, 0.012084, 0.012084, 0.012084, 
};

FLOAT_T sb_vol_table[1025];

void init_sb_vol_table(void)
{
	int i;
	
	for (i = 0; i < 1024; i++)
		sb_vol_table[i] = pow(10.0, (double)(1023 - i) * 960.0 / (1023.0 * -200.0));
	sb_vol_table[1024] = 1.0;
}

FLOAT_T modenv_vol_table[1025], sf2_concave_table[1025], sf2_convex_table[1025];
#define SF2_CONCAVE(a) (1.0 - sqrt(1.0 - (a) * (a)))
#define SF2_CONVEX(a) (sqrt(1.0 - (1.0 - a) * (1.0 - a)))
//#define SF2_CONCAVE(a) (log10((double)((1.0 - a) * (1.0 - a))) * (-20.0) * DIV_96 * 0.79737513840065736552677499445091)
//#define SF2_CONVEX(a) (1.0 - log10((double)((a) * (a))) * (-20.0) * DIV_96 * 0.79737513840065736552677499445091)
#define CURVE_MIX(a,b,c) ((a) * (c) + (b) * (1.0 - (c)))
// 0.591685

void init_modenv_vol_table(void)
{
	int i;
	double x, linear;
	
	modenv_vol_table[0] = (FLOAT_T)0;
	sf2_concave_table[0] = (FLOAT_T)0;
	sf2_convex_table[0] = (FLOAT_T)0;
	for (i = 1; i < 1023; i++) {
		linear = (double)i * DIV_1023;
		x = (1.0 - (-20.0 * DIV_96 * log((linear * linear) * DIV_LN10)));
		if (x < 0) {x = 0;}
		modenv_vol_table[i] = log(x + 1) * DIV_LN2;
	//	sf2_concave_table[i] = SF2_CONCAVE(linear);
	//	sf2_convex_table[i] = SF2_CONVEX(linear);
		sf2_concave_table[i] = CURVE_MIX(SF2_CONCAVE(linear), linear, 0.5);
		sf2_convex_table[i] = CURVE_MIX(SF2_CONVEX(linear), linear, 0.5);
	}
	modenv_vol_table[1024] = modenv_vol_table[1023] = (FLOAT_T)1.0;
	sf2_concave_table[1024] = sf2_concave_table[1023] = (FLOAT_T)1.0;
	sf2_convex_table[1024] = sf2_convex_table[1023] = (FLOAT_T)1.0;
}


float cb_to_amp_table[961] = 
{
    1.000000, 0.995677, 0.991373, 0.987088, 0.982821, 0.978572, 0.974342, 0.970130, 
    0.965936, 0.961761, 0.957603, 0.953464, 0.949342, 0.945238, 0.941152, 0.937084, 
    0.933033, 0.929000, 0.924984, 0.920985, 0.917004, 0.913040, 0.909093, 0.905163, 
    0.901250, 0.897355, 0.893475, 0.889613, 0.885768, 0.881939, 0.878126, 0.874330, 
    0.870551, 0.866787, 0.863040, 0.859310, 0.855595, 0.851896, 0.848214, 0.844547, 
    0.840896, 0.837261, 0.833642, 0.830038, 0.826450, 0.822878, 0.819321, 0.815779, 
    0.812252, 0.808741, 0.805245, 0.801764, 0.798298, 0.794848, 0.791412, 0.787990, 
    0.784584, 0.781192, 0.777816, 0.774453, 0.771105, 0.767772, 0.764453, 0.761149, 
    0.757858, 0.754582, 0.751320, 0.748072, 0.744839, 0.741619, 0.738413, 0.735221, 
    0.732043, 0.728878, 0.725728, 0.722590, 0.719467, 0.716357, 0.713260, 0.710177, 
    0.707107, 0.704050, 0.701007, 0.697976, 0.694959, 0.691955, 0.688964, 0.685986, 
    0.683020, 0.680068, 0.677128, 0.674201, 0.671286, 0.668384, 0.665495, 0.662618, 
    0.659754, 0.656902, 0.654062, 0.651235, 0.648420, 0.645617, 0.642826, 0.640047, 
    0.637280, 0.634525, 0.631783, 0.629051, 0.626332, 0.623625, 0.620929, 0.618245, 
    0.615572, 0.612911, 0.610262, 0.607624, 0.604997, 0.602382, 0.599778, 0.597185, 
    0.594604, 0.592033, 0.589474, 0.586926, 0.584389, 0.581862, 0.579347, 0.576843, 
    0.574349, 0.571866, 0.569394, 0.566933, 0.564482, 0.562042, 0.559612, 0.557193, 
    0.554785, 0.552387, 0.549999, 0.547621, 0.545254, 0.542897, 0.540550, 0.538213, 
    0.535887, 0.533570, 0.531264, 0.528967, 0.526681, 0.524404, 0.522137, 0.519880, 
    0.517632, 0.515395, 0.513167, 0.510949, 0.508740, 0.506541, 0.504351, 0.502171, 
    0.500000, 0.497839, 0.495687, 0.493544, 0.491410, 0.489286, 0.487171, 0.485065, 
    0.482968, 0.480880, 0.478802, 0.476732, 0.474671, 0.472619, 0.470576, 0.468542, 
    0.466516, 0.464500, 0.462492, 0.460493, 0.458502, 0.456520, 0.454547, 0.452582, 
    0.450625, 0.448677, 0.446738, 0.444807, 0.442884, 0.440969, 0.439063, 0.437165, 
    0.435275, 0.433394, 0.431520, 0.429655, 0.427798, 0.425948, 0.424107, 0.422274, 
    0.420448, 0.418631, 0.416821, 0.415019, 0.413225, 0.411439, 0.409660, 0.407889, 
    0.406126, 0.404371, 0.402623, 0.400882, 0.399149, 0.397424, 0.395706, 0.393995, 
    0.392292, 0.390596, 0.388908, 0.387227, 0.385553, 0.383886, 0.382227, 0.380574, 
    0.378929, 0.377291, 0.375660, 0.374036, 0.372419, 0.370809, 0.369207, 0.367611, 
    0.366021, 0.364439, 0.362864, 0.361295, 0.359733, 0.358178, 0.356630, 0.355088, 
    0.353553, 0.352025, 0.350503, 0.348988, 0.347480, 0.345977, 0.344482, 0.342993, 
    0.341510, 0.340034, 0.338564, 0.337100, 0.335643, 0.334192, 0.332748, 0.331309, 
    0.329877, 0.328451, 0.327031, 0.325617, 0.324210, 0.322808, 0.321413, 0.320024, 
    0.318640, 0.317263, 0.315891, 0.314526, 0.313166, 0.311812, 0.310464, 0.309122, 
    0.307786, 0.306456, 0.305131, 0.303812, 0.302499, 0.301191, 0.299889, 0.298593, 
    0.297302, 0.296017, 0.294737, 0.293463, 0.292194, 0.290931, 0.289674, 0.288421, 
    0.287175, 0.285933, 0.284697, 0.283466, 0.282241, 0.281021, 0.279806, 0.278597, 
    0.277392, 0.276193, 0.274999, 0.273811, 0.272627, 0.271448, 0.270275, 0.269107, 
    0.267943, 0.266785, 0.265632, 0.264484, 0.263340, 0.262202, 0.261068, 0.259940, 
    0.258816, 0.257697, 0.256583, 0.255474, 0.254370, 0.253270, 0.252175, 0.251085, 
    0.250000, 0.248919, 0.247843, 0.246772, 0.245705, 0.244643, 0.243585, 0.242533, 
    0.241484, 0.240440, 0.239401, 0.238366, 0.237336, 0.236310, 0.235288, 0.234271, 
    0.233258, 0.232250, 0.231246, 0.230246, 0.229251, 0.228260, 0.227273, 0.226291, 
    0.225313, 0.224339, 0.223369, 0.222403, 0.221442, 0.220485, 0.219532, 0.218583, 
    0.217638, 0.216697, 0.215760, 0.214827, 0.213899, 0.212974, 0.212053, 0.211137, 
    0.210224, 0.209315, 0.208411, 0.207510, 0.206613, 0.205719, 0.204830, 0.203945, 
    0.203063, 0.202185, 0.201311, 0.200441, 0.199575, 0.198712, 0.197853, 0.196998, 
    0.196146, 0.195298, 0.194454, 0.193613, 0.192776, 0.191943, 0.191113, 0.190287, 
    0.189465, 0.188646, 0.187830, 0.187018, 0.186210, 0.185405, 0.184603, 0.183805, 
    0.183011, 0.182220, 0.181432, 0.180648, 0.179867, 0.179089, 0.178315, 0.177544, 
    0.176777, 0.176013, 0.175252, 0.174494, 0.173740, 0.172989, 0.172241, 0.171496, 
    0.170755, 0.170017, 0.169282, 0.168550, 0.167822, 0.167096, 0.166374, 0.165655, 
    0.164938, 0.164225, 0.163516, 0.162809, 0.162105, 0.161404, 0.160706, 0.160012, 
    0.159320, 0.158631, 0.157946, 0.157263, 0.156583, 0.155906, 0.155232, 0.154561, 
    0.153893, 0.153228, 0.152565, 0.151906, 0.151249, 0.150595, 0.149944, 0.149296, 
    0.148651, 0.148008, 0.147368, 0.146731, 0.146097, 0.145466, 0.144837, 0.144211, 
    0.143587, 0.142967, 0.142349, 0.141733, 0.141121, 0.140511, 0.139903, 0.139298, 
    0.138696, 0.138097, 0.137500, 0.136905, 0.136313, 0.135724, 0.135138, 0.134553, 
    0.133972, 0.133393, 0.132816, 0.132242, 0.131670, 0.131101, 0.130534, 0.129970, 
    0.129408, 0.128849, 0.128292, 0.127737, 0.127185, 0.126635, 0.126088, 0.125543, 
    0.125000, 0.124460, 0.123922, 0.123386, 0.122853, 0.122322, 0.121793, 0.121266, 
    0.120742, 0.120220, 0.119700, 0.119183, 0.118668, 0.118155, 0.117644, 0.117135, 
    0.116629, 0.116125, 0.115623, 0.115123, 0.114626, 0.114130, 0.113637, 0.113145, 
    0.112656, 0.112169, 0.111684, 0.111202, 0.110721, 0.110242, 0.109766, 0.109291, 
    0.108819, 0.108348, 0.107880, 0.107414, 0.106949, 0.106487, 0.106027, 0.105568, 
    0.105112, 0.104658, 0.104205, 0.103755, 0.103306, 0.102860, 0.102415, 0.101972, 
    0.101532, 0.101093, 0.100656, 0.100221, 0.099787, 0.099356, 0.098926, 0.098499, 
    0.098073, 0.097649, 0.097227, 0.096807, 0.096388, 0.095972, 0.095557, 0.095144, 
    0.094732, 0.094323, 0.093915, 0.093509, 0.093105, 0.092702, 0.092302, 0.091903, 
    0.091505, 0.091110, 0.090716, 0.090324, 0.089933, 0.089545, 0.089158, 0.088772, 
    0.088388, 0.088006, 0.087626, 0.087247, 0.086870, 0.086494, 0.086120, 0.085748, 
    0.085378, 0.085008, 0.084641, 0.084275, 0.083911, 0.083548, 0.083187, 0.082827, 
    0.082469, 0.082113, 0.081758, 0.081404, 0.081052, 0.080702, 0.080353, 0.080006, 
    0.079660, 0.079316, 0.078973, 0.078631, 0.078292, 0.077953, 0.077616, 0.077281, 
    0.076947, 0.076614, 0.076283, 0.075953, 0.075625, 0.075298, 0.074972, 0.074648, 
    0.074325, 0.074004, 0.073684, 0.073366, 0.073049, 0.072733, 0.072418, 0.072105, 
    0.071794, 0.071483, 0.071174, 0.070867, 0.070560, 0.070255, 0.069952, 0.069649, 
    0.069348, 0.069048, 0.068750, 0.068453, 0.068157, 0.067862, 0.067569, 0.067277, 
    0.066986, 0.066696, 0.066408, 0.066121, 0.065835, 0.065550, 0.065267, 0.064985, 
    0.064704, 0.064424, 0.064146, 0.063869, 0.063592, 0.063318, 0.063044, 0.062771, 
    0.062500, 0.062230, 0.061961, 0.061693, 0.061426, 0.061161, 0.060896, 0.060633, 
    0.060371, 0.060110, 0.059850, 0.059591, 0.059334, 0.059077, 0.058822, 0.058568, 
    0.058315, 0.058062, 0.057811, 0.057562, 0.057313, 0.057065, 0.056818, 0.056573, 
    0.056328, 0.056085, 0.055842, 0.055601, 0.055360, 0.055121, 0.054883, 0.054646, 
    0.054409, 0.054174, 0.053940, 0.053707, 0.053475, 0.053244, 0.053013, 0.052784, 
    0.052556, 0.052329, 0.052103, 0.051877, 0.051653, 0.051430, 0.051208, 0.050986, 
    0.050766, 0.050546, 0.050328, 0.050110, 0.049894, 0.049678, 0.049463, 0.049249, 
    0.049037, 0.048825, 0.048613, 0.048403, 0.048194, 0.047986, 0.047778, 0.047572, 
    0.047366, 0.047161, 0.046958, 0.046755, 0.046552, 0.046351, 0.046151, 0.045951, 
    0.045753, 0.045555, 0.045358, 0.045162, 0.044967, 0.044772, 0.044579, 0.044386, 
    0.044194, 0.044003, 0.043813, 0.043624, 0.043435, 0.043247, 0.043060, 0.042874, 
    0.042689, 0.042504, 0.042320, 0.042138, 0.041955, 0.041774, 0.041593, 0.041414, 
    0.041235, 0.041056, 0.040879, 0.040702, 0.040526, 0.040351, 0.040177, 0.040003, 
    0.039830, 0.039658, 0.039486, 0.039316, 0.039146, 0.038977, 0.038808, 0.038640, 
    0.038473, 0.038307, 0.038141, 0.037976, 0.037812, 0.037649, 0.037486, 0.037324, 
    0.037163, 0.037002, 0.036842, 0.036683, 0.036524, 0.036366, 0.036209, 0.036053, 
    0.035897, 0.035742, 0.035587, 0.035433, 0.035280, 0.035128, 0.034976, 0.034825, 
    0.034674, 0.034524, 0.034375, 0.034226, 0.034078, 0.033931, 0.033784, 0.033638, 
    0.033493, 0.033348, 0.033204, 0.033060, 0.032918, 0.032775, 0.032634, 0.032492, 
    0.032352, 0.032212, 0.032073, 0.031934, 0.031796, 0.031659, 0.031522, 0.031386, 
    0.031250, 0.031115, 0.030980, 0.030846, 0.030713, 0.030580, 0.030448, 0.030317, 
    0.030186, 0.030055, 0.029925, 0.029796, 0.029667, 0.029539, 0.029411, 0.029284, 
    0.029157, 0.029031, 0.028906, 0.028781, 0.028656, 0.028533, 0.028409, 0.028286, 
    0.028164, 0.028042, 0.027921, 0.027800, 0.027680, 0.027561, 0.027441, 0.027323, 
    0.027205, 0.027087, 0.026970, 0.026853, 0.026737, 0.026622, 0.026507, 0.026392, 
    0.026278, 0.026164, 0.026051, 0.025939, 0.025827, 0.025715, 0.025604, 0.025493, 
    0.025383, 0.025273, 0.025164, 0.025055, 0.024947, 0.024839, 0.024732, 0.024625, 
    0.024518, 0.024412, 0.024307, 0.024202, 0.024097, 0.023993, 0.023889, 0.023786, 
    0.023683, 0.023581, 0.023479, 0.023377, 0.023276, 0.023176, 0.023075, 0.022976, 
    0.022876, 0.022777, 0.022679, 0.022581, 0.022483, 0.022386, 0.022289, 0.022193, 
    0.022097, 0.022002, 0.021906, 0.021812, 0.021717, 0.021624, 0.021530, 0.021437, 
    0.021344, 0.021252, 0.021160, 0.021069, 0.020978, 0.020887, 0.020797, 0.020707, 
    0.020617, 0.020528, 0.020439, 0.020351, 0.020263, 0.020176, 0.020088, 0.020001, 
    0.019915, 0.019829, 0.019743, 0.019658, 0.019573, 0.019488, 0.019404, 0.019320, 
    0.019237, 0.019153, 0.019071, 0.018988, 0.018906, 0.018824, 0.018743, 0.018662, 
    0.018581, 0.018501, 0.018421, 0.018341, 0.018262, 0.018183, 0.018105, 0.018026, 
    0.017948, 0.017871, 0.017794, 0.017717, 0.017640, 0.017564, 0.017488, 0.017412, 
    0.017337, 0.017262, 0.017187, 0.017113, 0.017039, 0.016966, 0.016892, 0.016819, 
    0.016746, 0.016674, 0.016602, 0.016530, 0.016459, 0.016388, 0.016317, 0.016246, 
    0.016176, 0.016106, 0.016036, 0.015967, 0.015898, 0.015829, 0.015761, 0.015693, 
    0.015625, 
};

/* phase lag between left and right ear. (in ms) */
float pan_delay_table[128] = 
{
    0.000000, 0.006136, 0.012271, 0.018404, 0.024534, 0.030660, 0.036782, 0.042899, 
    0.049009, 0.055111, 0.061205, 0.067290, 0.073365, 0.079429, 0.085481, 0.091520, 
    0.097545, 0.103556, 0.109551, 0.115529, 0.121490, 0.127433, 0.133356, 0.139260, 
    0.145142, 0.151003, 0.156841, 0.162655, 0.168445, 0.174209, 0.179948, 0.185659, 
    0.191342, 0.196996, 0.202621, 0.208215, 0.213778, 0.219308, 0.224806, 0.230269, 
    0.235698, 0.241092, 0.246449, 0.251769, 0.257051, 0.262295, 0.267499, 0.272662, 
    0.277785, 0.282866, 0.287904, 0.292899, 0.297850, 0.302756, 0.307616, 0.312430, 
    0.317197, 0.321916, 0.326586, 0.331208, 0.335779, 0.340300, 0.344770, 0.349188, 
    0.353553, 0.357865, 0.362124, 0.366327, 0.370476, 0.374568, 0.378604, 0.382584, 
    0.386505, 0.390369, 0.394173, 0.397918, 0.401604, 0.405229, 0.408792, 0.412295, 
    0.415735, 0.419112, 0.422427, 0.425678, 0.428864, 0.431986, 0.435043, 0.438035, 
    0.440961, 0.443820, 0.446612, 0.449337, 0.451995, 0.454584, 0.457105, 0.459557, 
    0.461940, 0.464253, 0.466496, 0.468670, 0.470772, 0.472804, 0.474764, 0.476653, 
    0.478470, 0.480215, 0.481888, 0.483488, 0.485016, 0.486470, 0.487851, 0.489159, 
    0.490393, 0.491553, 0.492639, 0.493651, 0.494588, 0.495451, 0.496240, 0.496953, 
    0.497592, 0.498156, 0.498645, 0.499059, 0.499398, 0.499661, 0.499849, 0.500000, 
};

/* for 0dB, 0.25dB, 0.5dB,..., 24dB. */
float chamberlin_filter_db_to_q_table[97] = 
{
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000, 1.029207, 1.113701, 1.205132, 
    1.304068, 1.411127, 1.526975, 1.652334, 1.787984, 1.934771, 2.093608, 2.265485, 
    2.451472, 2.652729, 2.870507, 3.106165, 3.361169, 3.637108, 3.935700, 4.258806, 
    4.608437, 4.986772, 5.396167, 5.839171, 6.318544, 6.837272, 7.398585, 8.005980, 
    8.663240, 9.374459, 10.144065, 10.976853, 11.878010, 12.853149, 13.908342, 15.050163, 
    16.285723, 17.622717, 19.069474, 20.635003, 22.329057, 24.162185, 26.145807, 28.292276, 
    30.614961, 33.128330, 35.848037, 38.791022, 41.975614, 45.421648, 49.150589, 53.185661, 
    57.551996, 62.276791, 67.389473, 72.921887, 78.908490, 85.386569, 92.396474, 99.981865, 
    108.189987, 117.071964, 126.683116, 137.083307, 148.337313, 160.515229, 173.692904, 187.952416, 
    203.382577, 220.079495, 238.147165, 257.698120, 278.854132, 301.746971, 326.519223, 353.325180, 
    382.331802, 
};


///r
/********** GS SYS **********/
/* Reverb Time in sec */
float reverb_time_table[128] = 
{
    0.410349, 0.440872, 0.468882, 0.494640, 0.518394, 0.540373, 0.560793, 0.579854, 
    0.597743, 0.614635, 0.630688, 0.646053, 0.660866, 0.675251, 0.689325, 0.703192, 
    0.716947, 0.730676, 0.744456, 0.758358, 0.772441, 0.786761, 0.801365, 0.816293, 
    0.831583, 0.847262, 0.863356, 0.879886, 0.896866, 0.914308, 0.932223, 0.950614, 
    0.969484, 0.988835, 1.008663, 1.028967, 1.049741, 1.070980, 1.092677, 1.114826, 
    1.137419, 1.160450, 1.183914, 1.207803, 1.232115, 1.256845, 1.281992, 1.307556, 
    1.333540, 1.359947, 1.386784, 1.414061, 1.441788, 1.469982, 1.498661, 1.527845, 
    1.557561, 1.587836, 1.618703, 1.650199, 1.682363, 1.715240, 1.748879, 1.783333, 
    1.818659, 1.854921, 1.892183, 1.930517, 1.970001, 2.010713, 2.052741, 2.096173, 
    2.141107, 2.187641, 2.235880, 2.285935, 2.337920, 2.391955, 2.448163, 2.506674, 
    2.567622, 2.631144, 2.697384, 2.766490, 2.838612, 2.913907, 2.992536, 3.074662, 
    3.160454, 3.250085, 3.343730, 3.441570, 3.543786, 3.650566, 3.762098, 3.878575, 
    4.000192, 4.127146, 4.259638, 4.397868, 4.542042, 4.692364, 4.849041, 5.012281, 
    5.182294, 5.359289, 5.543476, 5.735064, 5.934264, 6.141286, 6.356336, 6.356336, 
    6.356336, 6.356336, 6.356336, 6.356336, 6.356336, 6.356336, 6.356336, 6.356336, 
    6.356336, 6.356336, 6.356336, 6.356336, 6.356336, 6.356336, 6.356336, 6.356336, 
};

float chorus_delay_time_table[] =
{
    0.000000, 0.078740, 0.157480, 0.236220, 0.314961, 0.393701, 0.472441, 0.551181, 
    0.629921, 0.708661, 0.787402, 0.866142, 0.944882, 1.023622, 1.102362, 1.181102, 
    1.259843, 1.338583, 1.417323, 1.496063, 1.574803, 1.653543, 1.732283, 1.811024, 
    1.889764, 1.968504, 2.047244, 2.125984, 2.204724, 2.283465, 2.362205, 2.440945, 
    2.519685, 2.598425, 2.677165, 2.755906, 2.834646, 2.913386, 2.992126, 3.070866, 
    3.149606, 3.228346, 3.307087, 3.385827, 3.464567, 3.543307, 3.622047, 3.700787, 
    3.779528, 3.858268, 3.937008, 4.015748, 4.094488, 4.173228, 4.251969, 4.330709, 
    4.409449, 4.488189, 4.566929, 4.645669, 4.724409, 4.803150, 4.881890, 4.960630, 
    5.039370, 5.118110, 5.196850, 5.275591, 5.354331, 5.433071, 5.511811, 5.590551, 
    5.669291, 5.748031, 5.826772, 5.905512, 5.984252, 6.062992, 6.141732, 6.220472, 
    6.299213, 6.377953, 6.456693, 6.535433, 6.614173, 6.692913, 6.771654, 6.850394, 
    6.929134, 7.007874, 7.086614, 7.165354, 7.244094, 7.322835, 7.401575, 7.480315, 
    7.559055, 7.637795, 7.716535, 7.795276, 10.000000, 10.555556, 11.111111, 11.666667, 
    12.222222, 12.777778, 13.333333, 13.888889, 14.444444, 15.000000, 15.555556, 16.111111, 
    16.666667, 17.222222, 17.777778, 18.333333, 18.888889, 19.444444, 20.000000, 20.555556, 
    21.111111, 21.666667, 22.222222, 22.777778, 23.333333, 23.888889, 24.444444, 25.000000, 
};

float delay_time_center_table[] =
{	/* 0x00~0x73, 0.1ms~1000ms */
	0.1, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0, 1.1, 1.2, 1.3, 1.4, 1.5, 1.6, 1.7, 1.8, 1.9,
	2.0, 2.2, 2.4, 2.6, 2.8, 3.0, 3.2, 3.4, 3.6, 3.8, 4.0, 4.2, 4.4, 4.6, 4.8,
	5.0, 5.5, 6.0, 6.5, 7.0, 7.5, 8.0, 8.5, 9.0, 9.5,
	10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
	20, 22, 24, 26, 28, 30, 32, 34, 36, 38, 40, 42, 44, 46, 48,
	50, 55, 60, 65, 70, 75, 80, 85, 90, 95,
	100, 110, 120, 130, 140, 150, 160, 170, 180, 190,
	200, 220, 240, 260, 280, 300, 320, 340, 360, 380, 400, 420, 440, 460, 480,
	500, 550, 600, 650, 700, 750, 800, 850, 900, 950, 1000,
};


/********** GS INS **********/
// table 1
float pre_delay_time_table[] =
{
	0.0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0, 1.1, 1.2, 1.3, 1.4, 1.5, 1.6, 1.7, 1.8, 1.9,
	2.0, 2.1, 2.2, 2.3, 2.4, 2.5, 2.6, 2.7, 2.8, 2.9, 3.0, 3.1, 3.2, 3.3, 3.4, 3.5, 3.6, 3.7, 3.8, 3.9,
	4.0, 4.1, 4.2, 4.3, 4.4, 4.5, 4.6, 4.7, 4.8, 4.9,
	5.0, 5.5, 6.0, 6.5, 7.0, 7.5, 8.0, 8.5, 9.0, 9.5,
	10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29,
	30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49,
	50, 52, 54, 56, 58, 60, 62, 64, 66, 68, 70, 72, 74, 76, 78, 80, 82, 84, 86, 88,
	90, 92, 94, 96, 98, 100, 100, 100,
};

// table 2
int16 delay_time_1_table[128] =
{ 
	200,205,210,215,220,225,230,235,240,245,250,255,260,265,270,275,280,285,290,295,
	300,305,310,315,320,325,330,335,340,345,350,355,360,365,370,375,380,385,390,395,
	400,405,410,415,420,425,430,435,440,445,450,455,460,465,470,475,480,485,490,495,
	500,505,510,515,520,525,530,535,540,545,550,560,570,580,590,600,610,620,630,640,
	650,660,670,680,690,700,710,720,730,740,750,760,770,780,790,800,810,820,830,840,
	850,860,870,880,890,900,910,920,930,940,950,960,970,980,990,1000,1000,1000,1000,1000,
	1000,1000,1000,1000,1000,1000,1000,
};

// table 3
int16 delay_time_2_table[128] =
{ 
	200,205,210,215,220,225,230,235,240,245,250,255,260,265,270,275,280,285,290,295,
	300,305,310,315,320,325,330,335,340,345,350,355,360,365,370,375,380,385,390,395,
	400,405,410,415,420,425,430,435,440,445,450,455,460,465,470,475,480,485,490,495,
	500,505,510,515,520,525,530,535,540,545,550,555,560,565,570,575,580,585,590,595,
	600,610,620,630,640,650,660,670,680,690,700,710,720,730,740,750,760,770,780,790,
	800,810,820,830,840,850,860,870,880,890,900,910,920,930,940,950,960,970,980,990,
	1000,1000,1000,1000,1000,1000,1000,1000,
};

// table 4
float delay_time_3_table[128] =
{	/* 0x00~0x7F, 0.0ms~500ms */
	0.0,0.1,0.2,0.3,0.4,0.5,0.6,0.7,0.8,0.9,1.0,1.1,1.2,1.3,1.4,1.5,1.6,1.7,1.8,1.9,
	2.0,2.1,2.2,2.3,2.4,2.5,2.6,2.7,2.8,2.9,3.0,3.1,3.2,3.3,3.4,3.5,3.6,3.7,3.8,3.9,
	4.0,4.1,4.2,4.3,4.4,4.5,4.6,4.7,4.8,4.9,5.0,5.5,6.0,6.5,7.0,7.5,8.0,8.5,9.0,9.5,
	10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,
	30,31,32,33,34,35,36,37,38,39,40,50,60,70,80,90,100,110,120,130,
	140,150,160,170,180,190,200,210,220,230,240,250,260,270,280,290,300,320,340,360,
	380,400,420,440,460,480,500,500,
};

// table 5
float delay_time_4_table[128] =
{	/* 0x00~0x7F, 0.0ms~635ms */
	  0.0,   5.0,  10.0,  15.0,  20.0,  25.0,  30.0,  35.0,  40.0,  45.0, 
	 50.0,  55.0,  60.0,  65.0,  70.0,  75.0,  80.0,  85.0,  90.0,  95.0, 
	100.0, 105.0, 110.0, 115.0, 120.0, 125.0, 130.0, 135.0, 140.0, 145.0, 
	150.0, 155.0, 160.0, 165.0, 170.0, 175.0, 180.0, 185.0, 190.0, 195.0, 
	200.0, 205.0, 210.0, 215.0, 220.0, 225.0, 230.0, 235.0, 240.0, 245.0, 
	250.0, 255.0, 260.0, 265.0, 270.0, 275.0, 280.0, 285.0, 290.0, 295.0, 
	300.0, 305.0, 310.0, 315.0, 320.0, 325.0, 330.0, 335.0, 340.0, 345.0, 
	350.0, 355.0, 360.0, 365.0, 370.0, 375.0, 380.0, 385.0, 390.0, 395.0, 
	400.0, 405.0, 410.0, 415.0, 420.0, 425.0, 430.0, 435.0, 440.0, 445.0, 
	450.0, 455.0, 460.0, 465.0, 470.0, 475.0, 480.0, 485.0, 490.0, 495.0, 
	500.0, 505.0, 510.0, 515.0, 520.0, 525.0, 530.0, 535.0, 540.0, 545.0, 
	550.0, 555.0, 560.0, 565.0, 570.0, 575.0, 580.0, 585.0, 590.0, 595.0, 
	600.0, 605.0, 610.0, 615.0, 620.0, 625.0, 630.0, 635.0, 
};

// table 6
float rate1_table[] =
{
	0.05, 0.10, 0.15, 0.20, 0.25, 0.30, 0.35, 0.40, 0.45, 0.50, 0.55, 0.60, 0.65, 0.70, 0.75, 0.80,
	0.85, 0.90, 0.95, 1.00, 1.05, 1.10, 1.15, 1.20, 1.25, 1.30, 1.35, 1.40, 1.45, 1.50, 1.55, 1.60,
	1.65, 1.70, 1.75, 1.80, 1.85, 1.90, 1.95, 2.00, 2.05, 2.10, 2.15, 2.20, 2.25, 2.30, 2.35, 2.40,
	2.45, 2.50, 2.55, 2.60, 2.65, 2.70, 2.75, 2.80, 2.85, 2.90, 2.95, 3.00, 3.05, 3.10, 3.15, 3.20,
	3.25, 3.30, 3.35, 3.40, 3.45, 3.50, 3.55, 3.60, 3.65, 3.70, 3.75, 3.80, 3.85, 3.90, 3.95, 4.00,
	4.05, 4.10, 4.15, 4.20, 4.25, 4.30, 4.35, 4.40, 4.45, 4.50, 4.55, 4.60, 4.65, 4.70, 4.75, 4.80,
	4.85, 4.90, 4.95, 5.00, 5.10, 5.20, 5.30, 5.40, 5.50, 5.60, 5.70, 5.80, 5.90, 6.00, 6.10, 6.20,
	6.30, 6.40, 6.50, 6.60, 6.70, 6.80, 6.90, 7.00, 7.50, 8.00, 8.50, 9.00, 9.50, 10.00, 10.00, 10.00,
};

// table 7
float rate2_table[] =
{
	0.05, 0.10, 0.15, 0.20, 0.25, 0.30, 0.35, 0.40, 0.45, 0.50, 0.55, 0.60, 0.65, 0.70, 0.75, 0.80,
	0.85, 0.90, 0.95, 1.00, 1.05, 1.10, 1.15, 1.20, 1.25, 1.30, 1.35, 1.40, 1.45, 1.50, 1.55, 1.60,
	1.65, 1.70, 1.75, 1.80, 1.85, 1.90, 1.95, 2.00, 2.05, 2.10, 2.15, 2.20, 2.25, 2.30, 2.35, 2.40,
	2.45, 2.50, 2.55, 2.60, 2.65, 2.70, 2.75, 2.80, 2.85, 2.90, 2.95, 3.00, 3.05, 3.10, 3.15, 3.20,
	3.25, 3.30, 3.35, 3.40, 3.45, 3.50, 3.55, 3.60, 3.65, 3.70, 3.75, 3.80, 3.85, 3.90, 3.95, 4.00,
	4.05, 4.10, 4.15, 4.20, 4.25, 4.30, 4.35, 4.40, 4.45, 4.50, 4.55, 4.60, 4.65, 4.70, 4.75, 4.80,
	4.85, 4.90, 4.95, 5.00, 5.05, 5.10, 5.15, 5.20, 5.25, 5.30, 5.35, 5.40, 5.45, 5.50, 5.55, 5.60,
	5.65, 5.70, 5.75, 5.80, 5.85, 5.90, 5.95, 6.00, 6.05, 6.10, 6.15, 6.20, 6.25, 6.30, 6.35, 6.40,
};

// table 8
int16 HF_damp_freq_table_gs[] =
{
	315, 315, 315, 315, 315, 315, 315, 315,
	400, 400, 400, 400, 400, 400, 400, 400,
	500, 500, 500, 500, 500, 500, 500, 500,
	630, 630, 630, 630, 630, 630, 630, 630,
	800, 800, 800, 800, 800, 800, 800, 800,
	1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000,
	1250, 1250, 1250, 1250, 1250, 1250, 1250, 1250,
	1600, 1600, 1600, 1600, 1600, 1600, 1600, 1600,
	2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,
	2500, 2500, 2500, 2500, 2500, 2500, 2500, 2500,
	3150, 3150, 3150, 3150, 3150, 3150, 3150, 3150,
	4000, 4000, 4000, 4000, 4000, 4000, 4000, 4000,
	5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000,
	6300, 6300, 6300, 6300, 6300, 6300, 6300, 6300,
	8000, 8000, 8000, 8000, 8000, 8000, 8000, 8000,
	22000, 22000, 22000, 22000, 22000, 22000, 22000, 22000,
};

// table 9
int16 cutoff_freq_table_gs[] =
{
	250, 250, 250, 250, 250, 250, 250, 250,
	315, 315, 315, 315, 315, 315, 315, 315,
	400, 400, 400, 400, 400, 400, 400, 400,
	500, 500, 500, 500, 500, 500, 500, 500,
	630, 630, 630, 630, 630, 630, 630, 630,
	800, 800, 800, 800, 800, 800, 800, 800,
	1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000,
	1250, 1250, 1250, 1250, 1250, 1250, 1250, 1250,
	1600, 1600, 1600, 1600, 1600, 1600, 1600, 1600,
	2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,
	2500, 2500, 2500, 2500, 2500, 2500, 2500, 2500,
	3150, 3150, 3150, 3150, 3150, 3150, 3150, 3150,
	4000, 4000, 4000, 4000, 4000, 4000, 4000, 4000,
	5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000,
	6300, 6300, 6300, 6300, 6300, 6300, 6300, 6300,
	8000, 8000, 8000, 8000, 8000, 8000, 8000, 8000,
};

// table 10
int16 eq_freq_table_gs[] =
{
	200, 200, 200, 200, 200, 200, 200, 200,
	250, 250, 250, 250, 250, 250, 250, 250,
	315, 315, 315, 315, 315, 315, 315, 315,
	400, 400, 400, 400, 400, 400, 400, 400,
	500, 500, 500, 500, 500, 500, 500, 500,
	630, 630, 630, 630, 630, 630, 630, 630,
	800, 800, 800, 800, 800, 800, 800, 800,
	1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000,
	1250, 1250, 1250, 1250, 1250, 1250, 1250, 1250,
	1600, 1600, 1600, 1600, 1600, 1600, 1600, 1600,
	2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,
	2500, 2500, 2500, 2500, 2500, 2500, 2500, 2500,
	3150, 3150, 3150, 3150, 3150, 3150, 3150, 3150,
	4000, 4000, 4000, 4000, 4000, 4000, 4000, 4000,
	5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000,
	6300, 6300, 6300, 6300, 6300, 6300, 6300, 6300,
};

// table 11
int16 lpf_table_gs[] =
{
	250, 250, 250, 250, 250, 250, 250, 250,
	315, 315, 315, 315, 315, 315, 315, 315,
	400, 400, 400, 400, 400, 400, 400, 400,
	500, 500, 500, 500, 500, 500, 500, 500,
	630, 630, 630, 630, 630, 630, 630, 630,
	800, 800, 800, 800, 800, 800, 800, 800,
	1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000,
	1250, 1250, 1250, 1250, 1250, 1250, 1250, 1250,
	1600, 1600, 1600, 1600, 1600, 1600, 1600, 1600,
	2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000,
	2500, 2500, 2500, 2500, 2500, 2500, 2500, 2500,
	3150, 3150, 3150, 3150, 3150, 3150, 3150, 3150,
	4000, 4000, 4000, 4000, 4000, 4000, 4000, 4000,
	5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000,
	6300, 6300, 6300, 6300, 6300, 6300, 6300, 6300,
	-1, -1, -1, -1, -1, -1, -1, -1,
};

// table 12
int16 manual_table[] =
{
	100,110,120,130,140,150,160,170,180,190,200,210,220,230,240,250,260,270,280,290,
	300,320,340,360,380,400,420,440,460,480,500,520,540,560,580,600,620,640,660,680,
	700,720,740,760,780,800,820,840,860,880,900,920,940,960,980,1000,1100,1200,1300,1400,
	1500,1600,1700,1800,1900,2000,2100,2200,2300,2400,2500,2600,2700,2800,2900,3000,3100,3200,3300,3400,
	3500,3600,3700,3800,3900,4000,4100,4200,4300,4400,4500,4600,4700,4800,4900,5000,5100,5200,5300,5400,
	5500,5600,5700,5800,5900,6000,6100,6200,6300,6400,6500,6600,6700,6800,6900,7000,7100,7200,7300,7400,
	7500,7600,7700,7800,7900,8000,8000,8000,
};

// table 13
int16 azimuth_table[] =
{
	-180,-180,-180,-180,-180,-180,-168,-168,-168,-168,-156,-156,-156,-156,-144,-144,
	-144,-144,-132,-132,-132,-132,-120,-120,-120,-120,-108,-108,-108,-108, -96, -96,
	 -96, -96, -84, -84, -84, -84, -72, -72, -72, -72, -60, -60, -60, -60, -48, -48,
	 -48, -48, -36, -36, -36, -36, -24, -24, -24, -24, -12, -12, -12, -12,   0,   0,
	   0,   0,  12,  12,  12,  12,  24,  24,  24,  24,  36,  36,  36,  36,  48,  48,
	  48,  48,  60,  60,  60,  60,  72,  72,  72,  72,  84,  84,  84,  84,  96,  96,
	  96,  96, 108, 108, 108, 108, 120, 120, 120, 120, 132, 132, 132, 132, 144, 144,
	 144, 144, 156, 156, 156, 156, 168, 168, 168, 168, 180, 180, 180, 180, 180, 180,
};

// table 14
//float accel_table[] =
//{
//	0,
//};




/********** XG **********/
// table 1
float lfo_freq_table_xg[] = 
{
	0.00, 0.04, 0.08, 0.13, 0.17, 0.21, 0.25, 0.29, 0.34, 0.38, 0.42, 0.46, 0.51, 0.55, 0.59, 0.63,
	0.67, 0.72, 0.76, 0.80, 0.84, 0.88, 0.93, 0.97, 1.01, 1.05, 1.09, 1.14, 1.18, 1.22, 1.26, 1.30,
	1.35, 1.39, 1.43, 1.47, 1.51, 1.56, 1.60, 1.64, 1.68, 1.72, 1.77, 1.81, 1.85, 1.89, 1.94, 1.98,
	2.02, 2.06, 2.10, 2.15, 2.19, 2.23, 2.27, 2.31, 2.36, 2.40, 2.44, 2.48, 2.52, 2.57, 2.61, 2.65,
	2.69, 2.78, 2.86, 2.94, 3.03, 3.11, 3.20, 2.28, 3.37, 3.45, 3.53, 3.62, 3.70, 3.87, 4.04, 4.21,
	4.37, 4.54, 4.71, 4.88, 5.05, 5.22, 5.38, 5.55, 5.72, 6.06, 6.39, 6.73, 7.07, 7.40, 7.74, 8.08,
	8.41, 8.75, 9.08, 9.42, 9.76, 10.1, 10.8, 11.4, 12.1, 12.8, 13.5, 14.1, 14.8, 15.5, 16.2, 16.8,
	17.5, 18.2, 19.5, 20.9, 22.2, 23.6, 24.9, 26.2, 27.6, 28.9, 30.3, 31.6, 33.0, 34.3, 37.0, 39.7,
};

// table 2
float mod_delay_offset_table_xg[] =
{
	0.0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0, 1.1, 1.2, 1.3, 1.4, 1.5,
	1.6, 1.7, 1.8, 1.9, 2.0, 2.1, 2.2, 2.3, 2.4, 2.5, 2.6, 2.7, 2.8, 2.9, 3.0, 3.1, 
	3.2, 3.3, 3.4, 3.5, 3.6, 3.7, 3.8, 3.9, 4.0, 4.1, 4.2, 4.3, 4.4, 4.5, 4.6, 4.7,
	4.8, 4.9, 5.0, 5.1, 5.2, 5.3, 5.4, 5.5, 5.6, 5.7, 5.8, 5.9, 6.0, 6.1, 6.2, 6.3,
	6.4, 6.5, 6.6, 6.7, 6.8, 6.9, 7.0, 7.1, 7.2, 7.3, 7.4, 7.5, 7.6, 7.7, 7.8, 7.9,
	8.0, 8.1, 8.2, 8.3, 8.4, 8.5, 8.6, 8.7, 8.8, 8.9, 9.0, 9.1, 9.2, 9.3, 9.4, 9.5,
	9.6, 9.7, 9.8, 9.9, 10.0, 11.1, 12.2, 13.3, 14.4, 15.5, 17.1, 18.6, 20.2, 21.8,	23.3, 24.9,
	26.5, 28.0, 29.6, 31.2, 32.8, 34.3, 35.9, 37.5, 39.0, 40.6,	42.2, 43.7, 45.3, 46.9, 48.4, 50.0,
};

// table 3
float eq_freq_table_xg[] = 
{
	20, 22, 25, 28, 32, 36, 40, 45, 50, 56, 63, 70, 80, 90, 100, 110,
	125, 140, 160, 180, 200, 225, 250, 280, 315, 355, 400, 450, 500, 560, 630, 700, 
	800, 900, 1000, 1100, 1200, 1400, 1600, 1800, 2000, 2200, 2500, 2800, 3200, 3600, 4000, 4500,
	5000, 5600, 6300, 7000, 8000, 9000, 10000, 11000, 
	12000, 14000, 16000, 18000, 20000,
}; // param 60


// table 4
float reverb_time_table_xg[] =
{
	0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0, 1.1, 1.2, 1.3, 1.4, 1.5, 1.6, 1.7, 1.8,
	1.9, 2.0, 2.1, 2.2, 2.3, 2.4, 2.5, 2.6, 2.7, 2.8, 2.9, 3.0, 3.1, 3.2, 3.3, 3.4,
	3.5, 3.6, 3.7, 3.8, 3.9, 4.0, 4.1, 4.2, 4.3, 4.4, 4.5, 4.6, 4.7, 4.8, 4.9, 5.0,
	5.5, 6.0, 6.5, 7.0, 7.5, 8.0, 8.5, 9.0, 9.5, 10.0, 11.0, 12.0, 13.0, 14.0, 15.0, 16.0,
	17.0, 18.0, 19.0, 20.0, 25.0, 30.0,
};

// table 5
float delay_time_table_xg[] =
{
	0.1, 1.7, 3.2, 4.8, 6.4, 8.0, 9.5, 11.1, 12.7, 14.3, 15.8, 17.4, 19.0, 20.6, 22.1, 23.7,
	25.3, 26.9, 28.4, 30.0, 31.6, 33.2, 34.7, 36.3, 37.9, 39.5, 41.0, 42.6, 44.2, 45.7, 47.3, 48.9,
	50.5, 52.0, 53.6, 55.2, 56.8, 58.3, 59.9, 61.5, 63.1, 64.6, 66.2, 67.8, 69.4, 70.9, 72.5, 74.1,
	75.7, 77.2, 78.8, 80.4, 81.9, 83.5, 85.1, 86.7, 88.2, 89.8, 91.4, 93.0, 94.5, 96.1, 97.7, 99.3,
	100.8, 102.4, 104.0, 105.6, 107.1, 108.7, 110.3, 111.9, 113.4, 115.0, 116.6, 118.2, 119.7, 121.3, 122.9, 124.4,
	126.0, 127.6, 129.2, 130.7, 132.3, 133.9, 135.5, 137.0, 138.6, 140.2, 141.8, 143.3, 144.9, 146.5, 148.1, 149.6,
	151.2, 152.8, 154.4, 155.9, 157.5, 159.1, 160.6, 162.2, 163.8, 165.4, 166.9, 168.5, 170.1, 171.7, 173.2, 174.8,
	176.4, 178.0, 179.5, 181.1, 182.7, 184.3, 185.8, 187.4, 189.0, 190.6, 192.1, 193.7, 195.3, 196.9, 198.4, 200.0,
};

// table 6
float room_size_table_xg[] =
{
	// 0 ~ 44
	0.1, 0.3, 0,4, 0.6, 0.7, 0.9, 1.0, 1.2, 1.4, 1.5,
	1.7, 1.8, 2.0, 2.1, 2.3, 2.5, 2.6, 2.8, 2.9, 3.1,
	3.2, 3.4, 3.5, 3.7, 3.9, 4.0, 4.2, 4.3, 4.5, 4.6,
	4.8, 5.0, 5.1, 5.3, 5.4, 5.6, 5.7, 5.9, 6.1, 6.2,
	6.4, 6.5, 6.7, 6.8, 7.0,
};

// table 7
float delay_time_2_table_xg[] =
{
	  0.1,   3.2,   6.4,   9.5,  12.7,  15.8,  19.0,  22.1,  25.3,  28.4,
	 31.6,  34.7,  37.9,  41.0,  44.2,  47.3,  50.5,  53.6,  56.8,  59.9,
	 63.1,  66.2,  69.4,  72.5,  75.7,  78.8,  82.0,  85.1,  88.3,  91.4,
	 94.6,  97.7, 100.9, 104.0, 107.2, 110.3, 113.5, 116.6, 119.8, 122.9,
	126.1, 129.2, 132.4, 135.5, 138.6, 141.8, 144.9, 148.1, 151.2, 154.4,
	157.5, 160.7, 163.8, 167.0, 170.1, 173.3, 176.4, 179.6, 182.7, 185.9,
	189.0, 192.2, 195.3, 198.5, 201.6, 204.8, 207.9, 211.1, 214.2, 217.4,
	220.5, 223.7, 226.8, 300.0, 233.1, 236.3, 239.4, 242.6, 245.7, 248.9,
	252.0, 255.2, 258.3, 261.5, 254.6, 267.7, 270.9, 274.0, 277.2, 280.3,
	283.5, 286.6, 289.8, 292.9, 296.1, 299.2, 302.4, 305.5, 308.7, 311.8,
	315.0, 318.1, 321.3, 324.4, 327.6, 330.7, 333.9, 337.0, 340.2, 343.3,
	346.5, 349.6, 352.8, 355.9, 359.1, 362.2, 365.4, 368.5, 371.7, 374.8,
	378.0, 381.1, 384.3, 387.4, 390.6, 393.7, 396.9, 400.0, 
};

// table 8
float compressor_attack_time_table_xg[] =
{
	1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 
	12, 14, 16, 18, 20, 23, 26, 30, 35 ,40, 
};

// table 9
float compressor_release_time_table_xg[] =
{
	10, 15, 25, 35, 45, 55, 65, 75, 85, 100, 
	115, 140, 170, 230, 340, 680, 
};

// table 10
float compressor_ratio_table_xg[] =
{
	1.0, 1.5, 2.0, 3.0, 5.0, 7.0, 10.0, 20.0,
};

// table 11
float reverb_width_height_table_xg[] =
{
	// 0 ~ 104
	 0.5,  0.8,  1.0,  1.3,  1.5,  1.8,  2.0,  2.3,  2.6,  2.8,
	 3.1,  3.3,  3.6,  3.9,  4.1,  4.4,  4.6,  4.9,  5.2,  5.4,
	 5.7,  5.9,  6.2,  6.5,  6.7,  7.0,  7.2,  7.5,  7.8,  8.0,
	 8.3,  8.6,  8.8,  9.1,  9.4,  9.6,  9.9, 10.2, 10.4, 10.7,
	11.0, 11.2, 11.5, 11.8, 11.8, 12.1, 12.3, 12.6, 12.9, 13.1,
	13.4, 13.7, 14.0, 14.2, 14.5, 14.8, 15.1, 15.6, 15.9, 16.2,
	16.5, 16.8, 17.1, 17.3, 17.6, 17.9, 18.2, 18.5, 18.8, 19.1,
	19.4, 19.7, 20.0, 20.2, 20.5, 20.8, 21.1, 21.4, 21.7, 22.0,
	22.4, 22.7, 23.0, 23.3, 23.6, 23.9, 24.2, 24.5, 24.9, 25.2,
	25.5, 25.8, 26.1, 26.5, 26.8, 27.1, 27.5, 27.8, 28.1, 28.5,
	28.8, 29.2, 29.5, 29.9, 30.2,
};

// table 12
float wah_release_time_table_xg[] =
{
	// 52~67
	10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 
	10, 10, 10, 15, 25, 35, 45, 55, 65, 75, 
	85, 100, 115, 140, 170, 230, 340, 680, 680, 680, 
	680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 
	680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 
	680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 
	680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 
	680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 
	680, 680, 680, 680, 680, 680, 680, 680,
};

//table 13
float lofi_sampling_freq_table_xg[] =
{
	44100.0, 22100.0, 14700.0, 11000.0, 8800.0, 7400.0, 6300.0, 5500.0,
	4900.0, 4400.0, 4000.0, 3700.0, 3400.0, 3200.0, 2900.0, 2800.0,
	2600.0, 2500.0, 2300.0, 2200.0, 2100.0, 2000.0, 1920.0, 1840.0,
	1760.0, 1700.0, 1630.0, 1580.0, 1520.0, 1470.0, 1420.0, 1380.0,
	1340.0, 1300.0, 1260.0, 1230.0, 1190.0, 1160.0, 1130.0, 1110.0,
	1080.0, 1050.0, 1030.0, 1000.0, 980.0, 959.0, 938.0, 919.0,
	900.0, 882.0, 865.0, 848.0, 832.0, 817.0, 802.0, 788.0,
	774.0, 760.0, 747.0, 735.0, 723.0, 711.0, 700.0, 689.0,
	678.0, 668.0, 658.0, 649.0, 639.0, 630.0, 621.0, 613.0,
	604.0, 596.0, 588.0, 580.0, 573.0, 565.0, 558.0, 551.0,
	544.0, 538.0, 531.0, 525.0, 519.0, 513.0, 507.0, 501.0,
	496.0, 490.0, 485.0, 479.0, 474.0, 469.0, 464.0, 459.0,
	455.0, 450.0, 445.0, 441.0, 437.0, 432.0, 428.0, 424.0,
	420.0, 416.0, 412.0, 408.0, 405.0, 401.0, 397.0, 394.0,
	390.0, 387.0, 383.0, 380.0, 377.0, 374.0, 371.0, 368.0,
	364.0, 361.0, 359.0, 356.0, 353.0, 350.0, 347.0, 345.0,
};

// table 14
// 4th = 1.0
float tempo_table_xg[] = {
	DIV_8 * DIV_3,	// 64th/3
	DIV_16 * 1.5f,	// 64th.
	DIV_8,			// 32th
	DIV_4 * DIV_3,	// 32th/3
	DIV_8 * 1.5f,	// 32th.
	DIV_4,			// 16th
	DIV_2 * DIV_3,	// 16th/3
	DIV_4 * 1.5f,	// 16th.
	DIV_2,			// 8th
	1.0f * DIV_3,	// 8th/3
	DIV_2 * 1.5f,	// 8th.
	1.0f,			// 4th
	2.0f * DIV_3,	// 4th/3
	1.0f * 1.5f,	// 4th.
	2.0f,			// 2th
	4.0f * DIV_3,	// 2th/3
	2.0f * 1.5f,	// 2th.
	4.0f,			// 4thx4
	5.0f,			// 4thx5
	6.0f,			// 4thx6
	7.0f,			// 4thx7
	8.0f,			// 4thx8
	9.0f,			// 4thx9
	10.0f,			// 4thx10
	11.0f,			// 4thx11
	12.0f,			// 4thx12
	13.0f,			// 4thx13
	14.0f,			// 4thx14
	15.0f,			// 4thx15
	16.0f,			// 4thx16
	17.0f,			// 4thx17
	18.0f,			// 4thx18
	19.0f,			// 4thx19
	20.0f,			// 4thx20
	21.0f,			// 4thx21
	22.0f,			// 4thx22
	23.0f,			// 4thx23
	24.0f,			// 4thx24
	25.0f,			// 4thx25
	26.0f,			// 4thx26
	27.0f,			// 4thx27
	28.0f,			// 4thx28
	29.0f,			// 4thx29
	30.0f,			// 4thx30
	31.0f,			// 4thx31
	32.0f,			// 4thx32
	33.0f,			// 4thx33
	34.0f,			// 4thx34
	35.0f,			// 4thx35
	36.0f,			// 4thx36
	37.0f,			// 4thx37
	38.0f,			// 4thx38
	39.0f,			// 4thx39
	40.0f,			// 4thx40
	41.0f,			// 4thx41
	42.0f,			// 4thx42
	43.0f,			// 4thx43
	44.0f,			// 4thx44
	45.0f,			// 4thx45
	46.0f,			// 4thx46
	47.0f,			// 4thx47
	48.0f,			// 4thx48
	49.0f,			// 4thx49
	50.0f,			// 4thx50
	51.0f,			// 4thx51
	52.0f,			// 4thx52
	53.0f,			// 4thx53
	54.0f,			// 4thx54
	55.0f,			// 4thx55
	56.0f,			// 4thx56
	57.0f,			// 4thx57
	58.0f,			// 4thx58
	59.0f,			// 4thx59
	60.0f,			// 4thx60
	61.0f,			// 4thx61
	62.0f,			// 4thx62
	63.0f,			// 4thx63
	64.0f,			// 4thx64
	64.0f,			// 4thx64 //
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
	64.0f,			// 4thx64
};

// table 15
//float dyna_attack_time_table_xg[] =

// table 16
//float dyna_release_time_table_xg[] =

//table 17
float ring_mod_carrier_freq_course_xg[] = {
	0.7, 1.3, 2.0, 2.7, 3.4, 4.0, 4.7, 5.4, 6.1, 6.7, 
	7.4, 8.1, 8.7, 9.4, 10.1, 10.8, 11.4, 12.1, 12.8, 13.5, 
	14.1, 14.8, 15.5, 16.2, 16.8, 17.5, 18.2, 19.5, 20.9, 21.5, 
	22.9, 24.2, 25.6, 26.9, 28.9, 30.3, 32.3, 33.6, 35.7, 37.7, 
	39.7, 42.4, 44.4, 47.1, 49.8, 52.5, 55.9, 59.2, 62.6, 65.9, 
	70.0, 73.3, 78.1, 82.1, 86.8, 92.2, 96.9, 103.0, 108.3, 115.1, 
	121.1, 128.5, 135.9, 143.3, 151.4, 160.2, 169.6, 179.0, 189.1, 199.9, 
	211.3, 223.4, 236.2, 249.7, 263.8, 279.3, 294.7, 311.6, 329.7, 348.6, 
	368.1, 389.6, 411.8, 435.4, 459.6, 485.9, 514.1, 543.1, 574.0, 607.0, 
	642.0, 678.3, 717.3, 757.7, 801.5, 847.2, 895.0, 946.1, 1000.7, 1057.2, 
	1117.7, 1181.7, 1249.0, 1320.3, 1395.7, 1475.1, 1559.2, 1648.7, 1742.9, 1841.8, 
	1947.5, 2058.5, 2175.6, 2300.1, 2431.3, 2569.9, 2716.6, 2871.4, 3035.6, 4208.5, 
	3391.6, 3585.4, 3790.0, 4006.6, 4234.8, 4477.0, 4732.1, 5002.6, 
};

// table 18
//float v_flanger_delay_offset_table_xg[] =

uint8 multi_eq_block_table_xg[] =
{	/* Gain1, Freq1, Q1, Shape1, Gain2, Freq2, Q2, Not Used, Gain3, Freq3, Q3, Not Used,
	Gain4, Freq4, Q4, Not Used, Gain5, Freq5, Shape5 */
	64, 12, 7, 0, 64, 28, 7, 0, 64, 34, 7, 0, 64, 46, 7, 0, 64, 52, 7, 0,	/* Flat */
	58, 8, 7, 0, 66, 16, 3, 0, 68, 33, 3, 0, 60, 44, 5, 0, 58, 50, 7, 0,	/* Jazz */
	68, 16, 7, 0, 60, 24, 20, 0, 67, 34, 7, 0, 60, 40, 20, 0, 70, 48, 7, 0,	/* Pops */
	71, 16, 7, 0, 68, 20, 7, 0, 60, 36, 5, 0, 68, 41, 10, 0, 66, 50, 7, 0,	/* Rock */
	67, 12, 7, 0, 68, 24, 7, 0, 64, 34, 5, 0, 66, 50, 7, 0, 61, 52, 7, 0,	/* Concert */
};





/********** SD INS **********/

// 4th = 1.0
float note1_table_sd[10] = {
	DIV_4,			// 16th
	DIV_2 * DIV_3,	// 16th/3
	DIV_4 * 1.5f,	// 16th.
	DIV_2,			// 8th
	1.0f * DIV_3,	// 8th/3
	DIV_2 * 1.5f,	// 8th.
	1.0f,			// 4th
	2.0f * DIV_3,	// 4th/3
	1.0f * 1.5f,	// 4th.
	2.0f,			// 2th
};

float note2_table_sd[22] = {
	DIV_16 * DIV_3,	// 64th/3
	DIV_16,			// 64th
	DIV_8 * DIV_3,	// 32th/3
	DIV_8,			// 32th
	DIV_4 * DIV_3,	// 32th/3
	DIV_8 * 1.5f,	// 32th.
	DIV_4,			// 16th
	DIV_2 * DIV_3,	// 16th/3
	DIV_4 * 1.5f,	// 16th.
	DIV_2,			// 8th
	1.0f * DIV_3,	// 8th/3
	DIV_2 * 1.5f,	// 8th.
	1.0f,			// 4th
	2.0f * DIV_3,	// 4th/3
	1.0f * 1.5f,	// 4th.
	2.0f,			// 2th
	4.0f * DIV_3,	// 2th/3
	2.0f * 1.5f,	// 2th.
	4.0f,			// 1th
	8.0f * DIV_3,	// 1th/3
	4.0f * 1.5f,	// 1th.
	8.0f,			// 0.5th
};




///r
/********** delay_out type **********/

int8 delay_out_type_gs[18][128] = {
{// MSB 0x00
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x01
	1,1,3,1,0,0,0,0,0,0,0,0,0,0,0,0,2,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	3,1,2,3,3,2,2,0,0,0,0,0,0,0,0,0,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,5,5,4,0,0,0,0,0,0,0,0,
	2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x02
	3,3,4,3,3,4,3,3,4,4,4,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x03
	3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x04
	4,3,4,4,4,3,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x05
	4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x06
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x07
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x08
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x09
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x0A
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x0B
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x0C
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x0D
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x0E
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x0F
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x10
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},{// MSB 0x11
	4,4,3,2,2,3,2,3,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
},
};

int8 delay_out_type_xg[128][9] = {
{0,0,0,0,0,0,0,0,0,},{5,5,0,0,0,0,5,5,0,},{5,5,5,0,0,5,5,5,0,},{5,5,0,0,0,0,0,0,0,},{5,0,0,0,0,0,0,5,0,},
{4,0,0,0,0,0,0,0,0,},{4,0,0,0,0,0,0,0,0,},{4,0,0,0,0,0,0,0,0,},{4,0,0,0,0,0,0,0,0,},{4,4,0,0,0,0,0,0,0,},
{5,0,0,0,0,0,0,0,0,},{5,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
{0,0,0,0,0,0,0,0,0,},{5,0,0,0,0,0,0,0,0,},{5,0,0,0,0,0,0,0,0,},{5,0,0,0,0,0,0,0,0,},{5,0,0,0,0,0,0,0,0,},
// 20
{4,4,4,0,0,0,0,0,0,},{4,0,0,0,0,0,0,0,4,},{4,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
// 40
{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
// 60
{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
{3,3,3,3,3,3,3,3,3,},{3,3,3,0,0,0,0,0,3,},{3,3,0,0,0,0,0,3,3,},{3,0,0,0,0,0,0,0,0,},{2,2,2,2,0,0,0,0,0,},
{1,0,0,0,0,0,0,0,0,},{1,1,0,0,0,0,0,0,0,},{3,0,0,0,0,0,0,0,3,},{2,2,0,0,0,0,0,0,2,},{2,0,0,0,0,0,0,0,2,},
{2,2,0,0,0,0,0,0,2,},{1,0,0,0,0,0,0,0,0,},{1,0,0,0,0,0,0,0,0,},{1,2,2,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
// 80
{4,4,0,0,0,0,0,0,0,},{3,0,0,0,0,0,0,0,0,},{1,2,2,0,0,0,0,0,1,},{1,0,0,0,0,0,0,0,0,},{1,0,0,0,0,0,0,0,0,},
{1,0,0,0,0,0,0,0,0,},{2,2,2,2,0,0,0,0,0,},{3,0,0,0,0,0,0,0,0,},{4,0,0,0,0,0,0,0,0,},{3,0,0,0,0,0,0,0,0,},
{3,0,0,0,0,0,0,0,0,},{3,0,0,0,0,0,0,0,0,},{3,0,0,0,0,0,0,0,0,},{1,0,0,0,0,0,0,0,0,},{1,0,0,0,0,0,0,0,0,},
{4,4,0,0,0,0,0,0,0,},{4,4,0,0,0,0,0,0,0,},{4,4,0,0,0,0,0,0,0,},{2,4,2,4,0,0,0,0,0,},{2,2,0,0,0,0,0,0,0,},
// 100
{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{3,0,0,0,0,0,0,0,0,},
{2,0,0,0,0,0,0,0,0,},{2,0,0,0,0,0,0,0,0,},{3,0,0,0,0,0,0,0,0,},{3,0,0,0,0,0,0,0,0,},{2,0,0,0,0,0,0,0,0,},
{3,0,0,0,0,0,0,0,0,},{3,0,0,0,0,0,0,0,0,},{2,0,0,0,0,0,0,0,0,},{2,0,0,0,0,0,0,0,0,},{2,0,0,0,0,0,0,0,0,},
{2,0,0,0,0,0,0,0,0,},{2,0,0,0,0,0,0,0,0,},{3,0,0,0,0,0,0,0,0,},{3,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
// 120
{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},
{0,0,0,0,0,0,0,0,0,},{0,0,0,0,0,0,0,0,0,},{0,3,3,3,0,0,0,0,0,},
};

/*
// 0 : buffer mix / effect thru / no feedback (10ms
// 1 : filter (300ms
// 2 : multi filter + any (600ms
// 3 : pre_delay + feedback (chorus/flanger/phaser (1500ms
// 4 : delay // long delay + max feedbackƑȂ (6000ms
// 5 : reverb (12000ms
// 6 : system reverb (30000ms
*/

int8 delay_out_type_sd[] = {
	0,1,2,2,3,1,2,2,3,1,
	1,3,3,3,3,3,3,4,4,4,
	4,4,4,4,5,5,4,4,5,4,
	4,5,4,4,5,5,5,4,5,5,
	4,4,4,2,1,4,4,4,4,3,
	2,2,3,2,2,1,1,1,1,1,
	3,3,0,0,3,1,2,4,4,4,
	4,4,4,4,4,4,1,1,1,1,
	1,4,4,4,4,4,4,1,1,2,
	2,6,6,6,6,6,6,6,6,6,
};




