/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 */

#include "asm.h"
#include "apic.h"
#include "apic_regs.h"
#include "current.h"
#include "initfunc.h"
#include "localapic.h"
#include "mm.h"
#include "mmio.h"
#include "panic.h"
#include "vm.h"

/* #define APIC_DEBUG */
#include <core/printf.h>
#ifdef APIC_DEBUG
#define APIC_DBG(...)						\
	do {							\
		printf("APIC: " __VA_ARGS__);			\
	} while (0)
#else
#define APIC_DBG(...)
#endif

struct do_startup_data {
	struct vm *vcpu0;
	u32 sipi_vector, apic_id;
};

struct ipi_info {
	vector_t	vector;
	bool		self;
};

static void (*ap_start) (void);

static bool
do_startup (struct vcpu *p, void *q)
{
	struct do_startup_data *d;
	u32 sipi_vector;

	d = q;
	if (d->apic_id == 0xFF || p->apic_id == d->apic_id) {
		APIC_DBG("Send startup ipi to cpu 0x%x with vector 0x%x\n",
			 p->apic_id, d->sipi_vector);
		sipi_vector = p->localapic.sipi_vector;
		asm_lock_cmpxchgl (&p->localapic.sipi_vector, &sipi_vector,
				   d->sipi_vector);
		apic_send_sipi(d->sipi_vector, p->apic_id);
	}
	return false;
}

static void
apic_startup (u32 icr_low, u32 icr_high)
{
	struct do_startup_data d;

	switch (icr_low & APIC_ICR_LOW_DSH_MASK) {
	case APIC_ICR_LOW_DSH_DEST:
		if (icr_low & APIC_ICR_LOW_DM_LOGICAL_BIT) {
			APIC_DBG("Start-up IPI with a logical ID destination"
				 " is not yet supported");
			return;
		}
		d.apic_id = icr_high >> APIC_ICR_HIGH_DES_SHIFT;
		break;
	case APIC_ICR_LOW_DSH_SELF:
		APIC_DBG("Delivering start-up IPI to self");
		return;
	case APIC_ICR_LOW_DSH_ALL:
		APIC_DBG("Delivering start-up IPI to all including self");
		return;
	case APIC_ICR_LOW_DSH_OTHERS:
		d.apic_id = 0xFF; /* APIC ID 0xFF means a broadcast */
		break;
	}
	d.vcpu0 = current->vcpu0;
	d.sipi_vector = icr_low & APIC_ICR_LOW_VEC_MASK;
	vcpu_list_foreach (do_startup, &d);
}

u32
localapic_wait_for_sipi (void)
{
	u32 sipi_vector;

	sipi_vector = current->localapic.sipi_vector;
	asm_lock_cmpxchgl (&current->localapic.sipi_vector, &sipi_vector, ~0U);
	do {
		asm_pause ();
		asm_lock_cmpxchgl (&current->localapic.sipi_vector,
				   &sipi_vector, ~0U);
	} while (sipi_vector == ~0U);
	APIC_DBG("Cpu 0x%x receives startup ipi with vector 0x%x\n",
		 current->apic_id, sipi_vector);
	return sipi_vector;
}

void
localapic_change_base_msr (u64 msrdata)
{
	if (!(msrdata & MSR_IA32_APIC_BASE_MSR_APIC_GLOBAL_ENABLE_BIT))
		return;
	if ((msrdata & 0xFFFFFFFF00000000ULL) ||
	    (msrdata & MSR_IA32_APIC_BASE_MSR_APIC_BASE_MASK) != APIC_BASE)
		panic ("localapic_wait_for_sipi: Bad APIC base 0x%llX",
		       msrdata);
}

void
localapic_mmio_register (void)
{
	/*
	 * To make multiple OS run, VMM should always register mmio.
	 */
}

void
localapic_delayed_ap_start (void (*func) (void))
{
	ap_start = func;
}

static bool
call_send_init(struct vcpu *dest_vcpu, void *data)
{
	if (dest_vcpu == current) {
		return false;
	}

	apic_send_init(dest_vcpu->apic_id);
	return false;
}

void
apic_send_init_others(void)
{
	vcpu_list_foreach(call_send_init, NULL);
}

static bool
call_send_ipi (struct vcpu *dest_vcpu, void *data)
{
	struct ipi_info *ipi = (struct ipi_info *)data;

	if (!ipi->self && dest_vcpu == current) {
		return false;
	}

	apic_send_ipi(ipi->vector, dest_vcpu->apic_id);
	return false;
}

static int
apic_mmio_ro(void *data, phys_t gphys, bool wr, void *buf,
	       uint len, u32 flags)
{
	if (wr) {
		APIC_DBG("Ignore writing to off 0x%llx\n",
			 gphys - APIC_BASE);
	} else {
		mmio_memcpy(buf, apic_g2v(gphys), len);
	}
	return 1; /* emulated */
}

static int
apic_mmio_pass(void *data, phys_t gphys, bool wr, void *buf,
	       uint len, u32 flags)
{
	if (wr) {
		mmio_memcpy(apic_g2v(gphys), buf, len);
	} else {
		mmio_memcpy(buf, apic_g2v(gphys), len);
	}
	return 1; /* emulated */
}

static int
icr_low_handler(u32 *buf, bool wr)
{
	struct localapic_data *localapic = &current->localapic;
	u32 icr_low;
	u32 icr_high;
	u32 delivery_mode;
	struct vcpu *dest_vcpu;
	apic_id_t apic_id;
	u32 dsh;
	struct ipi_info ipi;

	if (!wr) {
		*buf = (localapic->icr_low & ~APIC_ICR_LOW_STATUS_BIT) |
			(apic_read32(APIC_ICR_LOW_OFFSET)
			 & APIC_ICR_LOW_STATUS_BIT);
		return 1 /* emulated */;
	}

	icr_high = localapic->icr_high;
	icr_low = localapic->icr_low = *buf;

	delivery_mode = icr_low & APIC_ICR_LOW_MT_MASK;
	dsh = icr_low & APIC_ICR_LOW_DSH_MASK;

	if (ap_start) {
		switch (delivery_mode) {
		case APIC_ICR_LOW_MT_INIT:
		case APIC_ICR_LOW_MT_STARTUP:
			ap_start ();
			ap_start = NULL;
			call_initfunc ("dbsp");
		}
	}

	switch (delivery_mode) {
	case APIC_ICR_LOW_MT_FIXED:
	case APIC_ICR_LOW_MT_LOWEST_PRI:
		switch (dsh) {
		case APIC_ICR_LOW_DSH_DEST:
			if ((icr_high & APIC_ICR_HIGH_DEST_MASK)
			    == APIC_ICR_HIGH_DEST_MASK) {
				APIC_DBG("Broadcast IPI: CPU%d ICR %08x %08x\n",
					 get_cpu_id(), icr_high, icr_low);
				ipi.vector = apic_get_vector (icr_low);
				ipi.self = true;
				vcpu_list_foreach(call_send_ipi, &ipi);
			}
			/* fall-through */
		case APIC_ICR_LOW_DSH_SELF:
			apic_wait_for_idle();
			apic_write32(APIC_ICR_HIGH_OFFSET, icr_high);
			apic_write32(APIC_ICR_LOW_OFFSET, icr_low);
			break;
		case APIC_ICR_LOW_DSH_ALL:
			ipi.vector = apic_get_vector (icr_low);
			ipi.self = true;
			vcpu_list_foreach(call_send_ipi, &ipi);
			break;
		case APIC_ICR_LOW_DSH_OTHERS:
			ipi.vector = apic_get_vector (icr_low);
			ipi.self = false;
			vcpu_list_foreach(call_send_ipi, &ipi);
			break;
		default:
			APIC_DBG("Specified DSH is not supported on fixed int: CPU%d ICR %08x %08x\n",
				 get_cpu_id(), icr_high, icr_low);
		}
		break;
	case APIC_ICR_LOW_MT_NMI:
		APIC_DBG("NMI is not supported: CPU%d ICR %08x %08x\n",
			 get_cpu_id(), icr_high, icr_low);
		break;
	case APIC_ICR_LOW_MT_INIT:
		if ((icr_low & APIC_ICR_LOW_LEVEL_ASSERT) == 0) {
			break;
		}
		switch (dsh) {
		case APIC_ICR_LOW_DSH_DEST:
			if (apic_is_logical_dest_mode(icr_low)) {
				APIC_DBG("Logical dest mode is not supported on INIT: CPU%d ICR %08x %08x\n",
					 get_cpu_id(), icr_high, icr_low);
				break;
			}
			apic_id = apic_get_apic_id(icr_high);
			dest_vcpu = find_vcpu_with_apic_id(current->vm,
							   apic_id);
			if (!dest_vcpu) {
				break;
			}
			apic_send_init(dest_vcpu->apic_id);
			break;
		case APIC_ICR_LOW_DSH_OTHERS:
			apic_send_init_others();
			break;
		default:
			APIC_DBG("Specified DSH is not supported on INIT: CPU%d ICR %08x %08x\n",
				 get_cpu_id(), icr_high, icr_low);
		}
		break;
	case APIC_ICR_LOW_MT_STARTUP:
		apic_startup(icr_low, icr_high);
		break;
	default:
		break;
	}
	return 1; /* emulated */;
}

static int
icr_high_handler(u32 *buf, bool wr)
{
	if (wr) {
		current->localapic.icr_high = *buf;
	} else {
		*buf = current->localapic.icr_high;
	}
	return 1; /* emul
ated */
}

static int
eoi_handler(u32 *buf, bool wr)
{
	if (!wr) {
		/* Read */
		return 0;
	}
	apic_write_eoi();
	return 1; /* emulated */
}

static int
logical_apic_id_handler(u32 *buf, bool wr)
{
	if (wr) {
		current->localapic.ldr = *buf;
		APIC_DBG("Writing ldr 0x%x\n", current->localapic.ldr);

	} else {
		*buf = current->localapic.ldr;
		APIC_DBG("Reading ldr 0x%x\n", current->localapic.ldr);
	}
	return 1; /* emulated */
}

static int
apic_privileged_mmio_handler(void *data, phys_t gphys, bool wr, void *buf,
			     uint len, u32 flags)
{
	int emulated = 0;

	if ((gphys & 0x3) != 0 || len != 4) {
		APIC_DBG("gphys 0x%llx len %d wr %d\n", gphys, len, wr);
		mmio_do_nothing(data, gphys, wr, buf, len, flags);
	}

	switch(gphys - APIC_BASE) {
	case APIC_APIC_ID_OFFSET:
		emulated = apic_mmio_ro(data, gphys, wr, buf, len, flags);
		break;
	case APIC_EOI_OFFSET:
		emulated = eoi_handler(buf, wr);
		break;
	case APIC_ICR_LOW_OFFSET:
		emulated = icr_low_handler(buf, wr);
		break;
	case APIC_ICR_HIGH_OFFSET:
		emulated = icr_high_handler(buf, wr);
		break;
	}
	if (!emulated) {
		apic_mmio_pass(data, gphys, wr, buf, len, flags);
	}
	return 1; /* emulated */
}

static int
apic_mmio_handler(void *data, phys_t gphys, bool wr, void *buf,
			     uint len, u32 flags)
{
	int emulated = 0;

	if ((gphys & 0x3) != 0 || len != 4) {
		mmio_do_nothing(data, gphys, wr, buf, len, flags);
	}

	switch(gphys - APIC_BASE) {
	case APIC_APIC_ID_OFFSET:
		emulated = apic_mmio_ro(data, gphys, wr, buf, len, flags);
		break;
	case APIC_LVT_LINT0_OFFSET:
	case APIC_LVT_LINT1_OFFSET:
		/* apic_maskall masks lint0 and lint1. So leave them. */
		emulated = apic_mmio_ro(data, gphys, wr, buf, len, flags);
		break;
	case APIC_EOI_OFFSET:
		emulated = eoi_handler(buf, wr);
		break;
	case APIC_LOGICAL_APIC_ID_OFFSET:
		emulated = logical_apic_id_handler(buf, wr);
		break;
	case APIC_ICR_LOW_OFFSET:
		emulated = icr_low_handler(buf, wr);
		break;
	case APIC_ICR_HIGH_OFFSET:
		emulated = icr_high_handler(buf, wr);
		break;
	}
	if (!emulated) {
		apic_mmio_pass(data, gphys, wr, buf, len, flags);
	}
	return 1; /* emulated */
}

static void
localapic_init (void)
{
	if (!current->vbsp)
		return;

	spinlock_init(&current->localapic.lock);

	if (vm_get_id() == 0) {
		mmio_register(APIC_BASE, APIC_LEN,
			      apic_privileged_mmio_handler, NULL);
	} else {
		mmio_register(APIC_BASE, APIC_LEN,
			      apic_mmio_handler, NULL);
	}
}

INITFUNC ("pass5", localapic_init);
