/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <common/calc.h>
#include <core/assert.h>
#include <core/cpu.h>
#include <core/exint.h>
#include <core/gmm.h>
#include <core/initfunc.h>
#include <core/mm.h>
#include <core/pci.h>
#include <core/process.h>
#include <core/printf.h>
#include <core/spinlock.h>
#include <core/string.h>
#include <core/vm_config.h>
#include "apic.h"
#include "acpi.h"
#include "callrealmode.h"
#include "cpu_emul.h"
#include "current.h"
#include "gbiosloader.h"
#include "int.h"
#include "loadbootsector.h"
#include "reload_os.h"
#include "panic.h"
#include "pcpu.h"
#include "sleep.h"
#include "vm.h"
#include "vmmcall_boot.h"
#include "uefi.h"

static size_t driver_data_size = 0;
static int new_vm_called = 0;

static LIST2_DEFINE_HEAD_INIT(vm_list, struct vm, vm_list);
static struct vm *vm0;
static u8 bios_boot_drive;

drvdata_hdl_t
vm_alloc_driver_data(size_t size)
{
	off_t offset;

	ASSERT(new_vm_called == 0);
	ASSERT(cpu_is_bsp());

	offset = driver_data_size;
	driver_data_size += ROUND_UP(size, sizeof(u64));

	return offset;
}

void *
vm_get_driver_data(drvdata_hdl_t handle)
{
	ASSERT(handle < driver_data_size);
	return (void *)((ulong)(current->vm->driver_data) + handle);
}

void *
vm_get_driver_data_of_vm(char *name, drvdata_hdl_t handle)
{
	struct vm *vm;
	ASSERT(handle < driver_data_size);

	vm = vm_find_by_name(name);
	if (vm == NULL) {
		return NULL;
	}
	return (void *)((ulong)(vm->driver_data) + handle);
}

unsigned int
vm_get_vcpu_count()
{
	struct vm *vm;
	unsigned int vcpu_count;

	vm = current->vm;
	spinlock_lock(&vm->vcpu_list_lock);
	vcpu_count = vm->vcpu_count;
	spinlock_unlock(&vm->vcpu_list_lock);
	return vcpu_count;
}

char *
vm_get_name()
{
	return current->vm->name;
}

u32
vm_get_id(void)
{
	return current->vm->id;
}

apic_id_t
vm_vbsp_apic_id(void)
{
	return current->vm->vbsp_apic_id;
}

struct vm *
vm_find(void)
{
	struct vm *vm;
	struct vm_config *vm_config;
	struct cpu_config *cpu_config;
	char *name = NULL;

	if (cpu_is_bsp()) {
		return vm0;
	}

	LIST2_FOREACH(vm_config_list.head, vm_config_list, vm_config) {
		LIST2_FOREACH(vm_config->cpu_list, cpu_list, cpu_config) {
			if (cpu_config->apic_id == get_apic_id()) {
				name = vm_config->name;
				goto found;
			}
		}
	}
	return vm0;
found:
	LIST2_FOREACH(vm_list, vm_list, vm) {
		if (strcmp(name, vm->name) == 0) {
			return vm;
		}
	}
	return NULL;
}

struct vm *
vm_find_by_name(char *name)
{
	struct vm *vm;

	LIST2_FOREACH(vm_list, vm_list, vm) {
		if (strcmp(name, vm->name) == 0) {
			return vm;
		}
	}
	return NULL;
}

bool
vm_current_is_vbsp(void)
{
	struct vm_config *vm_config;
	struct cpu_config *cpu_config;
	bool is_vbsp;

	if (cpu_is_bsp()) {
		/*
		 * physical CPU0 is always vBSP of VM0.
		 */
		return true;
	}

	LIST2_FOREACH(vm_config_list.head, vm_config_list, vm_config) {
		is_vbsp = true;
		LIST2_FOREACH(vm_config->cpu_list, cpu_list, cpu_config) {
			if (cpu_config->apic_id == get_apic_id()) {
				return is_vbsp;
			}
			is_vbsp = false;
		}
	}
	return false;
}

static struct vm *
vm_new(u32 id, struct vm_config *vm_config)
{
	struct vm *vm;
	struct cpu_config *cpu_config;

	ASSERT(cpu_is_bsp());

	new_vm_called = 1;

	vm = (struct vm *)alloc(sizeof(struct vm) + driver_data_size);
	if (vm == NULL) {
		return NULL;
	}
	memset(vm, 0, sizeof(struct vm) + driver_data_size);

	vm->id = id;
	snprintf(vm->name, VM_NAME_BUF_SIZE, "%s", vm_config->name);

	spinlock_init(&vm->vcpu_list_lock);
	LIST2_HEAD_INIT(vm->vcpu_list, vcpu_list);

	LIST2_ADD(vm_list, vm_list, vm);

	cpu_config = LIST2_NEXT(vm_config->cpu_list, cpu_list,
				(struct cpu_config *)NULL);
	if (cpu_config) {
		vm->vbsp_apic_id = cpu_config->apic_id;
	} else {
		vm->vbsp_apic_id = get_apic_id();
	}

	wait_init(&vm->wait);
	return vm;
}

void
vm_create_all(void)
{
	struct vm_config *vm_config;
	u32 id = 0;

	ASSERT(cpu_is_bsp());

	vm0 = vm_new(id++, &vm0_config);
	if (vm0 == NULL) {
		panic("Can't alloc vm0");
	}

	LIST2_FOREACH(vm_config_list.head, vm_config_list, vm_config) {
		if (strcmp(vm_config->name, VM0_NAME) == 0) {
			continue;
		}
		vm_new(id++, vm_config);
	}
}

void
vm_dump(void)
{
	static spinlock_t vm_dump_lock = SPINLOCK_INITIALIZER;
	struct vm *vm;
	struct vcpu *vcpu;
	int index = 0;
	phys_t gphys;
	phys_t len;
	u32 type;

	spinlock_lock(&vm_dump_lock);
	vm = current->vm;
	printf("%s:", vm->name);
	vcpu = NULL;
	while ((vcpu = vcpu_next(vm, vcpu)) != NULL) {
		printf(" 0x%x", vcpu->apic_id);
	}
	printf("\n");

	while (gmm_get_mem_map(index++, &gphys, &len, &type, NULL)) {
		printf("    %016llx-%016llx 0x%x\n",
		       gphys, gphys + len - 1, type);
	}
	pci_dump_assignment();
	spinlock_unlock(&vm_dump_lock);
	return;
}

void
vm_iommu_enabled(void)
{
	current->vm->iommu_enabled = true;
}

void
vm_wait_other_cpus(void)
{
	wait(&current->vm->wait, vm_get_vcpu_count());
}

bool
vm_resetting(void)
{
	return current->vm->resetting;
}

int
vm0_boot_method(u8 drive, u8 *boot_drive)
{
	static bool boot_once = false;
	int method;

	if (!boot_once) {
		bios_boot_drive = drive;
	}

	if (vm0_config.boot_int18) {
		if (boot_once) {
			printf("Boot from 0x%x drive.\n", 0x80);
			method = 1;
			*boot_drive = 0x80;
		} else {
			printf("Boot using int18.\n");
			method = 0;
		}
	} else {
		if (vm0_config.boot_drive != 0x00) {
			*boot_drive = vm0_config.boot_drive;
			printf("Boot from 0x%x drive.\n", *boot_drive);
			method = 1;
		} else {
			printf("Boot from 0x%x drive.\n",
			       bios_boot_drive);
			*boot_drive = bios_boot_drive;
			method = 1;
		}
		if (vm0_config.boot_partition) {
			printf("Use loader of active partition\n");
			*boot_drive = bios_boot_drive;
			method = 2;
		}
	}
	boot_once = true;
	return method;
}

static void
whole_machine_reboot(void)
{
	bool all_shutdown;
	struct vm *vm;
	int d;

	for (;;) {
		all_shutdown = true;
		LIST2_FOREACH(vm_list, vm_list, vm) {
			if (vm->id == 0) {
				continue;
			}
			if (vm->shutdown == false) {
				all_shutdown = false;
			}
		}
		if (all_shutdown) {
			break;
		}
		usleep(100);
	}

	d = msgopen ("reboot");
	if (d >= 0) {
		msgsendint (d, 0);
		msgclose (d);
	} else {
		printf ("reboot not found.\n");
	}
	panic("Reboot failed.\n");
}

vmmerr_t
vm_reset(void)
{
	struct vm *vm = current->vm;

	if (!current->vbsp) {
		return VMMERR_NOSUP;
	}

	printf("Reset %s\n", vm_get_name());

	if (uefi_booted && vm_get_id() == 0) {
		whole_machine_reboot();
	}

	vm->resetting = true;

	/*
	 * Initialize other cpus.
	 */
	apic_send_init_others();

	/*
	 * Reset vcpu and load guest BIOS.
	 */
	printf("Reset cpu 0x%x\n", get_cpu_id());
	vcpu_reset();
	apic_maskall();

	vm_wait_other_cpus();

	if (current->gmm.clear_mem) {
		current->gmm.clear_mem();
	}

	call_initfunc("reset");

	exint_reset_apic();

	vm->resetting = false;

	if (vm_get_id() == 0) {
		vmmcall_boot_enable(reload_os_thread, NULL);
	} else {
		if (gbios_load() != VMMERR_SUCCESS) {
			current->vmctl.init_signal ();
		}
	}

	return VMMERR_SUCCESS;
}

void
vm_shutdown(u8 state)
{
	struct vm *vm;
	bool all_shutdown;

	printf("Shutdown %s S%d\n", vm_get_name(), state);

	current->vm->shutdown = true;

	if (vm_get_id() != 0) {
		halt_cpu();
	}

	for (;;) {
		all_shutdown = true;
		LIST2_FOREACH(vm_list, vm_list, vm) {
			if (vm->shutdown == false) {
				all_shutdown = false;
			}
		}
		if (all_shutdown) {
			break;
		}
		usleep(100);
	}

	acpi_poweroff(state);
	halt_cpu();
}
