/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 */

/**
 * @file	drivers/pci.c
 * @brief	PCI driver (core)
 * @author	T. Shinagawa
 */

#include <common.h>
#include <core.h>
#include <core/mmio.h>
#include <core/pci.h>
#include "pci.h"
#include "pci_internal.h"
#include "pci_conceal.h"

/* #define PRINT_PCI_DEVICE */
/* #define PCI_DUMP_RESOURCES */

/* #define PCI_DEBUG */
#ifdef PCI_DEBUG
#define PCI_DBG(...)						\
	do {							\
		printf("PCI: " __VA_ARGS__);		\
	} while (0)
#else
#define PCI_DBG(...)
#endif

static spinlock_t pci_config_lock = SPINLOCK_INITIALIZER;
static spinlock_t pci_device_list_lock = SPINLOCK_INITIALIZER;

static struct resource	mmio_resource;
static struct resource	io_resource;

/********************************************************************************
 * PCI internal interfaces
 ********************************************************************************/

static LIST2_DEFINE_HEAD_INIT(pci_device_on_root_bus, struct pci_device,
			      sibling);
LIST_DEFINE_HEAD(pci_device_list);
LIST_DEFINE_HEAD(pci_driver_list);
struct pci_config_mmio_data *pci_config_mmio_data_head;

void pci_append_device(struct pci_device *dev)
{
	LIST_APPEND(pci_device_list, dev);
#ifdef PRINT_PCI_DEVICE
	pci_print_device(dev->address, &dev->config_space);
#endif
}

void
pci_init_root_resources(void)
{
	rm_init_resource(&mmio_resource, 0, 0xffffffffffffffffLL,
			 RESOURCE_TYPE_MMIO, 0, "mmio_resource");
	rm_init_resource(&io_resource, 0, 0x0000ffff,
			 RESOURCE_TYPE_IO, 0, "io_resource");

}

void
pci_dump_resources(void)
{
#ifdef PCI_DUMP_RESOURCES
	rm_dump_resources(&mmio_resource);
	rm_dump_resources(&io_resource);
#endif
}

static void
pci_insert_recouces(struct pci_device *dev)
{
	struct pci_device *parent;
	struct resource *resource, *presource;
	vmmerr_t err;
	int i, j;

	parent = dev->parent;

	for (i = 0; i < PCI_RESOURCE_NUM; i++) {
		resource = dev->resource + i;
		if (resource->type == RESOURCE_TYPE_INVALID) {
			continue;
		}

		if (parent == NULL) {
			switch (resource->type) {
			case RESOURCE_TYPE_MMIO:
				rm_insert_resource(&mmio_resource, resource);
				break;
			case RESOURCE_TYPE_IO:
				rm_insert_resource(&io_resource, resource);
				break;
			}
		} else {
			for (j = 0; j < PCI_RESOURCE_NUM; j++) {
				presource = parent->resource + j;
				if ((presource->data & PCI_RESOURCE_WINDOW)
				    == 0) {
					continue;
				}
				if (resource->type != presource->type) {
					continue;
				}
				if ((presource->data &
				     PCI_RESOURCE_PREFETCHABLE) &&
				    (resource->data &
				     PCI_RESOURCE_PREFETCHABLE) == 0) {
					continue;
				}
				err = rm_insert_resource(presource, resource);
				if (err == VMMERR_SUCCESS) {
					break;
				}
			}
		}
		if (resource->parent == NULL) {
			printf("Fail to insert resource. %s"
			       " %016llx-%016llx type %x data %x\n",
			       dev->name, resource->start,
			       resource->end, resource->type, resource->data);
		}
	}
}

vmmerr_t
pci_alloc_resource(struct pci_device *dev, struct resource *resource,
		  size_t size, phys_t align, u32 type, u32 data)
{
	struct pci_device *parent;
	struct resource *presource;
	vmmerr_t err;
	int j;

	parent = dev->parent;
	if (parent == NULL) {
		return VMMERR_NODEV;
	}

	for (j = 0; j < PCI_RESOURCE_NUM; j++) {
		presource = parent->resource + j;
		if ((presource->data & PCI_RESOURCE_WINDOW) == 0) {
			continue;
		}
		if (presource->type != type) {
			continue;
		}
		if ((presource->data & PCI_RESOURCE_PREFETCHABLE) &&
		    (data & PCI_RESOURCE_PREFETCHABLE) == 0) {
			continue;
		}
		err = rm_alloc_resource(presource, resource, size,
					align, type, data, dev->name);
		if (err == VMMERR_SUCCESS) {
			break;
		}
	}
	if (resource->parent == NULL) {
		return VMMERR_BUSY;
	}
	return VMMERR_SUCCESS;
}

void
pci_insert_device(struct pci_device *parent, struct pci_device *dev)
{
	if (parent) {
		dev->parent = parent;
		LIST2_ADD(parent->children, sibling, dev);
	} else {
		dev->parent = NULL;
		LIST2_ADD(pci_device_on_root_bus, sibling, dev);
	}
	pci_insert_recouces(dev);
}

struct pci_device *
pci_next_device_on_root_bus(struct pci_device *cur)
{
	return LIST2_NEXT(pci_device_on_root_bus, sibling, cur);
}

#define BIT_SET(flag, index)	(flag |=  (1 << index))
#define BIT_CLEAR(flag, index)	(flag &= ~(1 << index))
#define BIT_TEST(flag, index)	(flag &   (1 << index))

static int pci_config_emulate_base_address_mask(struct pci_device *dev, core_io_t io, union mem *data)
{
	struct pci_vm_data *pci_data = vm_get_driver_data(pci_handle);
	int index = pci_data->config_addr.reg_no
		- PCI_CONFIG_ADDRESS_GET_REG_NO(base_address[0]);

	if (! ((0 <= index && index <= 5) || index == 8) )
		return CORE_IO_RET_DEFAULT;

	if (index == 8) // expansion ROM base address
		index -= 2;

	if (io.dir == CORE_IO_DIR_OUT) {
		if (io.size == 4 && (data->dword & 0xFFFFFFF0) == 0xFFFFFFF0) {
			BIT_SET(dev->in_base_address_mask_emulation, index);
			return CORE_IO_RET_DONE;
		} else {
			BIT_CLEAR(dev->in_base_address_mask_emulation, index);
		}
	} else {
		if (io.size == 4 && BIT_TEST(dev->in_base_address_mask_emulation, index)) {
			data->dword = dev->base_address_mask[index];
			return CORE_IO_RET_DONE;
		}
	}
	return CORE_IO_RET_DEFAULT;
}

static bool
pci_config_mmio_emulate_base_address_mask (struct pci_device *dev,
					   unsigned int reg_offset, bool wr,
					   union mem *data, uint len)
{
	int index = (reg_offset - 0x10) >> 2;

	if (reg_offset < 0x10)
		return false;
	if (! ((0 <= index && index <= 5) || index == 8) )
		return false;
	if (index == 8)		/* expansion ROM base address */
		index -= 2;
	if (wr) {
		if (len == 4 && (data->dword & 0xFFFFFFF0) == 0xFFFFFFF0) {
			BIT_SET (dev->in_base_address_mask_emulation, index);
			return true;
		} else {
			BIT_CLEAR (dev->in_base_address_mask_emulation, index);
		}
	} else {
		if (len == 4 &&
		    BIT_TEST (dev->in_base_address_mask_emulation, index)) {
			data->dword = dev->base_address_mask[index];
			return true;
		}
	}
	return false;
}

static void
core_io_to_size_wr(core_io_t io, pci_off_t *size, bool *wr)
{
	switch (io.type) {
	case CORE_IO_TYPE_IN8:
		*wr = false;
		*size = 1;
		break;
	case CORE_IO_TYPE_IN16:
		*wr = false;
		*size = 2;
		break;
	case CORE_IO_TYPE_IN32:
		*wr = false;
		*size = 4;
		break;
	case CORE_IO_TYPE_OUT8:
		*wr = true;
		*size = 1;
		break;
	case CORE_IO_TYPE_OUT16:
		*wr = true;
		*size = 2;
		break;
	case CORE_IO_TYPE_OUT32:
		*wr = true;
		*size = 4;
		break;
	default:
		panic("iotype_to_size_wr: Invalid type. type %d", io.type);
	}
}

int pci_config_data_handler(core_io_t io, union mem *data, void *arg)
{
	int ioret = CORE_IO_RET_DEFAULT;
	struct pci_device *dev = NULL;
	pci_config_address_t caddr, caddr0;
	u8 offset;
	int (*func) (struct pci_device *dev, u8 iosize, u16 offset,
		     union mem *data);
	struct pci_vm_data *pci_data = vm_get_driver_data(pci_handle);
	int ret;
	pci_off_t size;
	bool wr;

	if (pci_data->config_addr.allow == 0)
		return CORE_IO_RET_NEXT;	// not configration access

	func = NULL;
	offset = pci_data->config_addr.reg_no * sizeof(u32) + (io.port - PCI_CONFIG_DATA_PORT);
	caddr = pci_data->config_addr; caddr.reserved = caddr.reg_no = caddr.type = 0;
	caddr0 = caddr, caddr0.func_no = 0;

	ret = pci_check_assignment(caddr.bus_no, caddr.devfn);

	PCI_DBG(PCI_LOCATION_FORMAT " 0x%04x assignment %d\n",
		caddr.bus_no, caddr.device_no, caddr.func_no,
		offset, ret);

	switch (ret) {
	case PCI_NOT_ASSIGNED:
		goto conceal;
	case PCI_ASSIGNED:
		if (vm_get_id() != 0) {
			spinlock_lock(&pci_config_lock);
			out32(PCI_CONFIG_ADDR_PORT,
			      pci_data->config_addr.value);
			core_io_handle_default(io, data);
			spinlock_unlock(&pci_config_lock);
			return CORE_IO_RET_DONE;
		}
		break;
	case PCI_DUMMY_FUNC0:
		core_io_to_size_wr(io, &size, &wr);
		pci_dummy_func0(offset, size, wr, data);
		return CORE_IO_RET_DONE;
	default:
		panic("pci_config_data_handler: Unknown assignment %d", ret);
	}

	spinlock_lock(&pci_device_list_lock);
	LIST_FOREACH (pci_device_list, dev) {
		if (dev->address.value == caddr.value) {
			spinlock_unlock(&pci_device_list_lock);
			goto found;
		}
		if (caddr.func_no != 0 && dev->address.value == caddr0.value &&
		    dev->config_space.multi_function == 0) {
			/* The guest OS is trying to access a PCI
			   configuration header of a single-function
			   device with function number 1 to 7. The
			   access will be concealed. */
			spinlock_unlock(&pci_device_list_lock);
			goto conceal;
		}
	}
	dev = pci_possible_new_device (caddr);
	spinlock_unlock(&pci_device_list_lock);
	if (dev && dev->conceal)
		goto found;
	if (dev) {
		u32 id = dev->config_space.regs32[0];
		u32 class = dev->config_space.class_code;
		struct pci_driver *driver;

		printf ("[%02X:%02X.%X] New PCI device found.\n",
			caddr.bus_no, caddr.device_no, caddr.func_no);
		LIST_FOREACH (pci_driver_list, driver) {
			if (idmask_match (id, driver->id) &&
			    idmask_match (class, driver->class)) {
				dev->driver = driver;
				driver->new (dev);
				goto found;
			}
		}
	}
	goto ret;
found:
	if (dev->conceal) {
	conceal:
		ioret = pci_conceal_config_data_handler (io, data, arg);
		goto ret;
	}
	if (dev->driver == NULL)
		goto ret;
	if (dev->driver->options.use_base_address_mask_emulation) {
		ioret = pci_config_emulate_base_address_mask(dev, io, data);
		if (ioret == CORE_IO_RET_DONE)
			goto ret;
	}

	func = io.dir == CORE_IO_DIR_IN ? dev->driver->config_read : dev->driver->config_write;
ret:
	if (func)
		ioret = func (dev, io.size, offset, data);
	if (ioret == CORE_IO_RET_DEFAULT) {
		spinlock_lock(&pci_config_lock);
		out32(PCI_CONFIG_ADDR_PORT, pci_data->config_addr.value);
		core_io_handle_default(io, data);
		spinlock_unlock(&pci_config_lock);
		ioret = CORE_IO_RET_DONE;
	}
	return ioret;
}

int pci_config_addr_handler(core_io_t io, union mem *data, void *arg)
{
	struct pci_vm_data *pci_data = vm_get_driver_data(pci_handle);

	switch (io.type) {
	case CORE_IO_TYPE_OUT32:
		pci_data->config_addr.value = data->dword;
		return CORE_IO_RET_DONE;
	case CORE_IO_TYPE_IN32:
		data->dword = pci_data->config_addr.value;
		return CORE_IO_RET_DONE;
	}
	return CORE_IO_RET_NEXT;
}

/********************************************************************************
 * PCI service functions exported to PCI device drivers
 ********************************************************************************/
/* ------------------------------------------------------------------------------
   PCI driver registration
 ------------------------------------------------------------------------------ */

/**
 * @brief		PCI driver registration function
 * @param  driver	pointer to struct pci_driver
 */
void pci_register_driver(struct pci_driver *driver)
{
	struct pci_device *dev;

	LIST_APPEND(pci_driver_list, driver);
	LIST_FOREACH (pci_device_list, dev) {
		u32 id = dev->config_space.regs32[0];
		u32 class = dev->config_space.class_code;

		if (dev->conceal)
			continue;
		if (idmask_match(id, driver->id) && idmask_match(class, driver->class)) {
			dev->driver = driver;
			driver->new(dev);
		}
	}
	if (driver->longname)
		printf ("%s registered\n", driver->longname);
	return;
}

/* ------------------------------------------------------------------------------
   PCI configuration registers access
 ------------------------------------------------------------------------------ */

#define DEFINE_pci_read_config_data_without_lock(size)		\
static inline u##size pci_read_config_data##size##_without_lock(pci_config_address_t addr, int offset) \
{								\
	u##size data;						\
	out32(PCI_CONFIG_ADDR_PORT, addr.value);		\
	in##size(PCI_CONFIG_DATA_PORT + offset, &data);		\
	return data;						\
}

#define DEFINE_pci_write_config_data_without_lock(size)		\
static inline void pci_write_config_data##size##_without_lock(pci_config_address_t addr, int offset, u##size data) \
{								\
	out32(PCI_CONFIG_ADDR_PORT, addr.value);		\
	out##size(PCI_CONFIG_DATA_PORT + offset, data);		\
}

DEFINE_pci_read_config_data_without_lock(8)
DEFINE_pci_read_config_data_without_lock(16)
DEFINE_pci_read_config_data_without_lock(32)
DEFINE_pci_write_config_data_without_lock(8)
DEFINE_pci_write_config_data_without_lock(16)
DEFINE_pci_write_config_data_without_lock(32)

#define DEFINE_pci_read_config_data(size)				\
	u##size pci_read_config_data##size(pci_config_address_t addr, int offset) \
	{								\
		u##size data;						\
		spinlock_lock(&pci_config_lock);			\
		data = pci_read_config_data##size##_without_lock(addr, offset);	\
		spinlock_unlock(&pci_config_lock);			\
		return data;						\
	}
#define DEFINE_pci_write_config_data(size)				\
	void pci_write_config_data##size(pci_config_address_t addr, int offset, u##size data) \
	{								\
		spinlock_lock(&pci_config_lock);			\
		pci_write_config_data##size##_without_lock(addr, offset, data);	\
		spinlock_unlock(&pci_config_lock);			\
	}
DEFINE_pci_read_config_data(8)
DEFINE_pci_read_config_data(16)
DEFINE_pci_read_config_data(32)
DEFINE_pci_write_config_data(8)
DEFINE_pci_write_config_data(16)
DEFINE_pci_write_config_data(32)

void
pci_readwrite_config_mmio (struct pci_config_mmio_data *p, bool wr,
			   uint bus_no, uint device_no, uint func_no,
			   uint offset, uint iosize, void *data)
{
	u8 *q = p->map;
	u64 phys = p->base;

	phys += (bus_no << 20) + (device_no << 15) + (func_no << 12) + offset;
	if (phys < p->phys || phys + iosize > p->phys + p->len)
		panic ("pci_readwrite_config_mmio: error"
		       " base=0x%llX seg_group=0x%X"
		       " bus_start=0x%X bus_end=0x%X"
		       " phys=0x%llX len=0x%X"
		       " bus_no=0x%X device_no=0x%X func_no=0x%X offset=0x%X",
		       p->base, p->seg_group, p->bus_start, p->bus_end,
		       p->phys, p->len, bus_no, device_no, func_no, offset);
	q += phys - p->phys;
	if (wr)
		mmio_memcpy (q, data, iosize);
	else
		mmio_memcpy (data, q, iosize);
}

static void
pci_readwrite_config_mmio_without_device(bool wr,
				 uint bus_no, uint device_no, uint func_no,
				 uint offset, uint iosize, void *data)
{
	struct pci_config_mmio_data *mmio_data;

	mmio_data = pci_search_config_mmio(0, bus_no);
	if (mmio_data == NULL) {
		if (!wr) {
			memset(data, 0xff, iosize);
		}
		return;
	}
	pci_readwrite_config_mmio(mmio_data, wr, bus_no, device_no, func_no,
				  offset, iosize, data);
}

void
pci_read_config_mmio (struct pci_config_mmio_data *p, uint bus_no,
		      uint device_no, uint func_no, uint offset, uint iosize,
		      void *data)
{
	pci_readwrite_config_mmio (p, false, bus_no, device_no, func_no,
				   offset, iosize, data);
}

void
pci_write_config_mmio (struct pci_config_mmio_data *p, uint bus_no,
		       uint device_no, uint func_no, uint offset, uint iosize,
		       void *data)
{
	pci_readwrite_config_mmio (p, true, bus_no, device_no, func_no,
				   offset, iosize, data);
}

u8
pci_read_config8(struct pci_device *pci_device, pci_off_t offset)
{
	pci_config_address_t addr;
	u8 val;
	if (pci_device->config_mmio) {
		pci_read_config_mmio (pci_device->config_mmio,
				      pci_device->address.bus_no,
				      pci_device->address.device_no,
				      pci_device->address.func_no,
				      offset, 1, &val);
	} else {
		addr = pci_device->address;
		addr.reg_no = offset >> 2;
		val = pci_read_config_data8(addr, offset & 0x3);
	}
	return val;
}

u16
pci_read_config16(struct pci_device *pci_device, pci_off_t offset)
{
	pci_config_address_t addr;
	u16 val;
	if (pci_device->config_mmio) {
		pci_read_config_mmio (pci_device->config_mmio,
				      pci_device->address.bus_no,
				      pci_device->address.device_no,
				      pci_device->address.func_no,
				      offset, 2, &val);
	} else {
		addr = pci_device->address;
		addr.reg_no = offset >> 2;
		val = pci_read_config_data16(addr, offset & 0x3);
	}
	return val;
}

u32
pci_read_config32(struct pci_device *pci_device, pci_off_t offset)
{
	pci_config_address_t addr;
	u32 val;
	if (pci_device->config_mmio) {
		pci_read_config_mmio (pci_device->config_mmio,
				      pci_device->address.bus_no,
				      pci_device->address.device_no,
				      pci_device->address.func_no,
				      offset, 4, &val);
	} else {
		addr = pci_device->address;
		addr.reg_no = offset >> 2;
		val = pci_read_config_data32(addr, offset & 0x3);
	}
	return val;
}

u64
pci_read_config64(struct pci_device *pci_device, pci_off_t offset)
{
	u64 val;
	if (pci_device->config_mmio) {
		pci_read_config_mmio (pci_device->config_mmio,
				      pci_device->address.bus_no,
				      pci_device->address.device_no,
				      pci_device->address.func_no,
				      offset, 8, &val);
	} else {
		val = pci_read_config32(pci_device, offset);
		val |= ((u64)pci_read_config32(pci_device, offset + 4) << 32);
	}
	return val;
}

void
pci_write_config8(struct pci_device *pci_device, pci_off_t offset, u8 val)
{
	pci_config_address_t addr;
	if (pci_device->config_mmio) {
		pci_write_config_mmio (pci_device->config_mmio,
				       pci_device->address.bus_no,
				       pci_device->address.device_no,
				       pci_device->address.func_no,
				       offset, 1, &val);
	} else {
		addr = pci_device->address;
		addr.reg_no = offset >> 2;
		pci_write_config_data8(addr, offset & 0x3, val);
	}
}

void
pci_write_config16(struct pci_device *pci_device, pci_off_t offset, u16 val)
{
	pci_config_address_t addr;
	if (pci_device->config_mmio) {
		pci_write_config_mmio (pci_device->config_mmio,
				       pci_device->address.bus_no,
				       pci_device->address.device_no,
				       pci_device->address.func_no,
				       offset, 2, &val);
	} else {
		addr = pci_device->address;
		addr.reg_no = offset >> 2;
		pci_write_config_data16(addr, offset & 0x3, val);
	}
}

void
pci_write_config32(struct pci_device *pci_device, pci_off_t offset, u32 val)
{
	pci_config_address_t addr;
	if (pci_device->config_mmio) {
		pci_write_config_mmio (pci_device->config_mmio,
				       pci_device->address.bus_no,
				       pci_device->address.device_no,
				       pci_device->address.func_no,
				       offset, 4, &val);
	} else {
		addr = pci_device->address;
		addr.reg_no = offset >> 2;
		pci_write_config_data32(addr, offset & 0x3, val);
	}
}

void
pci_write_config64(struct pci_device *pci_device, pci_off_t offset, u64 val)
{
	if (pci_device->config_mmio) {
		pci_write_config_mmio (pci_device->config_mmio,
				       pci_device->address.bus_no,
				       pci_device->address.device_no,
				       pci_device->address.func_no,
				       offset, 8, &val);
	} else {
		pci_write_config32(pci_device, offset, val);
		pci_write_config32(pci_device, offset + 4, val >> 32);
	}
}

/**
 * @brief		
 */
void
pci_handle_default_config_write (struct pci_device *pci_device, u8 iosize,
				 u16 offset, union mem *data)
{
	u32 reg;
	pci_config_address_t addr;
	core_io_t io;

	if (pci_device->config_mmio) {
		pci_write_config_mmio (pci_device->config_mmio,
				       pci_device->address.bus_no,
				       pci_device->address.device_no,
				       pci_device->address.func_no,
				       offset, iosize, data);
		if (offset >= 256)
			return;
		pci_read_config_mmio (pci_device->config_mmio,
				      pci_device->address.bus_no,
				      pci_device->address.device_no,
				      pci_device->address.func_no,
				      offset & ~3, 4, &reg);
		pci_device->config_space.regs32[offset >> 2] = reg;
		return;
	}
	if (offset >= 256)
		panic ("pci_handle_default_config_write: offset %u >= 256",
		       offset);
	addr = pci_device->address;
	addr.reg_no = offset >> 2;
	io.port = PCI_CONFIG_DATA_PORT + (offset & 3);
	io.dir = CORE_IO_DIR_OUT;
	io.size = iosize;
	spinlock_lock(&pci_config_lock);
	out32(PCI_CONFIG_ADDR_PORT, addr.value);
	core_io_handle_default(io, data);
	in32(PCI_CONFIG_DATA_PORT, &reg);
	spinlock_unlock(&pci_config_lock);
	pci_device->config_space.regs32[offset >> 2] = reg;
}

void
pci_handle_default_config_read (struct pci_device *pci_device, u8 iosize,
				u16 offset, union mem *data)
{
	pci_config_address_t addr;
	core_io_t io;

	if (pci_device->config_mmio) {
		pci_read_config_mmio (pci_device->config_mmio,
				      pci_device->address.bus_no,
				      pci_device->address.device_no,
				      pci_device->address.func_no,
				      offset, iosize, data);
		return;
	}
	if (offset >= 256)
		panic ("pci_handle_default_config_read: offset %u >= 256",
		       offset);
	addr = pci_device->address;
	addr.reg_no = offset >> 2;
	io.port = PCI_CONFIG_DATA_PORT + (offset & 3);
	io.dir = CORE_IO_DIR_IN;
	io.size = iosize;
	spinlock_lock(&pci_config_lock);
	out32(PCI_CONFIG_ADDR_PORT, addr.value);
	core_io_handle_default (io, data);
	spinlock_unlock(&pci_config_lock);
}

int
pci_config_mmio_handler (void *data, phys_t gphys, bool wr, void *buf,
			 uint len, u32 flags)
{
	struct pci_config_mmio_data *d = data;
	union {
		struct {
			unsigned int reg_offset : 12;
			unsigned int func_no : 3;
			unsigned int dev_no : 5;
			unsigned int bus_no : 8;
		} s;
		struct {
			unsigned int reg_offset : 12;
			unsigned int devfn  : 8;
			unsigned int bus_no : 8;
		} t;
		phys_t offset;
	} addr;
	enum core_io_ret ioret;
	struct pci_device *dev;
	int (*func) (struct pci_device *dev, u8 iosize, u16 offset,
		     union mem *data);
	int ret;

	addr.offset = gphys - d->base;

	ret = pci_check_assignment(addr.t.bus_no, addr.t.devfn);

	PCI_DBG(PCI_LOCATION_FORMAT " 0x%04x mm wr %d assignment %d\n",
		addr.s.bus_no, addr.s.dev_no, addr.s.func_no,
		addr.s.reg_offset, wr, ret);

	switch (ret) {
	case PCI_NOT_ASSIGNED:
		return mmio_do_nothing(data, gphys, wr, buf, len, flags);
	case PCI_ASSIGNED:
		if (vm_get_id() != 0) {
			pci_readwrite_config_mmio_without_device(wr,
				addr.s.bus_no, addr.s.dev_no, addr.s.func_no,
				addr.s.reg_offset, len, buf);
			return 1;
		}
		break;
	case PCI_DUMMY_FUNC0:
		return pci_dummy_func0(addr.s.reg_offset, len, wr, buf);
	default:
		panic("pci_config_mmio_handler: Unknown assignment %d", ret);
	}

	spinlock_lock(&pci_device_list_lock);
	LIST_FOREACH (pci_device_list, dev) {
		if (dev->address.bus_no == addr.s.bus_no &&
		    dev->address.device_no == addr.s.dev_no &&
		    dev->address.func_no == addr.s.func_no) {
			spinlock_unlock(&pci_device_list_lock);
			goto found;
		}
		if (addr.s.func_no != 0 &&
		    dev->address.bus_no == addr.s.bus_no &&
		    dev->address.device_no == addr.s.dev_no &&
		    dev->address.func_no == 0 &&
		    dev->config_space.multi_function == 0) {
			/* The guest OS is trying to access a PCI
			   configuration header of a single-function
			   device with function number 1 to 7. The
			   access will be concealed. */
			spinlock_unlock(&pci_device_list_lock);
			goto conceal;
		}
	}
	spinlock_unlock(&pci_device_list_lock);
	if (!wr)
		memset (buf, 0xFF, len);
	return 1;
found:
	if (dev->conceal) {
	conceal:
		if (!wr)
			memset (buf, 0xFF, len);
		return 1;
	}
	if (dev->driver == NULL) {
		pci_readwrite_config_mmio_without_device(wr,
			addr.s.bus_no, addr.s.dev_no, addr.s.func_no,
			addr.s.reg_offset, len, buf);
		return 1;
	}
	if (dev->driver->options.use_base_address_mask_emulation) {
		if (pci_config_mmio_emulate_base_address_mask (dev, addr.s.
							       reg_offset, wr,
							       buf, len))
		    return 1;
	}
	func = wr ? dev->driver->config_write : dev->driver->config_read;
	if (func) {
		ioret = func (dev, len, addr.s.reg_offset, buf);
		return ioret == CORE_IO_RET_DONE;
	}
	pci_readwrite_config_mmio_without_device(wr,
		addr.s.bus_no, addr.s.dev_no, addr.s.func_no,
		addr.s.reg_offset, len, buf);
	return 1;
}
