/*******************************************************************************
  TPI - flexible but useless plug-in framework.
  Copyright (C) 2002-2009 Silky

  This library is free software; you can redistribute it and/or modify it under
  the terms of the GNU Lesser General Public License as published by the Free
  Software Foundation; either version 2.1 of the License, or (at your option)
  any later version.

  This library is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
  for more details.

  You should have received a copy of the GNU Lesser General Public License along
  with this library; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

  $Id$
*******************************************************************************/

#include "lychee.h"
#include "dlg_process.h"
#include "dlg_make.h"
#include "frm_main.h"
#include "functions.h"

ProcessDialog * g_procDlg = nullptr;

//******************************************************************************
// ProcessDialog
//******************************************************************************

ProcessDialog::ProcessDialog(wxWindow * parent, wxULongLong_t n, wxString szPassword): wxDialog(), nFileCount(n), szPassword(szPassword), fOpen(false)
{
	::wxXmlResource::Get()->Load(L_DIR_S_XRC wxT("dlg_process.xrc"));
	::wxXmlResource::Get()->LoadDialog(this, this->GetParent(), wxT("dlg_process"));
	g_procDlg = this;
	this->hParent = parent;
	this->InitDialog();
}

ProcessDialog::~ProcessDialog()
{
	// ウインドウの機能を有効化。
	MainFrame * frm_main = (MainFrame *) this->hParent;
	frm_main->tree_ctrl->Thaw();
	frm_main->list_ctrl->Thaw();
	frm_main->menubar->EnableTop(0, true);
	frm_main->menubar->EnableTop(1, true);
	frm_main->menubar->EnableTop(2, true);
	frm_main->toolbar->Enable();

	// 書庫を開く途中に中断した場合、書庫を閉じる処理を行う。
	if (this->fOpen && this->fCancel)
	{
		wxCommandEvent e;
		frm_main->OnArcClose(e);
	}

#ifdef __WINDOWS__
	if (this->tlTaskbar != nullptr)
	{
		this->tlTaskbar->SetProgressState(this->hTBWnd, TBPF_NOPROGRESS);
		this->tlTaskbar->Release();
	}
#endif
	((MainFrame *) this->hParent)->gFrame->Show(false);
	g_procDlg = nullptr;
}

//******************************************************************************
// Event table.
//******************************************************************************

BEGIN_EVENT_TABLE(ProcessDialog, wxDialog)
	EVT_INIT_DIALOG(ProcessDialog::OnInit)
	EVT_CLOSE(ProcessDialog::OnClose)
END_EVENT_TABLE()

//******************************************************************************
// Event handler.
//******************************************************************************

void ProcessDialog::OnInit(wxInitDialogEvent &)
{
	if (this->nOverwriteMode == wxID_OK)
	{
		// 2回目以降は何もせず終了。
		return;
	}
	MainFrame * frm_main = (MainFrame *) this->hParent;

	// XRCと結びつけ。
	this->ebTarget          = XRCCTRL(* this, "ebTarget",     wxTextCtrl);
	this->ebSource          = XRCCTRL(* this, "ebSource",     wxTextCtrl);
	this->gFile             = XRCCTRL(* this, "gFile",        wxGauge);
	this->gArchive          = XRCCTRL(* this, "gArchive",     wxGauge);
	this->fCancel           = false;
	this->nOverwriteMode    = wxID_OK;
	::wxXmlResource::Get()->Unload(L_DIR_S_XRC wxT("dlg_process.xrc"));

	// コールバックを送信。
	TPI_PROCESSINFO piInfo;
	piInfo.eMessage = TPI_MESSAGE_STATUS;
	piInfo.eStatus = 0x1000;
	piInfo.fiInfo.fnFileName = frm_main->fnArchive;
	piInfo.fiInfo.nUnpackedSize = this->nFileCount;
	this->CallbackProc(TPI_NOTIFY_COMMON, & piInfo);

	if (frm_main->conf.ReadId(CONF_PROCESS_DLG, false))
	{
		// 旧来の進捗ダイアログを表示。
		// ここでOnInitが呼ばれるので、OnInitの最初で無効化している。
		this->Show();
	}

	// ウインドウの機能を無効化。
	frm_main->tree_ctrl->Freeze();
	frm_main->list_ctrl->Freeze();
	frm_main->menubar->EnableTop(0, false);
	frm_main->menubar->EnableTop(1, false);
	frm_main->menubar->EnableTop(2, false);
	frm_main->toolbar->Enable(false);
#ifdef __WINDOWS__
	// タスクバー。
	this->hTBWnd = frm_main->GetHandle();

	::CoCreateInstance(CLSID_TaskbarList, nullptr, CLSCTX_INPROC_SERVER, IID_ITaskbarList3, (void **) & this->tlTaskbar);
	if (this->tlTaskbar != nullptr)
	{
		this->tlTaskbar->HrInit();
	}
#endif
}

void ProcessDialog::OnClose(wxCloseEvent & e)
{
	if (e.CanVeto() && AskDlg(_("Really do you want to cancel this operation?"), this) == wxYES)
	{
		this->fCancel = true;
	}
	e.Veto();
}

int ProcessDialog::CallbackProc(unsigned int _uMsg, void * _pStructure)
{
	if (_uMsg != TPI_NOTIFY_COMMON)
	{
		return TPI_CALLBACK_UNSUPPORTED;
	}

	TPI_PROCESSINFO * piInfo = (TPI_PROCESSINFO *) _pStructure;
	if (piInfo == nullptr)
	{
		return TPI_CALLBACK_CONTINUE;
	}

	switch (piInfo->eMessage)
	{
	case TPI_MESSAGE_STATUS:
	{
		static int s_nProcess;
#ifdef __WINDOWS__
		static int s_nFileCount;
#endif
		static wxStopWatch sw;
		switch (piInfo->eStatus)
		{
		case TPI_STATUS_OPENARCHIVE:
			this->ebSource->ChangeValue(piInfo->fiInfo.fnFileName.GetFullPath());
			sw.Start();
			break;
		case TPI_STATUS_BEGINPROCESS:
			if (piInfo->fiInfo.nUnpackedSize > 10000)
			{
				this->gFile->SetRange(piInfo->fiInfo.nUnpackedSize);
			}
		case TPI_STATUS_INPROCESS:
			if (sw.Time() > 500)
			{
				if (this->IsShownOnScreen())
				{
					sw.Start();
					this->ebSource->ChangeValue(piInfo->fiInfo.fnFileName.GetFullPath());
					this->ebTarget->ChangeValue(piInfo->fnDestination.GetFullPath());
					if (piInfo->fiInfo.nUnpackedSize > 10000)
					{
						this->gFile->SetValue(piInfo->nProcessedSize);
					}
					this->gArchive->SetValue(s_nProcess);
					this->Update();
					::wxSafeYield(this, true);
				}
				else
				{
					::wxSafeYield(this->hParent, true);
				}
				((MainFrame *) this->hParent)->gFrame->SetValue(s_nProcess);
			}
#ifdef __WINDOWS__
			if (this->tlTaskbar != nullptr)
			{
				this->tlTaskbar->SetProgressValue(this->hTBWnd, s_nProcess, s_nFileCount);
			}
#endif
			break;
		case TPI_STATUS_ENDPROCESS:
			s_nProcess++;
			break;
		// Lycheeの独自仕様。
		case 0x1000: // 初期設定。
			this->ebSource->ChangeValue(piInfo->fiInfo.fnFileName.GetFullPath());
			this->Update();
			((MainFrame *) this->hParent)->gFrame->Show();
//			::wxSafeYield(this->hParent, true);
		case 0x1001: // 閲覧時は最初にファイル数が分からないので、ここで設定。
			this->gArchive->SetRange(piInfo->fiInfo.nUnpackedSize);
			((MainFrame *) this->hParent)->gFrame->SetRange(piInfo->fiInfo.nUnpackedSize);
			s_nProcess = 0;
#ifdef __WINDOWS__
			s_nFileCount = piInfo->fiInfo.nUnpackedSize;
#endif
			sw.Start();
			if (piInfo->eStatus == 0x1001)
			{
				this->fOpen = true;
			}
			break;
		case 0x1002: // 各ファイルの処理を開始。
			if (sw.Time() > 500)
			{
				if (this->IsShownOnScreen())
				{
					sw.Start();
					this->ebTarget->ChangeValue(piInfo->fiInfo.fnFileName.GetFullPath());
					this->gArchive->SetValue(piInfo->nProcessedSize);
					this->Update();
//					::wxSafeYield(this, true);
				}
				::wxSafeYield(this->hParent, true);
			}
			((MainFrame *) this->hParent)->gFrame->SetValue(piInfo->nProcessedSize);
#ifdef __WINDOWS__
			if (this->tlTaskbar != nullptr)
			{
				this->tlTaskbar->SetProgressValue(this->hTBWnd, piInfo->nProcessedSize, s_nFileCount);
			}
#endif
			break;
		}
		break;
	}
	case TPI_MESSAGE_ASK:
	{
#ifdef __WINDOWS__
		if (this->tlTaskbar != nullptr)
		{
			this->tlTaskbar->SetProgressState(this->hTBWnd, TBPF_PAUSED);
		}
#endif
		switch (piInfo->eStatus)
		{
		case TPI_PARAM_PASSWORD:
		{
			if (piInfo->fiInfo.fnFileName == fnFormer || this->szPassword.IsEmpty())
			{
				piInfo->szParam = ::wxGetPasswordFromUser(_("Password for:\n") + piInfo->fiInfo.fnFileName.GetFullPath(), wxT("Lychee"), wxEmptyString, this);
				if (piInfo->szParam.IsEmpty())
				{
					this->fCancel = true;
				}
				this->szPassword = piInfo->szParam;
			}
			else
			{
				piInfo->szParam = this->szPassword;
				fnFormer = piInfo->fiInfo.fnFileName;
			}
			break;
		}
		case TPI_PARAM_NEXTVOLUME:
		{
			wxFileDialog fd(this, _("Choose next volume of: ") + piInfo->fiInfo.fnFileName.GetFullName(), piInfo->fiInfo.fnFileName.GetPath(), wxEmptyString, wxFileSelectorDefaultWildcardStr, wxFD_OPEN | wxFD_FILE_MUST_EXIST);
			if (fd.ShowModal() == wxID_CANCEL)
			{
				this->fCancel = true;
			}
			piInfo->szParam = fd.GetPath();
			break;
		}
		case TPI_PARAM_DEST:
			if (piInfo->fnDestination.FileExists() || ::wxDirExists(piInfo->fnDestination.GetFullPath()))
			{
				// 上書き確認を行う。
				bool bPerm = false;
				if (this->nOverwriteMode == wxID_OK)
				{
					OverwriteDialog odDlg;
					odDlg.fnExist = & piInfo->fnDestination;
					odDlg.tmWrite = & piInfo->fiInfo.tmModify;
					odDlg.nWriteSize = piInfo->fiInfo.nUnpackedSize;

					this->nOverwriteMode = odDlg.ShowModal();
					bPerm = odDlg.cbApplyAll->IsChecked();
				}
				else
				{
					bPerm = true;
				}
				switch (this->nOverwriteMode)
				{
				case wxID_CANCEL:
					this->fCancel = true;
					break;
				case 0:
					// 自動上書き。既存のファイルを削除しておく。
					if (piInfo->fnDestination.FileExists())
					{
						::wxRemoveFile(piInfo->fnDestination.GetFullPath());
					}
					else if (::wxDirExists(piInfo->fnDestination.GetFullPath()))
					{
						// TODO: 再帰削除。
						::wxRmdir(piInfo->fnDestination.GetFullPath());
					}
					break;
				case 1:
					// 自動スキップ。
					piInfo->fnDestination.Clear();
					break;
				case 2:
					// 自動リネーム。
					break;
				}
				if (! bPerm)
				{
					this->nOverwriteMode = wxID_OK;
				}
			}
			break;
		default:
			return TPI_CALLBACK_UNSUPPORTED;
		}
#ifdef __WINDOWS__
		if (this->tlTaskbar != nullptr)
		{
			this->tlTaskbar->SetProgressState(this->hTBWnd, TBPF_NORMAL);
		}
#endif
		break;
	}
	default:
		return TPI_CALLBACK_UNSUPPORTED;
	}

	return this->fCancel ? TPI_CALLBACK_CANCEL : TPI_CALLBACK_CONTINUE;
}
//******************************************************************************
// OverwriteDialog
//******************************************************************************

OverwriteDialog::OverwriteDialog(): wxDialog()
{
	::wxXmlResource::Get()->Load(L_DIR_S_XRC wxT("dlg_overwrite.xrc"));
	::wxXmlResource::Get()->LoadDialog(this, this->GetParent(), wxT("dlg_overwrite"));	
}

//******************************************************************************
// Event table.
//******************************************************************************

BEGIN_EVENT_TABLE(OverwriteDialog, wxDialog)
	EVT_INIT_DIALOG(OverwriteDialog::OnInit)
	EVT_BUTTON(XRCID("btnOverwrite"), OverwriteDialog::OnBtnOverwrite)
	EVT_BUTTON(XRCID("btnSkip"), OverwriteDialog::OnBtnSkip)
	EVT_BUTTON(XRCID("btnAuto"), OverwriteDialog::OnBtnAuto)
	EVT_BUTTON(XRCID("btnBrowse"), OverwriteDialog::OnBtnBrowse)
	EVT_CLOSE(OverwriteDialog::OnClose)
END_EVENT_TABLE()

//******************************************************************************
// Event handler.
//******************************************************************************

void OverwriteDialog::OnInit(wxInitDialogEvent &)
{
	// XRCと結びつけ。
	this->ebTarget          = XRCCTRL(* this, "ebTarget",     wxTextCtrl);
	this->cbApplyAll        = XRCCTRL(* this, "cbApplyAll",   wxCheckBox);
	::wxXmlResource::Get()->Unload(L_DIR_S_XRC wxT("dlg_overwrite.xrc"));

	// ファイル情報を設定。
	this->ebTarget->SetValue(this->fnExist->GetFullPath());
	XRCCTRL(* this, "stExistTime", wxStaticText)->SetLabel(this->fnExist->GetModificationTime().Format(_("%Y/%m/%d %H:%M:%S modified, ")));
	XRCCTRL(* this, "stExistSize", wxStaticText)->SetLabel(this->fnExist->GetHumanReadableSize());
	XRCCTRL(* this, "stWriteTime", wxStaticText)->SetLabel(this->tmWrite->Format(_("%Y/%m/%d %H:%M:%S modified, ")));
	XRCCTRL(* this, "stWriteSize", wxStaticText)->SetLabel(wxFileName::GetHumanReadableSize((wxULongLong) this->nWriteSize));
	XRCCTRL(* this, "sbIcon", wxStaticBitmap)->SetIcon(GetFileTypeIcon(* this->fnExist));
}

void OverwriteDialog::OnClose(wxCloseEvent & e)
{
	if (e.CanVeto() && AskDlg(_("Really do you want to cancel this operation?"), this) == wxYES)
	{
		this->EndModal(wxID_CANCEL);
	}
	e.Veto();
}

void OverwriteDialog::OnBtnOverwrite(wxCommandEvent&)
{
	if (this->fnExist->GetFullPath() != this->ebTarget->GetValue())
	{
		// 保存先を変えているので、次回もダイアログを表示。
		this->cbApplyAll->SetValue(false);
		* this->fnExist = wxFileName(this->ebTarget->GetValue());
	}
	this->EndModal(0);
}

void OverwriteDialog::OnBtnSkip(wxCommandEvent&)
{
	this->EndModal(1);
}

void OverwriteDialog::OnBtnAuto(wxCommandEvent&)
{
	this->EndModal(2);
}

void OverwriteDialog::OnBtnBrowse(wxCommandEvent&)
{
	wxFileName fn(this->ebTarget->GetValue());
	wxFileDialog fd(this, _("Choose a file"), fn.GetPath(), fn.GetFullName(), wxFileSelectorDefaultWildcardStr, wxFD_SAVE | wxFD_OVERWRITE_PROMPT);
	if (fd.ShowModal() == wxID_OK)
	{
		this->ebTarget->SetValue(fd.GetPath());
	}
}

//******************************************************************************
//    ダイアログプロシージャ
//******************************************************************************

int __stdcall TPICallbackProc(unsigned int _uMsg, void * _pStructure)
{
	return g_procDlg == nullptr ? TPI_CALLBACK_CONTINUE : g_procDlg->CallbackProc(_uMsg, _pStructure);
}
