package org.ultramonkey.l7.controller;

import org.apache.log4j.Logger;
import org.ultramonkey.l7.model.*;
import java.util.regex.*;
import java.util.*;
import java.io.*;

/**
 * 
 * <p>
 * class L7vsAdmController
 * </p>
 * <p>
 * Copyright(c) NTT COMWARE 2008
 * </p>
 * 
 * @author tanuma
 */
public class L7vsAdmController {
	private static final String l7vsadm = "/usr/sbin/l7vsadm";

	private Logger ioCommandLogger = Logger.getLogger(LogCategorySet.GUI_IO_COMMAND);
	
	/**
	 * 
	 * <p>
	 * addVirtualService method
	 * </p>
	 * 
	 * @param vs
	 * @return result of l7vsadm add virtual service command.
	 */
	public boolean addVirtualService(VirtualSetting vs) {
        // --- debug log (in method) ---
        if (ioCommandLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf.append("L7vsAdmController::addVirtualService(VirtualSetting vs) in ");
            buf.append("vs=(" + vs + ")");
            ioCommandLogger.debug("11524 " + buf.toString());
        }
        // --- debug log (in method) ---
		
		synchronized (this) {
			if (vs == null || vs.virtual == null) {
				ioCommandLogger.warn("31003 Invalid null argument.");
		        // --- debug log (in method) ---
		        if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11525 L7vsAdmController::addVirtualService(VirtualSetting vs) out return=false");
		        }
		        // --- debug log (in method) ---
				return false;
			}
			
			StringBuffer command = new StringBuffer(l7vsadm + " -A -t "
					+ vs.virtual.host + ":" + vs.virtual.port + " -m "
					+ vs.protomod + " " + vs.option + " -s " + vs.sched);
			if (vs.maxconn > 0) {
				command.append(" -u " + vs.maxconn);
			}
			if (vs.sorryserver != null && vs.sorryserver.host.length() > 0 && vs.sorryserver.port.length() > 0) {
				command.append(" -b " + vs.sorryserver.host + ":"
						+ vs.sorryserver.port);
			}
			if (vs.sorryflag == true) {
				command.append(" -f 1");
			}
			if (vs.qosservice > 0) {
				if (vs.qosservice >= 1000000000L) {
					long qoss = vs.qosservice / 1000000000L;
					command.append(" -Q " + qoss + "G");
				}
				else if (vs.qosservice >= 1000000L) {
					long qoss = vs.qosservice / 1000000L;
					command.append(" -Q " + qoss + "M");
				}
				else if (vs.qosservice >= 1000L) {
					long qoss = vs.qosservice / 1000L;
					command.append(" -Q " + qoss + "K");
				}
				else {
					command.append(" -Q " + vs.qosservice);
				}
			}
			if (vs.qosclient > 0) {
				if (vs.qosclient >= 1000000000L) {
					long qosc = vs.qosclient / 1000000000L;
					command.append(" -q " + qosc + "G");
				}
				else if (vs.qosclient >= 1000000L) {
					long qosc = vs.qosclient / 1000000L;
					command.append(" -q " + qosc + "M");
				}
				else if (vs.qosclient >= 1000L) {
					long qosc = vs.qosclient / 1000L;
					command.append(" -q " + qosc + "K");
				}
				else {
					command.append(" -q " + vs.qosclient);
				}
			}
	
			String result = runProcess(command.toString());
			if (result == null || result.length() != 0) {
				ioCommandLogger.error("41220 Command error: command=" + command);
				ioCommandLogger.error("41221 Command error:  result=" + result);
				// --- debug log (out method) ---
				if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11526 L7vsAdmController::addVirtualService(VirtualSetting vs) out return=false");
				}
				// --- debug log (out method) ---
				return false;
			}
			
			if (vs.real != null) {
				for (int i = 0; i < vs.real.size(); i++) {
					EndPoint real = vs.real.elementAt(i);
					String realCommand = l7vsadm + " -a -t " + vs.virtual.host + ":" + vs.virtual.port + " -m "
						+ vs.protomod + " " + vs.option + " -r " + real.host + ":" + real.port + " -w " + real.weight;
					result = runProcess(realCommand);
					
					if (result == null || result.length() != 0) {
						ioCommandLogger.error("41222 Command error: command=" + command);
						ioCommandLogger.error("41223 Command error:  result=" + result);
						// --- debug log (out method) ---
						if (ioCommandLogger.isDebugEnabled()) {
				            ioCommandLogger.debug("11527 L7vsAdmController::addVirtualService(VirtualSetting vs) out return=false");
						}
						// --- debug log (out method) ---
						return false;
					}
				}
			}
	
			// --- debug log (out method) ---
			if (ioCommandLogger.isDebugEnabled()) {
	            ioCommandLogger.debug("11528 L7vsAdmController::addVirtualService(VirtualSetting vs) out return=true");
			}
			// --- debug log (out method) ---
			return true;
		}
	}

	/**
	 * 
	 * <p>
	 * delVirtualService method
	 * </p>
	 * 
	 * @param vs
	 * @return result of l7vsadm delete virtual service command.
	 */
	public boolean delVirtualService(VirtualSetting vs) {
        // --- debug log (in method) ---
        if (ioCommandLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf.append("L7vsAdmController::delVirtualService(VirtualSetting vs) in ");
            buf.append("vs=(" + vs + ")");
            ioCommandLogger.debug("11529 " + buf.toString());
        }
        // --- debug log (in method) ---

        synchronized (this) {
			if (vs == null || vs.virtual == null) {
				ioCommandLogger.warn("31004 Invalid null argument.");
		        // --- debug log (out method) ---
		        if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11530 L7vsAdmController::delVirtualService(VirtualSetting vs) out return=false");
		        }
		        // --- debug log (out method) ---
				return false;
			}
			String command = l7vsadm + " -D -t " + vs.virtual.host + ":"
					+ vs.virtual.port + " -m " + vs.protomod + " " + vs.option;
	
			String result = runProcess(command);
			if (result == null || result.length() != 0) {
				ioCommandLogger.error("41224 Command error: command=" + command);
				ioCommandLogger.error("41225 Command error:  result=" + result);
				// --- debug log (out method) ---
				if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11531 L7vsAdmController::delVirtualService(VirtualSetting vs) out return=false");
				}
				// --- debug log (out method) ---
				return false;
			}
	
			// --- debug log (out method) ---
			if (ioCommandLogger.isDebugEnabled()) {
	            ioCommandLogger.debug("11532 L7vsAdmController::delVirtualService(VirtualSetting vs) out return=true");
			}
			// --- debug log (out method) ---
			return true;
		}
	}

	/**
	 * 
	 * <p>
	 * getData method
	 * </p>
	 * 
	 * @return result of l7vsadm list command, including virtual service
	 *         setting.
	 */
	public L7vsAdmData getData() {
        // --- debug log (in method) ---
        if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11533 L7vsAdmController::getData() in");
        }
        // --- debug log (in method) ---
        
		synchronized (this) {
			L7vsAdmData admData = new L7vsAdmData();
	
			String command = l7vsadm + " -V -n";
			String result = runProcess(command);
	
			if (result == null) {
				ioCommandLogger.error("41226 Command error: command=" + command);
				ioCommandLogger.error("41227 Command error:  result=null");
				// --- debug log (out method) ---
				if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11534 L7vsAdmController::getData() out return=null");
				}
				// --- debug log (out method) ---
				return null;
			}
	
			// l7vsd log level regex
			Pattern l7vsdLogLevel = Pattern.compile(
					"^(l7vsd_[\\w\\.]+)\\s+(debug|info|warn|error|fatal)$",
					Pattern.MULTILINE);
			Matcher l7vsdLog = l7vsdLogLevel.matcher(result);
			while (l7vsdLog.find()) {
				LogData.LogLevel level;
				if (l7vsdLog.group(2).equals("debug"))
					level = LogData.LogLevel.DEBUG;
				else if (l7vsdLog.group(2).equals("info"))
					level = LogData.LogLevel.INFO;
				else if (l7vsdLog.group(2).equals("warn"))
					level = LogData.LogLevel.WARN;
				else if (l7vsdLog.group(2).equals("error"))
					level = LogData.LogLevel.ERROR;
				else if (l7vsdLog.group(2).equals("fatal"))
					level = LogData.LogLevel.FATAL;
				else {
					ioCommandLogger.info("21092 Unknown log level: " + l7vsdLog.group(2));
					continue;
				}
	
				if (l7vsdLog.group(1).equals("l7vsd_network"))
					admData.l7vsdLogLevel.put(LogData.LogCategory.L7VSD_NETWORK,
							level);
				else if (l7vsdLog.group(1).equals("l7vsd_network.bandwidth"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_NETWORK_BANDWIDTH, level);
				else if (l7vsdLog.group(1).equals("l7vsd_network.num_connection"))
					admData.l7vsdLogLevel
							.put(LogData.LogCategory.L7VSD_NETWORK_NUM_CONNECTION,
									level);
				else if (l7vsdLog.group(1).equals("l7vsd_network.qos"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_NETWORK_QOS, level);
				else if (l7vsdLog.group(1).equals("l7vsd_virtual_service"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_VIRTUAL_SERVICE, level);
				else if (l7vsdLog.group(1).equals("l7vsd_real_server"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_REAL_SERVER, level);
				else if (l7vsdLog.group(1).equals("l7vsd_sorry_server"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_SORRY_SERVER, level);
				else if (l7vsdLog.group(1).equals("l7vsd_real_server.balancing"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_REAL_SERVER_BALANCING, level);
				else if (l7vsdLog.group(1).equals("l7vsd_replication"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_REPLICATION, level);
				else if (l7vsdLog.group(1).equals("l7vsd_start_stop"))
					admData.l7vsdLogLevel.put(LogData.LogCategory.L7VSD_START_STOP,
							level);
				else if (l7vsdLog.group(1).equals("l7vsd_system"))
					admData.l7vsdLogLevel.put(LogData.LogCategory.L7VSD_SYSTEM,
							level);
				else if (l7vsdLog.group(1).equals("l7vsd_system.memory"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_SYSTEM_MEMORY, level);
				else if (l7vsdLog.group(1).equals("l7vsd_system.socket"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_SYSTEM_SOCKET, level);
				else if (l7vsdLog.group(1).equals("l7vsd_system.signal"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_SYSTEM_SIGNAL, level);
				else if (l7vsdLog.group(1).equals("l7vsd_environment"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_ENVIRONMENT, level);
				else if (l7vsdLog.group(1).equals("l7vsd_environment.parameter"))
					admData.l7vsdLogLevel.put(
							LogData.LogCategory.L7VSD_ENVIRONMENT_PARAMETER, level);
				else if (l7vsdLog.group(1).equals("l7vsd_logger"))
					admData.l7vsdLogLevel.put(LogData.LogCategory.L7VSD_LOGGER,
							level);
				else if (l7vsdLog.group(1).equals("l7vsd_parameter"))
					admData.l7vsdLogLevel.put(LogData.LogCategory.L7VSD_PARAMETER,
							level);
				else if (l7vsdLog.group(1).equals("l7vsd_event"))
					admData.l7vsdLogLevel.put(LogData.LogCategory.L7VSD_EVENT,
							level);
				else if (l7vsdLog.group(1).equals("l7vsd_schedule"))
					admData.l7vsdLogLevel.put(LogData.LogCategory.L7VSD_SCHEDULE,
							level);
				else if (l7vsdLog.group(1).equals("l7vsd_program"))
					admData.l7vsdLogLevel.put(LogData.LogCategory.L7VSD_PROGRAM,
							level);
				else if (l7vsdLog.group(1).equals("l7vsd_protocol"))
					admData.l7vsdLogLevel.put(LogData.LogCategory.L7VSD_PROTOCOL,
							level);
				else if (l7vsdLog.group(1).equals("l7vsd_module"))
					admData.l7vsdLogLevel.put(LogData.LogCategory.L7VSD_MODULE,
							level);
				else if (l7vsdLog.group(1).equals("l7vsd_all"))
					admData.l7vsdLogLevel.put(LogData.LogCategory.L7VSD_ALL, level);
				else {
					ioCommandLogger.info("21093 Unknown log category: " + l7vsdLog.group(1));
					continue;
				}
			}
	
			// SNMP Agent log level regex
			Pattern snmpLogLevel = Pattern.compile(
					"^(snmpagent_\\w+)\\s+(debug|info|warn|error|fatal)$",
					Pattern.MULTILINE);
			Matcher snmpLog = snmpLogLevel.matcher(result);
			while (snmpLog.find()) {
				LogData.LogLevel level;
				if (snmpLog.group(2).equals("debug"))
					level = LogData.LogLevel.DEBUG;
				else if (snmpLog.group(2).equals("info"))
					level = LogData.LogLevel.INFO;
				else if (snmpLog.group(2).equals("warn"))
					level = LogData.LogLevel.WARN;
				else if (snmpLog.group(2).equals("error"))
					level = LogData.LogLevel.ERROR;
				else if (snmpLog.group(2).equals("fatal"))
					level = LogData.LogLevel.FATAL;
				else {
					ioCommandLogger.info("21094 Unknown log level: " + snmpLog.group(2));
					continue;
				}
	
				if (snmpLog.group(1).equals("snmpagent_start_stop"))
					admData.snmpLogLevel.put(
							LogData.LogCategory.SNMPAGENT_START_STOP, level);
				else if (snmpLog.group(1).equals("snmpagent_manager_receive"))
					admData.snmpLogLevel.put(
							LogData.LogCategory.SNMPAGENT_MANAGER_RECEIVE, level);
				else if (snmpLog.group(1).equals("snmpagent_manager_send"))
					admData.snmpLogLevel.put(
							LogData.LogCategory.SNMPAGENT_MANAGER_SEND, level);
				else if (snmpLog.group(1).equals("snmpagent_l7vsd_receive"))
					admData.snmpLogLevel.put(
							LogData.LogCategory.SNMPAGENT_L7VSD_RECEIVE, level);
				else if (snmpLog.group(1).equals("snmpagent_l7vsd_send"))
					admData.snmpLogLevel.put(
							LogData.LogCategory.SNMPAGENT_L7VSD_SEND, level);
				else if (snmpLog.group(1).equals("snmpagent_logger"))
					admData.snmpLogLevel.put(LogData.LogCategory.SNMPAGENT_LOGGER,
							level);
				else if (snmpLog.group(1).equals("snmpagent_parameter"))
					admData.snmpLogLevel.put(
							LogData.LogCategory.SNMPAGENT_PARAMETER, level);
				else if (snmpLog.group(1).equals("snmpagent_all"))
					admData.snmpLogLevel.put(LogData.LogCategory.SNMPAGENT_ALL,
							level);
				else {
					ioCommandLogger.info("21095 Unknown log level: " + snmpLog.group(1));
					continue;
				}
			}
	
			// Replication mode regex
			Pattern replicationMode = Pattern.compile(
					"^Replication Mode:\\s+(\\w+)", Pattern.MULTILINE);
			Matcher rep = replicationMode.matcher(result);
			if (rep.find()) {
				if (rep.group(1).equals("MASTER"))
					admData.replicationMode = ReplicationStatus.MASTER;
				else if (rep.group(1).equals("MASTER_STOP"))
					admData.replicationMode = ReplicationStatus.MASTER_STOP;
				else if (rep.group(1).equals("SLAVE"))
					admData.replicationMode = ReplicationStatus.SLAVE;
				else if (rep.group(1).equals("SLAVE_STOP"))
					admData.replicationMode = ReplicationStatus.SLAVE_STOP;
				else if (rep.group(1).equals("SINGLE"))
					admData.replicationMode = ReplicationStatus.SINGLE;
				else if (rep.group(1).equals("OUT"))
					admData.replicationMode = ReplicationStatus.OUT;
				else {
					ioCommandLogger.info("21096 Unknown replication status: " + rep.group(1));
				}
			}
	
			// SNMP Agent connection status regex
			Pattern snmpStatus = Pattern.compile(
					"^SNMPAgent Connection Status:\\s+([-\\w]+)", Pattern.MULTILINE);
			Matcher snmp = snmpStatus.matcher(result);
			if (snmp.find()) {
				if (snmp.group(1).equals("connecting"))
					admData.snmpStatus = SnmpAgentStatus.CONNECTING;
				else if (snmp.group(1).equals("non-connecting"))
					admData.snmpStatus = SnmpAgentStatus.NON_CONNECTING;
				else {
					ioCommandLogger.info("21097 Unknown SNMP Agent status: " + snmp.group(1));
				}
			}
	
			VirtualSetting vs = null;
	
			// Virtual service regex
			String virtualLine = "TCP (\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}):(\\d{1,5}) (\\w+) "
					+ "(\\w+) (0|1) (.*)\\s+([^\\s]+) (\\d+) (0|1)\\s+(\\d+) (\\d+) (\\d+)\\s*";
			// Real server regex
			String realLine = "\\s+-> (\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}):(\\d{1,5})\\s+Masq\\s+"
					+ "(\\d+)\\s+(\\d+)\\s+(\\d+)\\s*";
	
			Pattern virtualService = Pattern.compile("^(?:" + virtualLine + "|"
					+ realLine + ")$", Pattern.MULTILINE);
			Matcher virtual = virtualService.matcher(result);
			while (virtual.find()) {
	
				if (virtual.group(1) != null) { // virtual service line found
					if (vs != null)
						admData.virtualSettings.add(vs);
					vs = new VirtualSetting();
	
					vs.virtual = new EndPoint(virtual.group(1), virtual.group(2));
					vs.protomod = virtual.group(3);
					vs.sched = virtual.group(4);
					vs.reschedule = virtual.group(5).equals("0") ? false : true;
					vs.option = virtual.group(6);
					String[] sorry = virtual.group(7).split(":");
					if (sorry.length == 2 && sorry[1] != null) {
						vs.sorryserver = new EndPoint(sorry[0], sorry[1]);
					}
					vs.maxconn = Integer.parseInt(virtual.group(8));
					vs.sorryflag = virtual.group(9).equals("0") ? false : true;
					vs.qosservice = Long.parseLong(virtual.group(10));
					vs.qosclient = Long.parseLong(virtual.group(11));
					vs.throughput = Long.parseLong(virtual.group(12));
				} else { // real server line found
					if (vs.real == null)
						vs.real = new Vector<EndPoint>();
					vs.real.add(new EndPoint(virtual.group(13), virtual.group(14), Integer.parseInt(virtual
							.group(15)), Long.parseLong(virtual.group(16)), Long
							.parseLong(virtual.group(17))));
				}
			}
			if (vs != null)
				admData.virtualSettings.add(vs);
	
			// --- debug log (out method) ---
			if (ioCommandLogger.isDebugEnabled()) {
	            ioCommandLogger.debug("11535 L7vsAdmController::getData() out return=" + admData);
			}
			// --- debug log (out method) ---
			return admData;
		}
	}

	/**
	 * 
	 * <p>
	 * setReplicationMode method
	 * </p>
	 * 
	 * @param mode
	 * @return result of l7vsadm change replication mode command.
	 */
	public boolean setReplicationMode(ReplicationStatus mode) {
        // --- debug log (in method) ---
        if (ioCommandLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf.append("L7vsAdmController::setReplicationMode(ReplicationStatus mode) in ");
            buf.append("mode=" + mode);
            ioCommandLogger.debug("11536 " + buf.toString());
        }
        // --- debug log (in method) ---

		synchronized (this) {
			if (mode == null) {
				ioCommandLogger.error("41228 Invalid replication status: " + mode);
		        // --- debug log (out method) ---
		        if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11537 L7vsAdmController::setReplicationMode(ReplicationStatus mode) out return=false");
		        }
		        // --- debug log (out method) ---
				return false;
			}
			
			String command = l7vsadm + " -R -s ";
			switch (mode) {
			case MASTER:
			case SLAVE:
				command += "start";
				break;
			case MASTER_STOP:
			case SLAVE_STOP:
				command += "stop";
				break;
			default:
				ioCommandLogger.error("41229 Invalid replication status: " + mode);
		        // --- debug log (out method) ---
		        if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11538 L7vsAdmController::setReplicationMode(ReplicationStatus mode) out return=false");
		        }
		        // --- debug log (out method) ---
				return false;
			}
	
			String result = runProcess(command);
			if (result == null || result.length() != 0) {
				ioCommandLogger.error("41230 Command error: command=" + command);
				ioCommandLogger.error("41231 Command error:  result=" + result);
		        // --- debug log (out method) ---
		        if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11539 L7vsAdmController::setReplicationMode(ReplicationStatus mode) out return=false");
		        }
		        // --- debug log (out method) ---
				return false;
			}
	
	        // --- debug log (out method) ---
	        if (ioCommandLogger.isDebugEnabled()) {
	            ioCommandLogger.debug("11540 L7vsAdmController::setReplicationMode(ReplicationStatus mode) out return=true");
	        }
	        // --- debug log (out method) ---
			return true;
		}
	}

	/**
	 * 
	 * <p>
	 * setLogLevel method
	 * </p>
	 * 
	 * @param category
	 * @return result of l7vsadm set log level command.
	 */
	public boolean setLogLevel(HashMap<LogData.LogCategory, LogData.LogLevel> category) {
        // --- debug log (in method) ---
        if (ioCommandLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf.append("L7vsAdmController::setLogLevel(HashMap<LogData.LogCategory, LogData.LogLevel> category) in ");
            buf.append("category=" + category);
            ioCommandLogger.debug("11541 " + buf.toString());
        }
        // --- debug log (in method) ---
		
		synchronized (this) {
			if (category == null) {
				ioCommandLogger.error("41232 Invalid log category map: " + category);
		        // --- debug log (out method) ---
		        if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11542 L7vsAdmController::setLogLevel(HashMap<LogData.LogCategory, LogData.LogLevel> category) out return=false");
		        }
		        // --- debug log (out method) ---
				return false;
			}
			
			Iterator<LogData.LogCategory> keys = category.keySet().iterator();
			while (keys.hasNext()) {
				String command = l7vsadm;
				LogData.LogCategory c = (LogData.LogCategory) keys.next();
				switch (c) {
				case L7VSD_NETWORK:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_network";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_NETWORK_BANDWIDTH:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_network.bandwidth";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_NETWORK_NUM_CONNECTION:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_network.num_connection";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_NETWORK_QOS:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_network.qos";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_VIRTUAL_SERVICE:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_virtual_service";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_REAL_SERVER:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_real_server";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_SORRY_SERVER:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_sorry_server";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_REAL_SERVER_BALANCING:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_real_server.balancing";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_REPLICATION:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_replication";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_START_STOP:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_start_stop";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_SYSTEM:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_system";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_SYSTEM_MEMORY:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_system.memory";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_SYSTEM_SOCKET:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_system.socket";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_SYSTEM_SIGNAL:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_system.signal";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_ENVIRONMENT:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_environment";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_ENVIRONMENT_PARAMETER:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_environment.parameter";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_LOGGER:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_logger";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_PARAMETER:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_parameter";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_EVENT:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_event";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_SCHEDULE:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_schedule";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_PROGRAM:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_program";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_PROTOCOL:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_protocol";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_MODULE:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c l7vsd_module";
						break;
					default:
						continue;
					}
					break;
				case L7VSD_ALL:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -L -c all";
						break;
					default:
						continue;
					}
					break;
				case SNMPAGENT_START_STOP:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -S -c snmpagent_start_stop";
						break;
					default:
						continue;
					}
					break;
				case SNMPAGENT_MANAGER_RECEIVE:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -S -c snmpagent_manager_receive";
						break;
					default:
						continue;
					}
					break;
				case SNMPAGENT_MANAGER_SEND:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -S -c snmpagent_manager_send";
						break;
					default:
						continue;
					}
					break;
				case SNMPAGENT_L7VSD_RECEIVE:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -S -c snmpagent_l7vsd_receive";
						break;
					default:
						continue;
					}
					break;
				case SNMPAGENT_L7VSD_SEND:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -S -c snmpagent_l7vsd_send";
						break;
					default:
						continue;
					}
					break;
				case SNMPAGENT_LOGGER:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -S -c snmpagent_logger";
						break;
					default:
						continue;
					}
					break;
				case SNMPAGENT_PARAMETER:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -S -c snmpagent_parameter";
						break;
					default:
						continue;
					}
					break;
				case SNMPAGENT_ALL:
					switch (category.get(c)) {
					case DEBUG:
					case INFO:
					case WARN:
					case ERROR:
					case FATAL:
						command += " -S -c all";
						break;
					default:
						continue;
					}
					break;
				default:
					ioCommandLogger.info("21098 Unknown log category: " + c);
			        // --- debug log (out method) ---
			        if (ioCommandLogger.isDebugEnabled()) {
			            ioCommandLogger.debug("11543 L7vsAdmController::setLogLevel(HashMap<LogData.LogCategory, LogData.LogLevel> category) out return=false");
			        }
			        // --- debug log (out method) ---
					return false;
				}
	
				switch (category.get(c)) {
				case DEBUG:
					command += " -l debug";
					break;
				case INFO:
					command += " -l info";
					break;
				case WARN:
					command += " -l warn";
					break;
				case ERROR:
					command += " -l error";
					break;
				case FATAL:
					command += " -l fatal";
					break;
				default:
					ioCommandLogger.info("21099 Unknown log level: " + category.get(c));
			        // --- debug log (out method) ---
			        if (ioCommandLogger.isDebugEnabled()) {
			            ioCommandLogger.debug("11544 L7vsAdmController::setLogLevel(HashMap<LogData.LogCategory, LogData.LogLevel> category) out return=false");
			        }
			        // --- debug log (out method) ---
					return false;
				}
				
				String result = runProcess(command);
				if (result == null || result.length() != 0) {
					ioCommandLogger.error("41233 Command error: command=" + command);
					ioCommandLogger.error("41234 Command error:  result=" + result);
			        // --- debug log (out method) ---
			        if (ioCommandLogger.isDebugEnabled()) {
			            ioCommandLogger.debug("11545 L7vsAdmController::setLogLevel(HashMap<LogData.LogCategory, LogData.LogLevel> category) out return=false");
			        }
			        // --- debug log (out method) ---
					return false;
				}
			}
	        // --- debug log (out method) ---
	        if (ioCommandLogger.isDebugEnabled()) {
	            ioCommandLogger.debug("11546 L7vsAdmController::setLogLevel(HashMap<LogData.LogCategory, LogData.LogLevel> category) out return=true");
	        }
	        // --- debug log (out method) ---
			return true;
		}
	}

	/**
	 * 
	 * <p>
	 * dumpReplicationData method
	 * </p>
	 * 
	 * @return result of l7vsadm replication dump command.
	 */
	public boolean dumpReplicationData() {
        // --- debug log (in method) ---
        if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11547 L7vsAdmController::dumpReplicationData() in");
        }
        // --- debug log (in method) ---

        synchronized (this) {
			String command = l7vsadm + " -R -d";
			String result = runProcess(command);
			if (result == null || result.length() != 0) {
				ioCommandLogger.error("41235 Command error: command=" + command);
				ioCommandLogger.error("41236 Command error:  result=" + result);
		        // --- debug log (out method) ---
		        if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11548 L7vsAdmController::dumpReplicationData() out return=false");
		        }
		        // --- debug log (out method) ---
				return false;
			}
	
	        // --- debug log (out method) ---
	        if (ioCommandLogger.isDebugEnabled()) {
	            ioCommandLogger.debug("11549 L7vsAdmController::dumpReplicationData() out return=true");
	        }
	        // --- debug log (out method) ---
			return true;
		}
	}

	/**
	 * 
	 * <p>
	 * reloadConf method
	 * </p>
	 * 
	 * @param c
	 * @return result of l7vsadm reload config command.
	 */
	public boolean reloadConf(L7vsConfCategory c) {
        // --- debug log (in method) ---
        if (ioCommandLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf.append("L7vsAdmController::reloadConf(L7vsConfCategory c) in ");
            buf.append("c=" + c);
            ioCommandLogger.debug("11550 " + buf.toString());
        }
        // --- debug log (in method) ---
		
		synchronized (this) {
			if (c == null) {
				ioCommandLogger.error("41237 Invalid argument: " + c);
		        // --- debug log (out method) ---
		        if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11551 L7vsAdmController::reloadConf(L7vsConfCategory c) out return=false");
		        }
		        // --- debug log (out method) ---
				return false;
			}
			
			String command = l7vsadm + " -P -r ";
			switch (c) {
			case ALL:
				command += "all";
				break;
			case L7VSD:
				command += "l7vsd";
				break;
			case IOMUX:
				command += "iomux";
				break;
			case LSOCK:
				command += "lsock";
				break;
			case CONN:
				command += "conn";
				break;
			case DEST:
				command += "dest";
				break;
			case SERVICE:
				command += "service";
				break;
			case MODULE:
				command += "module";
				break;
			case REPLICATION:
				command += "replication";
				break;
			case LOGGER:
				command += "logger";
				break;
			case L7VSADM:
				command += "l7vsadm";
				break;
			case SNMPAGENT:
				command += "snmpagent";
				break;
			default:
				ioCommandLogger.error("41238 Unknown log category: " + c);
		        // --- debug log (out method) ---
		        if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11552 L7vsAdmController::reloadConf(L7vsConfCategory c) out return=false");
		        }
		        // --- debug log (out method) ---
				return false;
			}
	
			String result = runProcess(command);
			if (result == null || result.length() != 0) {
				ioCommandLogger.error("41239 Command error: command=" + command);
				ioCommandLogger.error("41240 Command error:  result=" + result);
		        // --- debug log (out method) ---
		        if (ioCommandLogger.isDebugEnabled()) {
		            ioCommandLogger.debug("11553 L7vsAdmController::reloadConf(L7vsConfCategory c) out return=false");
		        }
		        // --- debug log (out method) ---
				return false;
			}
	
	        // --- debug log (out method) ---
	        if (ioCommandLogger.isDebugEnabled()) {
	            ioCommandLogger.debug("11554 L7vsAdmController::reloadConf(L7vsConfCategory c) out return=true");
	        }
	        // --- debug log (out method) ---
			return true;
		}
	}

	/**
	 * 
	 * <p>
	 * runProcess method
	 * </p>
	 * 
	 * @param command
	 * @return result string of command execution
	 */
	protected String runProcess(String command) {
		// --- debug log (in method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
			buf.append("L7vsAdmController::runProcess(String command) in ");
			buf.append("command=\"" + command + "\"");
			ioCommandLogger.debug("11555 " + buf.toString());
		}
		// --- debug log (in method) ---

		synchronized (this) {
			StringBuffer result = new StringBuffer();
	
			try {
				// TODO using sudo command temporally
				Process ps = Runtime.getRuntime().exec("sudo " + command);
				InputStream stderr = ps.getErrorStream();
				BufferedReader br = new BufferedReader(new InputStreamReader(stderr));
				String line = null;
				while ((line = br.readLine()) != null) {
					result.append(line + "\n");
				}
				InputStream stdout = ps.getInputStream();
				br = new BufferedReader(new InputStreamReader(stdout));
				while ((line = br.readLine()) != null) {
					result.append(line + "\n");
				}
	
				Pattern p = Pattern
						.compile("^COMMON ERROR.*timeout");
				Matcher m = p.matcher(result);
				if (m.find()) {
					Parameter param = Parameter.getInstance();
					param.setValue(Parameter.L7VSADM_TIMEOUT_FLAG, "yes");
				} else {
					Parameter param = Parameter.getInstance();
					param.setValue(Parameter.L7VSADM_TIMEOUT_FLAG, null);
				}
			} catch (Exception e) {
				ioCommandLogger.error("41241 Exception occured: " + e.getMessage());
				result = null;
			}
	
			// --- debug log (out method) ---
			if (ioCommandLogger.isDebugEnabled()) {
	            ioCommandLogger.debug("11556 L7vsAdmController::runProcess(String command) out return=" + result);
			}
			// --- debug log (out method) ---
			return (result == null) ? null : result.toString();
		}
	}
}
