/*
 * @file  sched.c
 * @brief the framework module of scheduer 
 * @brief it proceeds common function of scheduler 
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>

#include "l7vs_sched.h"
#include "l7vs_service.h"
#include "l7vs_module.h"
#include "l7vs_dest.h"
#include "logger_wrapper.h"

static struct l7vs_scheduler *l7vs_sched_load(char *name);
static void l7vs_sched_unload(struct l7vs_scheduler *sched);
static struct l7vs_scheduler *l7vs_sched_lookup(char *name);
static gint l7vs_sched_cmp(struct l7vs_scheduler *sched, char *name);

static GList *l7vs_sched_list = NULL;

/*!
 * Get scheduler module.
 * @param[in]	*name		scheduler module name
 * @return	struct l7vs_scheduler*	OK=scheduler pointer, NG=NULL
 */
struct l7vs_scheduler *
l7vs_sched_get(char *name)
{
        struct l7vs_scheduler *sched = NULL;

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,3,
	            "in_function: struct l7vs_scheduler* l7vs_sched_get(char* name): name=\"%s\"", name);
	}
	/*------ DEBUG LOG END ------*/

	// argment check
	if (name == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,2,
		    "Scheduler module name is NULL");
		goto get_out;
	}
        sched = l7vs_sched_lookup(name);
        if (sched == NULL) {
                sched = l7vs_sched_load(name);
                if (sched == NULL) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,3,
						 "Scheduler module not found maybe module problem)");
			goto get_out;
                }
                sched->refcnt = 0;
        }

        sched->refcnt++;

get_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,4,
	            "out_function: struct l7vs_scheduler* l7vs_sched_get(char* name): return_value=%p", sched);
	}
	/*------ DEBUG LOG END ------*/

        return sched;
}

/*!
 * Put scheduler module.
 * @param[in]	*sched		scheduler pointer
 * @return	void
 */
void
l7vs_sched_put(struct l7vs_scheduler *sched)
{
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,5,
	            "in_function: void l7vs_sched_put(struct l7vs_scheduler* sched): sched=%p", sched);
	}
	/*------ DEBUG LOG END ------*/

	// argument check
	if (sched == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,4,
		    "Arg(sched) is NULL pointer.");
		goto put_out;
	}

       	if (--sched->refcnt <= 0) {
       	        l7vs_module_remove(&l7vs_sched_list, sched);
       	        l7vs_sched_unload(sched);
       	}

put_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,6,
	            "out_function: void l7vs_sched_put(struct l7vs_scheduler* sched)");
	}
	/*------ DEBUG LOG END ------*/

}

/*!
 * Bind scheduler module to service.
 * @param[in]	*sched		scheduler pointer
 * @param[in]	*svc		service pointer
 * @return	void
 */
void
l7vs_sched_bind(struct l7vs_scheduler *sched,
                struct l7vs_service *svc)
{
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		char srv_str[DEBUG_STR_LEN] = {0};
		l7vs_service_c_str(srv_str, svc);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,7,
	            "in_function: void l7vs_sched_bind(struct l7vs_scheduler* sched, struct l7vs_service* svc): sched=%p, svc=&(%s)", sched, srv_str);
	}
	/*------ DEBUG LOG END ------*/

	// argument check
	if (sched && svc) {
        	svc->scheduler = sched;
        	if (sched->bind != NULL) {
                	sched->bind(svc);
        	}
	}

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,8,
	            "out_function: void l7vs_sched_bind(struct l7vs_scheduler* sched, struct l7vs_service* svc)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Unbind scheduler module form service.
 * @param[in]	*sched		scheduler pointer
 * @param[in]	*svc		service pointer
 * @return	void
 */
void
l7vs_sched_unbind(struct l7vs_scheduler *sched,
                  struct l7vs_service *svc)
{
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		char srv_str[DEBUG_STR_LEN] = {0};
		l7vs_service_c_str(srv_str, svc);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,9,
	            "in_function: void l7vs_sched_unbind(struct l7vs_scheduler* sched, struct l7vs_service* svc): sched=%p, svc=&(%s)", sched, srv_str);
	}
	/*------ DEBUG LOG END ------*/

	// argument check
	if (sched && svc) {
        	svc->scheduler = NULL;
        	if (sched->unbind != NULL) {
                	sched->unbind(svc);
        	}
	}

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,10,
	            "out_function: void l7vs_sched_unbind(struct l7vs_scheduler* sched, struct l7vs_service* svc)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Load scheduler module.
 * @param[in]	*name		scheduler module name
 * @return	struct l7vs_scheduler*	OK=scheduler pointer, NG=NULL
 */
static struct l7vs_scheduler *
l7vs_sched_load(char *name)
{
        struct l7vs_scheduler *sched = NULL;

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,11,
	            "in_function: struct l7vs_scheduler* l7vs_sched_load(char* name): name=\"%s\"", name);
	}
	/*------ DEBUG LOG END ------*/

	// argment check
	if (name == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,5,
		    "Scheduler module name is NULL");
		goto load_out;
	}
        sched = (struct l7vs_scheduler *)l7vs_module_load(name, "sched");
        if (sched == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,6,
		    "Module load error.");
		goto load_out;
        }
	sched->get_log_level = logger_get_log_level;
	sched->put_log_debug = logger_put_log_debug;
	sched->put_log_info = logger_put_log_info;
	sched->put_log_warn = logger_put_log_warn;
	sched->put_log_error = logger_put_log_error;
	sched->put_log_debug = logger_put_log_debug;

        l7vs_module_register(&l7vs_sched_list, sched);

load_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,12,
	            "out_function: struct l7vs_scheduler* l7vs_sched_load(char* name): return_value=%p", sched);
	}
	/*------ DEBUG LOG END ------*/

        return sched;
}

/*!
 * Unload scheduler module.
 * @param[in]	*sched		scheduler pointer
 * @return	void
 */
static void
l7vs_sched_unload(struct l7vs_scheduler *sched)
{
        void *h;

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,13,
	            "in_function: void l7vs_sched_unload(struct l7vs_scheduler* sched): sched=%p", sched);
	}
	/*------ DEBUG LOG END ------*/

	// argument check
	if (sched == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,7,
		    "Arg(sched) is NULL pointer.");
		goto unload_out;
	}

        h = sched->handle;
	if (sched->fini) {
	        sched->fini();
	}
        l7vs_module_unload(h);

unload_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,14,
	            "out_function: void l7vs_sched_unload(struct l7vs_scheduler* sched)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Lookup scheduler module in sched_list.
 * @param[in]	*name		scheduler module name
 * @return	struct l7vs_scheduler*	found=scheduler pointer, not found=NULL
 */
static struct l7vs_scheduler *
l7vs_sched_lookup(char *name)
{
	struct l7vs_scheduler* sched = NULL;

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,15,
	            "in_function: struct l7vs_scheduler* l7vs_sched_lookup(char* name): "
		    "name=\"%s\"", name);
	}
	/*------ DEBUG LOG END ------*/

	// argment check
	if (name == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,8,
		    "Scheduler module name is NULL");
		goto lookup_out;
	}

	sched = (struct l7vs_scheduler *) l7vs_module_lookup(l7vs_sched_list, name,
	    (GCompareFunc) l7vs_sched_cmp);

lookup_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,16,
	            "out_function: struct l7vs_scheduler* l7vs_sched_load(char* name): "
		    "return_value=%p", sched);
	}
	/*------ DEBUG LOG END ------*/

	return sched;
}

/*!
 * Compare scheduler module name.
 * @param[in]	*sched		scheduler pointer
 * @param[in]	*name		scheduler module name
 * @return	gint		compare result match=0, not match=-1 or other
 */
static gint
l7vs_sched_cmp(struct l7vs_scheduler *sched, char *name)
{
	gint return_value;

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,17,
	            "in_function: gint l7vs_sched_cmp(struct l7vs_scheduler* sched, char* name): "
		    "sched=%p, name=\"%s\"", sched, name);
	}
	/*------ DEBUG LOG END ------*/

	// argument check
	if (sched == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,9,
		    "Arg(sched) is NULL pointer.");
		return_value = -1;
		goto cmp_out;
	}
	if (name == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,10,
		    "Arg(name) is NULL pointer.");
		return_value = -1;
		goto cmp_out;
	}

	return_value = strcmp(sched->modname, name);

cmp_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,18,
	            "out_function: gint l7vs_sched_cmp(struct l7vs_scheduler* sched, char* name): "
		    "return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Sorry status of the specified service is checked.
 * @param[in]	*srv		service pointer
 * @param[in]	cc_check_flag	connection count check flag
 * @return	int		check result 0=not sorry or check NG , 1=sorry
 */
int
l7vs_sched_sorry_check(struct l7vs_service *srv, int cc_check_flag)
{
	GList *l;
	struct l7vs_dest *d;
	int dest_cc = 0;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		char srv_str[DEBUG_STR_LEN] = {0};
		l7vs_service_c_str(srv_str, srv);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,19,
	            "in_function: int l7vs_sched_sorry_check(struct l7vs_service* srv, int cc_check_flag): "
		    "svc=&(%s), cc_check_flag=%d", srv_str, cc_check_flag);
	}
	/*------ DEBUG LOG END ------*/

	// argument check
	if (!srv) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,11,
		    "Argument srv is NULL pointer");
		goto sorry_check_out;
	}
	if (cc_check_flag != 0 && cc_check_flag != 1) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,12,
		    "Invalid cc_check_flag value");
		goto sorry_check_out;
	}

	// check sorry-flag of service
	if (srv->sorry_flag) {
		return_value = 1;
		goto sorry_check_out;
	}

	// check valid destination of service
	if (!srv->dest_list) {
		return_value = 1;
		goto sorry_check_out;
	}

	// not continuous check if cc_check_flag==0
	if (!cc_check_flag) {
		goto sorry_check_out;
	}

	// srv->sorry_cc not set then not sorry
	if (srv->sorry_cc == 0) {
		goto sorry_check_out;
	}
	// check connection count in all real servers of service
	for (l = g_list_first(srv->dest_list); l != NULL; l = g_list_next(l)) {
		d = (struct l7vs_dest *)l->data;
		if (d) {
			dest_cc += d->nactive;
			if (dest_cc + 1 > srv->sorry_cc) {
				return_value = 1;
				goto sorry_check_out;
			}
		}
	}

sorry_check_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,20,
	            "out_function: int l7vs_sched_sorry_check(struct l7vs_service* srv, int cc_check_flag): "
		    "return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Get sorry-server destination or old real-server destination.
 * @param[in]	*srv		service pointer
 * @param[in]	*conn		connection pointer
 * @param[in]	reverse		specification which get sorry-server or old real-server
 * @return	l7vs_dest*	sorry-server or old real-server destination
 */
struct l7vs_dest *
l7vs_sched_sorry_dest(struct l7vs_service *srv, struct l7vs_conn *conn, int reverse)
{
	struct l7vs_dest* dest = NULL;

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		char srv_str[DEBUG_STR_LEN] = {0};
		char conn_str[DEBUG_STR_LEN] = {0};
		l7vs_service_c_str(srv_str, srv);
		l7vs_conn_c_str(conn_str, conn);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,21,
	            "in_function: struct l7vs_dest* l7vs_sched_sorry_dest(struct l7vs_service* srv, "
		    "struct l7vs_conn* conn, int reverse): srv=&(%s), conn=&(%s), reverse=%d",
		    srv_str, conn_str, reverse);
	}
	/*------ DEBUG LOG END ------*/

	// argument check
	if (reverse != 0 && reverse != 1) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,13,"Invalid reverse value");
		goto sorry_dest_out;
	}
	if (!reverse) {
		// argument check
		if (!srv) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,14,"Argument srv is NULL pointer");
			goto sorry_dest_out;
		}
		// return sorry-server destination
		dest = srv->sorry_dest;
	} else {
		// argument check
		if (!conn) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SCHEDULE,15,"Argument srv is NULL pointer");
			goto sorry_dest_out;
		}
		// return old real-server destination
		dest = conn->old_dest;
	}

sorry_dest_out:

	if( dest ){
		char dest_str[DEBUG_STR_LEN] = {0};
		l7vs_dest_c_str(dest_str, dest);
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_SORRY_SERVER,1, "SorryServer Information : %s",dest_str);
	}

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( LOG_CAT_L7VSD_SCHEDULE ) ){
		char dest_str[DEBUG_STR_LEN] = {0};
		l7vs_dest_c_str(dest_str, dest);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,22,
	            "out_function: struct l7vs_dest* l7vs_sched_sorry_dest(struct l7vs_service* srv, "
		    "struct l7vs_conn* conn, int reverse): return_value=&(%s)", dest_str);
	}
	/*------ DEBUG LOG END ------*/


	return dest;
}

