 /*
 * @file        sched_lc.c
 * @brief       Least-Connection Scheduling Module for UltraMonkey-L7
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <limits.h>
#include "l7vs_dest.h"
#include "l7vs_service.h"
#include "l7vs_conn.h"
#include "l7vs_sched.h"

#define	IS_SCHEDLC_DEBUG if( LOG_LV_DEBUG == IS_DEBUG<struct l7vs_scheduler>( sched_lc_scheduler, LOG_CAT_L7VSD_SCHEDULE ))
#define	SCHED_DEBUG(X,Y...) PUT_LOG_DEBUG( sched_lc_scheduler, LOG_CAT_L7VSD_SCHEDULE, X, ##Y )
#define	SCHED_INFO(X,Y...)  PUT_LOG_INFO(  sched_lc_scheduler, LOG_CAT_L7VSD_SCHEDULE, X, ##Y )
#define	SCHED_WARN(X,Y...)  PUT_LOG_WARN(  sched_lc_scheduler, LOG_CAT_L7VSD_SCHEDULE, X, ##Y )
#define SCHED_ERROR(X,Y...) PUT_LOG_ERROR( sched_lc_scheduler, LOG_CAT_L7VSD_SCHEDULE, X, ##Y )
#define	SCHED_FATAL(X,Y...) PUT_LOG_FATAL( sched_lc_scheduler, LOG_CAT_L7VSD_SCHEDULE, X, ##Y )

/*!
 * schedule module finalize function
 * @param void
 * @return void
 */
static void fini(void);

/*!
 * schedule module scheduling function
 * @param[in] service pointer
 * @param[in] connection pointer
 * @return l7vs_dest struct for scheduleing
 */
static struct l7vs_dest *l7vs_sched_lc_schedule(struct l7vs_service *srv,
                                                struct l7vs_conn *conn);

/*!
 *  Least-Connection Scheduling struct 
 */
static struct l7vs_scheduler sched_lc_scheduler = {
        NULL,				/*! schedule handle*/
        "lc",				/*! leastconnection schleduler module name*/
        0,				/*! refarence count*/
        l7vs_sched_lc_schedule,         /*! schedule function */
        NULL,                           /*! bind function */
        NULL,                           /*! unbind function */
        fini,                           /*! fini function */
	NULL,				/*! loglevel get function */
	NULL,				/*! debug log put function */
	NULL,				/*! info log put function */
	NULL,				/*! warn log put function */
	NULL,				/*! error log put function */
	NULL				/*! fatal log put function */
};


/*!
 *	LeastConnection schedule module initialize function
 *	@param[in] shared object handle
 *	@return    l7vs_scheduler struct pointer
 */
extern "C" struct l7vs_scheduler *
init(void *handle)
{
	IS_SCHEDLC_DEBUG{
		SCHED_DEBUG(23, "in function struct l7vs_scheduler* init( void* handle ) handle = %p", handle );
	}

        sched_lc_scheduler.handle = handle;

	IS_SCHEDLC_DEBUG{
		SCHED_DEBUG(24, "out_function: struct l7vs_scheduler* init( void* handle ) return: %p", &sched_lc_scheduler );
	}
        return &sched_lc_scheduler;
}

/*!
 *	LeastConnection schedule module finalize function.
 *	this function is empty.
 *	@param[in] void
 *	@return    void
 */
static void
fini(void)
{
	IS_SCHEDLC_DEBUG{
		SCHED_DEBUG(25, "in_function: void fini(void)" );
	}
	IS_SCHEDLC_DEBUG{
		SCHED_DEBUG(26, "out_function: void fini(void)" );
	}
}

/*!
 *	LeastConnection schedule module scheduling function
 *	@param[in] srv	l7vs_service struct pointer
 *	@param[in] conn	l7vs_connection struct pointer
 *	@return		scheduled dest pointer
 */
static struct l7vs_dest *
l7vs_sched_lc_schedule(struct l7vs_service *srv,
                       struct l7vs_conn *conn)
{
	IS_SCHEDLC_DEBUG{
		char	BUFF[DEBUG_STR_LEN];
		char	SRV_BUFF[DEBUG_STR_LEN];
		char	CON_BUFF[DEBUG_STR_LEN];
		l7vs_service_c_str( SRV_BUFF, srv );
		l7vs_conn_c_str( CON_BUFF, conn );
		snprintf( BUFF, DEBUG_STR_LEN, "in_function: static struct l7vs_dest*"
				      " l7vs_sched_lc_schedule( struct l7vs_service* srv ,"
				      " struct l7vs_conn* conn ) srv = "
		 );
		strncat( BUFF, SRV_BUFF, DEBUG_STR_LEN - strlen( BUFF)  );
		strncat( BUFF, " / conn = ", DEBUG_STR_LEN - strlen( BUFF ) );
		strncat( BUFF,  CON_BUFF, DEBUG_STR_LEN - strlen( BUFF ) );
		SCHED_DEBUG(27, BUFF );
	}
	int nactive = INT_MAX;
	GList* active_dest_list = NULL;
	GList* ptr = NULL;
	struct l7vs_dest* rect = NULL;
	struct l7vs_dest* target = NULL;
	if( !srv ) return NULL;


	//create active list
	 for( ptr = g_list_first( srv->dest_list ); ptr; ptr = g_list_next(ptr) ){
		struct l7vs_dest* dest = (struct l7vs_dest*)ptr->data;
		if(  dest->weight > 0 ) active_dest_list = g_list_append( active_dest_list, dest );
	}

	//active dest don't have
	if( !active_dest_list ){
		SCHED_INFO(1, "active dest don't have in dest-list" );
		return NULL;
	}

	for( ptr = g_list_first( active_dest_list ); ptr; ptr = g_list_next( ptr ) ){
		target = ( struct l7vs_dest* ) ptr->data;
		if( nactive > target->nactive ){
			rect = target;
			nactive = target->nactive;
		}	
	}
	IS_SCHEDLC_DEBUG{
		SCHED_DEBUG(28, "free active_dest_list  %p" , active_dest_list );
	}
	g_list_free( active_dest_list );

	IS_SCHEDLC_DEBUG{
		SCHED_DEBUG(29, "out_function static struct l7vs_dest* l7vs_sched_lc_schedule( struct l7vs_service* srv , struct l7vs_conn* conn ) return = %p", rect );
	}
	return rect;
}
