#include <QtGui>
#include <QTransform>
#include <algorithm>
#include "Scene.h"
#include "Node.h"
#include "UndoMgr.h"
#include <QtXml>
#include <QDebug>
#include "LinkDlg.h"
//#include "mainwindow.h"		//	for GlobalSettings
#include "GlobalSettings.h"

Scene::Scene(QObject *parent)
	: QGraphicsScene(parent)
{
	init();
}
Scene::Scene(qreal x, qreal y, qreal width, qreal height,
				const GlobalSettings *globalSettings, QObject *parent)
	: QGraphicsScene(x, y, width, height, parent)
	, m_defaultSceneRect(QRectF(x, y, width, height))
	, m_globalSettings(globalSettings)
{
	init();
}

void Scene::init()
{
	//setBackgroundBrush(QBrush(QColor(0xe0, 0xe0, 0xe0, 128), Qt::CrossPattern));
	m_rootNode = 0;
	m_editNode = 0;
	//m_globalSettings = 0;
	m_mode = COMMAND;
	m_orgLevel = 0;
	//m_rootNode = 0;
	m_selectedNode = 0;
	m_prevSelectedNode = 0;
	m_mousePressedNode = 0;
	m_dragSource = 0;
	m_dropTarget = 0;
	m_toDropAsChild = false;
	m_droppedToSameScene = false;
	//m_yankedNode = 0;
	m_undoMgr = new UndoMgr();
	connect(this, SIGNAL(selectionChanged()), this, SLOT(onSelectionChanged()));
	connect(m_undoMgr, SIGNAL(canUndoChanged(bool)), this, SIGNAL(canUndoChanged(bool)));
	connect(m_undoMgr, SIGNAL(canRedoChanged(bool)), this, SIGNAL(canRedoChanged(bool)));
	connect(m_undoMgr, SIGNAL(cleanChanged(bool)), this, SIGNAL(cleanChanged(bool)));
	//m_rootBackGroundItem = addEllipse(QRectF(0, 0, 0, 0));
	m_rootNode = createChildNode(0, tr("Root"));
	m_rootNode->setZValue(10);
	setSelectedNode(m_rootNode);
	setRootNodeXPos(sceneRect().width() / 2);
	//layoutAll();	//	setRootNodeXPos() R[̂ŕKvȂ
}
void Scene::updateBackground()
{
	if( m_globalSettings->m_backgroundGrid )
		setBackgroundBrush(QBrush(m_globalSettings->m_bgGridColor, Qt::CrossPattern));
	else
		setBackgroundBrush(QBrush());
}

Scene::~Scene()
{

}
void Scene::setDefaultFont(Node *node)
{
	node->setFont(QFont(m_globalSettings->m_fontName, m_globalSettings->m_fontSize));
}
void Scene::setRootNodeDefaultFont()
{
	setDefaultFont(m_rootNode);
}
void Scene::setClean()
{
	m_undoMgr->setClean();
}
QString Scene::toXmlText(const QRect &wRect) const
{
	QString buffer = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
						"<map version=\"0.9.0\" ";
	const QRectF sr = sceneRect();
	buffer += QString("WIDTH=\"%1\" HEIGHT=\"%2\" ROOTX=\"%3\" ")
				.arg(sr.width())
				.arg(sr.height())
				.arg(rootNodeXPos());
	buffer += QString("WINWD=\"%1\" WINHT=\"%2\" ")
				.arg(wRect.width())
				.arg(wRect.height());
	buffer += ">\n";
	if( m_rootNode != 0 )
		buffer += m_rootNode->toXmlText();
	foreach(Node *node, m_floatingNodes) {
		if( node )
			buffer += node->toXmlText();
	}
	buffer += "</map>\n";
	return buffer;
}
void Scene::onContentsChanged()
{
	layoutAll();
	if( m_selectedNode != 0 )
		m_selectedNode->ensureVisible();
   	//rootNode()->layoutChildren();
   	emit contentsChanged();
}
void Scene::onSelectionChanged()
{
	QList<QGraphicsItem *> items = selectedItems();
	if( items.isEmpty() ) {
		setSelectedNode(0, false);
		//m_selectedNode = 0;
	} else if( m_mousePressedNode != 0 && m_mousePressedNode->isSelected() ) {
		setSelectedNode(m_mousePressedNode, false);
	} else {
		setSelectedNode(dynamic_cast<Node *>(*items.begin()), false);
		//m_selectedNode = dynamic_cast<Node *>(*items.begin());
	}
	m_mousePressedNode = 0;
}
void Scene::setRootNodeXPos(qreal x)
{
	Node *ptr = rootNode();
	if( !ptr ) return;
	QRectF r = sceneRect();
	if( x < 0 ) x = r.width() / 2;
	ptr->setPos(m_rootNodeXPos = x, r.top() + r.height() / 2);
	layoutAll();
}
void Scene::setMode(Mode m, Cursor cp)
{
	if( m_mode == m ) return;
	m_mode = m;
	if( m_selectedNode != 0 ) {
		if( m == COMMAND ) {
			//	Z^Ô߂ɁAeLXgݒ
			m_selectedNode->setTextWidth(m_selectedNode->textWidthEx());
			//	I邽
			QKeyEvent keyEvent(QEvent::KeyPress, Qt::Key_Home, Qt::ControlModifier);
			sendEvent(m_selectedNode, &keyEvent);
			m_selectedNode->setTextCursor(QTextCursor());	//	eLXgJ[\
			m_selectedNode->setTextInteractionFlags(Qt::NoTextInteraction);
			if( m_selectedNode == m_editNode && m_editNode->toPlainText() != m_orgText ) {
				const QString &text = m_editNode->toPlainText();
				m_editNode->setPlainText(m_orgText);
				m_undoMgr->push(new UndoEditNodeTextCommand(this, m_editNode, text));
			}
			this->setFocus();
		} else if( m == INSERT ) {
			m_editNode = m_selectedNode;
			m_orgText = m_selectedNode->toPlainText();
			m_selectedNode->setTextWidth(-1);
			m_selectedNode->setTextInteractionFlags(Qt::TextEditorInteraction);
			m_selectedNode->setFocus();
			m_selectedNode->setTextCursor(QTextCursor(m_selectedNode->document()));	//	eLXgJ[\L
			//	Lbg\邽
			Qt::Key key = cp != END ? Qt::Key_Home : Qt::Key_End;
			QKeyEvent keyEvent(QEvent::KeyPress, key, Qt::ControlModifier);
			sendEvent(m_selectedNode, &keyEvent);
			if( cp == SELALL ) {
				QKeyEvent keyEvent(QEvent::KeyPress, Qt::Key_End, Qt::ShiftModifier);
				sendEvent(m_selectedNode, &keyEvent);
			}
			{	//	I\mɂ邽߂̋̍ 11/08/02
				m_selectedNode->setToInsertMode(true);
				m_selectedNode->clearFocus();
				m_selectedNode->setFocus();
				m_selectedNode->setToInsertMode(false);
			}
			//	XVtXV
			m_selectedNode->setModifiedDT();
		}
		m_selectedNode->update();
	}
	emit modeChanged();
}
Node *Scene::createChildNode(Node *parentNode, const QString title)
{
	return createNode(parentNode, LAST_CHILD, title);
}
Node *Scene::createNode(Node *ptr, CreatePosition crPos, const QString title, bool loading)
{
	Node *node = new Node(title);
	//const int wd = node->QGraphicsTextItem::boundingRect().width();
	//node->setTextWidth(node->textWidthEx());
	node->setFlags(QGraphicsItem::ItemIsSelectable 
						| QGraphicsItem::ItemIsFocusable 
						| QGraphicsItem::ItemAcceptsInputMethod);
	//node->setTextInteractionFlags(Qt::TextEditorInteraction);	//	ҏW\
	connect(node->document(), SIGNAL(contentsChanged()), this, SLOT(onContentsChanged()));
	connect(node, SIGNAL(selectedNodeClicked(Node *)), this, SLOT(selectedNodeClicked(Node *)));
	connect(node, SIGNAL(doDelete(Node *)), this, SLOT(doDelete(Node *)));
	//connect(node, SIGNAL(dragMove(Node *, Node *, bool, bool)), this, SLOT(dragMove(Node *, Node *, bool, bool)));
	connect(node, SIGNAL(rootNodeMoved(Node *, const QPointF &)), this, SLOT(rootNodeMoved(Node *, const QPointF &)));
	if( !ptr ) {
		addItem(node);		//	m[hV[ɒǉ
		if( !m_rootNode )
			setRootNode(node);
		else {
			m_floatingNodes.push_back(node);
			node->setFloatingNode(true);
		}
	} else {
		switch( crPos ) {
		case LAST_CHILD:
			ptr->addNode(node);		//	Ō̎qm[hƂĒǉ
			break;
		case JUST_BEFORE:
			ptr->insertBefore(node);
			break;
		case JUST_AFTER:
			ptr->insertAfter(node);
			break;
		}
		node->setRightSide(ptr->isRightSide());
		node->updateLinkIconPosRecursive();
		if( !loading )
			layoutAll();
		//rootNode()->layoutChildren();
	}
	//setSelectedNode(node);
	emit nodeCreated(node);
	return node;
}
void Scene::addToFloatingNodes(Node *node)
{
	m_floatingNodes.push_back(QPointer<Node>(node));
}
void Scene::removeFromFloatingNodes(Node *node)
{
	QList<QPointer<Node> >::iterator itr = std::find(m_floatingNodes.begin(), m_floatingNodes.end(), node);
	if( itr != m_floatingNodes.end() )
		m_floatingNodes.erase(itr);
}
void Scene::removeAll()
{
	//m_rootNode->removeFromScene();
	removeItem(m_rootNode);
	m_rootNode->freeNode();
	m_rootNode = m_selectedNode = 0;
}
void Scene::layoutAll()
{
	QRectF br = rootNode()->layoutChildren();
	foreach(Node *node, m_floatingNodes) {
		br |= node->layoutChildren();
	}
	m_nodeBoundingRectRaw = br;
	br.translate(-40, -40);
	br.setWidth(br.width() + 80);
	br.setHeight(br.height() + 80);
	m_nodeBoundingRect = br;
	//QRectF r = rootNode()->boundingRect();
	//m_rootBackGroundItem->setRect(r.left(), r.top() - r.height() / 2,
	//								r.width(), r.height() * 2);
	setSceneRect(br | m_defaultSceneRect);
	emit toUpdateView();
}
void Scene::flipSelection()
{
	if( m_selectedNode ) {
		setSelectedNode(0);
	} else if( m_prevSelectedNode )
		setSelectedNode(m_prevSelectedNode);
}
void Scene::setSelectedNode(Node *node, bool toClear)
{
	m_prevSelectedNode = m_selectedNode;
	if( m_selectedNode )
		m_selectedNode->update();
	if( toClear )
		clearSelection();
	m_selectedNode = node;
	if( node ) {
		if( !node->isVisible() ) {
			Node *pNode = node->parentNode();
			while( !pNode->isVisible() )
				pNode = pNode->parentNode();
			doCollapseExpand(pNode);
		}
		node->ensureVisible();
		node->setSelected(true);
		node->setFocus();
		emit showMessage(node->infoText(), 10*1000);
	}
	m_orgLevel = 0;
	emit curNodeChanged(node);
}
#if 1
void Scene::dragEnterEvent ( QGraphicsSceneDragDropEvent * event )
{
	if( event->mimeData()->hasHtml() ) {
		event->setAccepted(true);
	}
#if 0
	if( event->mimeData()->hasFormat("text/uri-list") ) {
		qDebug() << "text/uri-list";
		event->acceptProposedAction();
	}
#endif
}
void Scene::dragMoveEvent ( QGraphicsSceneDragDropEvent * event )
{
	QGraphicsItem *item = itemAt(event->scenePos(), QTransform());
	qDebug() << "item = " << item;
	Node *node = 0;
	if( item ) {
		qDebug() << "type = " << item->type();
		node = dynamic_cast<Node *>(item);
		if( m_dragSource != 0 &&
			(m_dragSource == node || m_dragSource->isDescendantNode(node)) )
		{
			node = 0;	//	̎qɂ̓hbvs
		}
	}
	qDebug() << "node = " << node;
	//const bool alt = (event->modifiers() & Qt::AltModifier) != 0;
	const bool shift = (event->modifiers() & Qt::ShiftModifier) != 0;
	setTopDropAsChild(shift);		//	hbv as  or q
	if( dropTarget() != node ) {
		if( dropTarget() != 0 )
			dropTarget()->update();
		setDropTarget(node);
	}
	if( node ) {
		//QPointF mp(event->scenePos() - node->scenePos());
		//Node::setMousePos(mp);
		m_mousePos = event->scenePos();
		node->update();	//	shift ԂςAbvf[gKv
	}
	if( event->mimeData()->hasHtml() /*&& alt*/ ) {
		event->setAccepted(true);
	} else
		event->setAccepted(node != 0);
}
void Scene::dropEvent ( QGraphicsSceneDragDropEvent * event )
{
	//bool alt = (event->modifiers() & Qt::AltModifier) != 0;
	bool shift = (event->modifiers() & Qt::ShiftModifier) != 0;
	Node *target = dropTarget();
	//Node::setDropTarget(0);
	bool toRightSide = true;
	if( target != 0 ) {
		toRightSide = target->isRightSide();
		if( target->isRootNode() )
			toRightSide = event->scenePos().x() >= target->scenePos().x() + target->boundingRect().width() / 2;
	}
	const bool sameScene = m_dragSource != 0 && m_dragSource->scene() == this;
	setDroppedToSameScene(sameScene);
	qDebug() << "sameScene = " << sameScene;
	if( sameScene ) {
		m_undoMgr->beginMacro("node DnD");
		doDelete(m_dragSource);
	}
	doDrop(event->mimeData()->html(), target, toRightSide, !shift, event->scenePos());
	//dragMove(m_dragSource, target, toRightSide, shift);
	if( sameScene ) {
		m_undoMgr->endMacro();
		event->setDropAction(Qt::CopyAction);
	}
#if 0
	qDebug() << "dropEvent";
	QList<QUrl> fileList = event->mimeData()->urls();
	foreach(QUrl url, fileList) {
		QString filePath = QDir::toNativeSeparators(url.toLocalFile());
		qDebug() << filePath;
	}
#endif
}
#endif
void Scene::contextMenuEvent ( QGraphicsSceneContextMenuEvent *event )
{
	if( mode() != Scene::COMMAND ) {
		event->ignore();
		return;
	}
	QMenu menu;

	QAction *undoAct = new QAction(QIcon(":viMind/Resources/undo.png"), tr("Undo"), this);
	undoAct->setStatusTip(tr("undo edit command"));
	connect(undoAct, SIGNAL(triggered()), this, SLOT(undo()));
	menu.addAction(undoAct);

	QAction *redoAct = new QAction(QIcon(":viMind/Resources/redo.png"), tr("Redo"), this);
	redoAct->setStatusTip(tr("redo edit command"));
	connect(redoAct, SIGNAL(triggered()), this, SLOT(redo()));
	menu.addAction(redoAct);

	QAction *deleteAct = new QAction(QIcon(":viMind/Resources/delete.png"), tr("delete"), this);
	deleteAct->setStatusTip(tr("Delete the current selected nodes"));
	connect(deleteAct, SIGNAL(triggered()), this, SLOT(deleteSelectedItems()));
	menu.addAction(deleteAct);

	menu.exec(event->screenPos());
}
void Scene::keyPressEvent ( QKeyEvent * keyEvent )
{
	int key = keyEvent->key();
	if( key == Qt::Key_Shift || key == Qt::Key_Control )
		return;
	if( key == Qt::Key_Return ) key = Qt::Key_Enter;
	const bool ctrl = (keyEvent->modifiers() & Qt::ControlModifier) != 0;
	const bool shift = (keyEvent->modifiers() & Qt::ShiftModifier) != 0;
	const bool alt = (keyEvent->modifiers() & Qt::AltModifier) != 0;
#if 0
	if( Node::dropTarget() != 0 ) {
		Node::setShift(shift);
		Node::dropTarget()->update();
	}
#endif
	const QString text = keyEvent->text();
#if 0
	if( text == "\x1b" )
		qDebug() << keyEvent->key() << " Esc";
	else
		qDebug() << keyEvent->key() << " " << text;
#endif
	switch( mode() ) {
	case COMMAND:
		if( !ctrl ) {
			if( !shift ) {	//	Ctrl, Shift 
				switch( key ) {
				case Qt::Key_Up: doCurUp(); return;
				case Qt::Key_Enter:
				case Qt::Key_Down: doCurDown(); return;
				case Qt::Key_Left: doCurLeft(); return;
				case Qt::Key_Right: doCurRight(); return;
				case Qt::Key_PageUp: doCurFirstSibling(); return;
				case Qt::Key_PageDown: doCurLastSibling(); return;
				case Qt::Key_Tab: doCurNextTree(); return;
				case Qt::Key_Delete: deleteSelectedItems(); return;
				case Qt::Key_Home: setMode(INSERT, HOME); return;
				case Qt::Key_End: setMode(INSERT, END); return;
				case Qt::Key_Insert: addChildNode(); return;
				//case Qt::Key_Escape: flipSelection(); return;
				}
			} else {	//	Shift ̂
				switch( key ) {
				case Qt::Key_Enter: openPrevNode(); return;
				case Qt::Key_Tab: doCurPrevTree(); return;
				case Qt::Key_Insert: addParentNode(); return;
				}
			}
		} else {
			if( !shift ) {	//	Ctrl ̂
				switch( key ) {
				case Qt::Key_Up: moveNodeUp(); return;
				case Qt::Key_Down: moveNodeDown(); return;
				case Qt::Key_Left: moveNodeLeft(); return;
				case Qt::Key_Right: moveNodeRight(); return;
				case Qt::Key_Enter: openNextNode(); return;
				case Qt::Key_Home: doCurRoot(); return;
				}
			} else {		//	Ctrl + Shift +
			}
		}
		if( text == "i" || text == "I" ) {
			setMode(INSERT, HOME);
		} else if( text == "a" || text == "A" ) {
			setMode(INSERT, END);
		} else if( text == "s" || text == "S" ) {
			setMode(INSERT, SELALL);
		} else if( text == "c" || text == "C" ) {
			addChildNode();
		} else if( text == "r" || text == "R" ) {
			addParentNode();
		} else if( text == "o" ) {
			openNextNode();
		} else if( text == "O" ) {
			openPrevNode();
		} else if( text == "p" ) {
			pasteNext();
		} else if( text == "P" ) {
			pastePrev();
		} else if( text == "t" ) {
			pasteChild();
		} else if( text == "h" ) {
			doCurLeft();
		} else if( text == "j" ) {
			doCurDown();
		} else if( text == "k" ) {
			doCurUp();
		} else if( text == "/" ) {
			emit toFind();
		} else if( text == "n" ) {
			emit toFindNext();
		} else if( text == "N" ) {
			emit toFindPrev();
		} else if( text == "l" ) {
			doCurRight();
		} else if( text == "{" ) {
			doCurFirstSibling();
		} else if( text == "}" ) {
			doCurLastSibling();
		} else if( text == "0" ) {
			doCurRoot();
		} else if( text == "G" ) {
			doCurLast();
		} else if( text == "d" ) {
			deleteSelectedItems();
		} else if( text == "X" ) {
			deleteSelectedItemsWithoutChildren();
		} else if( text == "y" ) {
			yankSelectedItems();
		} else if( text == " " ) {
			doCollapseExpand();
		} else if( text == "H" ) {
			moveNodeLeft();
		} else if( text == "J" ) {
			moveNodeDown();
		} else if( text == "K" ) {
			moveNodeUp();
		} else if( text == "L" ) {
			moveNodeRight();
		} else if( text == "u" ) {
			undo();
		} else if( text == "U" ) {
			redo();
		} else if( text == "v" ) {
			viewLink();
		}
		break;
	case INSERT:
		if( key == Qt::Key_Escape ) {
			setMode(COMMAND);
		} else if( ctrl && !shift && key == Qt::Key_Enter ) {
			setMode(COMMAND);
			openNextNode();
		} else if( !ctrl && shift && key == Qt::Key_Enter ) {
			setMode(COMMAND);
			openPrevNode();
		} else if( ctrl && !shift && key == Qt::Key_Insert ) {
			setMode(COMMAND);
			addChildNode();
		} else if( !ctrl && shift && key == Qt::Key_Insert ) {
			setMode(COMMAND);
			addParentNode();
		} else {
			QGraphicsScene::keyPressEvent( keyEvent );
		}
		break;
	}
}
void Scene::editNode()
{
	setMode(INSERT, HOME);
}
void Scene::editNodeAndSelect()
{
	setMode(INSERT, SELALL);
}
void Scene::addParentNode()
{
	Node *node = selectedNode();
	if( !node || node->isRootNode() ) return;
	m_undoMgr->beginMacro("add parent node");
	Node *ptr = openPrevNode();
	if( node->isRightSide() )
		moveNodeRight(node);
	else
		moveNodeLeft(node);
	m_undoMgr->endMacro();
	setSelectedNode(ptr);
}
Node *Scene::addChildNode(const QString nodeText)
{
	if( !m_selectedNode )
		return 0;
	setMode(COMMAND);
	if( !m_selectedNode->expanded() )
		m_selectedNode->setExpanded( true );
	Node *node = createNode(m_selectedNode, LAST_CHILD, nodeText);
	node->setNodeStyle(m_selectedNode->nodeStyle());
	//setDefaultFont(node);
	node->setFont(m_selectedNode->font());
	int ix = node->parentNode()->childIndexOf(node);
	removeNode(node);
	m_undoMgr->push(new UndoAddNodeCommand(this, node, ix));
	//setSelectedNode(node);
	setMode(INSERT);
	return node;
}
void Scene::pasteChild()
{
	pasteFromYankBuffer(LAST_CHILD);
}
void Scene::pasteNext()
{
	pasteFromYankBuffer(JUST_AFTER);
}
void Scene::pastePrev()
{
	pasteFromYankBuffer(JUST_BEFORE);
}
void Scene::addFloatingNode()
{
	Node *node = createChildNode(0, "");
	removeFromFloatingNodes(node);
	//node->setFloatingNode(true);
	node->setPos(m_nodeBoundingRectRaw.x() + m_nodeBoundingRectRaw.width()/2,
					m_nodeBoundingRectRaw.bottom() + globalSettings()->m_ccSpace);
	node->setFont(QFont(globalSettings()->m_fontName, globalSettings()->m_fontSize));
	m_undoMgr->push(new UndoAddNodeCommand(this, node, -1));
	//addItem(node);
	//m_floatingNodes.push_back(QPointer<Node>(node));
	setSelectedNode(node);
	setMode(INSERT);
	emit contentsChanged();
}
void Scene::openNextNode(const QString nodeText)
{
	if( m_selectedNode != 0 ) {
		if( m_selectedNode == rootNode() ) {
			addChildNode(nodeText);
			return;
		}
		setMode(COMMAND);
		Node *node = createNode(m_selectedNode, JUST_AFTER, nodeText);
		node->setNodeStyle(m_selectedNode->nodeStyle());
		node->setFont(m_selectedNode->font());
		//setDefaultFont(node);
		int ix = node->parentNode()->childIndexOf(node);
		removeNode(node);
		m_undoMgr->push(new UndoAddNodeCommand(this, node, ix));
		//setSelectedNode(node);
		setMode(INSERT);
		emit contentsChanged();
	}
}
Node *Scene::openPrevNode()
{
	if( !m_selectedNode ) return 0;
	if( m_selectedNode == rootNode() ) {
		return addChildNode();
	}
	setMode(COMMAND);
	Node *node = createNode(m_selectedNode, JUST_BEFORE);
	node->setNodeStyle(m_selectedNode->nodeStyle());
	node->setFont(m_selectedNode->font());
	//setDefaultFont(node);
	int ix = node->parentNode()->childIndexOf(node);
	removeNode(node);
	m_undoMgr->push(new UndoAddNodeCommand(this, node, ix));
	//setSelectedNode(node);
	setMode(INSERT);
	emit contentsChanged();
	return node;
}
void Scene::insertLink()
{
	Node *node = selectedNode();
	if( !node || node->isRootNode() ) return;
	LinkDlg dlg;
	dlg.m_linkText->setText(node->linkedFileName());
	dlg.m_width->setText(node->pixmapWidth());
	if( dlg.exec() && node->linkedFileName() != dlg.m_linkText->text() ) {
		m_undoMgr->push(new UndoLinkNodeCommand(this, node, dlg.m_linkText->text()));
		//node->setLink(dlg.m_linkText->text());
		//layoutAll();
		emit contentsChanged();
	}
}
void Scene::insertPicture()
{
	Node *node = selectedNode();
	if( !node || node->isRootNode() ) return;
    QString fileName = QFileDialog::getOpenFileName(0, tr("Select Picture File"));
    if( !fileName.isEmpty() ) {
    	//node->setPicture(fileName);
    	layoutAll();
    	setSelectedNode(node);
    }
}
void Scene::insertFileLink()
{
	Node *node = selectedNode();
	if( !node || node->isRootNode() ) return;
    QString fileName = QFileDialog::getOpenFileName(0, tr("Select Link File"));
    if( !fileName.isEmpty() ) {
    	node->setLink(fileName);
    	layoutAll();
    	setSelectedNode(node);
    }
}
void Scene::doCurUp()
{
	Node *node = selectedNode();
	if( !node || node->isRootNode() ) return;
	Node *node2;
	int lvl = m_orgLevel;
	for(;;) {
		if( (node2 = node->prevNode()) != 0 ) {	//	Zm[hꍇ
			const bool isRightSide = node->isRightSide();
			while( lvl && node2->expanded() ) {		//	܂܂Ăm[hɂ͈ړȂ
				Node *cNode = node2->lastChildNode(isRightSide);
				if( !cNode ) break;
				node2 = cNode;
				--lvl;
			}
			break;
		} else {	//	Zm[hꍇ
			Node *pNode = node->parentNode();
			if( pNode->isRootNode() )	//	[gm[hɒ킪ꍇ
				return;
			node = pNode;
			++lvl;
		}
	}
	setSelectedNode(node2);
	m_orgLevel = lvl;
}
void Scene::doCurDown()
{
	Node *node = selectedNode();
	if( !node || node->isRootNode() ) return;
	Node *node2;
	int lvl = m_orgLevel;
	for(;;) {
		if( (node2 = node->nextNode()) != 0 ) {	//	m[hꍇ
			const bool isRightSide = node->isRightSide();
			while( lvl && node2->expanded() ) {		//	܂܂Ăm[hɂ͈ړȂ
				Node *cNode = node2->firstChildNode(isRightSide);
				if( !cNode ) break;
				node2 = cNode;
				--lvl;
			}
			break;
		} else {	//	m[hꍇ
			Node *pNode = node->parentNode();
			if( pNode->isRootNode() )	//	[gm[hɒ킪ꍇ
				return;
			node = pNode;
			++lvl;
		}
	}
	setSelectedNode(node2);
	m_orgLevel = lvl;
}
void Scene::doCurFirstSibling()
{
	Node *node = selectedNode();
	if( !node || node->isRootNode() )
		return;
	setSelectedNode(node->parentNode()->firstChildNode(node->isRightSide()));
}
void Scene::doCurLastSibling()
{
	Node *node = selectedNode();
	if( !node || node->isRootNode() )
		return;
	setSelectedNode(node->parentNode()->lastChildNode(node->isRightSide()));
}
void Scene::doCurLeft()
{
	Node *node = selectedNode();
	if( !node ) return;
	if( !node->parentNode() ) {		//	[gm[h̏ꍇ
		Node *cNode = m_leftSideReturnNode;
		if( cNode && !cNode->isRightSide() && node->isChildNode(cNode) )		//	^[m[hꍇ
			node = cNode;
		else
			node = node->firstChildNode(Node::LEFT_SIDE);
	} else if( node->isRightSide() ) {	//	Em[h̏ꍇ
		Node *pNode = node->parentNode();
		pNode->setReturnNode(node);
		node = pNode;
	} else {		//	m[h̏ꍇ
		if( !node->expanded() )
			doCollapseExpand(node);
		else {
			Node *cNode = node->returnNode();
			if( cNode && node->isChildNode(cNode) )		//	^[m[hꍇ
				node = cNode;
			else
				node = node->firstChildNode();
		}
	}
	if( !node ) return;
	setSelectedNode(node);
}
void Scene::doCurRight()
{
	Node *node = selectedNode();
	if( !node ) return;
	if( !node->parentNode() ) {		//	[gm[h̏ꍇ
		Node *cNode = node->returnNode();
		if( cNode && cNode->isRightSide() && node->isChildNode(cNode) )		//	^[m[hꍇ
			node = cNode;
		else
			node = node->firstChildNode(Node::RIGHT_SIDE);
	} else if( node->isRightSide() ) {	//	Em[h̏ꍇ
		if( !node->expanded() )
			doCollapseExpand(node);
		else {
			Node *cNode = node->returnNode();
			if( cNode && node->isChildNode(cNode) )		//	^[m[hꍇ
				node = cNode;
			else
				node = node->firstChildNode();
		}
	} else {		//	m[h̏ꍇ
		Node *pNode = node->parentNode();
		if( !pNode->parentNode() )		//	[gm[hɈړꍇ
			m_leftSideReturnNode = QPointer<Node>(node);
		else
			pNode->setReturnNode(node);
		node = pNode;
	}
	if( !node ) return;
	setSelectedNode(node);
}
void Scene::doCurRoot()
{
	setSelectedNode(rootNode());
}
void Scene::doCurLast()
{
	Node *cur = selectedNode();
	Node *leftLastNode = rootNode()->lastDescendantNode(false);
	if( leftLastNode != 0 && leftLastNode != cur ) {
		setSelectedNode(leftLastNode);
	} else {
		Node *rightLastNode = rootNode()->lastDescendantNode(true);
		if( rightLastNode != 0 )
			setSelectedNode(rightLastNode);
	}
}
Node *Scene::lastNode() const
{
	Node *ptr = rootNode();
	Node *node = ptr->lastChildNode(false);		//	̍Ō
	if( !node ) {
		node = ptr->lastChildNode(true);	//	E̍Ō
		if( !node ) return 0;
	}
	ptr = node;
	while( (node = ptr->lastChildNode()) != 0 )
		ptr = node;
	return ptr;
}
void Scene::doCurNextTree()
{
	if( m_selectedNode == 0 ) {
		doCurRoot();
		return;
	}
	if( m_floatingNodes.isEmpty() ) return;
	Node *node = m_selectedNode->rootNode();
	if( node == rootNode() )
		node = m_floatingNodes[0];
	else {
		int ix = m_floatingNodes.indexOf(node);
		if( ++ix == m_floatingNodes.size() )
			node = rootNode();
		else
			node = m_floatingNodes[ix];
	}
	setSelectedNode(node);
}
void Scene::doCurPrevTree()
{
	if( m_selectedNode == 0 ) {
		doCurRoot();
		return;
	}
	if( m_floatingNodes.isEmpty() ) return;
	Node *node = m_selectedNode->rootNode();
	if( node == rootNode() )
		node = m_floatingNodes[m_floatingNodes.size() - 1];
	else {
		int ix = m_floatingNodes.indexOf(node);
		if( !ix )
			node = rootNode();
		else
			node = m_floatingNodes[ix - 1];
	}
	setSelectedNode(node);
}
void Scene::doCurForwardNode()
{
	Node *node = 0;
	if( !m_selectedNode )
		rootNode()->findNewestNode(node);
	else
		rootNode()->findNewerNode(m_selectedNode, node);
	if( node != 0 )
		setSelectedNode(node);
}
void Scene::doCurBackwardNode()
{
	Node *node = 0;
	if( !m_selectedNode )
		rootNode()->findOldestNode(node);
	else
		rootNode()->findOlderNode(m_selectedNode, node);
	if( node != 0 )
		setSelectedNode(node);
}
void Scene::selectAll()
{
	rootNode()->selectAllChildren();
	setSelectedNode(rootNode(), false);
}
void Scene::undo()
{
	if( !m_undoMgr->canUndo() ) return;
	//m_undoMgr->count();
	//m_undoMgr->index();
	const bool macro = m_undoMgr->command(m_undoMgr->index() - 1)->childCount() > 1;
	m_undoMgr->undo();
	if( macro )		//	R}hꍇ́A܂Ƃ߂čăCAEgs
		layoutAll();
}
void Scene::redo()
{
	if( !m_undoMgr->canRedo() ) return;
	//m_undoMgr->count();
	//m_undoMgr->index();
	const bool macro = m_undoMgr->command(m_undoMgr->index())->childCount() > 1;
	m_undoMgr->redo();
	if( macro )		//	R}hꍇ́A܂Ƃ߂čăCAEgs
		layoutAll();
}
#if 0
void Scene::insertNode(Node *node, Node *pNode, int ix)
{
	pNode->insert(ix, node);
	setSelectedNode(node);
}
#endif
void Scene::removeNode(Node *node)
{
	Node *node2 = node->nextNode();		//	폜ɑIm[h
	if( !node2 ) {
		if( !(node2 = node->prevNode()) )
			node2 = node->parentNode();
	}
	Node *pNode = node->parentNode();
	if( pNode )
		pNode->removeFromChildrenList(node);	//	e m_children 폜
	else {	//	t[eBOm[h[g̏ꍇ
		removeFromFloatingNodes(node);
	}
	//node->removeFromScene();	//	V[ node ȉ폜
	removeItem(node);
	setSelectedNode(node2);
}
void Scene::yankSelectedItems()
{
	Node *node = selectedNode();
	if( !node ) return;
	m_yankBuffer = "<map>\n";
	QList<QGraphicsItem *> items = selectedItems();
	foreach(QGraphicsItem *item, items) {
		Node *node = dynamic_cast<Node *>(item);
		if( !node->isRootNode() || node->isFloatingNode() ) {
			m_yankBuffer += node->toXmlText();
		}
	}
	m_yankBuffer += "</map>\n";
}

void Scene::doDelete(Node *node)
{
	m_undoMgr->push(new UndoDeleteNodeCommand(this, node));
}

//	qm[h폜AIm[ĥ݂폜
void Scene::deleteSelectedItemsWithoutChildren()
{
	Node *node = selectedNode();
	if( !node || node->isRootNode() )	//	[gm[h͍폜s
		return;
}
//	Im[hȉ폜
void Scene::deleteSelectedItems()
{
	Node *node = selectedNode();
	if( !node ) return;
	//	undone B em[hƎqm[hƂɑIĂꍇɑΉĂȂ
	QList<QGraphicsItem *> items = selectedItems();
	if( items.size() == 1 && node->isRootNode() && !node->isFloatingNode() )
		return;		//	c[̃[gm[ĥ݂IĂꍇ
	foreach(QGraphicsItem *item, items) {
		Node *node = dynamic_cast<Node *>(item);
		if( !node->isRootNode() && node->isAncesterNodeSelected() ) {		//	cm[hIĂꍇ
			node->setSelected(false);
		}
	}
	items = selectedItems();
	yankSelectedItems();
	m_undoMgr->beginMacro("delete nodes");		//	m[h폜ЂƂ undo/redo PʂƂ邽
	foreach(QGraphicsItem *item, items) {
		Node *node = dynamic_cast<Node *>(item);
		if( !node->isRootNode() || node->isFloatingNode() ) {
			m_undoMgr->push(new UndoDeleteNodeCommand(this, node));
		}
	}
	m_undoMgr->endMacro();
	emit contentsChanged();
}
void Scene::cut()
{
	Node *node = selectedNode();
	if( !node || node == rootNode() )	//	[gm[h͍폜s
		return;
	copy();
	deleteSelectedItems();
}
void Scene::copy()
{
	Node *node = selectedNode();
	if( !node ) return;
	bool emp = true;
	QMimeData *mime = new QMimeData();
	QString text = node->toOutlineText();
	if( !text.isEmpty() ) {
		mime->setText(text);
		emp = false;
	}
	text = node->toXmlText();
	if( !text.isEmpty() ) {
		text = "<map>\n" + text + "</map>\n";
		mime->setHtml(text);
		emp = false;
	}
	if( !emp ) {
		QClipboard *cb = QApplication::clipboard();
		cb->setMimeData(mime);
	}
}
void Scene::paste()
{
	pasteFromCB(JUST_AFTER);
}
void Scene::pasteChildCB()
{
	pasteFromCB(LAST_CHILD);
}
Node *Scene::doPaste(const QString &buffer, Node *node, CreatePosition crpos, bool toRightSide,
						bool bPos, const QPointF pos)
{
	Node *nNode = 0;
	if( buffer.startsWith("<map>") ) {
		QDomDocument doc;
		if( doc.setContent(buffer) ) {		//	XML  DOM ϊ
			QDomElement element = doc.documentElement();
			//qDebug() << element;
			if( node != 0 && node->isRootNode() || crpos == LAST_CHILD ) {
				if( node != 0 && !node->expanded() )
					node->setExpanded(true);
				nNode = addNode(node, element, toRightSide /*!node || node->isRightSide()*/);
			} else {
				nNode = addNode(node, crpos, element, toRightSide /*!node || node->isRightSide()*/);
			}
			if( nNode != 0 ) {
				if( nNode->parentNode() != 0 ) {
					int index = nNode->parentNode()->childIndexOf(nNode);
					removeNode(nNode);
					m_undoMgr->push(new UndoAddNodeCommand(this, nNode, index));
				} else {
					removeFromFloatingNodes(nNode);
					if( bPos )
						nNode->setPos(pos);
					else
						nNode->setPos(m_nodeBoundingRectRaw.x() + m_nodeBoundingRectRaw.width()/2,
										m_nodeBoundingRectRaw.bottom() + globalSettings()->m_ccSpace);
					nNode->setFont(QFont(globalSettings()->m_fontName, globalSettings()->m_fontSize));
					m_undoMgr->push(new UndoAddNodeCommand(this, nNode, -1));
				}
			}
		}
	}
	return nNode;
}
void Scene::pasteFromCB(CreatePosition crpos)
{
	Node *node = selectedNode();
	//if( !node ) return;
	QClipboard *cb = QApplication::clipboard();
	const QMimeData *mime = cb->mimeData();
	if( !mime ) return;
	if( mime->hasHtml() ) {
		QString buffer = mime->html();
		if( doPaste(buffer, node, crpos, !node || node->isRightSide()) != 0 ) return;
	}
	if( mime->hasText() ) {
		QString buffer = mime->text();
		if( crpos == LAST_CHILD )
			addChildNode(buffer);
		else
			openNextNode(buffer);
		return;
	}
}
void Scene::pasteFromYankBuffer(CreatePosition crpos)
{
	Node *node = selectedNode();
	//if( !node ) return;
	if( doPaste(m_yankBuffer, node, crpos, !node || node->isRightSide()) != 0 ) return;
}
//	parentNode Ɏqm[hǉ
Node *Scene::addNode(Node *parentNode, const QDomElement &element, bool isRightSide)
{
	return addNode(parentNode, LAST_CHILD, element, isRightSide);
}
Node *Scene::addNode(Node *ptr, CreatePosition addPos, const QDomElement &element, bool isRightSide)
{
	Node *node = 0;
	/*const*/ QDomElement childEle = element.firstChildElement();
	//const uint cdt = QDateTime::currentDateTime().toTime_t();
	while( !childEle.isNull() ) {
		const QString tagName = childEle.tagName();
		if( tagName == "node" ) {
			node = createNode(ptr, addPos, childEle, isRightSide);
			addNode(node, LAST_CHILD, childEle, isRightSide);
			if( addPos != LAST_CHILD )
				ptr = node;		//	OȄꍇ͊m[hXV
		}
		childEle = childEle.nextSiblingElement();
	}
	return node;
}
Node *Scene::createNode(Node *ptr,					//	m[h
						CreatePosition addPos,		//	ǉʒu
						const QDomElement &element,
						bool isRightSide,
						bool loading)				//	t@C[h
{
	//const QString tagName = element.tagName();
	const QString title = element.attribute("TEXT");
	Node *node = createNode(ptr, addPos, title, loading);
	QGraphicsItem *pNode = node->parentItem();
	if( addPos != LAST_CHILD )
		ptr = node;		//	OȄꍇ͊m[hXV
	node->setRightSide(isRightSide);
	const bool expanded = element.attribute("EXPANDED") != "false";
	node->setExpanded(expanded);
	node->setNodeDownward(element.attribute("DOWNWARD") == "true");
	uchar s = Node::RECT_STYLE;
	const QString style = element.attribute("STYLE");
	if( style == "fork" ) s = Node::FORK_STYLE;
	else if( style == "roundRect" ) s = Node::ROUND_RECT_STYLE;
	else if( style == "circleRect" ) s = Node::CIRCLE_RECT_STYLE;
	node->setNodeStyle(s);
	const QString fontName = element.attribute("FONTNAME", globalSettings()->m_fontName);
	const int fontSize = element.attribute("FONTSIZE", QString("%1").arg(globalSettings()->m_fontSize)).toInt();
	node->setFont(QFont(fontName, fontSize));
	const QString link = element.attribute("LINK");
	if( !link.isEmpty() ) {
		node->setLink(link);
		const QString pixmapWidth = element.attribute("PXMWIDTH");
		if( !pixmapWidth.isEmpty() )
			node->setPixmapWidth(pixmapWidth);
	}
	const QString align = element.attribute("ALIGN");
	if( align == "center" ) {
		node->setTextAlign(Qt::AlignHCenter);
		//node->document()->setDefaultTextOption(QTextOption(Qt::AlignHCenter));
	} else if( align == "right" ) {
		node->setTextAlign(Qt::AlignRight);
		//node->document()->setDefaultTextOption(QTextOption(Qt::AlignRight));
	}
	const bool alignColumn = element.attribute("ALIGNCOLUMN", "false") == "true";
	node->setAlignColumn(alignColumn);
	const QString colorName = element.attribute("FILLCOLOR");
	if( !colorName.isEmpty() ) {
		node->setFillColor(QColor(colorName));
	}
	const uint cdt = QDateTime::currentDateTime().toTime_t();
	QString dt = element.attribute("CREATED");
	node->setCreatedDT(dt.isEmpty() ? cdt : dt.toUInt());
	dt = element.attribute("MODIFIED");
	node->setModifiedDT(dt.isEmpty() ? cdt : dt.toUInt());
	return node;
}
#if 0
void Scene::doMoveUp(Node *node)
{
	Node *pNode;
	if( !node || !(pNode = node->parentNode()) )	//	[gm[h͍폜s
		return;
}
#endif
void Scene::moveNodeUp()
{
	Node *node = selectedNode();
	if( !node ) return;
	if( node->isRootNode() ) {
		//if( node->isFloatingNode() ) {
			QPointF pos = node->pos();
			pos.setY(pos.y() - 10);
			node->setPos(pos);
		//}
		return;
	}
	m_undoMgr->push(new UndoMoveNodeUpCommand(this, node));
}
void Scene::moveNodeDown()
{
	Node *node = selectedNode();
	if( !node ) return;
	if( node->isRootNode() ) {
		//if( node->isFloatingNode() ) {
			QPointF pos = node->pos();
			pos.setY(pos.y() + 10);
			node->setPos(pos);
		//}
		return;
	}
	m_undoMgr->push(new UndoMoveNodeDownCommand(this, node));
}
void Scene::doMoveNodeUp()
{
	Node *node = selectedNode();
	if( !node || node == rootNode() )	//	[gm[h͈ړs
		return;
	node->moveUp();
	layoutAll();
	node->ensureVisible();
	//rootNode()->layoutChildren();
	emit contentsChanged();
	emit curNodeChanged(node);
}
void Scene::doMoveNodeDown()
{
	Node *node = selectedNode();
	if( !node || node == rootNode() )	//	[gm[h͈ړs
		return;
	node->moveDown();
	layoutAll();
	node->ensureVisible();
	//rootNode()->layoutChildren();
	emit contentsChanged();
	emit curNodeChanged(node);
}
void Scene::moveNodeLeft()
{
	Node *node = selectedNode();
	if( !node ) return;
	if( node->isRootNode() ) {
		//if( node->isFloatingNode() ) {
			QPointF pos = node->pos();
			pos.setX(pos.x() - 10);
			node->setPos(pos);
		//}
		return;
	}
	moveNodeLeft(node);
	emit curNodeChanged(node);
}
void Scene::moveNodeLeft(Node *node)
{
#if 1
	Node *srcPN = node->parentNode();
	const int srcIx = srcPN->childIndexOf(node);
	const bool isRightSide = node->isRightSide();
	node->moveLeft();
	Node *dstPN = node->parentNode();
	UndoMoveNodeCommand *cmd = new UndoMoveNodeCommand(this, node, srcPN, isRightSide, srcIx, dstPN);
	m_undoMgr->push(cmd);
#else
	node->moveLeft();
	layoutAll();
#endif
	node->ensureVisible();
	//rootNode()->layoutChildren();
	emit contentsChanged();
}
void Scene::moveNodeRight()
{
	Node *node = selectedNode();
	if( !node ) return;
	if( node->isRootNode() ) {
		//if( node->isFloatingNode() ) {
			QPointF pos = node->pos();
			pos.setX(pos.x() + 10);
			node->setPos(pos);
		//}
		return;
	}
	moveNodeRight(node);
	emit curNodeChanged(node);
}
void Scene::moveNodeRight(Node *node)
{
#if 1
	Node *srcPN = node->parentNode();
	const int srcIx = srcPN->childIndexOf(node);
	const bool isRightSide = node->isRightSide();
	node->moveRight();
	Node *dstPN = node->parentNode();
	UndoMoveNodeCommand *cmd = new UndoMoveNodeCommand(this, node, srcPN, isRightSide, srcIx, dstPN);
	m_undoMgr->push(cmd);
#else
	node->moveRight();
	layoutAll();
#endif
	node->ensureVisible();
	//rootNode()->layoutChildren();
	emit contentsChanged();
}
void Scene::rootNodeMoved(Node *node, const QPointF &pos)
{
	m_undoMgr->push(new UndoChangeNodePosCommand(this, node, pos));
}
void Scene::doDrop(const QString &buffer, Node *dst, bool toRightSide, bool toPasteNext, const QPointF &pos)
{
	const CreatePosition crpos = toPasteNext ? JUST_AFTER : LAST_CHILD;
	Node *node = doPaste(buffer, dst, crpos, toRightSide, true, pos);
	setSelectedNode(node);
}
void Scene::dragMove(Node *src, Node *dst, bool toRightSide, bool toPasteNext)
{
	Node *srcPN = src->parentNode();
	const int srcIx = srcPN->childIndexOf(src);
	const bool isRightSide = src->isRightSide();
	Node *pNode = src->parentNode();
	if( pNode )
		pNode->removeFromChildrenList(src);	//	e m_children 폜
	else {	//	t[eBOm[h[g̏ꍇ
		removeFromFloatingNodes(src);
	}
	removeItem(src);
	if( dst->isRootNode() || !toPasteNext ) {
		if( !dst->expanded() )
			dst->setExpanded(true);
		dst->addNode(src);
	} else
		dst->insertAfter(src);
	src->setRightSide(toRightSide);
	src->updateLinkIconPosRecursive();
	UndoMoveNodeCommand *cmd = new UndoMoveNodeCommand(this, src, srcPN, isRightSide, srcIx, dst);
	m_undoMgr->push(cmd);
}
void Scene::selectedNodeClicked(Node *node)
{
	if( node->isRootNode() ) return;
	doCollapseExpand(node);
}
void Scene::doCollapseExpand()
{
	Node *node = selectedNode();
	if( !node || node == rootNode() )	//	[gm[h͐܏s
		return;
	doCollapseExpand(node);
}

//	?? Node NXɈړH
void Scene::doCollapseExpand(Node *node)
{
	if( node->isRootNode() || !node->hasChildren() )
		return;
	node->setExpanded( !node->expanded() );
	layoutAll();
	//rootNode()->layoutChildren();
	//emit contentsChanged();		11/08/05 WJE܏ŃfBt@CtOON͂߂
}
//	qm[h܏
void Scene::doCollapse1()
{
	doCollapse(1);
}
//	m[h܏
void Scene::doCollapse2()
{
	doCollapse(2);
}
//	Бm[h܏
void Scene::doCollapse3()
{
	doCollapse(3);
}
void Scene::doCollapse(int lvl)
{
	Node *node = selectedNode();
	if( !node ) return;
	node->collapse(lvl);
	layoutAll();
}

//	m[heLXgҏWȂ΁A̓t}
void Scene::editInputDate()
{
	Node *node = selectedNode();
	if( mode() != INSERT || !node ) return;
	const QString todayStr1 = QDate::currentDate().toString("yy/MM/dd" /*Qt::DefaultLocaleShortDate*/);
	node->textCursor().insertText(todayStr1);
}
//	m[heLXgҏWȂ΁A摜}
void Scene::editInputImage()
{
	Node *node = selectedNode();
	if( mode() != INSERT || !node ) return;
    QString fileName = QFileDialog::getOpenFileName(0);
    if (fileName.isEmpty()) return;
    QImage *img= new QImage(fileName);
    m_imageList.push_back(img);
    node->document()->addResource(QTextDocument::ImageResource, QUrl("myimage"), *img);
	node->textCursor().insertImage("myimage");

	//node->setHtml("<img src=\"http://gyazo.com/4a026007c2da8c04403d130799324cc3.png\">");
	//node->textCursor().insertText("<img src=\"http://gyazo.com/4a026007c2da8c04403d130799324cc3.png\">");
}
void Scene::setFillColor(const QColor &color)
{
#if 1
	QList<QGraphicsItem *> items = selectedItems();
	if( items.size() > 1 )
		m_undoMgr->beginMacro("change fill color");
	foreach(QGraphicsItem *ptr, items) {
		Node *node = dynamic_cast<Node *>(ptr);
		//node->setFillColor(color);
		m_undoMgr->push(new UndoColorNodeCommand(this, node, color));
	}
	if( items.size() > 1 )
		m_undoMgr->endMacro();
	update();
#else
	Node *node = selectedNode();
	if( !node ) return;
	node->setFillColor(color);
#endif
}
void Scene::setFont(const QFont &font)
{
	Node *node = selectedNode();
	if( !node ) return;
	//node->setFont(font);
	QList<QGraphicsItem *> items = selectedItems();
	const bool macro = items.size() > 1;
	if( macro )
		m_undoMgr->beginMacro("change font");
	foreach(QGraphicsItem *item, items) {
		Node *node = dynamic_cast<Node *>(item);
		m_undoMgr->push(new UndoFontNodeCommand(this, node, font, macro));
	}
	if( macro ) {
		m_undoMgr->endMacro();
		layoutAll();
	}
	emit contentsChanged();
}

bool Scene::findNext(const QString &text)
{
	Node *node = selectedNode();
	if( !node ) node = rootNode();
	if( !node ) return false;
	bool isRS = node->isRightSide();
	for(;;) {
		if( node->hasChildren() )
			node = node->firstChildNode(node->isRightSide());
		else {
			for(;;) {
				if( node->isRootNode() ) {
					if( isRS ) {
						node = node->firstChildNode(isRS = false);
						if( node != 0 ) break;
					}
					return false;	//	Ƃ肠[v͂ȂƂɂĂ
				}
				Node *next = node->nextNode();
				if( next != 0 ) {
					node = next;
					break;
				}
				node = node->parentNode();
			}
			//if( node->isRootNode() ) return false;		//	Ƃ肠[v͂ȂƂɂĂ
		}
		if( node->toPlainText().indexOf(text, 0, Qt::CaseInsensitive) >= 0 ) {
			setSelectedNode(node);
			return true;
		}
	}
}
bool Scene::findPrev(const QString &text)
{
	Node *node = selectedNode();
	if( !node ) node = lastNode();
	if( !node ) return false;
	bool isRS = node->isRightSide();
	for(;;) {
		if( node->isRootNode() ) {
			if( !isRS ) {
				node = node->lastChildNode(isRS = true);
				if( !node ) return false;
				while( node->hasChildren() ) {
					//	qꍇ͈Ԗ[̃m[hɈړ
					node = node->lastChildNode(isRS);
				}
			} else
				return false;	//	Ƃ肠[v͂ȂƂɂĂ
		} else {
			Node *prev = node->prevNode();
			if( !prev ) {		//	Zꍇ
				node = node->parentNode();
			} else {
				node = prev;
				while( node->hasChildren() ) {
					//	qꍇ͈Ԗ[̃m[hɈړ
					node = node->lastChildNode(isRS);
				}
			}
		}
		if( node->toPlainText().indexOf(text, 0, Qt::CaseInsensitive) >= 0 ) {
			setSelectedNode(node);
			return true;
		}
	}
}
void Scene::expandAll()
{
	foreach(QPointer<Node> ptr, rootNode()->children()) {
		if( ptr ) {
			ptr->setExpanded(true);
			ptr->expandAll(true);
		}
	}
	layoutAll();
	emit contentsChanged();
}
void Scene::collapseAll()
{
	foreach(QPointer<Node> ptr, rootNode()->children()) {
		if( ptr ) {
			ptr->setExpanded(false);
			ptr->expandAll(false);
		}
	}
	layoutAll();
	setSelectedNode(rootNode());
	emit contentsChanged();
}
void Scene::setTextALign(Qt::Alignment align)
{
	QList<QGraphicsItem *> items = selectedItems();
	foreach(QGraphicsItem *item, items) {
		Node *node = dynamic_cast<Node *>(item);
		node->setTextAlign(align);
		//node->document()->setDefaultTextOption(QTextOption(align));
		node->setTextWidth(node->textWidthEx());
	}
	update();
}
void Scene::formatALignLeft()
{
	setTextALign(Qt::AlignLeft);
}
void Scene::formatALignCenter()
{
	setTextALign(Qt::AlignHCenter);
}
void Scene::formatALignRight()
{
	setTextALign(Qt::AlignRight);
}
void Scene::formatALignColumn()
{
	Node *node = selectedNode();
	if( !node ) return;
	node->setAlignColumn(!node->isAlignColumn());
	node->resetChildrenTextWidthRecursive(node->isRightSide());
	layoutAll();
}
void Scene::nodeDownward(bool b)
{
	Node *node = selectedNode();
	if( !node ) return;
	node->setNodeDownwardRcsv(b);
	layoutAll();
}
void Scene::setNodeStyle(uchar s)
{
	Node *node = selectedNode();
	if( !node ) return;
	node->setNodeStyle(s);
	layoutAll();
}
void Scene::formatFork()
{
	setNodeStyle(Node::FORK_STYLE);
}
void Scene::formatRect()
{
	setNodeStyle(Node::RECT_STYLE);
}
void Scene::formatRoundRect()
{
	setNodeStyle(Node::ROUND_RECT_STYLE);
}
void Scene::formatCircleRect()
{
	setNodeStyle(Node::CIRCLE_RECT_STYLE);
}
void Scene::doShowMessage(const QString &text, int limit)
{
	emit showMessage(text, limit);
}
void Scene::getCurFileName(QString &path)
{
	emit curFileName(path);
}
void Scene::viewLink()
{
	Node *node = selectedNode();
	if( !node ) return;
	node->onLinkIconClicked();
}

#if 0
//	XVtԂɐVm[hT
Node *Scene::findNewerNode(const Node *rNode)
{
	Node *node = 0;

}
//	XVtԂɌÂm[hT
Node *Scene::findOlderNode(const Node *)
{
}
#endif
