package org.b2ox.math
{
	import org.b2ox.math.Number3D;

	/**
	 * org.papervision3d.core.math.Quaternion をベースにして弄ってます
	 */
	public class Quaternion
	{
		public static const EPSILON:Number = 0.000001;
		static public const toDEGREES:Number = 180/Math.PI;
		static public const toRADIANS:Number = Math.PI/180;

		/** imaginary x */
		public var x:Number;

		/** imaginary y */
		public var y:Number;

		/** imaginary z */
		public var z:Number;

		/** real part */
		public var w:Number;

		/**
		 * constructor.
		 *
		 * @param	x
		 * @param	y
		 * @param	z
		 * @param	w
		 * @return
		 */
		public function Quaternion( x:Number = 0, y:Number = 0, z:Number = 0, w:Number = 1 )
		{
			this.x = x;
			this.y = y;
			this.z = z;
			this.w = w;
		}

		/**
		 * for displaying
		 * @return
		 */
		public function toString():String{
			return "Quaternion: x:"+this.x+" y:"+this.y+" z:"+this.z+" w:"+this.w;
		}

		/**
		 * Clone.
		 *
		 */
		public function clone():Quaternion
		{
			return new Quaternion(this.x, this.y, this.z, this.w);
		}

		/**
		 * Copies the values of this Quaternion to the passed Quaternion.
		 */
		public function copyTo(q:Quaternion):void
		{
			q.x = x;
			q.y = y;
			q.z = z;
			q.w = w;
		}

		/**
		 * Copies the values of this Quaternion to the passed Quaternion.
		 */
		public function copyFrom(q:Quaternion):void
		{
			x = q.x;
			y = q.y;
			z = q.z;
			w = q.w;
		}

		/**
		 * Quick way to set the properties of the Quaternion.
		 *
		 * @param	nx
		 * @param	ny
		 * @param	nz
		 * @param	nw
		 */
		public function reset( nx:Number = 0, ny:Number = 0, nz:Number = 0, nw:Number = 1 ):Quaternion
		{
			x = nx;
			y = ny;
			z = nz;
			w = nw;
			return this;
		}

		//-------------------------------------------------
		// get scalers by operators

		/**
		 * Squared modulo.
		 *
		 * @return
		 */
		public function get moduloSquared():Number
		{
			return x*x + y*y + z*z + w*w;
		}

		/**
		 * Modulo.
		 *
		 * @return
		 */
		public function get modulo():Number
		{
			return Math.sqrt(x*x + y*y + z*z + w*w);
		}

		/**
		 * Dot product.
		 *
		 * @param	b
		 * @return	this . b
		 */
		public function dot( b:Quaternion ):Number
		{
			return (x * b.x) + (y * b.y) + (z * b.z) + (w * b.w);
		}

		//-------------------------------------------------
		// get new Quaternions by operators

		/**
		 * Conjugate.
		 *
		 * @return	new conjugate quaternion
		 */
		public function conjugate():Quaternion
		{
			return new Quaternion( -x, -y, -z, w );
		}

		/**
		 * add
		 *
		 * @param	b
		 * @return	this + b
		 */
		public function add(b:Quaternion):Quaternion
		{
			return new Quaternion(x + b.x, y + b.y, z + b.z, w + b.w);
		}

		/**
		 * sub
		 *
		 * @param	b
		 * @return	this - b
		 */
		public function sub(b:Quaternion):Quaternion
		{
			return new Quaternion(x - b.x, y - b.y, z - b.z, w - b.w);
		}

		/**
		 * Multiply.
		 *
		 * @param	b
		 * @return	this * b
		 */
		public function multiply( b:Quaternion ):Quaternion
		{
			var bx:Number = b.x;
			var by:Number = b.y;
			var bz:Number = b.z;
			var bw:Number = b.w;
			var nx:Number = w * bx + x * bw + y * bz - z * by;
			var ny:Number = w * by - x * bz + y * bw + z * bx;
			var nz:Number = w * bz + x * by - y * bx + z * bw;
			var nw:Number = w * bw - x * bx - y * by - z * bz;
			return new Quaternion(nx, ny, nz, nw);
		}

		//-------------------------------------------------
		// set values by operators

		/**
		 * this := a + b
		 * @param	a
		 * @param	b
		 */
		public function setAdd( a:Quaternion, b:Quaternion ):Quaternion
		{
			x = a.x + b.x;
			y = a.y + b.y;
			z = a.z + b.z;
			w = a.w + b.z;
			return this;
		}

		/**
		 * this := a - b
		 * @param	a
		 * @param	b
		 */
		public function setSub( a:Quaternion, b:Quaternion ):Quaternion
		{
			x = a.x - b.x;
			y = a.y - b.y;
			z = a.z - b.z;
			w = a.w - b.z;
			return this;
		}

		/**
		 * this := a * b
		 * @param	a
		 * @param	b
		 */
		public function setMultiply( a:Quaternion, b:Quaternion ):Quaternion
		{
			x = a.x;
			y = a.y;
			z = a.z;
			w = a.w;
			var bx:Number = b.x;
			var by:Number = b.y;
			var bz:Number = b.z;
			var bw:Number = b.w;
			var nx:Number = w * bx + x * bw + y * bz - z * by;
			var ny:Number = w * by - x * bz + y * bw + z * bx;
			var nz:Number = w * bz + x * by - y * bx + z * bw;
			var nw:Number = w * bw - x * bx - y * by - z * bz;
			x = nx;
			y = ny;
			z = nz;
			w = nw;
			return this;
		}

		/**
		 * this := a / |a|
		 * @param	a
		 */
		public function setNormalize( a:Quaternion ):Quaternion
		{
			var len:Number = a.modulo;

			if( Math.abs(len) < EPSILON )
			{
				x = y = z = 0.0;
				w = 1.0;
			}
			else
			{
				var m:Number = 1 / len;
				x = a.x * m;
				y = a.y * m;
				z = a.z * m;
				w = a.w * m;
			}
			return this;
		}

		/**
		 * this := log(a / |a|)
		 * @param	a
		 */
		public function setLog( a:Quaternion ):Quaternion
		{
			this.setNormalize(a);
			var theta:Number = Math.acos(w);
			if (Math.abs(theta) < EPSILON)
			{
				x = y = z = 0.0;
				w = 1.0;
			} else {
				theta /= Math.sin(theta);
				x *= theta;
				y *= theta;
				z *= theta;
				w = 0.0;
			}
			return this;
		}

		/**
		 * this := exp((a - a.conjugate)/2)
		 * @param	a	a.w = 0 として処理する
		 */
		public function setExp( a:Quaternion ):Quaternion
		{
			var theta:Number = Math.sqrt(a.x * a.x + a.y * a.y + a.z * a.z);
			w = Math.cos(theta);
			theta = Math.sin(theta);
			x = a.x * theta;
			y = a.y * theta;
			z = a.z * theta;
			return this;
		}

		/**
		 * this := (a / |a|)^k
		 * @param	a
		 */
		public function setPower( a:Quaternion, k:Number ):Quaternion
		{
			this.setNormalize(a);
			var theta:Number = Math.acos(w);
			if (Math.abs(theta) < EPSILON)
			{
				x = y = z = 0.0;
				w = 1.0;
			} else {
				theta *= k;
				w = Math.cos( theta );
				var m:Number = Math.sin( theta ) / Math.sqrt(x * x + y * y + z * z);
				x *= m;
				y *= m;
				z *= m;
			}
			return this;
		}

		//-------------------------------------------------
		// +=, -=, *= operators

		/**
		 * The same as += operator.
		 *
		 * @param	b
		 */
		public function addEq(b:Quaternion):Quaternion
		{
			x += b.x;
			y += b.y;
			z += b.z;
			w += b.w;
			return this;
		}

		/**
		 * The same as -= operator.
		 *
		 * @param	b
		 */
		public function subEq(b:Quaternion):Quaternion
		{
			x -= b.x;
			y -= b.y;
			z -= b.z;
			w -= b.w;
			return this;
		}

		/**
		 * Multiply by another Quaternion. The same as *= operator.
		 *
		 * @param	b
		 */
		public function multiplyEq( b:Quaternion ):Quaternion
		{
			var bx:Number = b.x;
			var by:Number = b.y;
			var bz:Number = b.z;
			var bw:Number = b.w;
			var nx:Number = w * bx + x * bw + y * bz - z * by;
			var ny:Number = w * by - x * bz + y * bw + z * bx;
			var nz:Number = w * bz + x * by - y * bx + z * bw;
			var nw:Number = w * bw - x * bx - y * by - z * bz;
			x = nx;
			y = ny;
			z = nz;
			w = nw;
			return this;
		}

		//-------------------------------------------------
		// other utilities

		/**
		 * Normalize.
		 */
		public function normalize():Quaternion
		{
			var len:Number = this.modulo;

			if( Math.abs(len) < EPSILON )
			{
				x = y = z = 0.0;
				w = 1.0;
			}
			else
			{
				var m:Number = 1 / len;
				x *= m;
				y *= m;
				z *= m;
				w *= m;
			}
			return this;
		}

		/**
		 * SLERP (Spherical Linear intERPolation).
		 *
		 * @param	qa		start quaternion
		 * @param	qb		end quaternion
		 * @param	alpha	a value between 0 and 1
		 *
		 * @return the interpolated quaternion.
		 */
		public static function slerp( qa:Quaternion, qb:Quaternion, alpha:Number ):Quaternion
		{
			return (new Quaternion()).setSLERP(qa, qb, alpha);
		}

		/**
		 * SLERP (Spherical Linear intERPolation). @author Trevor Burton
		 *
		 * @param	qa		start quaternion
		 * @param	qb		end quaternion
		 * @param	alpha	a value between 0 and 1
		 *
		 * @return the interpolated quaternion.
		 */
		public function setSLERP( qa:Quaternion, qb:Quaternion, alpha:Number ):Quaternion
		{
			var qa_x:Number, qa_y:Number, qa_z:Number, qa_w:Number;
			var qb_x:Number, qb_y:Number, qb_z:Number, qb_w:Number;
			var angle:Number = qa.dot(qb);

			if (angle < 0.0)
			{
				qa_x = - qa.x;
				qa_y = - qa.y;
				qa_z = - qa.z;
				qa_w = - qa.w;
				angle *= -1.0;
			} else {
				qa_x = qa.x;
				qa_y = qa.y;
				qa_z = qa.z;
				qa_w = qa.w;
			}

			var scale:Number;
			var invscale:Number;

			if ((angle + 1.0) > EPSILON) // Take the shortest path
			{
				if ((1.0 - angle) >= EPSILON)  // spherical interpolation
				{
					var theta:Number = Math.acos(angle);
					var invsintheta:Number = 1.0 / Math.sin(theta);
					scale = Math.sin(theta * (1.0-alpha)) * invsintheta;
					invscale = Math.sin(theta * alpha) * invsintheta;
				}
				else // linear interploation
				{
					scale = 1.0 - alpha;
					invscale = alpha;
				}
				qb_x = qb.x;
				qb_y = qb.y;
				qb_z = qb.z;
				qb_w = qb.w;
			}
			else // long way to go...
			{
				scale = Math.sin(Math.PI * (0.5 - alpha));
				invscale = Math.sin(Math.PI * alpha);
				qb_x = qa.x;
				qb_y = -qa.y;
				qb_z = qa.z;
				qb_w = -qa.w;
			}

			x = scale * qa_x + invscale * qb_x;
			y = scale * qa_y + invscale * qb_y;
			z = scale * qa_z + invscale * qb_z;
			w = scale * qa_w + invscale * qb_w;
			return this;
		}

		/**
		 * get SLERP function
		 *
		 * @param	qa		start quaternion
		 * @param	qb		end quaternion
		 *
		 * @return	interpolate function (alpha:Number, target:Quaternion):Quaternion
		 */
		public static function getSLERPer( qa:Quaternion, qb:Quaternion ):Function
		{
			var qa_x:Number, qa_y:Number, qa_z:Number, qa_w:Number;
			var qb_x:Number, qb_y:Number, qb_z:Number, qb_w:Number;
			var angle:Number = qa.dot(qb);

			if (angle < 0.0)
			{
				qa_x = - qa.x;
				qa_y = - qa.y;
				qa_z = - qa.z;
				qa_w = - qa.w;
				angle *= -1.0;
			} else {
				qa_x = qa.x;
				qa_y = qa.y;
				qa_z = qa.z;
				qa_w = qa.w;
			}

			var getScale:Function, getInvScale:Function;

			if ((angle + 1.0) > EPSILON) // Take the shortest path
			{
				if ((1.0 - angle) >= EPSILON)  // spherical interpolation
				{
					var theta:Number = Math.acos(angle);
					var invsintheta:Number = 1.0 / Math.sin(theta);
					getScale = function (alpha:Number):Number { return Math.sin(theta * (1.0 - alpha)) * invsintheta; };
					getInvScale = function (alpha:Number):Number { return Math.sin(theta * alpha) * invsintheta; };
				}
				else // linear interploation
				{
					getScale = function (alpha:Number):Number { return 1.0 - alpha; };
					getInvScale = function (alpha:Number):Number { return alpha; };
				}
				qb_x = qb.x;
				qb_y = qb.y;
				qb_z = qb.z;
				qb_w = qb.w;
			}
			else // long way to go...
			{
				getScale = function (alpha:Number):Number { return Math.sin(Math.PI * (0.5 - alpha)); };
				getInvScale = function (alpha:Number):Number { return Math.sin(Math.PI * alpha); };
				qb_x = qa.x;
				qb_y = -qa.y;
				qb_z = qa.z;
				qb_w = -qa.w;
			}

			return function (alpha:Number, target:Quaternion):Quaternion {
				var scale:Number = getScale(alpha);
				var invscale:Number = getInvScale(alpha);
				target.x = scale * qa_x + invscale * qb_x;
				target.y = scale * qa_y + invscale * qb_y;
				target.z = scale * qa_z + invscale * qb_z;
				target.w = scale * qa_w + invscale * qb_w;
				return target;
			};
		}

		/**
		 * Sets this Quaternion from a axis and a angle.
		 *
		 * @param	x 	X-axis
		 * @param	y 	Y-axis
		 * @param	z 	Z-axis
		 * @param	angle	angle
		 * @param	useDegrees
		 */
		public function setFromAxisAngle( x:Number, y:Number, z:Number, angle:Number, useDegrees:Boolean = false ):Quaternion
		{
			if (useDegrees) angle *= toRADIANS;
			var sin:Number = Math.sin( angle / 2 );
			var cos:Number = Math.cos( angle / 2 );
			this.x = x * sin;
			this.y = y * sin;
			this.z = z * sin;
			this.w = cos;
			return this.normalize();
		}

		/**
		 * Creates a Quaternion from a axis and a angle.
		 *
		 * @param	x 	X-axis
		 * @param	y 	Y-axis
		 * @param	z 	Z-axis
		 * @param	angle	angle
		 * @param	useDegrees
		 *
		 * @return
		 */
		public static function createFromAxisAngle( x:Number, y:Number, z:Number, angle:Number, useDegrees:Boolean = false ):Quaternion
		{
			var q:Quaternion = new Quaternion();
			q.setFromAxisAngle(x, y, z, angle, useDegrees);
			return q;
		}

		/**
		 * Sets this Quaternion from Euler angles.
		 *
		 * @param	ax	X-angle in radians.
		 * @param	ay	Y-angle in radians.
		 * @param	az	Z-angle in radians.
		 * @param	useDegrees
		 */
		public function setFromEuler(ax:Number, ay:Number, az:Number, useDegrees:Boolean=false):void
		{
			if( useDegrees ) {
				ax *= toRADIANS;
				ay *= toRADIANS;
				az *= toRADIANS;
			}
			ax *= 0.5;
			ay *= 0.5;
			az *= 0.5;

			var fSinPitch       :Number = Math.sin( ax );
			var fCosPitch       :Number = Math.cos( ax );
			var fSinYaw         :Number = Math.sin( ay );
			var fCosYaw         :Number = Math.cos( ay );
			var fSinRoll        :Number = Math.sin( az );
			var fCosRoll        :Number = Math.cos( az );
			var fCosPitchCosYaw :Number = fCosPitch * fCosYaw;
			var fSinPitchSinYaw :Number = fSinPitch * fSinYaw;

			this.x = fSinRoll * fCosPitchCosYaw     - fCosRoll * fSinPitchSinYaw;
			this.y = fCosRoll * fSinPitch * fCosYaw + fSinRoll * fCosPitch * fSinYaw;
			this.z = fCosRoll * fCosPitch * fSinYaw - fSinRoll * fSinPitch * fCosYaw;
			this.w = fCosRoll * fCosPitchCosYaw     + fSinRoll * fSinPitchSinYaw;
		}

		/**
		 * Creates a Quaternion from Euler angles.
		 *
		 * @param	ax	X-angle in radians.
		 * @param	ay	Y-angle in radians.
		 * @param	az	Z-angle in radians.
		 *
		 * @return
		 */
		public static function createFromEuler( ax:Number, ay:Number, az:Number, useDegrees:Boolean = false ):Quaternion
		{
			var q:Quaternion = new Quaternion();
			q.setFromEuler(ax, ay, az, useDegrees);
			return q;
		}

		public function toEuler(useDegrees:Boolean = false):Number3D
		{
			return eulerSetTo(null, useDegrees);
		}

		/**
		 * オイラー角をeulerに出力し、返す
		 *
		 * @param	euler
		 * @param	useDegrees
		 * @return
		 */
		public function eulerSetTo(euler:Number3D = null, useDegrees:Boolean = false):Number3D
		{
			if (euler == null) euler = new Number3D();
			var q1:Quaternion = this;

			var test :Number = q1.x*q1.y + q1.z*q1.w;
			if (test > 0.499) { // singularity at north pole
				euler.x = 2 * Math.atan2(q1.x,q1.w);
				euler.y = Math.PI/2;
				euler.z = 0;
				return euler;
			}
			if (test < -0.499) { // singularity at south pole
				euler.x = -2 * Math.atan2(q1.x,q1.w);
				euler.y = - Math.PI/2;
				euler.z = 0;
				return euler;
			}

			var sqx	:Number = q1.x*q1.x;
			var sqy	:Number = q1.y*q1.y;
			var sqz	:Number = q1.z*q1.z;

			euler.x = Math.atan2(2*q1.y*q1.w-2*q1.x*q1.z , 1 - 2*sqy - 2*sqz);
			euler.y = Math.asin(2*test);
			euler.z = Math.atan2(2*q1.x*q1.w-2*q1.y*q1.z , 1 - 2*sqx - 2*sqz);
			if( useDegrees ) {
				euler.x *= toDEGREES;
				euler.y *= toDEGREES;
				euler.z *= toDEGREES;
			}

			return euler;
		}

		public static var tmpV:Number3D = new Number3D();

		/**
		 * v0からv1への回転クォータニオンにする
		 *
		 * @param	v0
		 * @param	v1
		 */
		public function setRotation(v0:Number3D, v1:Number3D):void
		{
			Number3D.crossNormalize(v0, v1, tmpV);
			setFromAxisAngle(tmpV.x, tmpV.y, tmpV.z, v0.angle(v1));
		}

		/**
		 * v := Q v P where Q := this, P := Q.conj, |Q|=1
		 */
		public function applyLeft(v:Number3D):void
		{
			/*
			qv = (qx,qy,qz), q = qv \mu + qw, v = (x,y,z) のとき u\mu = q v\mu q^* とすると u=(ux,uy,uz)
			(記法は http://cid-9da0fa00ac5a8258.skydrive.live.com/self.aspx/.Public/Document/DualQuaternion.pdf を参照)

			|q| = 1に限定して計算すると u = 2(qv v)qv + (2 qw^2 - 1)v + 2qw(qv × v).

			ここで、A = 2(qv v) = 2(qx x + qy y + qz z), B = qw^2 - qv^2 = qw^2 - qx^2 - qy^2 - qz^2, C = 2qw とすると
			ux = A qx + B x + C (qy z - qz y)
			uy = A qy + B y + C (qz x - qx z)
			uz = A qz + B z + C (qx y - qy x)
			*/
			var A:Number = 2 * (x * v.x + y * v.y + z * v.z);
			var C:Number = 2 * w;
			var B:Number = C * w - 1;
			var vx:Number = v.x;
			var vy:Number = v.y;
			var vz:Number = v.z;
			var ux:Number = A * x + B * vx + C * (y * vz - z * vy);
			var uy:Number = A * y + B * vy + C * (z * vx - x * vz);
			var uz:Number = A * z + B * vz + C * (x * vy - y * vx);
			v.x = ux;
			v.y = uy;
			v.z = uz;
		}

		/**
		 * v := P v Q where Q := this, P := Q.conj, |Q|=1
		 */
		public function applyRight(v:Number3D):void
		{
			/* applyLeftの計算でqx,qy,qzを-qx,-qy,-qzに置き換えればOK
			* 最終的にはCの符号を入れ替えればOK
			*/
			var A:Number = 2 * (x * v.x + y * v.y + z * v.z);
			var B:Number = 2 * w * w - 1;
			var C:Number = - 2 * w;
			var vx:Number = v.x;
			var vy:Number = v.y;
			var vz:Number = v.z;
			var ux:Number = A * x + B * vx + C * (y * vz - z * vy);
			var uy:Number = A * y + B * vy + C * (z * vx - x * vz);
			var uz:Number = A * z + B * vz + C * (x * vy - y * vx);
			v.x = ux;
			v.y = uy;
			v.z = uz;
		}

		/**
		 * v := k*(P(v-v0)Q + v0') where Q := this, P := Q.conj, |Q|=1
		 */
		public function transformVector(v:Number3D, v0:Number3D, v0_:Number3D, k:Number=1.0):void
		{
			v.minusEq(v0); // v := v-v0
			this.applyRight(v); // v := P v Q
			v.x = k*(v.x + v0_.x); // v := k*(v + v0')
			v.y = k*(v.y + v0_.y);
			v.z = k*(v.z + v0_.z);
		}

		/**
		 * v := k*(P(v-v0)Q + v0') where Q := this, P := Q.conj, |Q|=1
		 * tv += v の計算用.
		 *
		 * f = Q.getVectorTransformer(v0, v0');
		 * f(v,k); で上記の計算ができる
		 */
		public function getVectorTransformer(v0:Number3D, v0_:Number3D):Function
		{
			var B:Number = 2 * w * w - 1;
			var C:Number = - 2 * w;
			var v0x:Number = v0.x;
			var v0y:Number = v0.y;
			var v0z:Number = v0.z;
			var v0_x:Number = v0_.x;
			var v0_y:Number = v0_.y;
			var v0_z:Number = v0_.z;
			return function (tv:Number3D, sv:Number3D, k:Number = 1.0):void
			{
				// tv: ターゲット
				// sv: ソース
				var nx:Number = sv.x - v0x;
				var ny:Number = sv.y - v0y;
				var nz:Number = sv.z - v0z;
				var A:Number = 2 * (x * nx + y * ny + z * nz);
				tv.x += k * (A * x + B * nx + C * (y * nz - z * ny) + v0_x);
				tv.y += k * (A * y + B * ny + C * (z * nx - x * nz) + v0_y);
				tv.z += k * (A * z + B * nz + C * (x * ny - y * nx) + v0_z);
			}
		}

		/**
		 * 回転行列(3x3)の生成
		 */
		public function get matrix():Vector.<Vector.<Number>>
		{
			var xx:Number = x * x;
			var xy:Number = x * y;
			var xz:Number = x * z;
			var xw:Number = x * w;
			var yy:Number = y * y;
			var yz:Number = y * z;
			var yw:Number = y * w;
			var zz:Number = z * z;
			var zw:Number = z * w;

			var n11:Number = 1 - 2 * ( yy + zz );
			var n12:Number =     2 * ( xy - zw );
			var n13:Number =     2 * ( xz + yw );

			var n21:Number =     2 * ( xy + zw );
			var n22:Number = 1 - 2 * ( xx + zz );
			var n23:Number =     2 * ( yz - xw );

			var n31:Number =     2 * ( xz - yw );
			var n32:Number =     2 * ( yz + xw );
			var n33:Number = 1 - 2 * ( xx + yy );

			return new Vector.<Vector.<Number>>(new Vector.<Number>(n11,n12,n13), new Vector.<Number>(n21,n22,n23), new Vector.<Number>(n31,n32,n33));
		}
	}
}

