﻿/**
 * @author b2ox
 */
package org.b2ox.flash3d
{
	import flash.display.*;
	import flash.events.Event;
	import org.b2ox.flash3d.MikuMikuDance.*;
	import org.libspark.thread.*;
	import org.papervision3d.core.geom.renderables.Vertex3D;
	import org.papervision3d.core.geom.renderables.Triangle3D;
	import org.papervision3d.core.geom.TriangleMesh3D;
	import org.papervision3d.core.math.NumberUV;
	import org.papervision3d.materials.BitmapMaterial;
	import org.papervision3d.materials.BitmapColorMaterial;
	// import org.papervision3d.materials.utils.MaterialsList;
	import org.papervision3d.objects.DisplayObject3D;
	import org.tarotaro.flash.net.*;

	/**
	 * MikuMikuDance (model and motions) API Class.
	 *
	 * MikuMikuDanceのモデルデータ(*.pmd)とモーションデータ(*.vmd)を読み込み、再生するためのユーザー用APIクラス
	 */
	public class MikuMikuDance extends TriangleMesh3D
	{
		//---------------------------------------------------------------------
		// 内部的なパラメータ類
		private var _pmdController:PMDController = null;
		/**
		 * PMDControllerの取得.
		 * PMDLoaderThreadやモーション定義のときにつかう
		 */
		public function get pmdController():PMDController { return _pmdController; }
		private var _vmdControllers:Vector.<IVMDController> = new Vector.<IVMDController>();
		private var _vmdNames:Vector.<String> = new Vector.<String>();
		private var _vmdID:int = -1;
		private var _animation:IVMDController = null;
		private var _scaling:Number = 1.0;
		public function set scaling(val:Number):void { if (val > 0) _scaling = val; }
		public function get scaling():Number { return _scaling; }

		//---------------------------------------------------------------------
		/**
		 * コンストラクタ
		 */
		public function MikuMikuDance():void
		{
			super(null, new Array(), new Array(), null);
			_pmdController = new PMDController(this);
			if (!Thread.isReady) Thread.initialize(new EnterFrameThreadExecutor());
		}

		//---------------------------------------------------------------------
		// モーション再生用API
		public function play():void { if(_animation) _animation.play(); }
		public function stop():void { if(_animation) _animation.stop(); }
		public function togglePause():void { if(_animation) _animation.togglePause(); }
		public function gotoAndPlay(pos:Number):void { if (_animation) _animation.gotoAndPlay(pos); }
		public function gotoAndStop(pos:Number):void { if (_animation) _animation.gotoAndStop(pos); }

		public function get playing():Boolean { return _animation ? _animation.playing : false; }
		public function get looping():Boolean { return _animation ? _animation.looping : false; }
		public function set looping(loop:Boolean):void { if (_animation) _animation.looping = loop; }

		private var _motionName:String = "";
		/**
		 * モーション名からモーションIDを得る
		 * @param	motionName
		 * @return
		 */
		public function motionID_of(motionName:String):int { return _vmdNames.indexOf(motionName); }
		/**
		 * モーションIDからモーション名を得る
		 * @param	motionID
		 * @return
		 */
		public function motionName_of(motionID:int):String { return _vmdNames[motionID]; }

		/**
		 * モーション変更イベント用
		 */
		public static const MOTION_CHANGED:String = "MMD_MOTION_CHANGED";
		/**
		 * モーション追加イベント用
		 */
		public static const MOTION_ADDED:String = "MMD_MOTION_ADDED";
		/**
		 * モーション名を元にモーション変更
		 * @param	motion
		 */
		public function changeMotion(motion:String):void {
			var mID:int = motionID_of(motion);
			if (mID >= 0) {
				_animation = _vmdControllers[mID];
				_motionName = motion;
				_vmdID = mID;
				dispatchEvent(new Event(MOTION_CHANGED));
				trace("[DEBUG] MikuMikuDance.changeMotion: ID:" + _vmdID + " Name:" + _motionName );
			}
		}
		/**
		 * モーションIDを元にモーション変更
		 * @param	motion
		 */
		public function changeMotionByID(motionID:int):void {
			_animation = _vmdControllers[motionID];
			_motionName = motionName_of(motionID);
			_vmdID = motionID;
			dispatchEvent(new Event(MOTION_CHANGED));
			trace("[DEBUG] MikuMikuDance.changeMotion: ID:" + _vmdID + " Name:" + _motionName );
		}
		/**
		 * 次のIDのモーションに変更.
		 * 最後の場合は最初に戻る
		 */
		public function changeNextMotion():void {
			if (_vmdControllers.length > 1) {
				var mID:int = (_vmdID + 1) % _vmdNames.length;
				changeMotionByID(mID);
			}
		}
		/**
		 * モーションの追加.
		 * @param	motion	登録用のモーション名
		 * @param	ctrl	モーションコントローラ
		 */
		public function addMotion(motion:String, ctrl:IVMDController):void {
			trace("addMotion: "+motion);
			var mID:int = motionID_of(motion);
			if (mID < 0) {
				_vmdControllers.push(ctrl);
				_vmdNames.push(motion);
			} else {
				_vmdControllers[mID] = ctrl;
			}
			_vmdID = mID;
			_motionName = motion;
			_animation = ctrl;
			dispatchEvent(new Event(MOTION_ADDED));
		}
		/**
		 * モーションを非選択状態にする
		 */
		public function noMotion():void { _animation = null; _motionName = ""; }
		/**
		 * 現在のモーション名
		 */
		public function get motionName():String { return _motionName; }

		public function getMotion(motion:String):IVMDController {
			var mID:int = motionID_of(motion);
			return (mID >= 0) ? _vmdControllers[mID] : null;
		}

		/**
		 * モーションの再生位置(秒).
		 */
		public function get pos():Number { return _animation ? _animation.tween.position : 0; }

		//---------------------------------------------------------------------
		/**
		 * ポーズ変更の適用
		 */
		public function update():void { _pmdController.update(); }

		//---------------------------------------------------------------------
		/**
		 * InteractiveScene3DEventを受け取るかどうかを設定(Metasequoiaクラスのまね。よくわかってない)
		 */
		public var interactive:Boolean = false;

		/**
		 * PMDのバージョン
		 */
		public var version:Number;
		/**
		 * PMDのモデル名
		 */
		public var modelName:String;
		/**
		 * PMDのコメント
		 */
		public var comment:String;

		//---------------------------------------------------------------------
		/**
		 * PMDファイルの読み込み.
		 *
		 * @param	url			pmdファイルのurl
		 * @param	scaling		読み込み時のスケーリング
		 * @param	afterLoad	読み込み後に実行する関数
		 */
		public function loadPMD(url:String, scaling:Number=1.0, afterLoad:Function=null):void
		{
			makePMDLoader(url, scaling).$next( function ():void { if (afterLoad != null) afterLoad(); } ).start();
		}

		//---------------------------------------------------------------------
		/**
		 * PMDファイルの読み込みスレッドの作成.
		 *
		 * @param	url		pmdファイルのurl
		 * @param	scaling	読み込み時のスケーリング
		 * @return	スレッド
		 */
		public function makePMDLoader(url:String, scaling:Number=1.0):PMDLoaderThread
		{
			this._scaling = scaling;
			return new PMDLoaderThread(this, url, scaling);
		}

		//---------------------------------------------------------------------
		// 以下はPMDの読み込みで使用する頂点や面情報の登録用メソッド
		private var _uvarray:Vector.<NumberUV>;

		/**
		 * 頂点,UV配列の初期化.
		 * @param	vertex_count	頂点の個数
		 */
		public function initVertexUVarrays(vertex_count:uint):void
		{
			geometry.vertices = new Array(vertex_count);
			_uvarray = new Vector.<NumberUV>(vertex_count);
		}

		/**
		 * 頂点の登録.
		 * @param	i	頂点番号
		 * @param	x	X座標
		 * @param	y	Y座標
		 * @param	z	Z座標
		 */
		public function regVertex(i:uint, x:Number, y:Number, z:Number):void
		{
			geometry.vertices[i] = new Vertex3D(x, y, z);
			_pmdController.regVertex(i, x, y, z);
		}

		/**
		 * UVの登録.
		 * @param	i	頂点番号
		 * @param	u	U座標
		 * @param	v	V座標
		 */
		public function regUV(i:uint, u:Number, v:Number):void
		{
			_uvarray[i] = new NumberUV(u, v);
		}

		/**
		 * (三角形)面配列の初期化.
		 * @param	face_count	面数
		 */
		public function initTriangleArray(face_count:uint):void
		{
			geometry.faces = new Array(face_count);
		}

		/**
		 * 三角形の登録.
		 * 頂点は頂点配列のインデックスで指定
		 * @param	i	面番号
		 * @param	v0	頂点番号
		 * @param	v1	頂点番号
		 * @param	v2	頂点番号
		 */
		public function regTriangle(i:uint, v0:int, v1:int, v2:int):void
		{
			geometry.faces[i] = new Triangle3D(this, [geometry.vertices[v0], geometry.vertices[v1], geometry.vertices[v2]], null, [_uvarray[v0], _uvarray[v1], _uvarray[v2]]);
		}

		/**
		 * 材質リストの初期化.
		 */
		public function initMaterials():void {
			// materials = new MaterialsList();
		}

		/**
		 * 材質の登録.
		 *
		 * @param	matName		材質名
		 * @param	tex			テクスチャのBitmapData
		 * @param	fillColor	面の色
		 * @param	faceOffset	材質開始位置の面番号
		 * @param	count		対象面の個数
		 */
		public function regMaterialBase(matName:String, tex:BitmapData, fillColor:uint, faceOffset:uint, count:uint):void
		{
			var material:BitmapMaterial;
			if (tex != null)
			{
				material = new BitmapMaterial(tex, true);
			} else {
				material = new BitmapColorMaterial(fillColor);
			}
			material.interactive = this.interactive;
			material.smooth = true;
			material.name = matName;
			// materials.addMaterial(material, material.name);
			for (var i:int = 0; i < count; i++) geometry.faces[faceOffset + i].material = material;
		}

		private var spa_re:RegExp = /\*.+\.sp[ah]$/i; // ～.bmp*～.spa や .sph への対策
		/**
		 * 材質の登録.
		 *
		 * @param	matName		材質名
		 * @param	texPath		テクスチャファイルのURL
		 * @param	fillColor	面の色
		 * @param	faceOffset	材質開始位置の面番号
		 * @param	count		対象面の個数
		 */
		public function regMaterial(matName:String, texPath:String, fillColor:uint, faceOffset:uint, count:uint):void
		{
			if (texPath != "")
			{
				if (spa_re.test(texPath))
				{
					trace("警告: " + texPath + " の読み込みは現在未対応です");
					texPath = texPath.replace(spa_re, "");
				}
				var manager:ImageManager = new ImageManager(texPath);
				manager.addEventListener( ImageManagerEvent.LOAD_COMPLETE, function (evt:ImageManagerEvent):void
				{
					regMaterialBase(matName, evt.manager.bitmapData, fillColor, faceOffset, count);
				}
				);
				manager.getImage(texPath);
			} else {
				regMaterialBase(matName, null, fillColor, faceOffset, count);
			}
		}

		//---------------------------------------------------------------------
		private var _vmdScaling:Number = -1;
		/**
		 * VMD読み込み時のスケーリング設定.
		 *
		 * 未設定もしくは0以下の値の場合はPMD読み込み時のスケーリング値を使う
		 */
		public function set vmdScaling(v:Number):void { _vmdScaling = v; }
		public function get vmdScaling():Number { return _vmdScaling > 0 ? _vmdScaling : _scaling; }
		//---------------------------------------------------------------------
		/**
		 * VMDファイルを読み込んでモーションを登録する.
		 *
		 * @param	url			vmdファイルのurl
		 * @param	motion_name	モーション登録名
		 * @param	afterLoad	vmd読み込み後に実行する関数
		 */
		public function loadVMD(url:String, motion_name:String, afterLoad:Function = null):void
		{
			makeVMDLoader(url, motion_name).$next(
			function (vmd:VMDLoaderThread):void {
				if (afterLoad != null) afterLoad();
			} ).start();
		}

		//---------------------------------------------------------------------
		/**
		 * VMDファイルを読み込んでモーションを登録するスレッドの作成.
		 *
		 * @param	url			vmdファイルのurl
		 * @param	motion_name	モーション登録名
		 * @return	スレッド
		 */
		public function makeVMDLoader(url:String, motion_name:String):VMDLoaderThread
		{
			return new VMDLoaderThread(url, _pmdController, vmdScaling).$next(
			function (vmd:VMDLoaderThread):void { addMotion(motion_name, vmd.controller); }
			);
		}

		//---------------------------------------------------------------------
		/**
		 * ボーンにアクセサリを取り付ける.
		 *
		 * @param	boneName	取り付け先のボーン名
		 * @param	mdl			アクセサリ
		 * @param	mdlName		アクセサリの登録名
		 * @return	対象ボーンが存在しないときはfalse
		 */
		public function attachModel(boneName:String, mdl:DisplayObject3D, mdlName:String):Boolean
		{
			return _pmdController.attachModel(boneName, mdl, mdlName);
		}
		/**
		 * ボーンからアクセサリを取り外す.
		 *
		 * @param	boneName	ボーン名
		 * @param	mdlName		アクセサリ名
		 */
		public function removeModel(boneName:String, mdlName:String):void
		{
			_pmdController.removeModel(boneName, mdlName);
		}

		//---------------------------------------------------------------------
		/**
		 * 指定skinNameのweightを設定する.
		 * @param	skinName
		 * @param	weight
		 */
		public function setSkinWeight(skinName:String, weight:Number=1.0):void { _pmdController.setSkinWeight(skinName, weight); }

		/**
		 * 口の表情を設定する.
		 * @param	skinName
		 * @param	weight
		 */
		public function setLip(skinName:String, weight:Number = 1.0):void { setSkinType(skinName, weight, PMDSkin.TYPE_LIP); }
		/**
		 * 目の表情を設定する
		 * @param	skinName
		 * @param	weight
		 */
		public function setEye(skinName:String, weight:Number = 1.0):void { setSkinType(skinName, weight, PMDSkin.TYPE_EYE); }
		/**
		 * 眉の表情を設定する
		 * @param	skinName
		 * @param	weight
		 */
		public function setEyeBrow(skinName:String, weight:Number = 1.0):void { setSkinType(skinName, weight, PMDSkin.TYPE_EYEBROW); }
		/**
		 * 一旦skinTypeのパラメータをリセットしてからskinNameのweightを設定する.
		 * @param	skinName
		 * @param	weight
		 * @param	skinType
		 */
		private function setSkinType(skinName:String, weight:Number=1.0, skinType:int=-1):void
		{
			_pmdController.resetSkinWeightsByType(skinType);
			_pmdController.setSkinWeight(skinName, weight);
		}
		/**
		 * 指定skinTypeのパラメータを全てリセットする.
		 * @param	skinType
		 */
		public function resetSkinWeightsByType(skinType:int):void { _pmdController.resetSkinWeightsByType(skinType); }
		//---------------------------------------------------------------------
		//public function setBoneParam(boneName:String, rot:Quaternion, dv:Number3D):void { _pmdController.setVMDBoneParam(boneName, rot, dv); }
		/**
		 * 各ボーンの基点に立方体を表示する(基本的にデバッグ用).
		 */
		public function showBone():void { _pmdController.showBone(); }
	}
}

