﻿/*
  ==============================================================================

   This file is part of the async
   Copyright 2005-11 by Satoshi Fujiwara.

   async can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   async is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with async; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ==============================================================================
*/

// SDKのサンプルを改造した作ったもの。内容はほぼそのまま。

// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
// PARTICULAR PURPOSE.
//
// Copyright (c) Microsoft Corporation. All rights reserved.

//-----------------------------------------------------------
//
// wavefile.cpp
//
//   Wave file reader.
//
//-----------------------------------------------------------

#include "stdafx.h"
#include <stdio.h>
#include <assert.h>
#include <wtypes.h>
#include <mmreg.h>
#include <ks.h>
#include <ksmedia.h>
#include "wavefile.h"

namespace sf {
  // To parse a wave file, it is helpful to have two user-defined types.
  // The first will receive all the information in the file header and
  // in the header of the format chunk. The second will receive the
  // information in the headers for the data chunks, and, if the file
  // contains additional chunks, the headers for those chunks as well.

  typedef struct
  {
    ULONG Riff4CC;      // "RIFF" 4-character code
    ULONG FileSize;     // total file size in bytes
    ULONG Wave4CC;      // "WAVE" 4-character code
    ULONG Fmt4CC;       // "fmt " 4-character code
    ULONG FormatSize;   // wave format size in bytes
  } FileHeader;

  typedef struct
  {
    ULONG ChunkType;
    ULONG ChunkSize;
  } ChunkHeader;

  // Any file smaller than this cannot possibly contain wave data.
#define MIN_WAVE_FILE_SIZE (sizeof(FileHeader)+sizeof(PCMWAVEFORMAT)+sizeof(ChunkHeader)+1)

  // Macro to build FOURCC from first four characters in ASCII string
#define FOURCC(s)  ((ULONG)(s[0] | (s[1]<<8) | (s[2]<<16) | (s[3]<<24)))

  //
  // Constructor -- Open wave file and parse file header.
  //
  wave_file_reader::wave_file_reader(const std::wstring file_name, bool repeat_mode)
  {
    file_ = NULL;
    stream_status_ = S_OK;
    data_chunk_position = 0;
    total_data_bytes_ = 0;
    data_bytes_remaining_ = 0;
    repeat_mode_ = repeat_mode;
    ZeroMemory(&wfx_, sizeof(wfx_));
    try{
      // Try to open the wave file.
      if (_wfopen_s(&file_, file_name.c_str(), L"rb") != 0)
      {
        throw exception(std::wstring(L"ファイルをオープンできません。"));
      }

      // Copy header from wave file.
      FileHeader fileHdr;

      if (fread(&fileHdr, sizeof(fileHdr), 1, file_) != 1)
      {
        throw exception(std::wstring(L"ファイルを読み込むことができません。"));
      }

      // Verify that wave file header is valid.
      if (fileHdr.Riff4CC != FOURCC("RIFF") ||
        fileHdr.FileSize < MIN_WAVE_FILE_SIZE ||
        fileHdr.Wave4CC != FOURCC("WAVE") ||
        fileHdr.Fmt4CC != FOURCC("fmt ") ||
        fileHdr.FormatSize < sizeof(PCMWAVEFORMAT))
      {
        throw exception(std::wstring(L"サポートしていない.WAVファイルです。"));
      }

      // Copy wave format descriptor from file.
      if (fread(&wfx_, min(fileHdr.FormatSize,sizeof(wfx_)), 1, file_) != 1)
      {
        throw exception(std::wstring(L"サポートしていない.WAVファイルです。"));
      }

      // Skip over any padding at the end of the format in the format chunk.
      if (fileHdr.FormatSize > sizeof(wfx_))
      {
        if (fseek(file_, fileHdr.FormatSize-sizeof(wfx_), SEEK_CUR) != 0)
        {
          throw exception(std::wstring(L"ヘッダファイルのサイズが不正です。"));
        }
      }

      // If format type is PCMWAVEFORMAT, convert to valid WAVEFORMATEX structure.
      if (wfx_.Format.wFormatTag == WAVE_FORMAT_PCM)
      {
        wfx_.Format.cbSize = 0;
      }

      // If format type is WAVEFORMATEX, convert to WAVEFORMATEXTENSIBLE.
      if (wfx_.Format.wFormatTag == WAVE_FORMAT_PCM ||
        wfx_.Format.wFormatTag == WAVE_FORMAT_IEEE_FLOAT)
      {
        if (wfx_.Format.wFormatTag == WAVE_FORMAT_PCM)
        {
          wfx_.SubFormat = KSDATAFORMAT_SUBTYPE_PCM;
        }
        else
        {
          wfx_.SubFormat = KSDATAFORMAT_SUBTYPE_IEEE_FLOAT;
        }
        wfx_.Format.wFormatTag = WAVE_FORMAT_EXTENSIBLE;

        // Note that the WAVEFORMATEX structure is valid for
        // representing wave formats with only 1 or 2 channels.
        if (wfx_.Format.nChannels == 1)
        {
          wfx_.dwChannelMask = SPEAKER_FRONT_CENTER;
        }
        else if (wfx_.Format.nChannels == 2)
        {
          wfx_.dwChannelMask = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT;
        }
        else
        {
          throw exception(std::wstring(L"サポートしていない.WAVファイルです。"));
        }
        wfx_.Format.cbSize = sizeof(WAVEFORMATEXTENSIBLE) - sizeof(WAVEFORMATEX);
        wfx_.Samples.wValidBitsPerSample = wfx_.Format.wBitsPerSample;
      }

      // This wave file reader understands only PCM and IEEE float formats.
      if (wfx_.Format.wFormatTag != WAVE_FORMAT_EXTENSIBLE ||
        wfx_.SubFormat != KSDATAFORMAT_SUBTYPE_PCM &&
        wfx_.SubFormat != KSDATAFORMAT_SUBTYPE_IEEE_FLOAT)
      {
        throw exception(std::wstring(L"サポートしていない.WAVファイルです。"));
      }

      // Find chunk header for wave data. Skip past any other kinds of chunks.
      ChunkHeader chunkHdr;   // buffer for chunk header
      for (;;)
      {
        // Remember the file position of the data chunk (which this might be
        // -- we don't know yet). That way, if we need to play the file more
        // than once, we won't have to parse the header again each time.
        if (fgetpos(file_, &data_chunk_position) != 0)
        {
          throw exception(std::wstring(L"サポートしていない.WAVファイルです。"));
        }

        // Read header at start of next chunk of file.
        if (fread(&chunkHdr, sizeof(ChunkHeader), 1, file_) != 1)
        {
          throw exception(std::wstring(L"サポートしていない.WAVファイルです。"));
        }
        if (chunkHdr.ChunkType == FOURCC("data"))
        {
          break;  // found start of data chunk
        }
        // This is *not* a data chunk. Skip this chunk and go to the next chunk.
        if (fseek(file_, chunkHdr.ChunkSize, SEEK_CUR) != 0)
        {
          throw exception(std::wstring(L"サポートしていない.WAVファイルです。"));
        }
      }

      // We've found the start of the data chunk. We're ready to start
      // playing wave data...
      total_data_bytes_ = chunkHdr.ChunkSize;
      data_bytes_remaining_ = total_data_bytes_;
      if (total_data_bytes_ == 0)
      {
        throw exception(std::wstring(L"ファイルサイズが不正です。"));
      }
      stream_status_ = true;
    } catch (exception& e) {
      stream_status_ = false;
      throw;
    }
  }

  //
  // Destructor
  //
  wave_file_reader::~wave_file_reader()
  {
    if (file_)
    {
      fclose(file_);
    }
  }

  //
  // Reset the file pointer to the start of the wave data.
  //
  void wave_file_reader::reset_data_position()
  {
    if (!stream_status_ )
    {
      throw wave_file_reader::exception(std::wstring(L"ファイルがオープンされていません。"));
    }

    // Move to the header info at the start of the data chunk.
    if (fsetpos(file_, &data_chunk_position) != 0)
    {
      throw exception(std::wstring(L"不正なWAVファイルです。"));
    }

    // Read the header for the data chunk.
    ChunkHeader chunkHdr;
    if (fread(&chunkHdr, sizeof(chunkHdr), 1, file_) != 1)
    {
      throw exception(std::wstring(L"不正なWAVファイルです。"));
    }

    // Sanity check: The chunk header shouldn't have changed.
    if (chunkHdr.ChunkType != FOURCC("data") ||
      chunkHdr.ChunkSize != total_data_bytes_)
    {
      throw exception(std::wstring(L"不正なWAVファイルです。"));
    }
    data_bytes_remaining_ = total_data_bytes_;

    // At this point, the file pointer is positioned at
    // the beginning of the wave data.
  }

  void wave_file_reader::seek(uint64_t pos)
  {
    if (!stream_status_ )
    {
      throw wave_file_reader::exception(std::wstring(L"ファイルがオープンされていません。"));
    }
    fpos_t current;
    fgetpos(file_,&current);
    pos = (pos / get_wave_format().Format.nBlockAlign) * get_wave_format().Format.nBlockAlign;
    data_bytes_remaining_ = total_data_bytes_ - pos;
    pos = pos + data_chunk_position + sizeof(ChunkHeader);
    fsetpos(file_,reinterpret_cast<fpos_t*>(&pos));
  }


  //
  // Load next block of wave data from file into playback buffer.
  // In repeat mode, when we reach the end of the wave data in the
  // file, we just reset the file pointer back to the start of the
  // data and continue filling the caller's buffer until it is full.
  // In single-play mode, once we reach the end of the wave data in
  // the file, we just fill the buffer with silence instead of with
  // real data.
  //
  void wave_file_reader::read_data(BYTE *buffer, uint64_t numbytes)
  {

    if (!stream_status_ )
    {
      throw wave_file_reader::exception(std::wstring(L"ファイルがオープンされていません。"));
    }

    if (buffer == NULL)
    {
      throw exception(L"バッファアドレスが無効です。");
    }

    if (numbytes == 0)
    {
      throw exception(L"読み込みバイト数の指定が0です。");
    }

    BYTE *current = (BYTE*)buffer;
    ULONG numbytes_to_copy = numbytes;

    while (numbytes_to_copy > data_bytes_remaining_)
    {
      if (fread(current, 1, data_bytes_remaining_, file_) != data_bytes_remaining_)
      {
        throw exception(L"不正なWAVファイルです。");
      }
      current += data_bytes_remaining_;
      numbytes_to_copy -= data_bytes_remaining_;
      data_bytes_remaining_ = 0;

      // The file pointer now sits at the end of the data chunk.
      // Are we operating in repeat mode?
      if (!repeat_mode_)
      {
        // Nope, we're operating in single-play mode. Fill
        // the rest of the buffer with silence and return.
        BYTE silence = (wfx_.Format.wBitsPerSample==8) ? 0x80 : 0;
        memset(current, silence, numbytes_to_copy);
        return;  // yup, we're done
      }
      // Yes, we're operating in repeat mode, so loop back to
      // the start of the wave data in the file's data chunk
      // and continue loading data into the caller's buffer.
      reset_data_position();

    }

    assert(numbytes_to_copy > 0);
    assert(numbytes_to_copy <= data_bytes_remaining_);

    // The remainder of the data chunk is big enough to
    // completely fill the remainder of the caller's buffer.
    if (fread(buffer, 1, numbytes_to_copy, file_) != numbytes_to_copy)
    {
      throw exception(L"不正なWAVファイルです。");
    }
    data_bytes_remaining_ -= numbytes_to_copy;
    current += numbytes_to_copy;
  }
}
