﻿/*
  win32_key.cpp
  screen driver for win32

  Made by Studio Breeze. 2002

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
 */
#include "stdafx.h"

#include "sted_screen_win32.h"

// nofitication from main window
void
CSTedScreenWin32::NotifyKeyPressed(int in_key)
{
  fKeyEventNotified = TRUE;

  if (in_key == fKeyConv[EShift])
    fKeyShiftPressed = TRUE;
  else if (in_key == fKeyConv[EControl])
    fKeyControlPressed = TRUE;
  else if (in_key == fKeyConv[EOPT1])
    fKeyOPT1Pressed = TRUE;
  else if (in_key == fKeyConv[EOPT2])
    fKeyOPT2Pressed = TRUE;
  else if (in_key == fKeyConv[EKana])
    fKeyKanaPressed = TRUE;
  else if (in_key == fKeyConv[EInsert])
    fKeyInsertPressed = TRUE;

  else {
	  switch (in_key) {
		  case VK_PRIOR:
		  case VK_NEXT:
		  case VK_HOME:
		  case VK_END:
		  case VK_UP:
		  case VK_DOWN:
		  case VK_LEFT:
		  case VK_RIGHT:
		  case VK_F1:
		  case VK_F2:
		  case VK_F3:
		  case VK_F4:
		  case VK_F5:
		  case VK_F6:
		  case VK_F7:
		  case VK_F8:
		  case VK_F9:
		  case VK_F10:
		  case VK_F11:
		  case VK_F12:
		  case VK_LMENU:
		  case VK_RMENU:
		  case VK_LCONTROL:
		  case VK_RCONTROL:
		  case VK_NONCONVERT:
		  case 0xd8: // for Kigo
		  case 0xd9: // for Toroku
			fKeyBuffer[fKeyBufferEndPtr] = (in_key | 0x100);
			fKeyBufferEndPtr = (fKeyBufferEndPtr+1)%fKeyBufferLen;
			if (fKeyBufferEndPtr==fKeyBufferStartPtr) {
				fKeyBufferStartPtr = (fKeyBufferStartPtr+1)%fKeyBufferLen;
			}
			fKeyBufferAvailable++;
			if (fKeyBufferAvailable>=fKeyBufferLen) {
				fKeyBufferAvailable = fKeyBufferLen;
			}
			break;
		  default:
			  fKeyEventNotified = FALSE;
			  break;
	  }
  }
}

void
CSTedScreenWin32::NotifyKeyReleased(int in_key)
{
  fKeyEventNotified = TRUE;

  if (in_key == fKeyConv[EShift])
    fKeyShiftPressed = FALSE;
  else if (in_key == fKeyConv[EControl])
    fKeyControlPressed = FALSE;
  else if (in_key == fKeyConv[EOPT1])
    fKeyOPT1Pressed = FALSE;
  else if (in_key == fKeyConv[EOPT2])
    fKeyOPT2Pressed = FALSE;
  else if (in_key == fKeyConv[EKana])
    fKeyKanaPressed = FALSE;
  else if (in_key == fKeyConv[EInsert])
    fKeyInsertPressed = FALSE;
}

void
CSTedScreenWin32::NotifyChar(int in_char)
{
	int len=0;
	WCHAR buf[2];
	unsigned char str[6];
	int ptr;

	fKeyEventNotified = TRUE;

	buf[0] = in_char;
	buf[1] = '\0';

#ifdef _UNICODE
	len = ::WideCharToMultiByte(932, 0, (LPCWSTR)buf, 1, (LPSTR)str, 6, NULL, NULL);
#else
	len = 1;
	str[0] = (unsigned char)in_char;
#endif

	ptr=0;
	while (len>0) {
		fKeyBuffer[fKeyBufferEndPtr] = str[ptr++];
		fKeyBufferEndPtr = (fKeyBufferEndPtr+1)%fKeyBufferLen;
		if (fKeyBufferEndPtr==fKeyBufferStartPtr) {
			fKeyBufferStartPtr = (fKeyBufferStartPtr+1)%fKeyBufferLen;
		}
		fKeyBufferAvailable++;
		if (fKeyBufferAvailable>=fKeyBufferLen) {
			fKeyBufferAvailable = fKeyBufferLen;
		}
		len--;
	}
}

// key
int 
CSTedScreenWin32::KeyInit(void)
{
  fKeyBufferStartPtr = 0;
  fKeyBufferEndPtr = 0;
  fKeyBufferAvailable = 0;

  fKeyShiftPressed = FALSE;
  fKeyControlPressed = FALSE;
  fKeyOPT1Pressed = FALSE;
  fKeyOPT2Pressed = FALSE;
  fKeyKanaPressed = FALSE;
  fKeyInsertPressed = FALSE;

  fKeyEventNotified = FALSE;

  // default configuration for IBM-106

  fKeyConv[EShift] = VK_SHIFT;
  fKeyConv[EControl] = VK_CONTROL;
  fKeyConv[EOPT1] = VK_F11;
  fKeyConv[EOPT2] = VK_F12;
  fKeyConv[EXF1] = VK_LMENU;
  fKeyConv[EXF2] = VK_NONCONVERT;
  fKeyConv[EXF3] = VK_CONVERT;
  fKeyConv[EXF4] = VK_KANA;
  fKeyConv[EXF5] = VK_RMENU;
  fKeyConv[EKana] = VK_LCONTROL;
  fKeyConv[EKigo] = 0xd8; // unassigned
  fKeyConv[EToroku] = 0xd9; // unassigned
  fKeyConv[EInsert] = VK_INSERT;
  fKeyConv[EDel] = VK_DELETE;
  fKeyConv[EHome] = VK_HOME;
  fKeyConv[EUndo] = VK_END;
  fKeyConv[ERollUp] = VK_PRIOR;
  fKeyConv[ERollDown] = VK_NEXT;
  fKeyConv[EF1] = VK_F1;
  fKeyConv[EF2] = VK_F2;
  fKeyConv[EF3] = VK_F3;
  fKeyConv[EF4] = VK_F4;
  fKeyConv[EF5] = VK_F5;
  fKeyConv[EF6] = VK_F6;
  fKeyConv[EF7] = VK_F7;
  fKeyConv[EF8] = VK_F8;
  fKeyConv[EF9] = VK_F9;
  fKeyConv[EF10] = VK_F10;

  return 0;
}

// blocking
int 
CSTedScreenWin32::KeyIn(int in_code)
{
  int result;
  int key;

 loop:
  if (fKeyBufferAvailable<=0) return 0;

  key = fKeyBuffer[fKeyBufferStartPtr];
  result = ConvertWinKeyToX68Key(key);

  if (result<=0) {
    fKeyBufferAvailable--;
	fKeyBufferStartPtr = (fKeyBufferStartPtr+1)%fKeyBufferLen;
	goto loop;
  }


  if (in_code!=0xfe) {
    fKeyBufferAvailable--;
	fKeyBufferStartPtr = (fKeyBufferStartPtr+1)%fKeyBufferLen;
  }

  return result;
}


// キー入力blockingモード
int
CSTedScreenWin32::KeyInp(void)
{
  BOOL result;
  int ascii, code;
  int key;

loop:
  do {
	  if (!fKeyEventNotified) 
	  {
		  DoMessageLoop();
	  }
	  fKeyEventNotified = FALSE;
  } while(fKeyBufferAvailable <= 0);

  key = fKeyBuffer[fKeyBufferStartPtr];
  result = ConvertWinKeyToX68Key(key, &ascii, &code);
  if (!result) {
    fKeyBufferAvailable--;
	fKeyBufferStartPtr = (fKeyBufferStartPtr+1)%fKeyBufferLen;
	if (fKeyBufferAvailable>0)
		fKeyEventNotified = TRUE;
    goto loop;
  }

  fKeyBufferAvailable--;
  fKeyBufferStartPtr = (fKeyBufferStartPtr+1)%fKeyBufferLen;
  if (fKeyBufferAvailable>0) fKeyEventNotified = TRUE;

  return (ascii | code<<8);
}

int
CSTedScreenWin32::SftSense(void)
{
  int ret=0;

  if (fKeyShiftPressed) ret|=1;
  if (fKeyControlPressed) ret|=2;
  if (fKeyOPT1Pressed) ret|=4;
  if (fKeyOPT2Pressed) ret|=8;
  if (fKeyKanaPressed) ret|=16+256;
  if (fKeyInsertPressed) ret|=4096;

  if (!(::GetKeyState(VK_SHIFT)&0xff00)) {
	  fKeyShiftPressed = FALSE;
  }

  return ret;
}

BOOL
CSTedScreenWin32::CheckKeyBuffer(int in_key)
{
    int val;
    val = ::GetKeyState(in_key);
    if (val&0xff00) return TRUE;
    return FALSE;
}

int 
CSTedScreenWin32::BitSense(int in_group)
{
  int ret = 0;

  switch ( in_group ) {
  case 0:  /* 1 - 6 */
    if (CheckKeyBuffer('1')) ret|=4;
    if (CheckKeyBuffer('2')) ret|=8;
    if (CheckKeyBuffer('3')) ret|=16;
    if (CheckKeyBuffer('4')) ret|=32;
    if (CheckKeyBuffer('5')) ret|=64;
    if (CheckKeyBuffer('6')) ret|=128;
    break;

  case 1:
    if (CheckKeyBuffer('7')) ret|=1;
    if (CheckKeyBuffer('8')) ret|=2;
    if (CheckKeyBuffer('9')) ret|=4;
    if (CheckKeyBuffer('0')) ret|=8;
    if (CheckKeyBuffer('-')) ret|=16;
    if (CheckKeyBuffer('^')) ret|=32;
    if (CheckKeyBuffer('\\')) ret|=64;
    if (CheckKeyBuffer(VK_BACK)) ret|=128;
    break;

  case 2:
    if (CheckKeyBuffer(VK_TAB)) ret|=1;
    if (CheckKeyBuffer('Q')) ret|=2;
    if (CheckKeyBuffer('W')) ret|=4;
    if (CheckKeyBuffer('E')) ret|=8;
    if (CheckKeyBuffer('R')) ret|=16;
    if (CheckKeyBuffer('T')) ret|=32;
    if (CheckKeyBuffer('Y')) ret|=64;
    if (CheckKeyBuffer('U')) ret|=128;
    break;

  case 3:
    if (CheckKeyBuffer('I')) ret|=1;
    if (CheckKeyBuffer('O')) ret|=2;
    if (CheckKeyBuffer('P')) ret|=4;
    if (CheckKeyBuffer('@')) ret|=8;
    if (CheckKeyBuffer('[')) ret|=16;
    if (CheckKeyBuffer(VK_RETURN)) ret|=32;
    if (CheckKeyBuffer('A')) ret|=64;
    if (CheckKeyBuffer('S')) ret|=128;
    break;

  case 4:
    if (CheckKeyBuffer('D')) ret|=1;
    if (CheckKeyBuffer('F')) ret|=2;
    if (CheckKeyBuffer('G')) ret|=4;
    if (CheckKeyBuffer('H')) ret|=8;
    if (CheckKeyBuffer('J')) ret|=16;
    if (CheckKeyBuffer('K')) ret|=32;
    if (CheckKeyBuffer('L')) ret|=64;
    if (CheckKeyBuffer(';')) ret|=128;
    break;

  case 5:
    if (CheckKeyBuffer(':')) ret|=1;
    if (CheckKeyBuffer(']')) ret|=2;
    if (CheckKeyBuffer('Z')) ret|=4;
    if (CheckKeyBuffer('X')) ret|=8;
    if (CheckKeyBuffer('C')) ret|=16;
    if (CheckKeyBuffer('V')) ret|=32;
    if (CheckKeyBuffer('B')) ret|=64;
    if (CheckKeyBuffer('N')) ret|=128;
    break;

  case 6:
    if (CheckKeyBuffer('M')) ret|=1;
    if (CheckKeyBuffer(',')) ret|=2;
    if (CheckKeyBuffer('.')) ret|=4;
    if (CheckKeyBuffer('/')) ret|=8;
    if (CheckKeyBuffer('\\')) ret|=16;
    if (CheckKeyBuffer(' ')) ret|=32;
    break;

  case 7: /* R_UP,R_DOWN, Cursors */
    if (CheckKeyBuffer(VK_PRIOR)) ret|=1;
    if (CheckKeyBuffer(VK_NEXT)) ret|=2;
    if (CheckKeyBuffer(VK_END)) ret|=4;
    if (CheckKeyBuffer(VK_LEFT)) ret|=8;
    if (CheckKeyBuffer(VK_UP)) ret|=16;
    if (CheckKeyBuffer(VK_RIGHT)) ret|=32;
    if (CheckKeyBuffer(VK_DOWN)) ret|=64;
    break;
#if 0
  case 0x0a: /* XF1-XF3 */
    if (check_keycode(k_xf1      , key_buffer)) ret|=32;
    if (check_keycode(k_xf2      , key_buffer)) ret|=64;
    if (check_keycode(k_xf3      , key_buffer)) ret|=128;
    break;

  case 0x0b: /* XF4-XF5 */
    if (check_keycode(k_xf4      , key_buffer)) ret|=1;
    if (check_keycode(k_xf5      , key_buffer)) ret|=2;
    break;
#endif
  case 0x0c: /* F0-F4 */
    if (CheckKeyBuffer(VK_F1)) ret|=8;
    if (CheckKeyBuffer(VK_F2)) ret|=16;
    if (CheckKeyBuffer(VK_F3)) ret|=32;
    if (CheckKeyBuffer(VK_F4)) ret|=64;
    if (CheckKeyBuffer(VK_F5)) ret|=128;
    break;

  case 0x0d: /* F5-F9 */
    if (CheckKeyBuffer(VK_F6)) ret|=1;
    if (CheckKeyBuffer(VK_F7)) ret|=2;
    if (CheckKeyBuffer(VK_F8)) ret|=4;
    if (CheckKeyBuffer(VK_F9)) ret|=8;
    if (CheckKeyBuffer(VK_F10)) ret|=16;
    break;

  case 0x0e: /* OPT.1-OPT.2 */
    if (CheckKeyBuffer(VK_F11)) ret|=4;
    if (CheckKeyBuffer(VK_F12)) ret|=8;
    break;

  default:
    break;
  }

  return ret;
}

int
CSTedScreenWin32::KeySense(void)
{
	if (fKeyBufferAvailable==0) return 0;
	return 1;
}

void
CSTedScreenWin32::KeyWait(void)
{
	while (!fKeyEventNotified && fKeyBufferAvailable==0) {
		DoMessageLoop();
	}
	fKeyEventNotified = FALSE;
}

void
CSTedScreenWin32::LedMode(int in_code, int in_onoff)
{
}

void
CSTedScreenWin32::ClearKeyBuffer(void)
{
	fKeyBufferAvailable = 0;
	fKeyBufferStartPtr = 0;
	fKeyBufferEndPtr = 0;
}

// key conversion
int
CSTedScreenWin32::ConvertWinKeyToX68Key(int in_winkey)
{
  int result;

  if (in_winkey<0x100) { // char
	  return in_winkey;
  }

  in_winkey&=0xff;
  switch (in_winkey) {
  case VK_UP:
    result = fFncKey[4][0];
    break;
  case VK_DOWN:
    result = fFncKey[7][0];
    break;
  case VK_LEFT:
    result = fFncKey[5][0];
    break;
  case VK_RIGHT:
    result = fFncKey[6][0];
    break;
  case VK_BACK:
    result = 0x08;
    break;
  default:
    if (in_winkey == fKeyConv[EHome])
      result = fFncKey[10][0];
    else if (in_winkey == fKeyConv[EUndo])
      result = fFncKey[11][0];
    else if (in_winkey == fKeyConv[EDel])
      result = fFncKey[3][0];
    else if (in_winkey == fKeyConv[EInsert])
      result = fFncKey[2][0];
    else if (in_winkey == fKeyConv[ERollDown])
      result = fFncKey[0][0];
    else if (in_winkey == fKeyConv[ERollUp])
      result = fFncKey[1][0];
    else {
      result = in_winkey;
    }
    break;
  }

  return result;
}

int
CSTedScreenWin32::ConvertWinKeyToX68Key(int in_winkey, int* out_ascii, int* out_code)
{
  int ascii=0, code=0;
  int result=1;

  if (in_winkey<0x100) {
	  if (in_winkey>=0x20 && in_winkey<=0x7f) {
		  code = fX68ScanCode[in_winkey-0x20];
	  } else {
		  switch (in_winkey) {
			  case 0x1b: code = 0x01;break;
			  case 0x0d: code = 0x1d;break;
			  case 0x09: code = 0x10;break;
			  default: code = 0; break;
		  }
	  }
	  if (out_ascii) *out_ascii = in_winkey;
	  if (out_code) *out_code = code;
	  return 1;
  }

  in_winkey&=0xff;

  switch (in_winkey) {
  case VK_UP:
    ascii=0; code=0x3c; break;
    break;
  case VK_DOWN:
    ascii=0; code=0x3e; break;
    break;
  case VK_LEFT:
    ascii=0; code=0x3b; break;
    break;
  case VK_RIGHT:
    ascii=0; code=0x3d; break;
    break;
  case VK_BACK:
    ascii=0; code=0x0f; break;
    break;
  case VK_RETURN:
    ascii=0x0d; code=0x1d; break;
    break;
  case VK_ESCAPE:
    ascii=0x1b; code=0x01; break;
    break;
  case VK_TAB:
    ascii=0x09; code=0x10; break;
    break;

  case VK_SHIFT:
    ascii=0x00; code=fKeyShiftPressed?0x70:0xf0; break;
    break;
  case VK_CONTROL:
    ascii=0x00; code=fKeyControlPressed?0x71:0xf1; break;
    break;
  default:
    if (in_winkey == fKeyConv[EF1]) {
      ascii=0x00; code=0x63;
    } else if (in_winkey == fKeyConv[EF2]) {
      ascii=0x00; code=0x64;
    } else if (in_winkey == fKeyConv[EF3]) {
      ascii=0x00; code=0x65;
    } else if (in_winkey == fKeyConv[EF4]) {
      ascii=0x00; code=0x66;
    } else if (in_winkey == fKeyConv[EF5]) {
      ascii=0x00; code=0x67;
    } else if (in_winkey == fKeyConv[EF6]) {
      ascii=0x00; code=0x68;
    } else if (in_winkey == fKeyConv[EF7]) {
      ascii=0x00; code=0x69;
    } else if (in_winkey == fKeyConv[EF8]) {
      ascii=0x00; code=0x6a;
    } else if (in_winkey == fKeyConv[EF9]) {
      ascii=0x00; code=0x6b;
    } else if (in_winkey == fKeyConv[EF10]) {
      ascii=0x00; code=0x6c;

    } else if (in_winkey == fKeyConv[EKigo]) {
      ascii=0x00; code=0x52;
    } else if (in_winkey == fKeyConv[EToroku]) {
      ascii=0x00; code=0x53;

    } else if (in_winkey == fKeyConv[EXF1]) {
      ascii=0x00; code=0x55;
    } else if (in_winkey == fKeyConv[EXF2]) {
      ascii=0x00; code=0x56;
    } else if (in_winkey == fKeyConv[EXF3]) {
      ascii=0x00; code=0x57;
    } else if (in_winkey == fKeyConv[EXF4]) {
      ascii=0x00; code=0x58;
    } else if (in_winkey == fKeyConv[EXF5]) {
      ascii=0x00; code=0x59;

    } else if (in_winkey == fKeyConv[EHome]) {
      ascii=0x00; code=0x36;
    } else if (in_winkey == fKeyConv[EDel]) {
      ascii=0x00; code=0x37;
    } else if (in_winkey == fKeyConv[EInsert]) {
      ascii=0x00; code=0x5e;
    } else if (in_winkey == fKeyConv[ERollDown]) {
      ascii=0x00; code=0x38;
    } else if (in_winkey == fKeyConv[ERollUp]) {
      ascii=0x00; code=0x39;
    } else if (in_winkey == fKeyConv[EUndo]) {
      ascii=0x00; code=0x3a;
    } else if (in_winkey == fKeyConv[EOPT1]) {
      ascii=0x00; code=(fKeyOPT1Pressed)?0x72:0xf2;
    } else if (in_winkey == fKeyConv[EOPT2]) {
      ascii=0x00; code=(fKeyOPT2Pressed)?0x72:0xf2;

    } else if (in_winkey == VK_NUMPAD0) {
      ascii='0'; code=0x4f;
    } else if (in_winkey == VK_NUMPAD1) {
      ascii='1'; code=0x4b;
    } else if (in_winkey == VK_NUMPAD2) {
      ascii='2'; code=0x4c;
    } else if (in_winkey == VK_NUMPAD3) {
      ascii='3'; code=0x4d;
    } else if (in_winkey == VK_NUMPAD4) {
      ascii='4'; code=0x47;
    } else if (in_winkey == VK_NUMPAD5) {
      ascii='5'; code=0x48;
    } else if (in_winkey == VK_NUMPAD6) {
      ascii='6'; code=0x49;
    } else if (in_winkey == VK_NUMPAD7) {
      ascii='7'; code=0x43;
    } else if (in_winkey == VK_NUMPAD8) {
      ascii='8'; code=0x44;
    } else if (in_winkey == VK_NUMPAD9) {
      ascii='9'; code=0x45;
    } else if (in_winkey == VK_DECIMAL) {
      ascii='.'; code=0x51;
    } else if (in_winkey == VK_DIVIDE) {
      ascii='/'; code=0x40;
    } else if (in_winkey == VK_MULTIPLY) {
      ascii='*'; code=0x41;
    } else if (in_winkey == VK_SUBTRACT) {
      ascii='-'; code=0x42;
    } else if (in_winkey == VK_ADD) {
      ascii='+'; code=0x46;

	} else if (in_winkey == VK_OEM_2) {
		ascii='/'; code=fX68ScanCode[ascii-0x20];
	} else if (in_winkey == VK_OEM_102) {
		ascii='\\'; code=fX68ScanCode[ascii-0x20];

    } else if (in_winkey>=0x20 && in_winkey<0x7f) {
      ascii = in_winkey;
      code = fX68ScanCode[in_winkey-0x20];

    } else {
      ascii=0; code=0;
	  result = 0;
    }
    break;
  }

  if (out_ascii) *out_ascii = ascii;
  if (out_code) *out_code = code;
  return result;
}
