package jp.gr.java_conf.ykhr.eclipse.plugin.autocp;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.ui.IStartup;


public class LibraryManager implements IStartup {

    private static final List<String> LIB_EXT = Collections.unmodifiableList(
            Arrays.asList("jar", "zip"));
    private static final String LIBRARY_DIR_NAME = "lib";
    
    public void earlyStartup() {
        ResourcesPlugin.getWorkspace().addResourceChangeListener(new IResourceChangeListener() {
            public void resourceChanged(IResourceChangeEvent event) {
                try {
                    event.getDelta().accept(new IResourceDeltaVisitor() {
                        public boolean visit(IResourceDelta delta) throws CoreException {
                            boolean removed = (delta.getKind() & IResourceDelta.REMOVED) != 0;
                            boolean added = (delta.getKind() & IResourceDelta.ADDED) != 0;
                            
                            if (removed || added) {
                                IResource resource = delta.getResource();
                                if ((resource.getType() & IResource.FILE) != 0 
                                        && isTargetLibrary((IFile) resource)) {
                                    
                                    if (removed) {
                                        removeClasspath((IFile) resource);
                                    } else if (added) {
                                        addClasspath((IFile) resource);
                                    }
                                    return false;
                                }
                            }
                            return true;
                        }
                    });
                } catch (CoreException e) {
                    Activator.logError(e);
                }
            }
        }, IResourceChangeEvent.PRE_BUILD);
    }
    
    private boolean isTargetLibrary(IFile file) throws CoreException {
        IProject project = file.getProject();
        IPath dirPath = file.getParent().getFullPath();
        IPath libPath = project.getFullPath().append(LIBRARY_DIR_NAME);
        
        return project.exists() 
                && project.hasNature(Activator.NATURE_ID)
                && LIB_EXT.contains(file.getFileExtension())
                && dirPath.equals(libPath);
    }
    
    private void removeClasspath(IFile resource) throws JavaModelException {
        IProject project = resource.getProject();
        IJavaProject javaProject = (IJavaProject) project.getAdapter(IJavaElement.class);
        
        IClasspathEntry[] classpathEntries = javaProject.getRawClasspath();
        if (classpathEntries == null) {
            return;
        }
        
        List<IClasspathEntry> newEntries = new ArrayList<IClasspathEntry>();
        for (IClasspathEntry entry : classpathEntries) {
            if (entry.getEntryKind() == IClasspathEntry.CPE_LIBRARY 
                    && entry.getPath().equals(resource.getFullPath())) {
                continue;    
            }
            newEntries.add(entry);
        }
        
        setClasspath(javaProject, newEntries);
    }
    
    private void addClasspath(IFile resource) throws JavaModelException {
        IProject project = resource.getProject();
        IJavaProject javaProject = (IJavaProject) project.getAdapter(IJavaElement.class);
        
        IClasspathEntry[] classpathEntries = javaProject.getRawClasspath();
        if (classpathEntries == null) {
            classpathEntries = new IClasspathEntry[0];
        }
        
        boolean exist = false;
        List<IClasspathEntry> newEntries = new ArrayList<IClasspathEntry>();
        
        for (IClasspathEntry entry : classpathEntries) {
            if (entry.getEntryKind() == IClasspathEntry.CPE_LIBRARY 
                    && entry.getPath().equals(resource.getFullPath())) {
                exist = true;    
            }
            newEntries.add(entry);
        }
        
        if (!exist) {
            IClasspathEntry entry = JavaCore.newLibraryEntry(resource.getFullPath(), null, null);
            newEntries.add(entry);
            setClasspath(javaProject, newEntries);
        }
    }
    
    private void setClasspath(IJavaProject javaProject, List<IClasspathEntry> newEntries) throws JavaModelException {
        IClasspathEntry[] entries = new IClasspathEntry[newEntries.size()];
        newEntries.toArray(entries);
        javaProject.setRawClasspath(entries, null);
    }
    
}
