
#include <assert.h>

#include <vector>

#include "gl/lineDrawer8_Bresenham.h"
#include "gl/lineDrawer8_DDA.h"
#include "gl/lineDrawer8_Wu.h"
#include "gl/lineDrawer8_Nelson.h"
#include "gl/lineDrawer8_Shift.h"
#include "gl/clipper.h"

#include "shift.h"
#include <boost/random.hpp>
#include <boost/integer_traits.hpp>
#include "timer.h"
#include "xs_Float.h"

namespace {

HWND hWnd;
HBITMAP hBMP;
std::vector<uint8_t> bmiBuff(sizeof(BITMAPINFO) + sizeof(RGBQUAD) * 256);
//BITMAPINFO bmi;
void* pBits;
HDC hMemDC;

gl::Buffer2D<uint8_t> buff;
gl::LineDrawer8_Bresenham<float> lineDrawer_Bresenham;
gl::LineDrawer8_DDA<float> lineDrawer_DDA;
gl::LineDrawer8_Wu<float> lineDrawer_Wu;
gl::LineDrawer8_Nelson<float> lineDrawer_Nelson;
gl::LineDrawer8_Shift<float> lineDrawer_Shift;

gl::IBufferLineDrawer<float, uint8_t>* pLineDrawer = &lineDrawer_Bresenham;
gl::Clipper<float> clipper;
Timer timer;

float x1 = 512;
float y1 = 512;
float x2 = 1024;
float y2 = 1024;
float* px = &x1;
float* py = &y1;
float* pax = &x2;
float* pay = &y2;

void (*pRenderRoutine)();
void callRenderRoutine()
{
	::SetBkMode(hMemDC, TRANSPARENT);
	::SetTextColor(hMemDC, RGB(255,255,255));
	gl::Buffer2D_Fill<uint8_t>(buff, 0);
	pLineDrawer->SetBuffer(&buff);
	
	timer.Start();
	(*pRenderRoutine)();
	int64_t elapsed = timer.Elapsed();
	TCHAR buff[64];
	
	LARGE_INTEGER li;
	QueryPerformanceFrequency(&li);
	RECT rec = {10, 1000, 140, 1100};
	_stprintf(buff, _T("%f ms"), elapsed * 1000.0 / (double)li.QuadPart);
	::DrawText(hMemDC, buff, -1, &rec, DT_LEFT);
	
	::InvalidateRect(hWnd, 0, 0);
	::UpdateWindow(hWnd);
}

void renderLines()
{
	gl::IBufferLineDrawer<float, uint8_t>& ld = *pLineDrawer;
	float x = 10;
	
	float offset = 0;
	for (int s=0; s<16; ++s) {
		int slide = s * 120;
		int ei = 21- s / 4;
		for (int i=0; i<ei; ++i) {
			int y = 30 + 40 * i;
			ld.DrawLine(
				0xff,
				x+slide, y,
				x+100+slide, y+offset
				);
			
			TCHAR str[32];
			RECT rec;
			rec.left = x + slide;
			rec.right = rec.left + 100;
			rec.top = y - 20;
			rec.bottom = rec.top + 20;
			_stprintf(str, _T("%.1f"), offset);
			::DrawText(hMemDC, str, -1, &rec, DT_LEFT);
			offset += 1.0;
		}
	}
}

void renderLine()
{
	float lx1 = x1;
	float lx2 = x2;
	float ly1 = y1;
	float ly2 = y2;
	if (clipper.ClipLinePoint(&lx1, &ly1, &lx2, &ly2) >= 0) {
		pLineDrawer->DrawLine(0xff, lx1, ly1, lx2, ly2);
	}
}

struct Data {
	uint32_t x;
	float y;
};

const size_t NUM_LINES = 100000;
Data datas[NUM_LINES];

struct Initer {
	Initer()
	{
		using namespace boost;
		minstd_rand    gen( 42 );
		uniform_real<> dst( -1.0, +1.0 );
		variate_generator<
			minstd_rand&, uniform_real<>
		> rand( gen, dst );
		for (size_t i=0; i<countof(datas); ++i) {
			Data& d = datas[i];
			d.x = i;
			d.y = rand();
		}
	}
} initer;

void renderVerticalLines1(
	const Data* pDatas, size_t cnt,
	float xa, float xb, float ya, float yb
	)
{
	gl::IBufferLineDrawer<float, uint8_t>& lineDrawer = *pLineDrawer;
	const Data& ld = pDatas[0];
	Data rd = pDatas[1];
	float lx = xa * ld.x + xb;
	float ly = ya * ld.y + yb;
	for (size_t i=1; i<cnt-1; ++i) {
		const Data& nd = pDatas[i+1];
		float rx = xa * rd.x + xb;
		float ry = ya * rd.y + yb;
		
		lineDrawer.DrawLine(0xff, lx, ly, rx, ry);
		
		lx = rx;
		ly = ry;
		rd = nd;
	}
	
}

struct MinMaxEntry {
	typedef int16_t value_t;
	value_t min;
	value_t max;
};

// TODO: 2Cɂ܂c̕`̍Č
void renderVerticalLines2(
	const Data* pDatas, size_t cnt,
	float xa, float xb, float ya, float yb,
	MinMaxEntry* yMinMaxEntries
	)
{
	for (size_t i=0; i<2000; ++i) {
		yMinMaxEntries[i].min = boost::integer_traits<MinMaxEntry::value_t>::const_max;
		yMinMaxEntries[i].max = 0;
	}
	const Data& ld = pDatas[0];
	Data rd = pDatas[1];
	int ilx = xs_RoundToInt(xa * ld.x + xb);
	int ily = xs_RoundToInt(ya * ld.y + yb);
	for (size_t i=1; i<cnt-1; ++i) {
		const Data& nd = pDatas[i+1];
		MinMaxEntry& entry = yMinMaxEntries[ilx];
		int irx = xs_RoundToInt(xa * rd.x + xb);
		assert(irx - ilx <= 1);
		int iry = xs_RoundToInt(ya * rd.y + yb);
		entry.min = gl::min(entry.min, gl::min(ily, iry));
		entry.max = gl::max(entry.max, gl::max(ily, iry));
		ilx = irx;
		ily = iry;
		rd = nd;
	}
	const int xMin = xs_RoundToInt(xa * pDatas[0].x + xb);
	const int xMax = xs_RoundToInt(xa * pDatas[cnt-1].x + xb);
	gl::IBufferLineDrawer<float, uint8_t>& lineDrawer = *pLineDrawer;
	for (int x=xMin; x<=xMax; ++x) {
		const MinMaxEntry& e = yMinMaxEntries[x];
		lineDrawer.DrawLine(0xff, x, e.min, x, e.max);
	}
}

void callRenderVerticalLines1()
{
	renderVerticalLines1(datas, NUM_LINES, 1300.0/NUM_LINES, 10.0, -300.0, 510.0);
}

MinMaxEntry yMinMaxEntries[2048];

void callRenderVerticalLines2()
{
	renderVerticalLines2(datas, NUM_LINES, 1300.0/NUM_LINES, 10.0, -300.0, 510.0, yMinMaxEntries);
}

void switchRender(UINT vk, uint16_t repeatCount)
{
//	buff.SetPixel(100, 220, 0xff);
	switch (vk) {
	case 'B':
		pLineDrawer = &lineDrawer_Bresenham;
		break;
	case 'D':
		pLineDrawer = &lineDrawer_DDA;
		break;
	case 'W':
		pLineDrawer = &lineDrawer_Wu;
		break;
	case 'N':
		pLineDrawer = &lineDrawer_Nelson;
		break;
	case 'S':
		pLineDrawer = &lineDrawer_Shift;
		break;

	case '1':
		pRenderRoutine = &renderLines;
		break;
	case '2':
		pRenderRoutine = &renderLine;
		break;
	case '3':
		pRenderRoutine = &callRenderVerticalLines1;
		break;
	case '4':
		pRenderRoutine = &callRenderVerticalLines2;
		break;
	case VK_LEFT:
	case VK_RIGHT:
	case VK_UP:
	case VK_DOWN:
		{
			float& x = *px;
			float& y = *py;
//			TRACE(_T("%d\n"), repeatCount);
			float offset = 0.1 * repeatCount;
			if (GetKeyState(VK_SHIFT) < 0) {
				offset *= 10;
			}
			if (GetKeyState(VK_CONTROL) < 0) {
				offset *= 100;
			}
			switch (vk) {
			case VK_LEFT:
				x -= offset;
				break;
			case VK_RIGHT:
				x += offset;
				break;
			case VK_UP:
				y -= offset;
				break;
			case VK_DOWN:
				y += offset;
				break;
			}
		}
		break;
	}
	callRenderRoutine();
}

void OnKeyDown(WPARAM vk, LPARAM lparam)
{
	switch (vk) {
	case 'B':
	case 'D':
	case 'W':
	case 'N':
	case 'S':
	case '1':
	case '2':
	case '3':
	case '4':
	case VK_LEFT:
	case VK_RIGHT:
	case VK_UP:
	case VK_DOWN:
//		TRACE(_T("%0x\n"), lparam);
		switchRender(vk, lparam & 0x7FFF);
		break;
	}
}

// ̍Wƃ}EXJ[\dȂƌ̂ŁA󂯂B
void changePositions(LPARAM lParam)
{
	int xPos = lParam & 0xFFFF;
	int yPos = lParam >> 16;
	double dx = *pax - xPos;
	double dy = *pay - yPos;
	
	// to determine which quadrant the angle is in, use atan2
	// http://iphone-3d-programming.labs.oreilly.com/ch06.html#OrientationSensors
	// Example 6.31. Accelerometer Response Handler

	// atan͊px -/2 ` +2/ ͈̔͌Ȃ̂ŎgÂ炢B
	// atan2ł΂ǂ̏یłǂ늄ōlĂB
	// ݂ɂȂŗǂB
#if 0
	double rad = atan(dy / abs(dx));
	double adjacentRatio = cos(rad);
	if (dx < 0) {
		adjacentRatio = -adjacentLen;
	}
#else
	double rad = atan2(dy, dx);
	double adjacentRatio = cos(rad);	// ڕ
#endif
	double oppositeRatio = sin(rad);	// Ε
	double hypotenuseLen = 30;	// Ε
	*px = xPos + hypotenuseLen * adjacentRatio;
	*py = yPos + hypotenuseLen * oppositeRatio;

	// trigonometryiOp@j
	// Op`̊eӂ̉p̌Ăѕ̗R̉B
	// http://www.youtube.com/watch?v=hJH_FDbcmSg
}

void OnMouseButtonDown(WPARAM wParam, LPARAM lParam)
{
	int fwKeys = wParam;
	double dx, dy;
	switch (fwKeys) {
	case MK_LBUTTON:
		px = &x1;
		py = &y1;
		pax = &x2;
		pay = &y2;
		break;
	case MK_RBUTTON:
		px = &x2;
		py = &y2;
		pax = &x1;
		pay = &y1;
		break;
	}
	changePositions(lParam);
	callRenderRoutine();
}

void OnMouseMove(WPARAM wParam, LPARAM lParam)
{
	int fwKeys = wParam;
	if (fwKeys & (MK_LBUTTON|MK_RBUTTON)) {
		changePositions(lParam);
		callRenderRoutine();
	}
}

HBITMAP CreateDIB8(int width, int height, BITMAPINFO& bmi, void*& pBits)
{
	if (width % 4) {
		width += 4 - (width % 4);
	}
	assert(width % 4 == 0);
	BITMAPINFOHEADER& header = bmi.bmiHeader;
	header.biSize = sizeof(BITMAPINFOHEADER);
	header.biWidth = width;
	header.biHeight = height;
	header.biPlanes = 1;
	header.biBitCount = 8;
	header.biCompression = BI_RGB;
	header.biSizeImage = width * abs(height);
	header.biXPelsPerMeter = 0;
	header.biYPelsPerMeter = 0;
	header.biClrUsed = 0;
	header.biClrImportant = 0;

	return ::CreateDIBSection(
		(HDC)0,
		&bmi,
		DIB_RGB_COLORS,
		&pBits,
		NULL,
		0
	);
}

HBITMAP CreateDIB32(int width, int height, BITMAPINFO& bmi, void*& pBits)
{
	BITMAPINFOHEADER& header = bmi.bmiHeader;
	header.biSize = sizeof(BITMAPINFOHEADER);
	header.biWidth = width;
	header.biHeight = height;
	header.biPlanes = 1;
	header.biBitCount = 32;
	header.biCompression = BI_RGB;
	header.biSizeImage = width * abs(height) * 4;
	header.biXPelsPerMeter = 0;
	header.biYPelsPerMeter = 0;
	header.biClrUsed = 0;
	header.biClrImportant = 0;

	return ::CreateDIBSection(
		(HDC)0,
		&bmi,
		DIB_RGB_COLORS,
		&pBits,
		NULL,
		0
	);
}

void SetupDIB(HWND arg_hWnd)
{
	hWnd = arg_hWnd;
	pRenderRoutine = &renderLine;

	HDC hDC = ::GetDC(NULL);
	int width = GetDeviceCaps(hDC, HORZRES) + 8;
	int height = GetDeviceCaps(hDC, VERTRES) + 10;
	::ReleaseDC(NULL, hDC);
	
	if (hBMP) {
		::DeleteObject(hBMP);
		hBMP = 0;
	}
	BITMAPINFO* pBMI = (BITMAPINFO*) &bmiBuff[0];
	BITMAPINFO& bmi = *pBMI;
#if 1
	RGBQUAD rgb;
	for (size_t i=0; i<256; ++i) {
		rgb.rgbBlue = rgb.rgbGreen = rgb.rgbRed = i;
		bmi.bmiColors[i] = rgb;
	}
	hBMP = CreateDIB8(width, -height, bmi, pBits);
#else
	hBMP = CreateDIB32(width, -height, bmi, pBits);
#endif
	buff = gl::Buffer2D<uint8_t>(width, height, bmi.bmiHeader.biWidth, pBits);
	static float slopeCorrectionTable[256];
	gl::SetupSlopeCorrectionTable(slopeCorrectionTable, countof(slopeCorrectionTable));
	static float filterTable[256];
	gl::SetupFilterTable(filterTable, countof(filterTable));
	
	lineDrawer_Nelson.SetSlopeCorrectionTable(slopeCorrectionTable);
	lineDrawer_Nelson.SetFilterTable(filterTable);

	lineDrawer_Shift.SetSlopeCorrectionTable(slopeCorrectionTable);
	clipper.SetClipRegion(0, 0, width, height);

	if (hMemDC) {
		::DeleteDC(hMemDC);
	}
	HDC hWndDC = ::GetDC(hWnd);
	hMemDC = ::CreateCompatibleDC(hWndDC);
	::SetMapMode(hMemDC, ::GetMapMode(hWndDC));
	::ReleaseDC(hWnd, hWndDC);
	::SelectObject(hMemDC, hBMP);	

}

void PaintDIB(HDC hDC, PAINTSTRUCT& ps)
{
	const RECT& rec = ps.rcPaint;
	::BitBlt(
		hDC,
		rec.left,
		rec.top,
		rec.right - rec.left,
		rec.bottom - rec.top,
		hMemDC,
		rec.left,
		rec.top,
		SRCCOPY
	);
}

} // anonymous namespace


