﻿///////////////////////////////////////////////////////////////////////////////
// This file is part of SharpWebBrowser.
///////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2009-2011 bis5 <bis5@users.sourceforge.jp>
// All rights reserved.
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the Microsoft Public Licence (Ms-PL) as published by 
// Microsoft Corp.
//
// You should have received a copy of the Microsoft Public License along 
// with this program. 
// If not, see <http://www.microsoft.com/opensource/licenses.mspx>
//
///////////////////////////////////////////////////////////////////////////////
//
// File information
// Name: IConfMgr.cs
// Author: bis5
//
///////////////////////////////////////////////////////////////////////////////

namespace Bis5Products.SharpWebBrowser {
    /// <summary>SharpWebBrowserで使用するConfigurationManagerモジュールに必要な機能を実装します。</summary>
    public interface IConfMgr : ISWBModules {
        /// <summary>読み取り専用かどうかを表す<see cref="boolean"/>値。</summary>
        bool ReadOnly{ get; }
        /// <summary>設定値を追加します。</summary>
        /// <param name="key">設定値の名称</param>
        /// <param name="value">設定地</param>
        bool Add(string key, object value);
        /// <summary>すでに存在している設定値を変更します。</summary>
        /// <param name="key">設定値の名称</param>
        /// <param name="value">設定値</param>
        bool Set(string key, object value);
        /// <summary>設定値を一覧から取り除きます。</summary>
        /// <param name="key">設定の名称</param>
        bool Remove(string key);
        /// <summary>設定値の一覧を空にします。</summary>
        void Clear();
        
        /// <summary>設定ファイルから設定値を読み込みます。</summary>
        bool Load();
        /// <summary>設定ファイルに設定値を書き込みます。</summary>
        bool Save();
        /// <summary>一覧に設定値が含まれているかを確認します。</summary>
        /// <param name="key"設定値の名称></param>
        /// <returns>設定値が含まれているかどうかを表す<see cref="boolean"/>値。</returns>
        bool IsContains(string key);
        /// <summary>設定値を取得します。</summary>
        /// <param name="key">設定値の名称</param>
        /// <returns>設定値</returns>
        object Get(string key);
        /// <summary>設定値を取得します。</summary>
        /// <param name="id">設定値固有の番号</param>
        /// <returns>設定値</returns>
        object Get(int id);
        /// <summary>設定値を指定した型として取得します。</summary>
        /// <param name="key">設定値の名称</param>
        /// <param name="type">設定値の型を表す<see cref="System.Type"/>。</param>
        /// <returns>設定値</returns>
        object GetObj(string key, System.Type type);
        /// <summary>設定値を指定した型として取得します。</summary>
        /// <param name="id">設定値固有の番号</param>
        /// <param name="type">設定値の型を表す<see cref="System.Type"/。></param>
        /// <returns>設定値</returns>
        object GetObj(int id, System.Type type);
        /// <summary>設定値を文字列として取得します。</summary>
        /// <param name="key">設定値の名称</param>
        /// <returns>設定値の文字列形式</returns>
        string GetStr(string key);
        /// <summary>設定値を文字列として取得します。・</summary>
        /// <param name="id">設定値固有の番号</param>
        /// <returns>設定値の文字列形式</returns>
        string GetStr(int id);
        /// <summary>設定値を<see cref="int"/>型整数として取得します。</summary>
        /// <param name="key">設定値の名称</param>
        /// <returns>設定値の整数形式</returns>
        int GetInt32(string key);
        /// <summary>設定値を<see cref="int"/>型整数として取得します。</summary>
        /// <param name="id">設定値固有の番号</param>
        /// <returns>設定値の整数形式</returns>
        int GetInt32(int id);
        /// <summary>設定値を<see cref="bool"/>値として取得します。</summary>
        /// <param name="key">設定値の名称</param>
        /// <returns>設定値の真偽値形式</returns>
        bool GetBool(string key);
        /// <summary>設定値を<see cref="bool"/>値として取得します。</summary>
        /// <param name="id">設定値固有の番号</param>
        /// <returns>設定値の真偽値形式</returns>
        bool GetBool(int id);
        
        /// <summary>一覧に登録されている設定項目の数を表す整数値。</summary>
        int Count{ get; }
        /// <summary>設定ダイアログを表示します。</summary>
        /// <returns>設定ダイアログの終了状態を表す<see cref="System.Windows.Forms.DialogResult"/>値。</returns>
        System.Windows.Forms.DialogResult ShowConfigDialog();
        /// <summary>設定ダイアログに表示するプロパティシートを登録します。</summary>
        /// <param name="prop">設定ダイアログに追加するプロパティシート</param>
        void AddPropertySheet(ISWBPropertySheet prop);
        /// <summary>設定ダイアログを初期状態に戻します。</summary>
        void ResetConfigDialog();
        /// <summary>設定ダイアログに表示するプロパティシートの一覧を空にします。</summary>
        void ResetPropertyList();
        /// <summary>与えられた値を持つ設定値の名称を解決します。</summary>
        /// <param name="value">名称を解決する設定値</param>
        /// <returns>設定値の名称</returns>
        string SolveIndex(object value);
        /// <summary>設定値の名称を変更します。</summary>
        /// <param name="oldIndex">変更前の設定値の名称</param>
        /// <param name="newIndex">変更後の設定値の名称</param>
        bool ChangeIndex(string oldIndex, string newIndex);

        void SetButtonUac();

    }
}
