//
//  main.c
//  chncpu
//
//  Created by 西田　耀 on 2014/06/04.
//  Copyright (c) 2014年 CHNOSProject. All rights reserved.
//

#include "chncpu.h"

int main(int argc, const char *argv[])
{
    CHNCPU_RuntimeEnvironment *env;
	CHNCPU_VMArgs *vmArgs;
	
	vmArgs = bindVMArgs(argc, argv);
	env = CHNCPU_CreateRuntimeEnvironment(CHNCPU_CreateOpTableSet(), CHNCPU_CreateBIOS());
	
	if(vmArgs->VMFlags & CHNCPU_VM_FLAG_PRINT_INFO){
		printf("chncpu, inspired by OSECPU-VM. \nCompiled at %s, %s.\n", __TIME__, __DATE__);
		return 0;
	}
    
    CHNCPU_LoadBinaryFromHexStringFilePath(env, vmArgs->appPath);
	CHNCPU_VMArgs_BindDataToRuntimeEnv(env, vmArgs);
    if(!env->errFlags){
        CHNCPU_Execute(env);
    }
	if(!env->errFlags){
        CHNCPU_VMArgs_BindDataFromRuntimeEnv(env, vmArgs);
    }
	
	free(vmArgs);
    return 0;
}

int decodeHexString(char *src0, char *src1, unsigned char *dst0, unsigned char *dst1)
{
    // ASCII文字のみ対応
    // ダブルクオート内の文字は自動でバイナリに変換する。
    char *p;
    unsigned char *wp;
    int phase = 0;
    int wlen = 0;
    // 0: first 4-bit
    // 1: second 4-bit
    // 2: in string literal (byte aligned)
	// 3: in string literal (not aligned)
    unsigned char d = 0;
    char c;
    
    wp = dst0;
    
    for(p = src0; p != src1; p++){
        c = *p;
		if(c == '"'){
			if(phase == 0){
				phase = 2;
			} else if(phase == 1){
				phase = 3;
			} else if(phase == 2){
				phase = 0;
			} else if(phase == 3){
				phase = 1;
			}
			continue;
		} else if(phase == 0 || phase == 1){
			if('0' <= c && c <= '9'){
				c = c - '0';
			} else if('A' <= c && c <= 'F'){
				c = c - 'A' + 0x0A;
			} else if('a' <= c && c <= 'f'){
				c = c - 'a' + 0x0a;
			} else if(c == ' ' || c == '\t' || c == '\n'){
                continue;
            } else{
				puts("decodeHexString: unexpected character :");
				putchar(c);
				putchar('\n');
				exit(EXIT_FAILURE);
				break;
			}
        }
		if(phase == 0){
            d = c & 0x0f;
            phase = 1;
        } else if(phase == 1){
            d <<= 4;
            d |= c & 0x0f;
            *wp = d;
            wp++;
            wlen++;
            phase = 0;
        } else if(phase == 2){
            *wp = c;
            wp++;
            wlen++;
		} else if(phase == 3){
            d <<= 4;
            d |= (c >> 4) & 0x0f;
            *wp = d;
            wp++;
            wlen++;
			//
			d = c & 0x0f;
        }
        
        if(wp >= dst1){
            puts("chncpu:decodeHexString dst overrun, abort.\n");
            exit(EXIT_FAILURE);
        }
    }
    if(phase == 1){
        d <<= 4;
        d |= 0xf;
        *wp = d;
        wlen++;
    }
    
    return wlen;
}

CHNCPU_RuntimeEnvironment *CHNCPU_CreateRuntimeEnvironment(CHNCPU_OpTableSet *opSet, CHNCPU_BIOS *bios)
{
    CHNCPU_RuntimeEnvironment *env;
    int i;
    
    env = malloc(sizeof(CHNCPU_RuntimeEnvironment));
    if(!env){
        puts("CHNCPU_CreateRuntimeEnvironment: malloc error, abort.\n");
        exit(EXIT_FAILURE);
    }
	
	env->opSet = opSet;
	env->bios = bios;
    
	// レジスタ初期化
    for(i = 0; i < CHNCPU_NUMBER_OF_IREG; i++){
        env->iReg[i] = 0;
        env->iRegBits[i] = 0;
    }
    for(i = 0; i < CHNCPU_LENGTH_OF_MAIN_MEMORY; i++){
		env->mainmemory[i].opCode = CHNCPU_OPCODE_INVALID;
    }
	for(i = 0; i < CHNCPU_NUMBER_OF_PREG; i++){
        env->pReg[i].type = 0;
		env->pReg[i].mindex = 0;
		env->pReg[i].pindex = 0;
		env->pReg[i].data = NULL;
	}
	
	// ラベル管理初期化
	env->labelSet = CHNCPU_CreateLabelSet();
    
	// 実行バイナリ関連初期化
    env->appbin0 = malloc(CHNCPU_SIZE_APPBIN);
    env->appbinsize = 0;
    env->appbinReader = malloc(sizeof(CH4Reader));
	
    env->errFlags = 0;
    
    return env;
}

int CHNCPU_LoadBinaryFromHexStringFilePath(CHNCPU_RuntimeEnvironment *env, const char *path)
{
    FILE *fp;
    char *tmpdata0;
    size_t len;
    
    fp = fopen(path, "rb");
    if(!fp){
        puts("chncpu:Error > File open error, abort.\n");
        exit(EXIT_FAILURE);
    }
    
    tmpdata0 = malloc(SIZE_TMPDATA);
    
    len = fread(tmpdata0, 1, SIZE_TMPDATA, fp);
    if(len >= SIZE_TMPDATA){
        puts("chncpu:Error > File too large, abort.\n");
        exit(EXIT_FAILURE);
    }
    
    env->appbinsize = decodeHexString(tmpdata0, tmpdata0 + len, env->appbin0, env->appbin0 + CHNCPU_SIZE_APPBIN);
#if DEBUG_PRINT_APPBIN_BEFORE_EXECUTION
	printf("appbin = %d Bytes\n", env->appbinsize);
    CHNCPU_DumpAppBin(env);
#endif

    CH4Reader_Initialize(env->appbinReader, env->appbin0, env->appbinsize);
    
    CHNCPU_PrepareBinaryForExecution(env);
    
    return 0;
}

int CHNCPU_PrepareBinaryForExecution(CHNCPU_RuntimeEnvironment *env)
{
    ch4_uint opcode = 0;
    CHNCPU_OpTag *op;
	unsigned int prefix = 0;
	CHNCPU_OpTableSet *opSet = env->opSet;
    
    // env->appbinReaderから読み込んで、実行可能な状態にする。
    // これはコンパイラ版におけるコンパイルに相当する。
#if DEBUG_PRINT_OP_EXECUTING
    puts("< Start preparing for execution >");
#endif
    env->currentIndex = 0;
    for(env->currentIndex = 0; env->currentIndex < CHNCPU_LENGTH_OF_MAIN_MEMORY; env->currentIndex++){
        opcode = CH4Reader_ReadNextAsUINT(env->appbinReader);
        if(CH4Reader_IsEndOfBinary(env->appbinReader)){
            break;
        }
		switch(opcode){
            case 0x00:
                // NOP
#if DEBUG_PRINT_OP_BINDING
				puts("NOP()");
#endif
                env->currentIndex--;	// メモリには追加しない
                break;
			case 0x2F:
				// Prefix
				opcode = CH4Reader_ReadNextAsUINT(env->appbinReader);
#if DEBUG_PRINT_OP_BINDING
				printf("Prefix-%X\n", opcode);
#endif
				if(opcode > CHNCPU_PREFIX_MAX){
					env->errFlags |= CHNCPU_ERR_INVALID_PREFIX;
					break;
				}
				prefix |= (0x01 << opcode);
				env->currentIndex--;	// メモリには追加しない
				break;
            default:
                // ごく一部の特殊な命令以外は、命令テーブルを参照する
				op = &env->mainmemory[env->currentIndex];
				op->opCode = opcode;
                if(opcode <= CHNCPU_OPECODE_MAX && opSet->bindFuncTable[opcode]){
					opSet->bindFuncTable[opcode](env, op, prefix);
                } else{
                    env->errFlags |= CHNCPU_ERR_INVALID_OPCODE;
                }
#if DEBUG_PRINT_OP_BINDING
				if(op->opCode <= CHNCPU_OPECODE_MAX && opSet->printFuncTable[op->opCode]){
					opSet->printFuncTable[op->opCode](env, op, stdout);
				} else{
					printf("Unknown op: 0x%X", op->opCode);
				}
#endif
				prefix = 0;
                break;
        }
		if(env->appbinReader->errorFlags){
			env->errFlags |= CHNCPU_ERR_INVALID_BINARY;
		}
        if(env->errFlags){
            break;
        }
    }
    if(env->currentIndex >= CHNCPU_LENGTH_OF_MAIN_MEMORY && !CH4Reader_IsEndOfBinary(env->appbinReader)){
        env->errFlags |= CHNCPU_ERR_INTERNAL;
        puts("INVALID-C: Internal error (low on memory).");
    }
    if(env->errFlags){
		CHNCPU_PrintErrorMessage(env);
    }
	env->currentIndex = 0;
	env->currentLabel = 0;
    return 0;
}

int CHNCPU_Execute(CHNCPU_RuntimeEnvironment *env)
{
    int count = 0;
    CHNCPU_OpTag *op;
	CHNCPU_OpTableSet *opSet = env->opSet;
    
    if(env->errFlags){
        puts(">>> Can't execute binary because of there is some error.");
    }
#if DEBUG_PRINT_OP_EXECUTING
    puts("< Start execution >");
#endif
    for(; env->currentIndex < CHNCPU_LENGTH_OF_MAIN_MEMORY; env->currentIndex++){
        op = &env->mainmemory[env->currentIndex];
        if(op->opCode == CHNCPU_OPCODE_INVALID){
            // End of Binary
            break;
        }
#if DEBUG_PRINT_OP_EXECUTING
		if(op->opCode <= CHNCPU_OPECODE_MAX && opSet->printFuncTable[op->opCode]){
			opSet->printFuncTable[op->opCode](env, op, stdout);
		} else{
			printf("Unknown op: 0x%X", op->opCode);
		}
#endif
		
        if(op->opCode <= CHNCPU_OPECODE_MAX && opSet->execFuncTable[op->opCode]){
            if(opSet->execFuncTable[op->opCode](env, op)){
				// 命令実行時にエラーが発生
				break;
			}
        } else{
            // ロード時にチェックしたはずなのに不明な命令が来た
            env->errFlags |= CHNCPU_ERR_INTERNAL;
            break;
        }
    }
    if(env->errFlags){
        CHNCPU_PrintErrorMessage(env);
    }
	
#if (DEBUG_PRINT_IREG_AFTER_EXECUTION || DEBUG_PRINT_PREG_AFTER_EXECUTION)
	puts("\n< End of execution >");
#endif
#if DEBUG_PRINT_IREG_AFTER_EXECUTION
    CHNCPU_DumpIReg(env);
#endif
#if DEBUG_PRINT_PREG_AFTER_EXECUTION
    CHNCPU_DumpPReg(env);
#endif
    
    
    return count;
}

int CHNCPU_CHK_IsAvailableBits(CHNCPU_RuntimeEnvironment *env, ch4_uint bits)
{
    if(bits > 32){
        env->errFlags |= CHNCPU_ERR_INVALID_BITS;
        return 0;
    }
    return 1;
}

int CHNCPU_AdjustValueForBit(CHNCPU_RuntimeEnvironment *env, int *value, ch4_uint bit, int prefix)
{
	// retv: isTruncated
	int regMax, regMin, regMask;
	regMask = (-1) << bit;
	regMax = 1 << (bit - 1);
	regMax--;
	regMin = -regMax - 1;
	
	if(regMin <= *value && *value <= regMax){
		return 0;
	}
	if(!(prefix & CHNCPU_PREFIX_ALLOW_TRUNCATE)){
		env->errFlags |= CHNCPU_ERR_TRUNCATED_VALUE;
		return 1;
	}
	
	if(*value & (1 << bit)){
		// -
		*value |=  regMask;
	} else{
		// +
		*value &= ~regMask;
	}
	
	return 1;
}
