//
//  opcode.c
//  chncpu
//
//  Created by 西田　耀 on 2014/06/05.
//  Copyright (c) 2014年 CHNOSProject. All rights reserved.
//

#include "chncpu.h"
#include "opcache.h"

CHNCPU_OpTableSet *CHNCPU_CreateOpTableSet(void)
{
	CHNCPU_OpTableSet *opSet;
	
	opSet = malloc(sizeof(CHNCPU_OpTableSet));
	if(!opSet){
        puts("CHNCPU_CreateOpTableSet: malloc error, abort.\n");
        exit(EXIT_FAILURE);
    }
	
	CHNCPU_Op_Init(opSet);
	
	return opSet;
}

int CHNCPU_Op_Init(CHNCPU_OpTableSet *opSet)
{
	int i;
	
	// FuncTables
    for(i = 0; i <= CHNCPU_OPECODE_MAX; i++){
        opSet->bindFuncTable[i] = NULL;
		opSet->execFuncTable[i] = NULL;
		opSet->printFuncTable[i] = NULL;
    }
		
	// LB
	opSet->bindFuncTable[0x01] = CHNCPU_Op_LB_BindOperand;
	opSet->execFuncTable[0x01] = CHNCPU_Op_LB_Execute;
	opSet->printFuncTable[0x01] = CHNCPU_Op_LB_PrintCode;
	
	// LIMM
    opSet->bindFuncTable[0x02] = CHNCPU_Op_LIMM_BindOperand;
	opSet->execFuncTable[0x02] = CHNCPU_Op_LIMM_Execute;
	opSet->printFuncTable[0x02] = CHNCPU_Op_LIMM_PrintCode;
	
	// PLIMM
    opSet->bindFuncTable[0x03] = CHNCPU_Op_PLIMM_BindOperand;
	opSet->execFuncTable[0x03] = CHNCPU_Op_PLIMM_Execute;
	opSet->printFuncTable[0x03] = CHNCPU_Op_PLIMM_PrintCode;
	
	// CND
    opSet->bindFuncTable[0x04] = CHNCPU_Op_CND_BindOperand;
	opSet->execFuncTable[0x04] = CHNCPU_Op_CND_Execute;
	opSet->printFuncTable[0x04] = CHNCPU_Op_CND_PrintCode;
	
	// CALLBIOS
	opSet->bindFuncTable[0x05] = CHNCPU_Op_CALLBIOS_BindOperand;
	opSet->execFuncTable[0x05] = CHNCPU_Op_CALLBIOS_Execute;
	opSet->printFuncTable[0x05] = CHNCPU_Op_CALLBIOS_PrintCode;
	
	// LMEM
	opSet->bindFuncTable[0x08] = CHNCPU_Op_LMEM_BindOperand;
	opSet->execFuncTable[0x08] = CHNCPU_Op_LMEM_Execute;
	opSet->printFuncTable[0x08] = CHNCPU_Op_LMEM_PrintCode;
	
	// TernaryRegBitwise
	for(i = 0x10; i <= 0x12; i++){
		opSet->bindFuncTable[i] = CHNCPU_Op_TernaryReg_BindOperand;
		opSet->execFuncTable[i] = CHNCPU_Op_TernaryRegBitwise_Execute;
		opSet->printFuncTable[i] = CHNCPU_Op_TernaryReg_PrintCode;
	}
	// TernaryRegArithmetic
	for(i = 0x14; i <= 0x16; i++){
		opSet->bindFuncTable[i] = CHNCPU_Op_TernaryReg_BindOperand;
		opSet->execFuncTable[i] = CHNCPU_Op_TernaryRegArithmetic_Execute;
		opSet->printFuncTable[i] = CHNCPU_Op_TernaryReg_PrintCode;
	}
	for(i = 0x18; i <= 0x1B; i++){
		opSet->bindFuncTable[i] = CHNCPU_Op_TernaryReg_BindOperand;
		opSet->execFuncTable[i] = CHNCPU_Op_TernaryRegArithmetic_Execute;
		opSet->printFuncTable[i] = CHNCPU_Op_TernaryReg_PrintCode;
	}
	// CompareIReg
	for(i = 0x20; i <= 0x27; i++){
		opSet->bindFuncTable[i] = CHNCPU_Op_CompareIReg_BindOperand;
		opSet->execFuncTable[i] = CHNCPU_Op_CompareIReg_Execute;
		opSet->printFuncTable[i] = CHNCPU_Op_CompareIReg_PrintCode;
	}
	// DATA
	opSet->bindFuncTable[0x2E] = CHNCPU_Op_DATA_BindOperand;
	opSet->execFuncTable[0x2E] = CHNCPU_Op_DATA_Execute;
	opSet->printFuncTable[0x2E] = CHNCPU_Op_DATA_PrintCode;
	return 0;
}

//
// 01 LB
//

int CHNCPU_Op_LB_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix)
{
    CHNCPU_OpCache_LB *opCache;
    
    opCache = malloc(sizeof(CHNCPU_OpCache_LB));
    op->opCache = opCache;
    
    opCache->labelID = CH4Reader_ReadNextAsUINT(env->appbinReader);
    opCache->opt = CH4Reader_ReadNextAsUINT(env->appbinReader);
	
	env->currentLabel = opCache->labelID;
	
	if(CHNCPU_Label_Add(env->labelSet, env->currentIndex, opCache->labelID, opCache->opt, CHNCPU_PType_Label)){
		env->errFlags |= CHNCPU_ERR_DUPLICATED_LABEL_ID;
	}
    
	if(prefix != 0){
		env->errFlags |= CHNCPU_ERR_INVALID_PREFIX;
        return -1;
	}
    return 0;
}
int CHNCPU_Op_LB_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op)
{
    CHNCPU_OpCache_LB *opCache;
    opCache = op->opCache;
	
	env->currentLabel = opCache->labelID;
	
    return 0;
}

int CHNCPU_Op_LB_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file)
{
    CHNCPU_OpCache_LB *opCache;
    opCache = op->opCache;
	
    fprintf(file, "LB(id:0x%X, opt:%d);\n", opCache->labelID, opCache->opt);
	
    return 0;
}

//
// 02 LIMM
//

int CHNCPU_Op_LIMM_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix)
{
    CHNCPU_OpCache_LIMM *opCache;
    
	opCache = malloc(sizeof(CHNCPU_OpCache_LIMM));
    op->opCache = opCache;
    
    opCache->imm = CH4Reader_ReadNextAsSINT(env->appbinReader);
    opCache->r = CH4Reader_ReadNextAsUINT(env->appbinReader);
    opCache->bit = CH4Reader_ReadNextAsUINT(env->appbinReader);
    
    if(opCache->r >= CHNCPU_NUMBER_OF_IREG){
        env->errFlags |= CHNCPU_ERR_INVALID_REGNUM;
        return -1;
    }
    if(!CHNCPU_CHK_IsAvailableBits(env, opCache->bit)){
        return -1;
    }
    if(opCache->bit == 0 && opCache->imm != 0){
        env->errFlags |= CHNCPU_ERR_INVALID_EXPRESSION;
        return -1;
    }
	if(prefix != 0){
		env->errFlags |= CHNCPU_ERR_INVALID_PREFIX;
        return -1;
	}
	CHNCPU_AdjustValueForBit(env, &opCache->imm, opCache->bit, CHNCPU_PREFIX_ALLOW_TRUNCATE);
	if(env->errFlags){
		return -1;
	}
	
    return 0;
}
int CHNCPU_Op_LIMM_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op)
{
    CHNCPU_OpCache_LIMM *opCache;
    
    opCache = op->opCache;
    
    env->iReg[opCache->r] = opCache->imm;
	env->iRegBits[opCache->r] = opCache->bit;
    
    return 0;
}

int CHNCPU_Op_LIMM_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file)
{
    CHNCPU_OpCache_LIMM *opCache;
    
    opCache = op->opCache;
    fprintf(file, "LIMM(imm:0x%X, r:%02X, bit:%d);\n", opCache->imm, opCache->r, opCache->bit);

    return 0;
}

//
// 03 PLIMM
//
int CHNCPU_Op_PLIMM_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix)
{
    CHNCPU_OpCache_PLIMM *opCache;
    
    opCache = malloc(sizeof(CHNCPU_OpCache_PLIMM));
    op->opCache = opCache;
    
    opCache->labelID = CH4Reader_ReadNextAsUINT(env->appbinReader);
    opCache->r = CH4Reader_ReadNextAsUINT(env->appbinReader);
    
    if(opCache->r >= CHNCPU_NUMBER_OF_PREG){
        env->errFlags |= CHNCPU_ERR_INVALID_REGNUM;
        return -1;
    }
	if(prefix != 0){
		env->errFlags |= CHNCPU_ERR_INVALID_PREFIX;
        return -1;
	}
    return 0;
}
int CHNCPU_Op_PLIMM_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op)
{
    CHNCPU_OpCache_PLIMM *opCache;
	CHNCPU_LabelTag *label;
    
    opCache = op->opCache;
    
	label = CHNCPU_Label_GetTagFromLabelID(env->labelSet, opCache->labelID);
	if(!label){
		env->errFlags |= CHNCPU_ERR_NOT_FOUND_LABEL;
		return -1;
	}
	
	env->pReg[opCache->r].type = label->pType;
    env->pReg[opCache->r].mindex = label->mindex;
	env->pReg[opCache->r].pindex = 0;
	
	if(0 <= label->mindex && label->mindex < CHNCPU_LENGTH_OF_MAIN_MEMORY){
		env->pReg[opCache->r].data = CHNCPU_Op_DATA_GetData(&env->mainmemory[label->mindex]);
	}
	
	if(opCache->r == 0x3F){
		if(env->pReg[0x3F].type == CHNCPU_PType_Label){
			env->currentIndex = env->pReg[0x3F].mindex - 1;
		}
	}
    
    return 0;
}

int CHNCPU_Op_PLIMM_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file)
{
    CHNCPU_OpCache_PLIMM *opCache;
    
    opCache = op->opCache;
    fprintf(file, "PLIMM(labelID:0x%X, r:%02X);\n", opCache->labelID, opCache->r);
	
    return 0;
}

//
// 04 CND
//
int CHNCPU_Op_CND_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix)
{
    CHNCPU_OpCache_CND *opCache;
    
    opCache = malloc(sizeof(CHNCPU_OpCache_CND));
    op->opCache = opCache;
    
    opCache->r = CH4Reader_ReadNextAsUINT(env->appbinReader);
    
    if(opCache->r >= CHNCPU_NUMBER_OF_PREG){
        env->errFlags |= CHNCPU_ERR_INVALID_REGNUM;
        return -1;
    }
	if(prefix != 0){
		env->errFlags |= CHNCPU_ERR_INVALID_PREFIX;
        return -1;
	}
    return 0;
}
int CHNCPU_Op_CND_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op)
{
    CHNCPU_OpCache_CND *opCache;
    
    opCache = op->opCache;
    
	if((env->iReg[opCache->r] & 0x1) != 0x1){
		// 次の命令は実行しない
		env->currentIndex++;
	}
	return 0;
}

int CHNCPU_Op_CND_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file)
{
    CHNCPU_OpCache_CND *opCache;
    
    opCache = op->opCache;
    fprintf(file, "CND(r:%02X);\n", opCache->r);
	
    return 0;
}

//
// 05 CALLBIOS
//
int CHNCPU_Op_CALLBIOS_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix)
{
	ch4_uint fid;
	
	CHNCPU_OpTag *opCache;
    
    opCache = malloc(sizeof(CHNCPU_OpTag));
    op->opCache = opCache;
	
	opCache->opCode = CH4Reader_ReadNextAsUINT(env->appbinReader);
	
	fid = opCache->opCode;
	if(fid <= CHNCPU_BIOS_OP_MAX && env->bios->bindFuncTable[fid]){
		env->bios->bindFuncTable[fid](env, opCache, prefix);
	} else{
		env->errFlags |= CHNCPU_ERR_INVALID_BIOS_FUNC_ID;
	}
	
	if(env->errFlags){
		return -1;
	}
    return 0;
}
int CHNCPU_Op_CALLBIOS_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op)
{
    CHNCPU_OpTag *opCache;
	ch4_uint fid;
    
    opCache = op->opCache;
	
	fid = opCache->opCode;
	if(fid <= CHNCPU_BIOS_OP_MAX && env->bios->execFuncTable[fid]){
		env->bios->execFuncTable[fid](env, opCache);
	} else{
		// ロード時にチェックしたはずなのに不明な命令が来た
		env->errFlags |= CHNCPU_ERR_INTERNAL;
	}
	
    return 0;
}

int CHNCPU_Op_CALLBIOS_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file)
{
    CHNCPU_OpTag *opCache;
    ch4_uint fid;
	
    opCache = op->opCache;
	
	fid = opCache->opCode;
	if(fid <= CHNCPU_BIOS_OP_MAX && env->bios->printFuncTable[fid]){
		env->bios->printFuncTable[fid](env, opCache, stdout);
	} else{
		fprintf(file, "CALLBIOS(func:0x%X);\n", fid);
	}
	
    return 0;
}

//
// 08 LMEM
//
int CHNCPU_Op_LMEM_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix)
{
    CHNCPU_OpCache_LMEM *opCache;
    
    opCache = malloc(sizeof(CHNCPU_OpCache_LMEM));
    op->opCache = opCache;
    
    opCache->p = CH4Reader_ReadNextAsUINT(env->appbinReader);
    opCache->pType = CH4Reader_ReadNextAsSINT(env->appbinReader);
    opCache->pDiff = CH4Reader_ReadNextAsSINT(env->appbinReader);
	opCache->r = CH4Reader_ReadNextAsUINT(env->appbinReader);
	opCache->bitDst = CH4Reader_ReadNextAsUINT(env->appbinReader);
    
    if(opCache->r >= CHNCPU_NUMBER_OF_IREG ||
	   opCache->p >= CHNCPU_NUMBER_OF_PREG){
        env->errFlags |= CHNCPU_ERR_INVALID_REGNUM;
        return -1;
    }
    if(!CHNCPU_CHK_IsAvailableBits(env, opCache->bitDst)){
        return -1;
    }
	if(prefix != 0){
		env->errFlags |= CHNCPU_ERR_INVALID_PREFIX;
        return -1;
	}
    return 0;
}
int CHNCPU_Op_LMEM_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op)
{
    CHNCPU_OpCache_LMEM *opCache;
	CHNCPU_PointerTag *p;
	int value;
    
    opCache = op->opCache;
    p = &env->pReg[opCache->p];
    
	if(p->type != opCache->pType){
		env->errFlags |= CHNCPU_ERR_NOT_MATCHED_PTYPE;
        return -1;
	}
	
	value = CHNCPU_DATA_ReadAtIndex(p->data, p->pindex, &env->errFlags);
	if(env->errFlags){
		return -1;
	}
	
	CHNCPU_AdjustValueForBit(env, &value, opCache->bitDst, CHNCPU_PREFIX_ALLOW_TRUNCATE);
	if(env->errFlags){
		return -1;
	}
	env->iReg[opCache->r] = value;
	env->iRegBits[opCache->r] = opCache->bitDst;
	
	p->pindex += opCache->pDiff;
    
    return 0;
}
int CHNCPU_Op_LMEM_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file)
{
    CHNCPU_OpCache_LMEM *opCache;
    
    opCache = op->opCache;
    fprintf(file, "LMEM(p:0x%02X, pType:0x%X, pDiff:%d, r:%02X, bitDst:%d);\n", opCache->p, opCache->pType, opCache->pDiff, opCache->r, opCache->bitDst);
	
    return 0;
}

//
// Ternary Register Operation
//
int CHNCPU_Op_TernaryReg_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix)
{
    CHNCPU_OpCache_TernaryReg *opCache;
    
    opCache = malloc(sizeof(CHNCPU_OpCache_TernaryReg));
    op->opCache = opCache;
    
    opCache->r1 = CH4Reader_ReadNextAsUINT(env->appbinReader);
    opCache->r2 = CH4Reader_ReadNextAsUINT(env->appbinReader);
    opCache->r0 = CH4Reader_ReadNextAsUINT(env->appbinReader);
	opCache->bit = CH4Reader_ReadNextAsUINT(env->appbinReader);
	opCache->prefix = prefix;
    	
    if(opCache->r0 >= CHNCPU_NUMBER_OF_IREG ||
	   opCache->r1 >= CHNCPU_NUMBER_OF_IREG ||
	   opCache->r2 >= CHNCPU_NUMBER_OF_IREG){
        env->errFlags |= CHNCPU_ERR_INVALID_REGNUM;
        return -1;
    }
    if(!CHNCPU_CHK_IsAvailableBits(env, opCache->bit)){
        return -1;
    }
	if((prefix | CHNCPU_PREFIX_MASK_Op_TernaryReg) != CHNCPU_PREFIX_MASK_Op_TernaryReg){
		env->errFlags |= CHNCPU_ERR_INVALID_PREFIX;
        return -1;
	}
    return 0;
}
int CHNCPU_Op_TernaryRegBitwise_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op)
{
    CHNCPU_OpCache_TernaryReg *opCache;
	int opCode;
	ch4_sint result, checked;
    
    opCache = op->opCache;
	opCode = op->opCode;

	if(opCode == 0x10){
		if(opCache->r1 == opCache->r2){
			// CP
			result = env->iReg[opCache->r1];
		} else{
			// OR
			result = env->iReg[opCache->r1] | env->iReg[opCache->r2];
		}
	} else if(opCode == 0x11){
		// XOR
		result = env->iReg[opCache->r1] ^ env->iReg[opCache->r2];
	} else if(opCode == 0x12){
		// AND
		result = env->iReg[opCache->r1] & env->iReg[opCache->r2];
	} else{
		env->errFlags |= CHNCPU_ERR_INTERNAL;
		return -1;
	}
	// check
	checked = result & (0xFFFFFFFF >> (32 - opCache->bit));
	if(!(opCache->prefix & CHNCPU_PREFIX_ALLOW_TRUNCATE) && checked != result){
		env->errFlags |= CHNCPU_ERR_TRUNCATED_VALUE;
		return -1;
	}
	// store
	if(!env->errFlags){
		env->iReg[opCache->r0] = checked;
		env->iRegBits[opCache->r0] = opCache->bit;
	}
	return 0;
}
int CHNCPU_Op_TernaryRegArithmetic_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op)
{
    CHNCPU_OpCache_TernaryReg *opCache;
	int opCode;
	ch4_sint result;
    
    opCache = op->opCache;
	opCode = op->opCode;
	
	if(opCode == 0x14){	// ADD
		result = env->iReg[opCache->r1] + env->iReg[opCache->r2];
	} else if(opCode == 0x15){	// SUB
		result = env->iReg[opCache->r1] - env->iReg[opCache->r2];
	} else if(opCode == 0x16){	// MUL
		result = env->iReg[opCache->r1] * env->iReg[opCache->r2];
	} else if(opCode == 0x18){	// SHL
		result = env->iReg[opCache->r1] << env->iReg[opCache->r2];
	} else if(opCode == 0x19){	// SAR
		result = env->iReg[opCache->r1] >> env->iReg[opCache->r2];
	} else if(opCode == 0x1A){	// DIV
		result = env->iReg[opCache->r1] / env->iReg[opCache->r2];
	} else if(opCode == 0x1B){	// MOD
		result = env->iReg[opCache->r1] % env->iReg[opCache->r2];
	} else{
		env->errFlags |= CHNCPU_ERR_INTERNAL;
		return -1;
	}
	// check
	CHNCPU_AdjustValueForBit(env, &result, opCache->bit, opCache->prefix);
	// store
	if(!env->errFlags){
		env->iReg[opCache->r0] = result;
		env->iRegBits[opCache->r0] = opCache->bit;
	}
	
	return 0;
	
	return 0;
}

char *CHNCPU_Op_TernaryReg_MnemonicList0[12] = {
	// 10 - 1B
	"OR",
	"XOR",
	"AND",
	"SBX",
	"ADD",
	"SUB",
	"MUL",
	"(Reserved)",
	"SHL",
	"SAR",
	"DIV",
	"MOD"
};

int CHNCPU_Op_TernaryReg_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file)
{
    CHNCPU_OpCache_TernaryReg *opCache;
    
    opCache = op->opCache;
    
	if(op->opCode == 0x10 && (opCache->r1 == opCache->r2)){
		// CP
		fprintf(file, "CP");
	} else if(0x10 <= op->opCode && op->opCode <= 0x1B){
		fprintf(file, "%s", CHNCPU_Op_TernaryReg_MnemonicList0[op->opCode - 0x10]);
	} else{
		fprintf(file, "(%02X)", op->opCode);
	}
	
	fprintf(file, "(r0:%02X, r1:%02X, r2:%02X, bit:%d);\n", opCache->r0, opCache->r1, opCache->r2, opCache->bit);
	
    return 0;
}

//
// Compare Integer Register Operation
//
int CHNCPU_Op_CompareIReg_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix)
{
    CHNCPU_OpCache_CompareIReg *opCache;
    
    opCache = malloc(sizeof(CHNCPU_OpCache_CompareIReg));
    op->opCache = opCache;
    
    opCache->rSrc1 = CH4Reader_ReadNextAsUINT(env->appbinReader);
    opCache->rSrc2 = CH4Reader_ReadNextAsUINT(env->appbinReader);
    opCache->bitDiff = CH4Reader_ReadNextAsUINT(env->appbinReader);
	opCache->rDst = CH4Reader_ReadNextAsUINT(env->appbinReader);
	opCache->bitDst = CH4Reader_ReadNextAsUINT(env->appbinReader);
	opCache->prefix = prefix;
	
    if(opCache->rSrc1 >= CHNCPU_NUMBER_OF_IREG ||
	   opCache->rSrc2 >= CHNCPU_NUMBER_OF_IREG ||
	   opCache->rDst >= CHNCPU_NUMBER_OF_IREG){
        env->errFlags |= CHNCPU_ERR_INVALID_REGNUM;
        return -1;
    }
    if(!CHNCPU_CHK_IsAvailableBits(env, opCache->bitDiff) ||
	   !CHNCPU_CHK_IsAvailableBits(env, opCache->bitDst)){
        return -1;
    }
	if((prefix | CHNCPU_PREFIX_MASK_Op_CompareIReg) != CHNCPU_PREFIX_MASK_Op_CompareIReg){
		env->errFlags |= CHNCPU_ERR_INVALID_PREFIX;
        return -1;
	}
    return 0;
}
int CHNCPU_Op_CompareIReg_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op)
{
    CHNCPU_OpCache_CompareIReg *opCache;
	int opCode;
	ch4_sint result;
	//ch4_sint diff;
	ch4_sint r1, r2;
    
    opCache = op->opCache;
	opCode = op->opCode;
	
	// check with bitDiff
	/*
	// この機能は未実装
	diff = env->iReg[opCache->rSrc1] - env->iReg[opCache->rSrc2];
	CHNCPU_AdjustValueForBit(env, &diff, opCache->bitDiff, opCache->prefix);
	if(env->errFlags){
		return -1;
	}
	 */
	
	r1 = env->iReg[opCache->rSrc1];
	r2 = env->iReg[opCache->rSrc2];
	
	if(opCode == 0x20){
		result = (r1 == r2) ? -1 : 0;
	} else if(opCode == 0x21){
		result = (r1 != r2) ? -1 : 0;
	} else if(opCode == 0x22){
		result = (r1 < r2) ? -1 : 0;
	} else if(opCode == 0x23){
		result = (r1 >= r2) ? -1 : 0;
	} else if(opCode == 0x24){
		result = (r1 <= r2) ? -1 : 0;
	} else if(opCode == 0x25){
		result = (r1 > r2) ? -1 : 0;
	} else if(opCode == 0x26){
		result = ((r1 & r2) == 0) ? -1 : 0;
	} else if(opCode == 0x27){
		result = ((r1 & r2) != 0) ? -1 : 0;
	} else{
		env->errFlags |= CHNCPU_ERR_INTERNAL;
		return -1;
	}
	// check with bitDst
	if(opCache->bitDst == 0){
		env->errFlags |= CHNCPU_ERR_TRUNCATED_VALUE;
		return -1;
	}
	// store
	if(!env->errFlags){
		env->iReg[opCache->rDst] = result;
		env->iRegBits[opCache->rDst] = opCache->bitDst;
	}
	return 0;
}

char *CHNCPU_Op_CompareIReg_MnemonicList0[8] = {
	// 20-27
	"CMPE",
	"CMPNE",
	"CMPL",
	"CMPGE",
	"CMPLE",
	"CMPG",
	"TSTZ",
	"TSTNZ",
};

int CHNCPU_Op_CompareIReg_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file)
{
    CHNCPU_OpCache_CompareIReg *opCache;
    
    opCache = op->opCache;
    
	if(0x20 <= op->opCode && op->opCode <= 0x27){
		fprintf(file, "%s", CHNCPU_Op_CompareIReg_MnemonicList0[op->opCode - 0x20]);
	} else{
		fprintf(file, "(%02X)", op->opCode);
	}
	
	fprintf(file, "(rDst:%02X, bitDst:%d r1:%02X, r2:%02X, bitDiff:%d);\n", opCache->rDst, opCache->bitDst, opCache->rSrc1, opCache->rSrc2, opCache->bitDiff);
	
    return 0;
}

//
// 2E DATA
//
int CHNCPU_Op_DATA_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix)
{
    CHNCPU_OpCache_DATA *opCache;
	int i;
	ch4_sint type;
	ch4_uint count;
    
    opCache = malloc(sizeof(CHNCPU_OpCache_DATA));
    op->opCache = opCache;
    
    opCache->labelID = CH4Reader_ReadNextAsUINT(env->appbinReader);
    opCache->labelOpt = CH4Reader_ReadNextAsUINT(env->appbinReader);
	type = CH4Reader_ReadNextAsSINT(env->appbinReader);
	count = CH4Reader_ReadNextAsUINT(env->appbinReader);
	
	opCache->data = CHNCPU_DATA_CreateEmptyData();
	CHNCPU_DATA_AllocateRawDataMemory(opCache->data, type, count, &env->errFlags);
	if(env->errFlags){
		return -1;
	}
	
	// Bind rawData
	if(type == CHNCPU_PType_UINT8){
		for(i = 0; i < count; i++){
			((unsigned char *)opCache->data->rawData)[i] = CH4Reader_ReadNextUINTn(env->appbinReader, 8);
		}
	}
	
	if(CHNCPU_Label_Add(env->labelSet, env->currentIndex, opCache->labelID, opCache->labelOpt, type)){
		env->errFlags |= CHNCPU_ERR_DUPLICATED_LABEL_ID;
	}
    
	if(prefix != 0){
		env->errFlags |= CHNCPU_ERR_INVALID_PREFIX;
        return -1;
	}
    return 0;
}
int CHNCPU_Op_DATA_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op)
{
	// 何もしない
    return 0;
}
int CHNCPU_Op_DATA_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file)
{
    CHNCPU_OpCache_DATA *opCache;
	opCache = op->opCache;
	
    fprintf(file, "DATA(labelID:0x%X, labelOpt:%d, dataType:%d, dataCount:%d);\n", opCache->labelID, opCache->labelOpt, opCache->data->type, opCache->data->count);
	
	CHNCPU_DATA_PrintData(opCache->data, file);
	
    return 0;
}
CHNCPU_Data *CHNCPU_Op_DATA_GetData(CHNCPU_OpTag *op)
{
	if(!op || op->opCode != CHNCPU_OP_DATA || !op->opCache){
        return NULL;
	}
	return ((CHNCPU_OpCache_DATA *)op->opCache)->data;
}

//
// 32 MALLOC
//
/*
int CHNCPU_Op_MALLOC_BindOperand(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, unsigned int prefix)
{
    CHNCPU_OpCache_MALLOC *opCache;
    
    opCache = malloc(sizeof(CHNCPU_OpCache_MALLOC));
    op->opCache = opCache;
    
    opCache->rType = CH4Reader_ReadNextAsUINT(env->appbinReader);
    opCache->bitType = CH4Reader_ReadNextAsUINT(env->appbinReader);
    opCache->rCount = CH4Reader_ReadNextAsUINT(env->appbinReader);
	opCache->bitCount = CH4Reader_ReadNextAsUINT(env->appbinReader);
	opCache->p = CH4Reader_ReadNextAsUINT(env->appbinReader);
    
    if(opCache->rType >= CHNCPU_NUMBER_OF_IREG ||
	   opCache->rCount >= CHNCPU_NUMBER_OF_IREG ||
	   opCache->p >= CHNCPU_NUMBER_OF_PREG){
        env->errFlags |= CHNCPU_ERR_INVALID_REGNUM;
        return -1;
    }
    if(!CHNCPU_CHK_IsAvailableBits(env, opCache->bitType) ||
	   !CHNCPU_CHK_IsAvailableBits(env, opCache->bitCount)){
        return -1;
    }
	if(prefix != 0){
		env->errFlags |= CHNCPU_ERR_INVALID_PREFIX;
        return -1;
	}
    return 0;
}

int CHNCPU_Op_MALLOC_Execute(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op)
{
	CHNCPU_OpTag *dataOpTag;
    CHNCPU_OpCache_MALLOC *opCache;
	ch4_sint type;
	ch4_uint count;
    
    opCache = op->opCache;
    
    type = env->iReg[opCache->rType];
	count = env->iReg[opCache->rCount];
	
	dataOpTag = CHNCPU_Op_DATA_CreateEmptyData();
	if(CHNCPU_Op_DATA_AllocateRawDataMemory(dataOpTag, type, count, &env->errFlags)){
		return -1;
	}
	
	return 0;
}
int CHNCPU_Op_MALLOC_PrintCode(CHNCPU_RuntimeEnvironment *env, CHNCPU_OpTag *op, FILE *file)
{
    CHNCPU_OpCache_LMEM *opCache;
    
    opCache = op->opCache;
    fprintf(file, "LMEM(p:0x%02X, pType:0x%X, pDiff:%d, r:%02X, bitDst:%d);\n", opCache->p, opCache->pType, opCache->pDiff, opCache->r, opCache->bitDst);
	
    return 0;
}
*/