﻿using System.Drawing;
using System.Diagnostics;
using MikuMikuMoving;
using MikuMikuMoving.Plugin;
using DxMath;
using System.Collections.Generic;
using System.Windows.Forms;
using System;

namespace MMM_GraphEditor
{
    partial class GraphPort
    {
        public class OutOfViewPort : Exception {}

        public class GraphPoint
        {
            public float X { get; set; }
            public float Y { get; set; }
            public int Type { get; set; }
            public CMotionFrameData Frame { get; set; }
            public CMotionFrameData PrevFrame { get; set; }
            public bool isRot()
            {
                if (Type == IDX_ROT_R || Type == IDX_ROT_P || Type == IDX_ROT_Y)
                    return true;
                return false;
            }
        }


        /// <summary> 
        /// 必要なデザイナー変数です。
        /// </summary>
        private System.ComponentModel.IContainer components = null;

        private MikuMikuPlugin.Scene scene;
        public MikuMikuPlugin.Scene Scene {
            get { return this.scene; }
            set { this.scene = value; updateTarget(); }
        }
        public int graph_padding = 30;
        public int frame_width = 12;
        public float graph_height = 20f;
        public float graph_max = 0f;
        public float graph_min = 0f;
        public float graph_scale = 10.0f;
        public float rot_scale = 1f;
        public long max_frame = 0;
        public static float Graph_PenSize = 2f;
        public static Pen Ip_Pen = Pens.Black;
        public float ip_mark_size = 4f;
        public Point cur_mouse_pos = new Point(0, 0);
        public static Color[] Graph_Colors = { Color.Red, Color.Green, Color.Blue, Color.DeepPink, Color.YellowGreen, Color.DodgerBlue };
        public static Brush[] Graph_Brushes = { Brushes.Red, Brushes.Green, Brushes.Blue,
                                                   Brushes.DeepPink, Brushes.YellowGreen, Brushes.DodgerBlue };
        public static Pen[] Graph_Pens = {
                                            new Pen(Graph_Brushes[0], Graph_PenSize),
                                            new Pen(Graph_Brushes[1], Graph_PenSize),
                                            new Pen(Graph_Brushes[2], Graph_PenSize),
                                            new Pen(Graph_Brushes[3], Graph_PenSize),
                                            new Pen(Graph_Brushes[4], Graph_PenSize),
                                            new Pen(Graph_Brushes[5], Graph_PenSize),
                                        };
        public static Pen Graph_Selected_Pen = new Pen(new SolidBrush(Color.FromArgb(128, 255, 255, 0)), 4);
        public const int IDX_POS_X = 0;
        public const int IDX_POS_Y = 1;
        public const int IDX_POS_Z = 2;
        public const int IDX_ROT_R = 3;
        public const int IDX_ROT_P = 4;
        public const int IDX_ROT_Y = 5;
        public const int IDX_POS_IXA = 6;
        public const int IDX_POS_IXB = 7;
        public const int IDX_POS_IYA = 8;
        public const int IDX_POS_IYB = 9;
        public const int IDX_POS_IZA = 10;
        public const int IDX_POS_IZB = 11;
        public static IEnumerable<int> XYZ = new int[3] { IDX_POS_X, IDX_POS_Y, IDX_POS_Z };
        public static IEnumerable<int> RPY = new int[3] { IDX_ROT_R, IDX_ROT_P, IDX_ROT_Y };
        public static IEnumerable<int> XYZRPY = new int[6] { IDX_POS_X, IDX_POS_Y, IDX_POS_Z, IDX_ROT_R, IDX_ROT_P, IDX_ROT_Y };
        public MotionLayer target_layer = null;
        public Bone target_bone = null;
        public Quaternion target_local_rot;
        public Quaternion target_local_rot_rev;
        public const int IDX_EDIT_POS = 0;
        public const int IDX_EDIT_POS_IP = 1;
        public const int IDX_EDIT_ROT = 2;
        public bool ip_val_limit = true;
        GraphPoint key_pointed = null;

        public bool[] render_flags
        {
            get { return config.render_flags; }
        }
        public bool[] edit_flags
        {
            get { return config.edit_flags; }
        }
        public int euler_rot_order
        {
            get { return config.euler_rot_order; }
        }
        public GraphEditor.Config config
        {
            get { return MainControl.config; }
        }

        /// <summary> 
        /// 使用中のリソースをすべてクリーンアップします。
        /// </summary>
        /// <param name="disposing">マネージ リソースが破棄される場合 true、破棄されない場合は false です。</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing && (components != null))
            {
                components.Dispose();
                this.scene = null;
                this.target_layer = null;
                this.target_bone = null;
            }
            base.Dispose(disposing);
        }

        protected override void OnPaint(System.Windows.Forms.PaintEventArgs e)
        {
            Trace.WriteLine("GraphPane: OnPaint");

            base.OnPaint(e);
            Graphics g = e.Graphics;
            renderGrid(g);
            renderGraph(g);
        }

        private void renderGrid(Graphics g) {
            if (scene == null) return;
            MotionLayer layer = target_layer;
            this.max_frame = scene.MarkerPosition;
            if (layer != null && layer.Frames.Count > 0)
            {
                long fnum = layer.Frames[layer.Frames.Count - 1].FrameNumber;
                Trace.WriteLine("Max frame number: " + fnum.ToString());
                if (fnum > this.max_frame)
                    this.max_frame = fnum;
            }

            autoSetHeight();

            int width = (int) (this.max_frame +1) * frame_width -1;
            this.Width = width;

            g.FillRectangle(Brushes.LightGray, 0, 0, width, Height);
            
            int cur;
            bool has_label = false;
            bool is_em_tic = false;
            for (cur = 0; cur * frame_width < width; cur++) {
                has_label = (cur > 0 && cur % 10 == 0);
                is_em_tic = (cur % 5 == 0);
                g.DrawLine(scene != null && cur == scene.MarkerPosition ? Pens.YellowGreen :
                    has_label ? Pens.White : 
                    is_em_tic ? Pens.WhiteSmoke : Pens.Gainsboro,
                    cur * frame_width, 0, cur * frame_width, Height);
            }
            g.DrawLine(Pens.Gray, 0, graph_R((float)Math.PI), width, graph_R((float)Math.PI));
            g.DrawLine(Pens.Gray, 0, graph_R((float)-Math.PI), width, graph_R((float)-Math.PI));
            g.DrawLine(Pens.WhiteSmoke, 0, graph_Y(0f), width, graph_Y(0f));
        }

        private void renderGraph(Graphics g)
        {
            if (target_layer == null) return;
            
            this.key_pointed = null;
            CMotionFrameData prev = null;
            foreach (CMotionFrameData fd in target_layer.Frames)
            {
                Trace.WriteLine("Frame: " + fd.FrameNumber.ToString() +
                    " X = " + fd.Position.X.ToString() +
                    " Y = " + fd.Position.Y.ToString() +
                    " Z = " + fd.Position.Z.ToString());
                Trace.WriteLine("Frame: " + fd.FrameNumber.ToString() +
                    " RX = " + fd.Quaternion.X.ToString() +
                    " RY = " + fd.Quaternion.Y.ToString() +
                    " RZ = " + fd.Quaternion.Z.ToString() +
                    " RAngle = " + fd.Quaternion.Angle.ToString() 
                    );
                try
                {
                    plotKeyFrame(g, fd, prev);
                }
                catch (OutOfViewPort)
                {
                    break;
                }
                prev = fd;
            }

            int h_size = 8;
            GraphPoint h_key = null;
            if (key_move_target != null)
                h_key = key_move_target;
            else if (key_pointed != null)
                h_key = key_pointed;
            if (h_key != null)
                g.DrawEllipse(Graph_Selected_Pen, h_key.X - h_size, h_key.Y - h_size, h_size * 2, h_size * 2);
        }

        private float graph_Y(float v) {
            return graph_height + graph_padding - (v - graph_min) * graph_scale;
        }

        private float graph_R(float v)
        {
            return graph_Y(v*rot_scale);
        }

        private float pos_val_from_graph_Y(float y)
        {
            return (-y + graph_height + graph_padding) / graph_scale + graph_min;
        }

        private float rot_val_from_graph_Y(float y)
        {
            return (-y + graph_height + graph_padding + graph_min * graph_scale) / graph_scale / rot_scale;
        }

        private MikuMikuPlugin.InterpolatePoint ip_val(Point pos, int axis, CMotionFrameData to, CMotionFrameData from)
        {
            float span_x = (to.FrameNumber - from.FrameNumber) * frame_width;
            float span_y = graph_Y(to.Position[axis]) - graph_Y(from.Position[axis]);
            float ip_x = (pos.X - from.FrameNumber * frame_width) / span_x * 128f;
            float ip_y = (pos.Y - graph_Y(from.Position[axis])) / span_y * 128f;
            if (ip_x < 0) ip_x = 0;
            if (ip_x > 128) ip_x = 128;
            if (ip_val_limit && ip_y < 0) ip_y = 0;
            if (ip_val_limit && ip_y >= 128) ip_y = 128;

            return new MikuMikuPlugin.InterpolatePoint((int)ip_x, (int)ip_y);
        }

        private void plotKeyFrame(Graphics g, CMotionFrameData fd, CMotionFrameData prev)
        {
            try
            {
                int cur = -PanelOuter.AutoScrollPosition.X;
                if (fd.FrameNumber * frame_width < cur)
                    return; // skip plotting outside view port (prev)
                if (prev != null && prev.FrameNumber * frame_width > cur + PanelOuter.Width)
                    throw new OutOfViewPort();
            }
            catch (System.Exception)
            {
                // ignore
                return;
            }

            int size = 8;
            float cur_x = fd.FrameNumber * frame_width;

            if (target_bone.BoneFlags.HasFlag(MikuMikuPlugin.BoneType.XYZ) &&
                (render_flags[IDX_POS_X] || render_flags[IDX_POS_Y] || render_flags[IDX_POS_Z]))
            {
                Vector3 pos = scene.MarkerPosition == fd.FrameNumber ? target_layer.CurrentLocalMotion.Move : fd.Position;
                foreach (int i in XYZ)
                {
                    if (!render_flags[i]) continue;
                    float cur_y = graph_Y(pos[i]);
                    g.FillRectangle(Graph_Brushes[i], cur_x - (size / 2), cur_y - (size / 2), size, size);
                    if (!edit_flags[IDX_EDIT_POS]) continue;
                    if (cur_mouse_pos.X - frame_width/2 < cur_x && cur_x < cur_mouse_pos.X + frame_width/2 &&
                        cur_mouse_pos.Y - size < cur_y && cur_y < cur_mouse_pos.Y + size)
                    {
                        this.key_pointed = new GraphPoint{Type =i, Frame = fd, X = cur_x, Y = cur_y};
                    }
                    if (key_move_target != null && key_move_target.Type == i && key_move_target.Frame.FrameNumber == fd.FrameNumber)
                        this.key_move_target = new GraphPoint { Type = i, Frame = fd, X = cur_x, Y = cur_y };
                }
                drawInterpolateLine_Pos(g, prev, fd);
            }


            if (target_bone.BoneFlags.HasFlag(MikuMikuPlugin.BoneType.Rotate) &&
                (render_flags[IDX_ROT_P] || render_flags[IDX_ROT_R] || render_flags[IDX_ROT_Y]))
            {
                plotRot(g, fd, prev);
            }
        }

        private void plotRot(Graphics g, CMotionFrameData fd, CMotionFrameData prev)
        {
            int size = 8;
            float cur_x = fd.FrameNumber * frame_width;

            Quaternion rot = scene.MarkerPosition == fd.FrameNumber ? target_layer.CurrentLocalMotion.Rotation : fd.Quaternion;
            rot = Rot_World2Local(rot);
            int rot_type = euler_rot_order;
            Vector3 erot = Quaternion2Euler(rot, rot_type);
            Trace.WriteLine("Euler Rot: " + erot.ToString());
            foreach (int i in RPY)
            {
                int a = i - 3;
                if (!render_flags[i]) continue;
                float cur_y = graph_R(erot[a]);
                g.FillEllipse(Graph_Brushes[i], cur_x - (size / 2), cur_y - (size / 2), size, size);
                if (!edit_flags[IDX_EDIT_ROT]) continue;
                if (cur_mouse_pos.X - frame_width / 2 < cur_x && cur_x < cur_mouse_pos.X + frame_width / 2 &&
                    cur_mouse_pos.Y - size < cur_y && cur_y < cur_mouse_pos.Y + size)
                {
                    this.key_pointed = new GraphPoint{Type = i, Frame = fd, X = cur_x, Y= cur_y};
                }
                if (key_move_target != null && key_move_target.Type == i && key_move_target.Frame.FrameNumber == fd.FrameNumber)
                    this.key_move_target = new GraphPoint { Type = i, Frame = fd, X = cur_x, Y = cur_y }; ;
            }
            drawInterpolateLine_Rot(g, prev, fd, rot_type);
        }

        private Quaternion CancelCascadedParentsRot(float frameNumber, Quaternion cur_rot, int parent_bone_id)
        {
            if (parent_bone_id < 0) return cur_rot;
            MikuMikuPlugin.Bone parent_bone = scene.ActiveModel.Bones[parent_bone_id];
            if (parent_bone == null) return cur_rot;
            foreach (MotionLayer ml in parent_bone.Layers)
            {
                if (ml.Frames.Count == 0) continue;

                CMotionFrameData from = null, to = null;
                foreach (CMotionFrameData mfd in ml.Frames)
                {
                    if (mfd.FrameNumber <= frameNumber)
                    {
                        to = from = mfd;
                        continue;
                    }
                    if (mfd.FrameNumber > frameNumber)
                    {
                        to = mfd;
                        break;
                    }
                }
                if (from == null) continue;
                if (cur_rot.Axis.Equals(Vector3.Zero)) continue;

                if (from.FrameNumber == to.FrameNumber || from.FrameNumber == frameNumber)
                {
                    if (from.Quaternion.Axis.Equals(Vector3.Zero)) continue;
                    Vector4 new_axis = Vector3.Transform(cur_rot.Axis, from.Quaternion);
                    cur_rot = Quaternion.Normalize(Quaternion.RotationAxis(new Vector3(new_axis.X, new_axis.Y, new_axis.Z), cur_rot.Angle));
                }
                else
                {
                    if (from.Quaternion.Axis.Equals(Vector3.Zero) && to.Quaternion.Axis.Equals(Vector3.Zero)) continue;
                    double x = (frameNumber - from.FrameNumber) / (to.FrameNumber - from.FrameNumber);
                    double t = 0;
                    double tx = 0;
                    while (tx < x && t < 1)
                    {
                        t += 0.01f;
                        tx = (t * t * t + 3 * t * t * (1 - t) * to.InterpolRB.X / 128f + 3 * t * (1 - t) * (1 - t) * to.InterpolRA.X / 128f);
                    }
                    float amount = (float) (t * t * t + 3 * t * t * (1 - t) * to.InterpolRB.Y / 128 + 3 * t * (1 - t) * (1 - t) * to.InterpolRA.Y / 128);
                    Vector4 new_axis = Vector3.Transform(cur_rot.Axis, Quaternion.Slerp(from.Quaternion, to.Quaternion, amount));
                    cur_rot = Quaternion.Normalize(Quaternion.RotationAxis(new Vector3(new_axis.X, new_axis.Y, new_axis.Z), cur_rot.Angle));
                }
            }
            cur_rot = CancelCascadedParentsRot(frameNumber, cur_rot, parent_bone.ParentBoneID);
            return cur_rot;
        }

        private void drawInterpolateLine_Pos(Graphics g, CMotionFrameData from, CMotionFrameData to)
        {
            if (from == null) return;
            if (!(target_bone.BoneFlags.HasFlag(MikuMikuPlugin.BoneType.XYZ) &&
                (render_flags[IDX_POS_X] || render_flags[IDX_POS_Y] || render_flags[IDX_POS_Z])))
                return;
            MikuMikuPlugin.InterpolatePoint[] interpolations = { to.InterpolXA, to.InterpolXB, to.InterpolYA, to.InterpolYB, to.InterpolZA, to.InterpolZB };

            float span_x = (to.FrameNumber - from.FrameNumber) * frame_width;
            float from_x = from.FrameNumber * frame_width;
            float to_x = to.FrameNumber * frame_width;

            foreach (int i in XYZ)
            {
                if (!render_flags[i]) continue;
                Vector3 pos_from = scene.MarkerPosition == from.FrameNumber ? target_layer.CurrentLocalMotion.Move : from.Position;
                Vector3 pos_to = scene.MarkerPosition == to.FrameNumber ? target_layer.CurrentLocalMotion.Move : to.Position;
                float span_y = graph_Y(pos_to[i]) - graph_Y(pos_from[i]);
                float from_y = graph_Y(pos_from[i]);
                float ip_ax  = from_x + (interpolations[i * 2].X / 128f * span_x);
                float ip_ay  = from_y + (interpolations[i * 2].Y / 128f * span_y);
                float ip_bx  = from_x + (interpolations[i*2+1].X / 128f * span_x);
                float ip_by  = from_y + (interpolations[i*2+1].Y / 128f * span_y);
                float to_y   = graph_Y(pos_to[i]);
                g.DrawLine(Ip_Pen, from_x, from_y, ip_ax, ip_ay);
                g.DrawLine(Ip_Pen, ip_bx, ip_by, to_x, to_y);
                g.DrawRectangle(Ip_Pen, ip_ax - ip_mark_size/2, ip_ay - ip_mark_size/2, ip_mark_size, ip_mark_size);
                g.DrawRectangle(Ip_Pen, ip_bx - ip_mark_size/2, ip_by - ip_mark_size/2, ip_mark_size, ip_mark_size);
                g.DrawBezier(Graph_Pens[i], from_x, from_y, ip_ax, ip_ay, ip_bx, ip_by, to_x, to_y);

                if (!edit_flags[IDX_EDIT_POS_IP]) continue;
                if (cur_mouse_pos.X - frame_width / 2 < ip_ax && ip_ax < cur_mouse_pos.X + frame_width / 2 &&
                    cur_mouse_pos.Y - frame_width / 2 < ip_ay && ip_ay < cur_mouse_pos.Y + frame_width / 2)
                {
                    this.key_pointed = new GraphPoint { Type = IDX_POS_IXA + i * 2, Frame = to, PrevFrame = from, X = ip_ax, Y = ip_ay };
                }
                else if (cur_mouse_pos.X - frame_width / 2 < ip_bx && ip_bx < cur_mouse_pos.X + frame_width / 2 &&
                         cur_mouse_pos.Y - frame_width / 2 < ip_by && ip_by < cur_mouse_pos.Y + frame_width / 2)
                {
                    this.key_pointed = new GraphPoint { Type = IDX_POS_IXA + i * 2 + 1, Frame = to, PrevFrame = from, X = ip_bx, Y = ip_by };
                }
                if (key_move_target != null && key_move_target.Frame.FrameNumber == to.FrameNumber)
                {
                    if (key_move_target.Type == IDX_POS_IXA + i * 2)
                    {
                        this.key_move_target = new GraphPoint { Type = IDX_POS_IXA + i * 2, Frame = to, PrevFrame = from, X = ip_ax, Y = ip_ay };
                    }
                    else if (key_move_target.Type == IDX_POS_IXA + i * 2 + 1)
                    {
                        this.key_move_target = new GraphPoint { Type = IDX_POS_IXA + i * 2 + 1, Frame = to, PrevFrame = from, X = ip_bx, Y = ip_by };
                    }
                }
            }
        }

        private void drawInterpolateLine_Rot(Graphics g, CMotionFrameData from, CMotionFrameData to, int rot_type)
        {
            if (from == null) return;
            if (!(target_bone.BoneFlags.HasFlag(MikuMikuPlugin.BoneType.Rotate) &&
                (render_flags[IDX_ROT_Y] || render_flags[IDX_ROT_R] || render_flags[IDX_ROT_P])))
                return;

            int panel_start_x = -PanelOuter.AutoScrollPosition.X;
            int panel_end_x = panel_start_x + PanelOuter.Width;
            Quaternion rot_from = scene.MarkerPosition == from.FrameNumber ? target_layer.CurrentLocalMotion.Rotation : from.Quaternion;
            Quaternion rot_to = scene.MarkerPosition == to.FrameNumber ? target_layer.CurrentLocalMotion.Rotation : to.Quaternion;
            rot_from = Rot_World2Local(rot_from);
            rot_to = Rot_World2Local(rot_to);
            Vector3 erot_from = Quaternion2Euler(rot_from, rot_type);
            float from_x = from.FrameNumber * frame_width;
            float span_x = (to.FrameNumber - from.FrameNumber) * frame_width;
            float to_x = to.FrameNumber * frame_width;

            double t = 0;
            double tx = 0;
            double amount = 0;
            for (float i = 0; i < span_x; i++)
            {
                if (from_x + i < panel_start_x) continue;
                if (from_x + i > panel_end_x) throw new OutOfViewPort();

                while (tx < i && t < 1)
                {
                    t += 0.01f;
                    tx = (t * t * t + 3 * t * t * (1 - t) * to.InterpolRB.X / 128 + 3 * t * (1 - t) * (1 - t) * to.InterpolRA.X / 128) * span_x;
                }
                amount = (t * t * t + 3 * t * t * (1 - t) * to.InterpolRB.Y / 128 + 3 * t * (1 - t) * (1 - t) * to.InterpolRA.Y / 128);
                Quaternion mid_rq = Quaternion.Slerp(rot_from, rot_to, (float) amount);
                Vector3 mid_erot = Quaternion2Euler(mid_rq, rot_type);

                foreach (int ax in RPY)
                {
                    if (!render_flags[ax]) continue;
                    g.FillRectangle(Graph_Brushes[ax], from_x + i, graph_R(mid_erot[ax - 3]), 1, 2);
                }
            }
        }

        public Quaternion Rot4LocalGeo(MikuMikuPlugin.Bone bone)
        {
            Vector3 axis_world, axis_local;
            if (bone.LocalAxisX.X != 1.0f)
            {
                axis_world = new Vector3(1, 0, 0);
                axis_local = bone.LocalAxisX;
            }
            else if (bone.LocalAxisY.Y != 1.0f)
            {
                axis_world = new Vector3(0, 1, 0);
                axis_local = bone.LocalAxisY;
            }
            else if (bone.LocalAxisZ.Z != 1.0f)
            {
                axis_world = new Vector3(0, 0, 1);
                axis_local = bone.LocalAxisZ;
            }
            else
            {
                return Quaternion.Identity;
            }
            return Quaternion.Normalize(Quaternion.RotationAxis(Vector3.Cross(axis_world, axis_local), (float) Math.Acos(Vector3.Dot(axis_world, axis_local))));
        }


        public Quaternion Rot_World2Local(Quaternion rot)
        {
            Vector4 axis = Vector3.Transform(rot.Axis, target_local_rot_rev);
            return Quaternion.Normalize(Quaternion.RotationAxis(new Vector3(axis.X, axis.Y, axis.Z), rot.Angle));
        }

        private void updateTarget()
        {
            MotionLayer layer = null;
            Bone bone = null;
            if (this.scene == null || this.scene.ActiveModel == null)
            {
                this.target_bone = null;
                this.target_layer = null;
                return;
            }
            if (scene.ActiveModel != null)
            {
                foreach (Bone b in scene.ActiveModel.Bones)
                {
                    foreach (MotionLayer l in b.SelectedLayers)
                    {
                        bone = b;
                        layer = l;
                    }
                }
            }
            
            if (layer != null)
            {
                this.target_layer = layer;
                this.target_bone = bone;
                this.target_local_rot = Rot4LocalGeo(target_bone);
                this.target_local_rot_rev = Quaternion.Conjugate(target_local_rot);
            }
            else
            {
                this.target_bone = null;
                this.target_layer = null;
            }
        }

        private void autoSetGraphRange() {
            MotionLayer layer = target_layer;
            if (layer == null) return;
            float max =  3.2f * rot_scale;
            float min = -3.2f * rot_scale;
            bool check_cur_move = false;
            foreach (CMotionFrameData fd in layer.Frames)
            {
                if (fd.FrameNumber == this.scene.MarkerPosition)
                    check_cur_move = true;
                foreach (int i in XYZ)
                {
                    if (max < fd.Position[i])
                        max = fd.Position[i];
                    if (min > fd.Position[i])
                        min = fd.Position[i];
                }
            }
            if (check_cur_move)
            {
                foreach (int i in XYZ)
                {
                    float cur = this.target_layer.CurrentLocalMotion.Move[i];
                    if (max < cur)
                        max = cur;
                    if (min > cur)
                        min = cur;
                }
            }
            this.graph_height = (max - min) * graph_scale;
            this.graph_min = min;
            this.graph_max = max;
        }

        private void autoSetHeight()
        {
            autoSetGraphRange();
            this.Height = (int) graph_height + graph_padding * 2;
        }

        private int getMinDeltaAxis(Quaternion from, Quaternion to)
        {
            float span_x = Math.Abs(from.X - to.X);
            float span_y = Math.Abs(from.Y - to.Y);
            float span_z = Math.Abs(from.Z - to.Z);

            if (span_x <= span_y && span_x <= span_z)
                return 0;
            if (span_y <= span_x && span_y <= span_z)
                return 1;
            if (span_z <= span_x && span_z <= span_y)
                return 2;
            return 0;
        }

        private Vector3 Quaternion2Euler(Quaternion q, int min_axis)
        {
            Matrix rm = Quaternion2RotMatrix(q);
            double r, p, y;
            if (min_axis == 0) // Z-X-Y
            {
                r = -Math.Asin(rm.M32);
                p = Math.Atan2(rm.M31, rm.M33);
                y = Math.Atan2(rm.M12, rm.M22);
            }
            else if (min_axis == 1) // X-Y-Z
            {
                r = Math.Atan2(-rm.M32, rm.M33);
                p = -Math.Asin(rm.M13);
                y = Math.Atan2(rm.M12, rm.M11);
            }
            else // X-Z-Y
            {
                r = Math.Atan2(-rm.M32, rm.M22);
                p = Math.Atan2(-rm.M13, rm.M11);
                y = Math.Asin(rm.M12);
            }
            return new Vector3((float)-r, (float)-p, (float)-y);
        }

        private Matrix Quaternion2RotMatrix(Quaternion q)
        {
            Matrix m = new Matrix();
            m.M11 = (float) (1 - 2 * Math.Pow(q.Y, 2) - 2 * Math.Pow(q.Z, 2));
            m.M12 = 2 * q.X * q.Y - 2 * q.W * q.Z;
            m.M13 = 2 * q.X * q.Z + 2 * q.W * q.Y;
            m.M14 = 0;
            m.M21 = 2 * q.X * q.Y + 2 * q.W * q.Z;
            m.M22 = (float) (1 - 2 * Math.Pow(q.X, 2) - 2 * Math.Pow(q.Z, 2));
            m.M23 = 2 * q.Y * q.Z - 2 * q.W * q.X;
            m.M24 = 0;
            m.M31 = 2 * q.X * q.Z - 2 * q.W * q.Y;
            m.M32 = 2 * q.Y * q.Z + 2 * q.W * q.X;
            m.M33 = (float) (1 - 2 * Math.Pow(q.X, 2) - 2 * Math.Pow(q.Y, 2));
            m.M34 = 0;
            m.M41 = 0;
            m.M42 = 0;
            m.M43 = 0;
            m.M44 = 1;
            return m;
        }

        #region コンポーネント デザイナーで生成されたコード

        /// <summary> 
        /// デザイナー サポートに必要なメソッドです。このメソッドの内容を 
        /// コード エディターで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.SuspendLayout();
            // 
            // GraphPort
            // 
            this.AutoScaleDimensions = new System.Drawing.SizeF(6F, 12F);
            this.AutoScaleMode = System.Windows.Forms.AutoScaleMode.Font;
            this.CausesValidation = false;
            this.DoubleBuffered = true;
            this.Name = "GraphPort";
            this.Size = new System.Drawing.Size(100, 100);
            this.MouseDown += new System.Windows.Forms.MouseEventHandler(this.mouse_down);
            this.MouseMove += new System.Windows.Forms.MouseEventHandler(this.mouse_move);
            this.MouseUp += new System.Windows.Forms.MouseEventHandler(this.mouse_up);
            this.PreviewKeyDown += new System.Windows.Forms.PreviewKeyDownEventHandler(this.key_down);
            this.ResumeLayout(false);

        }

        #endregion

    }
}
