#!/usr/bin/python
# -*- coding: utf-8 -*-

import os.path
import pickle
import datetime

from winazurestorage import CLOUD_TABLE_HOST
import wazs
from config import config

class Tweets(object):
    def __init__(self, ids=[], upper={}):
        self.tweet_of = upper
        self.sorted_id = ids

    def __iter__(self):
        self.index = self.sorted_id.__iter__()
        return self

    def next(self):
        if self.index == None:
            raise StopIteration
        next_id  = self.index.next()
        return self.tweet_of[next_id]

    def get_subtweets(self, start, length):
        ids = self.sorted_id[start:start+length]
        return Tweets(ids, self.tweet_of)

    def __len__(self):
        return len(self.sorted_id)

    def __getitem__(self, key):
        tid = self.sorted_id[key]
        return self.tweet_of[tid]

class TweetDB(Tweets):
    def __init__(self):
        Tweets.__init__(self)
        self._is_dirty = {}
        self._ts = None

    def sort_localcache(self):
        self.sorted_id.sort()
        self.sorted_id.reverse()

    def append(self, tweet):
        "tweet: Tweet"
        if not tweet.tweet_id in self.tweet_of:
            self.sorted_id.append(tweet.tweet_id)
        self.tweet_of[tweet.tweet_id] = tweet
        self._is_dirty[tweet.tweet_id] = False
        e = tweet.to_entity()
        self._update_table(e)

    @classmethod
    def from_cache(cls):
        cache_pathname = os.path.join(config["cache_dir"], "tweetcache")
        try:
            f = open(cache_pathname, "rb")
            db = pickle.load(f)
            f.close()
        except (IOError, EOFError, ValueError, TypeError):
            db = TweetDB()
            db.load_from_db()
            db.create_cache()
        return db

    def create_cache(self):
        cache_pathname = os.path.join(config["cache_dir"], "tweetcache")
        try:
            f = open(cache_pathname, "wb")
            pickle.dump(self, f)
            f.close()
        except IOError, e:
            print e

    def load_from_db(self):
        ts = self._get_ts()
        entities = ts.get_all(config["tweet_db"])
        for e in entities:
            tw = Tweet.from_entity(e)
            self.tweet_of[tw.tweet_id] = tw
            self.sorted_id.append(tw.tweet_id)
            self._is_dirty[tw.tweet_id] = False

    def update_db(self):
        for tid in self.tweet_of:
            if self._is_dirty[tid]:
                e = tweet_of[tid].to_entity()
                self._update_table(e)
                self._is_dirty[tid] = False

    def update_db_all(self):
        for tid in self.tweet_of:
            e = tweet_of[tid].to_entity()
            self._update_table(e)
            self._is_dirty[tid] = False

    def _get_ts(self):
        if self._ts == None:
            self._ts = wazs.TableStorage(CLOUD_TABLE_HOST,
                                         config["storage_account"],
                                         config["storage_key"])
        return self._ts

    def _update_table(self, entity):
        ts = self._get_ts()
        ts.update_entity(config["tweet_db"],
                         entity.partition_key,
                         entity.row_key,
                         entity)

class Tweet(object):
    def __init__(self, image_url, user_id, tweet_id, created_at, text, original_url):
        self.image_url = image_url
        self.user_id = user_id
        self.tweet_id = tweet_id
        self.created_at = created_at
        self.text = text
        self.original_url = original_url

    def to_entity(self):
        e = wazs.TableEntity("", "", {})
        e.partition_key = "tweet"
        e.row_key = self.tweet_id
        e.add_property("ImageURL", self.image_url)
        e.add_property("UserID", self.user_id)
        e.add_property("CreatedAt", self.created_at)
        e.add_property("Text", self.text)
        e.add_property("OriginalImageUrl", self.original_url)
        return e

    def get_localtime(self):
        t = self.created_at + datetime.timedelta(hours=9)
        return t

    @classmethod
    def from_entity(cls, entity):
        tw = Tweet(entity.ImageURL,
                   entity.UserID,
                   long(entity.RowKey),
                   entity.CreatedAt,
                   entity.Text,
                   entity.OriginalImageUrl)
        return tw

