// export.js
/*
 * usage:
 * node export-single.js <target directory> <output directory>
 */

var config = require('../../config');
var path = require('path');
var fs = require('fs');
var child_process = require('child_process');
var util = require('util');

var HTML_HEADER = 'header.txt';
var HTML_FOOTER = 'footer.txt';
var TEXT_HEADER = 'honmon_header.txt';
var IMAGE_DIR = 'figure';

var headerFullPath = path.join(path.dirname(module.filename), HTML_HEADER);
var footerFullPath = path.join(path.dirname(module.filename), HTML_FOOTER);
var textHeaderFullPath = path.join(path.dirname(module.filename), TEXT_HEADER);

var HTML_HEAD = fs.readFileSync(headerFullPath, {encoding: 'utf8'});
var HTML_FOOT = fs.readFileSync(footerFullPath, {encoding: 'utf8'});
var TEXT_HEAD = fs.readFileSync(textHeaderFullPath, {encoding: 'utf8'});

var usage = 'node export-single.js <target file> <output directory>';


function copyFiles(filelist, from, dest) {
  // create output directory
  if (!fs.existsSync(dest)) {
    fs.mkdirSync(dest);
  }

  filelist.forEach(function (target) {
    var fromPathname = path.join(from, target);
    var destPathname = path.join(dest, target);

    if (!fs.existsSync(fromPathname)) {
      console.log(fromPathname, 'not exists');
    } else {
      // copy file
      fs.createReadStream(fromPathname).
        pipe(fs.createWriteStream(destPathname));
    }
  });
}

function exportSingle(target, output, chapter, pageStart, pageCount) {
  chapter = chapter || 1;
  pageStart = pageStart || 0;
  pageCount = pageCount || 2;
  var pageEnd = pageStart + pageCount - 1;

  var textHeader = TEXT_HEAD;
  textHeader = textHeader.replace('<CHAPTER>', chapter)
    .replace('<PAGE_START>', pageStart)
    .replace('<PAGE_END>', pageEnd)
    .replace('<PAGE_COUNT>', pageCount);

  fs.readFile(target, 'utf8', function (err, data) {
    console.log(target);
    if (err) {
      console.log('Cannot read file: ' + target);
      return;
    }

    var result = parseText(data, target);

    // create output directory
    if (!fs.existsSync(output)) {
      fs.mkdirSync(output);
    }

    // output honmon, caption
    fs.writeFileSync(path.join(output, 'honmon.txt'),
                     textHeader + result.bodies.join('\n'));

    if (result.captions.length != 0) {
      fs.writeFileSync(path.join(output, 'caption.txt'),
                       result.captions.join('\n'));
      // output html
      var out = fs.createWriteStream(path.join(output, 'figures.html'), {flags: 'w', encoding: 'utf8', mode: 0666});
      out.write(HTML_HEAD);
      out.write('\n');
      out.write(result.htmls.join('\n'));
      out.write('\n');
      out.write(HTML_FOOT);
      out.end();
    }

    // file copy
    var copyFrom = path.join(path.dirname(target), 'figure');
    if (result.figures.length != 0) {
      copyFiles(result.figures, copyFrom, path.join(output, 'figures'));
    }

  });
}

function findFigureFile(caption, imageDir) {
  var figId = caption.match(/^☆図([0-9]+-[0-9]+)/);
  var figName = 'fig' + figId[1] + '.fw.png';
  if (isExistsFigure(figName, imageDir)) {
    return figName;
  }
  return '';
}

function findImageFile(name, targetDir) {
  if (fs.existsSync(path.join(targetDir, name))) {
    return name;
  }
  var files = fs.readdirSync(targetDir);
  var exts = config.figureFormat;
  for (var i = 0; i < files.length; i++) {
    for (var j = 0; j < exts.length; j++) {
      if (files[i].indexOf(name + exts[j]) != 0) {
        return name + exts[j];
      }
    }
  }
  return '';
}

// split body-text and others
function splitBody(lines) {
  // split honmon and caption
  var counter = 0;
  var isCaption = false;

  var bodies = [];
  var captions = [];

  for (i = 0; i < lines.length; i++) {
    if (lines[i].match(/^＞＞＞＞/)) {
      captions.push(lines[i]);
      i++;
      while (!lines[i].match(/^＞＞＞＞/)) {
        captions.push(lines[i]);
        i++;
      }
      continue;
    }

    if (i >= lines.length) {
      break;
    }

    // cleanup blank line
    if (lines[i].match(/^\s+$/)) {
      lines[i] = '';
    }

    // end
    if ( lines[i] === ''
         && bodies.length > 0
         && bodies[bodies.length-1] === '' ) {
      // do nothing
    } else {
      // *図、*表を置換
      lines[i] = lines[i].replace(/\*図/g, '★図');
      lines[i] = lines[i].replace(/\*表/g, '★表');
      bodies.push(lines[i]);
    }
  }

  var result = {
    bodies: bodies,
    blockItems: captions
  }
  return result;
}

function createFigureList(lines, imageDir) {
  var figureList = [];
  var figMode = false;
  var currentFigure;

  for (var i = 0; i < lines.length; i++) {
    if (lines[i].match(/^＞＞＞＞/)) {
      figMode = false;
      continue;
    }

    if (lines[i].match(/^＝＝＝＝/)) {
      figMode = false;
      continue;
    }

    if (lines[i].match(/^☆図/)) {
      figMode = true;
      currentFigure = findFigureFile(lines[i], imageDir);
      if (currentFigure != '') {
        figureList.push(currentFigure);
      }
      continue;
    }

    if (figMode) {
      if (lines[i].match(/^\s*$/)) {
        continue;
      }
      var fullPath = findImageFile(lines[i], imageDir);
      figureList.push(fullPath);
      continue;
    }
  }
  return figureList;
}

function extractCaptions(lines, imageDir) {
  var captions = [];
  var figMode = false;
  var currentFigure;

  for (var i = 0; i < lines.length; i++) {
    if (lines[i].match(/^＞＞＞＞/)) {
      figMode = false;
      continue;
    }

    if (lines[i].match(/^＝＝＝＝/)) {
      figMode = false;
      captions.push('');
      continue;
    }

    if (lines[i].match(/^☆図/)) {
      captions.push('');
      captions.push(lines[i]);
      figMode = true;
      currentFigure = findFigureFile(lines[i], imageDir);
      /*
      if (currentFigure != '') {
        captions.push(currentFigure);
      }
      */
      continue;
    }

    if (figMode) {
      if (currentFigure != '') {
        continue;
      }
      if (lines[i].match(/^\s*$/)) {
        continue;
      }
      var fullPath = findImageFile(lines[i], imageDir);
      /*
      captions.push(fullPath);
      */
      continue;
    }
    captions.push(lines[i]);
  }
  return captions;
}

function isExistsFigure(figName, targetDir) {
  var fullPath = path.join(targetDir, figName);
  return fs.existsSync(fullPath);
}

function makeHtmls(lines, imageDir) {
  var html = [];
  var figMode = false;
  var figTextMode = false;
  var assumedFigName;
  var figName
  var figId;
  for (var i = 0; i < lines.length; i++) {

    if (lines[i].match(/^＞＞＞＞/)) {
      figMode = false;
      continue;
    }

    // figure
    if (lines[i].match(/^☆図/)) {
      figId = lines[i].match(/^☆図([0-9]+-[0-9]+)/);
      console.log(lines[i]);
      assumedFigName = 'fig' + figId[1] + '.fw.png';
      figMode = true;
      if (isExistsFigure(assumedFigName, imageDir)) {
        html.push('<span class="caption">' + lines[i] + '</span>');
        html.push('<img src="figures/' + assumedFigName + '">');
        html.push('<span class="filename">' + assumedFigName + '</span>');
        i++;
        while((i < lines.length) && (!lines[i].match(/^＞＞＞＞/))) {
          if (lines[i].match(/^\s*$/)) {
            i++;
            continue;
          }
          if (lines[i].match(/^＝＝＝＝/)) {
            figMode = false;
          }
          if (figMode) {
            figName = findImageFile(lines[i], imageDir);
            if ((figName != '') && (figName != assumedFigName)) {
              html.push('<span class="filename">' + figName + '</span>');
            }
          }
          i++;
        }
        continue;
      }
      figMode = true;
      html.push('<span class="caption">' + lines[i] + '</span>');
      continue;
    }

    if (lines[i].match(/^＝＝＝＝/)) {
      i++;
      while(!lines[i].match(/^＞＞＞＞/)) {
        i++;
      }
      continue;
    }

    // figure mode
    if (figMode) {
      if (lines[i].match(/^\s*$/)) {
        continue;
      }
      figName = findImageFile(lines[i], imageDir);
      if (figName != '') {
        html.push('<img src="figures/' + figName + '">');
        html.push('<span class="filename">' + figName + '</span>');
      }
      continue;
    }

    // table 
    if (lines[i].match(/^☆表/)) {
      html.push('<span class="caption">' + lines[i] + '</span>');
      html.push('<table>');
      i++;
      while((i < lines.length) && (!lines[i].match(/^\s*$/))) {
        if (lines[i].match(/^＞＞＞＞/)) {
          break;
        }
        html.push(lines[i].replace(/^/, '<tr><td>')
                  .replace(/$/, '</td></tr>')
                  .replace(/\t/, '</td><td>'));
        i++;
      }
      html.push('</table>');
      continue;
    }

    // others
    html.push(lines[i]);
  }
  return html;
}

// parse main text to separate body text and captions/tables
function parseText(data, target) {
  var lines = data.split('\n');
  var targetDir = path.dirname(target);
  var imageDir = path.join(targetDir, 'figure');

  // split body and others
  // result.bodies:
  // result.blockItems: 
  var result = splitBody(lines);

  // generate captions
  result.captions = extractCaptions(result.blockItems, imageDir)

  // generate htmls
  result.htmls = makeHtmls(result.blockItems, imageDir);

  // generate Figure List
  result.figures = createFigureList(result.blockItems, imageDir);

  return result;
}

module.exports = exportSingle;

// main action
if (require.main == module) {
  if (process.argv.length < 4) {
    process.stdout.write(usage + '\n');
    process.exit(1);
  }
  var target = process.argv[2];
  var output = process.argv[3];
  exportSingle(target, output);
}
