/* toc-parser.js */

var parser = {};

/**
tocの例：
{ directories: {
  '.': { name: '.', contents: [], pageStart: 0 },
  chapter1: {
    name: 'chapter1',
    contents: [
  { title: 'トビラ1', pages: 1 },
  { title: 'バージョン管理とは', pages: 2 },
  { title: 'Gitとは', pages: 2 },
  { title: '章末コラム1：Git以外のバージョン管理システム', pages: 1 }
    ],
    pageStart: 11,
    chapter: 1
  },
...

**/


// Class: Directory
function Directory(name) {
  this.name = name;
  this.contents = [];
  this.pageStart = 0;
  return this;
}

parser.parseToc = function parseToc(text) {
  var texts = text.split('\n');
  var current = new Directory('.');
  var toc = {
    directories: {}
  };
  toc.forEachDirectories = function (f) {
    for (directory in toc.directories) {
      f(toc.directories[directory]);
    }
  };
  toc.directories['.'] = current;

  for (var i = 0; i < texts.length; i++) {
    var text = texts[i];

    // コメント/プラグマ処理
    if (text.charAt(0) == '#') {
      if (text.charAt(1) == '@') {
        // @プラグマの処理
        // directoryプラグマ
        var match = text.match(/^#@directory:\s*(.*?)\s*$/);
        if (match) {
          var directory = match[1];
          if ((directory.length > 1)
              && (directory.charAt(directory.length - 1) == '/')) {
            directory = directory.slice(0, -1);
          }
          if (toc.directories[directory] !== undefined) {
            current = toc.directories[directory];
          } else {
            // create new directory data
            current = new Directory(directory);
            toc.directories[directory] = current;
          }
        }

        //pageStartプラグマ
        var match = text.match(/^#@pageStart:\s*(\d+)\s*$/);
        if (match) {
          var pageStart = Number(match[1]);
          if (!isNaN(pageStart)) {
            current.pageStart = pageStart;
          }
        }

        //chapterNumberプラグマ
        var match = text.match(/^#@chapterNumber:\s*(\d+)\s*$/);
        if (match) {
          var chapter = Number(match[1]);
          if (!isNaN(chapter)) {
            current.chapter = chapter;
          }
        }
      }
      continue;
    } // コメント/プラグマ処理ここまで
    if (text.match(/^\s*$/)) {
      continue;
    }

    var parsed = parseLine(text);
    current.contents.push(parsed);
  }

  return toc;
};

function parseLine(text) {
  var texts = text.replace('\n', '').split('\t');
  var ret = {
    title: '',
    pages: 2
  };
  ret.title = texts[0];
  if (texts.length < 2) {
    return ret;
  }
  var n = Number(texts[1]);
  if ((n !== NaN) && (texts[1] !== '')) {
    ret.pages = n;
  }
  return ret;
}

module.exports = parser;

// main action
if (require.main == module) {

  // test code
  var data = '';
  process.stdin.on('readable', function () {
    var chunk = process.stdin.read();
    if (chunk !== null) {
      data = data + chunk;
    }
  });
  process.stdin.on('end', function() {
    var toc = parser.parseToc(data)
    console.log(toc);
    var directory = toc.directories['chapter1'];
    var pageStart = directory.pageStart;
    directory.contents.forEach(function (content) {
      console.log(content);
    });
  });
  
}
