/*********************************************************************
 *
 *  SMTP Client Demonstrations
 *	  - SMTP Client (short message)
 *    - SMTP Client (long message, not enabled)
 *
 *********************************************************************
 * FileName:        SMTPDemo.c
 * Dependencies:    TCP/IP stack
 * Processor:       PIC18, PIC24F, PIC24H, dsPIC30F, dsPIC33F, PIC32
 * Compiler:        Microchip C32 v1.05 or higher
 *					Microchip C30 v3.12 or higher
 *					Microchip C18 v3.30 or higher
 *					HI-TECH PICC-18 PRO 9.63PL2 or higher
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * Copyright (C) 2002-2009 Microchip Technology Inc.  All rights
 * reserved.
 *
 * Microchip licenses to you the right to use, modify, copy, and
 * distribute:
 * (i)  the Software when embedded on a Microchip microcontroller or
 *      digital signal controller product ("Device") which is
 *      integrated into Licensee's product; or
 * (ii) ONLY the Software driver source files ENC28J60.c, ENC28J60.h,
 *		ENCX24J600.c and ENCX24J600.h ported to a non-Microchip device
 *		used in conjunction with a Microchip ethernet controller for
 *		the sole purpose of interfacing with the ethernet controller.
 *
 * You should refer to the license agreement accompanying this
 * Software for additional information regarding your rights and
 * obligations.
 *
 * THE SOFTWARE AND DOCUMENTATION ARE PROVIDED "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT
 * LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * MICROCHIP BE LIABLE FOR ANY INCIDENTAL, SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF
 * PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR SERVICES, ANY CLAIMS
 * BY THIRD PARTIES (INCLUDING BUT NOT LIMITED TO ANY DEFENSE
 * THEREOF), ANY CLAIMS FOR INDEMNITY OR CONTRIBUTION, OR OTHER
 * SIMILAR COSTS, WHETHER ASSERTED ON THE BASIS OF CONTRACT, TORT
 * (INCLUDING NEGLIGENCE), BREACH OF WARRANTY, OR OTHERWISE.
 *
 *
 * Author               Date      Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * E. Wood     			4/26/08	  Moved from MainDemo.c
 ********************************************************************/
#define __SMTPDEMO_C

#include "TCPIPConfig.h"
#include "FreeRTOS.h"
#include "task.h"
#include "queue.h"
#include "lcd.h"
#include "node.h"

#if defined(STACK_USE_SMTP_CLIENT)

#include "TCPIP Stack/TCPIP.h"

extern	xQueueHandle xDstarLogQueue;
extern	xQueueHandle xDstarDprsEmailMsgQueue;
extern	xQueueHandle xDstarLogQueue;

extern	struct DPRSinfo	DPRS;
extern	BYTE	TimeDate[20];
extern	BOOL	DebugSW;

BYTE	EmailTemp[138];

/*****************************************************************************
  Function:
	void SMTPDemo(void)

  Summary:
	Demonstrates use of the e-mail (SMTP) client.
	
  Description:
	This function demonstrates the use of the SMTP client.  The function is
	called periodically by the stack, and checks if BUTTON2 and BUTTON3 are
	pressed simultaneously.  If they are, it attempts to send an e-mail 
	message using parameters hard coded in the function below.
	
	While the client is executing, LED1 will be used as a busy indicator.  
	LED2 will light when the transmission has been completed successfully.  
	If both LEDs extinguish, an error occurred.
	
	For an example of sending a longer message (one that does not exist in
	RAM all at once), see the commented secondary implementation of this
	function in this file (SMTPDemo.c) below.  For an example of sending
	a message using parameters gathered at run time, and/or a message with 
	attachments, see the implementation of HTTPPostEmail in CustomHTTPApp.c.

  Precondition:
	The SMTP client is initialized.

  Parameters:
	None

  Returns:
  	None
  ***************************************************************************/
void	DprsSMTP()
{
	static	BYTE	returnAddr[70];
	static	BYTE	RAMStringSubject[50];
	static	BYTE	text[120];

	static enum
	{
		MAIL_HOME = 0,
		MAIL_BEGIN,
		MAIL_SMTP_FINISHING,
	} MailState = MAIL_HOME;
 

	switch(MailState)
	{
		case MAIL_HOME:
			if (xQueueReceive (xDstarDprsEmailMsgQueue, EmailTemp, 0) != pdTRUE) break;

			MailState++;
			if (DebugSW)
			{
				sprintf ((char *)text, "%19.19s APRS Mail Sending Start\r\n",TimeDate);
				xQueueSend( xDstarLogQueue, &text, 0 );
			}

		case MAIL_BEGIN:
			if(SMTPBeginUsage())
			{
				sprintf ((char *)RAMStringSubject, "APRS Message from %9.9s to %9.9s", (char *)&EmailTemp, (char *)&EmailTemp[10]);
				SMTPClient.Server.szROM = (ROM BYTE*)DPRS.SmtpServer;	// SMTP server address
				SMTPClient.ROMPointers.Server = 1;
				if (DPRS.smtpAuth)
				{
					SMTPClient.ROMPointers.Username = 1;
					SMTPClient.ROMPointers.Password = 1;
					SMTPClient.Username.szROM = DPRS.User;
				    SMTPClient.Password.szROM = DPRS.Password;
				}
				else
				{
					SMTPClient.ROMPointers.Username = 0;
					SMTPClient.ROMPointers.Password = 0;
				}
				SMTPClient.To.szRAM = &EmailTemp[90];
				sprintf ((char *)returnAddr, "\"SAAP SMTP Client\" <%s>", &EmailTemp[90]);
				SMTPClient.From.szROM = (ROM BYTE*)returnAddr;
				SMTPClient.ROMPointers.From = 1;
				SMTPClient.Subject.szROM = RAMStringSubject;
				SMTPClient.ROMPointers.Subject = 1;
				SMTPClient.Body.szRAM = &EmailTemp[20];
				SMTPSendMail();
				MailState++;
				if (DebugSW)
				{
					sprintf ((char *)text, "%19.19s APRS Mail Sending  from %9.9s to %9.9s\r\n",TimeDate, (char *)&EmailTemp, (char *)&EmailTemp[10]);
					xQueueSend( xDstarLogQueue, &text, 0 );
				}
			}
			break;

		case MAIL_SMTP_FINISHING:
			if(!SMTPIsBusy())
			{
				// Finished sending mail
				SMTPEndUsage();
				MailState = MAIL_HOME;
				if (DebugSW)
				{
					sprintf ((char *)text, "%19.19s APRS Mail Sending Done.\r\n",TimeDate);
					xQueueSend( xDstarLogQueue, &text, 0 );
				}
			}
			break;
	}
}

#endif //#if defined(STACK_USE_SMTP_CLIENT)
