# mq-series-list-colour.sh
# ------------------------------------------------------------------------------
#
# Colour control API for MQ series and guards listings; abstracted from the
# original "mq-series-list.sh" support module.
#
# ------------------------------------------------------------------------------
#
# $Id$
#
# Written by Keith Marshall <keith@users.osdn.me>
# Copyright (C) 2018, 2019, Keith Marshall
#
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# ------------------------------------------------------------------------------
#
# Expand possible abbreviations for the colour control setting; expected
# expansions are "auto", "always", or "never", with one possible ambiguity,
# namely "a" could expand to either "auto" or "always".
#
  for option in auto always never
  do case $option in ${mq_colour-invalid}*)
	test ${mq_colour_option-x} = x ||
	  ${fatal-die} "ambiguous colour option '$mq_colour';"\
	    "could be 'auto', or 'always'"
	mq_colour_option=$option
     esac
  done

# Confirm that any given colour control option, whether specified in full,
# or expanded from an abbreviation, represents one of the valid settings.
#
  test "x$mq_colour_option" = x || mq_colour=$mq_colour_option
  test "x$mq_colour" = x || case $mq_colour in auto | always | never) ;;
    *) ${fatal-die} "unknown colour option '$mq_colour';"\
	 "expected 'auto', 'always', or 'never'"
    esac

# Promote "auto" colour mode (the default) to "always", when /dev/stdout
# is connected to a TTY, then enable SGR colour settings for the "always"
# mode, whether selected explicitly, or by such promotion...
#
  test "${mq_colour-auto}" = auto && test -t 1 && mq_colour=always
  if test "${mq_colour-never}" = always
  then
    sgr_normal="${sgr_prefix=`echo '^[[' | tr -s '^[' '\033['`}0m"

    sgr_applied_colour="${sgr_prefix}32m"
    sgr_unapplied_colour="${sgr_prefix}35m"
    sgr_guarded_colour="${sgr_prefix}33m"
    sgr_missing_colour="${sgr_prefix}31m"

    mq_colour='
      function mq_apply_colour( text, state_flag ) {
	switch( state_flag ? state_flag : state[text] )
	{ case "A": return "'"$sgr_applied_colour"'" text "'"$sgr_normal"'";
	  case "U": return "'"$sgr_unapplied_colour"'" text "'"$sgr_normal"'";
	  case "G": return "'"$sgr_guarded_colour"'" text "'"$sgr_normal"'";
	  case "D": return "'"$sgr_missing_colour"'" text "'"$sgr_normal"'";
	}
	return text;
      }'
    mq_error(){ echo "$sgr_missing_colour*** ERROR *** $@$sgr_normal"; }

# ...otherwise, colour mode "never" was explicitly specified, or "auto"
# mode promotion to "always" did not occur; disable coloured output, (by
# the simple expedient of implementing "mq_apply_colour" and "mq_error"
# to return their arguments, without inserting SGR control codes).
#
  else
    mq_colour='
      function mq_apply_colour( text, unused_arg ) { return text; }'
    mq_error(){ echo "*** ERROR *** $@"; }
  fi
#
# ------------------------------------------------------------------------------
# $RCSfile$: end of file
